/**
 * Summary: udi.h
 * contains the main header file for the uniform driver interface
 *
 * Authors:
 *     Marcel Sondaar
 *     Bogdan Barbu
 *
 * License:
 *     <Public Domain>
 */

#ifndef __UDI_H__
#define __UDI_H__

#ifndef UDI_VERSION
#error "UDI_VERSION not defined."
#elif UDI_VERSION != 0x101
#error "UDI_VERSION not supported."
#endif

// configurations for fixed-sized types
#ifndef _PDCLIB_INT_H
#define _PDCLIB_INT_H _PDCLIB_INT_H
#include <_PDCLIB_int.h>
#endif

// todo: move to correct section
typedef _PDCLIB_int8_t udi_sbit8_t;
typedef _PDCLIB_int16_t udi_sbit16_t;
typedef _PDCLIB_int32_t udi_sbit32_t;
typedef _PDCLIB_uint8_t udi_ubit8_t;
typedef _PDCLIB_uint16_t udi_ubit16_t;
typedef _PDCLIB_uint32_t udi_ubit32_t;
typedef _PDCLIB_size_t udi_size_t;
typedef _PDCLIB_uint32_t udi_index_t;
typedef _PDCLIB_size_t udi_boolean_t;

#define TRUE 1
#define FALSE 0

// this is a feature of this UDI implementation:
// if UDI_ANNOY_ME is set all ISO C integers are
// made undefined and will cause errors to
// enforce a high standard of portability
#if defined(UDI_ANNOY_ME)
  #define int    Oops YouUsedInt
  #define short  Oops YouUsedShort
  #define long   Oops YouUsedLong
  #define float  Oops YouUsedFloat
  #define double Oops YouUsedDouble
#elif defined(UDI_FORCE_NO_FLOAT)
  #define float  Oops YouUsedFloat
  #define double Oops YouUsedDouble
#endif

// ------------------------------------------
// UDI core Ch. 9 - Fundamental types
// ------------------------------------------ (partial)

// Todo: remainder of UDI Core 9.*

#define offsetof(type, member) ((_PDCLIB_size_t) &(((type *)0)->member))

// UDI core 1.9.2 9.2
#ifndef NULL
#define NULL ((void *)0)
#endif

// UDI core 1.9.6 9.10
typedef _PDCLIB_intptr_t udi_channel_t;
/* NULL channel handle constant */
#define UDI_NULL_CHANNEL 0

// UDI core 1.9.6 9.11
typedef _PDCLIB_intptr_t udi_buf_path_t;
/* NULL buffer path handle constant */
#define UDI_NULL_BUF_PATH 0

// UDI core 1.9.6 9.12
typedef _PDCLIB_intptr_t udi_origin_t;
/* NULL origin handle constant */
#define UDI_NULL_ORIGIN 0

// UDI core 1.9.9 9.16-20
typedef udi_ubit32_t udi_status_t;
/* Mask Values and Flags for udi_status_t */
#define UDI_STATUS_CODE_MASK 0x0000FFFF
#define UDI_STAT_META_SPECIFIC 0x00008000
#define UDI_SPECIFIC_STATUS_MASK 0x00007FFF
#define UDI_CORRELATE_OFFSET 16
#define UDI_CORRELATE_MASK 0xFFFF0000
/* Common Status Values */
#define UDI_OK 0
#define UDI_STAT_NOT_SUPPORTED 1
#define UDI_STAT_NOT_UNDERSTOOD 2
#define UDI_STAT_INVALID_STATE 3
#define UDI_STAT_MISTAKEN_IDENTITY 4
#define UDI_STAT_ABORTED 5
#define UDI_STAT_TIMEOUT 6
#define UDI_STAT_BUSY 7
#define UDI_STAT_RESOURCE_UNAVAIL 8
#define UDI_STAT_HW_PROBLEM 9
#define UDI_STAT_NOT_RESPONDING 10
#define UDI_STAT_DATA_UNDERRUN 11
#define UDI_STAT_DATA_OVERRUN 12
#define UDI_STAT_DATA_ERROR 13
#define UDI_STAT_PARENT_DRV_ERROR 14
#define UDI_STAT_CANNOT_BIND 15
#define UDI_STAT_CANNOT_BIND_EXCL 16
#define UDI_STAT_TOO_MANY_PARENTS 17
#define UDI_STAT_BAD_PARENT_TYPE 18
#define UDI_STAT_TERMINATED 19
#define UDI_STAT_ATTR_MISMATCH 20

// UDI core 1.9.9 9.22-9.26
typedef const udi_ubit8_t udi_layout_t;
/* Specific-Length Layout Type Codes */
#define UDI_DL_UBIT8_T 1
#define UDI_DL_SBIT8_T 2
#define UDI_DL_UBIT16_T 3
#define UDI_DL_SBIT16_T 4
#define UDI_DL_UBIT32_T 5
#define UDI_DL_SBIT32_T 6
#define UDI_DL_BOOLEAN_T 7
#define UDI_DL_STATUS_T 8
/* Abstract Element Layout Type Codes */
#define UDI_DL_INDEX_T 20
/* Opaque Handle Element Layout Type Codes */
#define UDI_DL_CHANNEL_T 30
#define UDI_DL_ORIGIN_T 32
/* Indirect Element Layout Type Codes */
#define UDI_DL_BUF 40
#define UDI_DL_CB 41
#define UDI_DL_INLINE_UNTYPED 42
#define UDI_DL_INLINE_DRIVER_TYPED 43
#define UDI_DL_MOVABLE_UNTYPED 44
/* Nested Element Layout Type Codes */
#define UDI_DL_INLINE_TYPED 50
#define UDI_DL_MOVABLE_TYPED 51
#define UDI_DL_ARRAY 52
#define UDI_DL_END 0

// -------------------------------------------
// UDI core Ch. 11 - Control Block Management
// ------------------------------------------- (partial)

// Todo: remainder of UDI Core 11.*

// UDI core 1.11.2 11.3-4
typedef struct {
    udi_channel_t   channel;
    void *          context;
    void *          scratch;
    void *          initiator_context;
    udi_origin_t    origin;
} udi_cb_t;

// UDI core 1.11.2 11.5-6

typedef void udi_cb_alloc_call_t(udi_cb_t * gcb, udi_cb_t * new_cb);

void udi_cb_alloc(udi_cb_alloc_call_t * callback, udi_cb_t * gcb,
                  udi_index_t cb_idx, udi_channel_t default_channel);

// UDI core 1.11.2 11.10
void udi_cb_free ( udi_cb_t *cb );

// UDI core 1.11.2 11.11
#define UDI_GCB(mcb) (&(mcb)->gcb)

// UDI core 1.11.2 11.12
#define UDI_MCB(gcb, cb_type) ((cb_type *)(gcb))

// ------------------------------------------------
// UDI core Ch. 12 - Memory Management
// ------------------------------------------------ (partial)

// Todo: rest of UDI Core 12.*

// UDI core 1.12.2 12.3-6

#define UDI_MEM_NOZERO (1U << 0)
#define UDI_MEM_MOVABLE (1U << 1)

typedef void udi_mem_alloc_call_t(udi_cb_t * gcb, void * new_mem);

void udi_mem_alloc(udi_mem_alloc_call_t * callback, udi_cb_t * gcb, udi_size_t size, udi_ubit8_t flags);
void udi_mem_free(void * target_mem);

// ------------------------------------------------
// UDI core Ch. 15 - Instance attribute management
// ------------------------------------------------ (partial)

// Todo: rest of UDI Core 15.*

// UDI core 1.15.5 15.7
typedef udi_ubit8_t udi_instance_attr_type_t;
/* Instance Attribute Types */
#define UDI_ATTR_NONE 0
#define UDI_ATTR_STRING 1
#define UDI_ATTR_ARRAY8 2
#define UDI_ATTR_UBIT32 3
#define UDI_ATTR_BOOLEAN 4
#define UDI_ATTR_FILE 5

// UDI core 1.15.5 15.13
#define UDI_MAX_ATTR_NAMELEN 32
#define UDI_MAX_ATTR_SIZE 64
typedef struct {
    char                        attr_name[UDI_MAX_ATTR_NAMELEN];
    udi_ubit8_t                 attr_value[UDI_MAX_ATTR_SIZE];
    udi_ubit8_t                 attr_length;
    udi_instance_attr_type_t    attr_type;
} udi_instance_attr_list_t;



// --------------------------------------
// UDI core Ch. 17 - Tracing and logging
// -------------------------------------- (partial)

// Todo: remainder of UDI Core 17.*

// UDI core 1.17.2 17.3-5
typedef udi_ubit32_t udi_trevent_t;
/* Common Trace Events */
#define UDI_TREVENT_LOCAL_PROC_ENTRY (1U<<0)
#define UDI_TREVENT_LOCAL_PROC_EXIT (1U<<1)
#define UDI_TREVENT_EXTERNAL_ERROR (1U<<2)
/* Common Metalanguage-Selectable Trace Events */
#define UDI_TREVENT_IO_SCHEDULED (1U<<6)
#define UDI_TREVENT_IO_COMPLETED (1U<<7)
/* Metalanguage-Specific Trace Events */
#define UDI_TREVENT_META_SPECIFIC_1 (1U<<11)
#define UDI_TREVENT_META_SPECIFIC_2 (1U<<12)
#define UDI_TREVENT_META_SPECIFIC_3 (1U<<13)
#define UDI_TREVENT_META_SPECIFIC_4 (1U<<14)
#define UDI_TREVENT_META_SPECIFIC_5 (1U<<15)
/* Driver-Specific Trace Events */
#define UDI_TREVENT_INTERNAL_1 (1U<<16)
#define UDI_TREVENT_INTERNAL_2 (1U<<17)
#define UDI_TREVENT_INTERNAL_3 (1U<<18)
#define UDI_TREVENT_INTERNAL_4 (1U<<19)
#define UDI_TREVENT_INTERNAL_5 (1U<<20)
#define UDI_TREVENT_INTERNAL_6 (1U<<21)
#define UDI_TREVENT_INTERNAL_7 (1U<<22)
#define UDI_TREVENT_INTERNAL_8 (1U<<23)
#define UDI_TREVENT_INTERNAL_9 (1U<<24)
#define UDI_TREVENT_INTERNAL_10 (1U<<25)
#define UDI_TREVENT_INTERNAL_11 (1U<<26)
#define UDI_TREVENT_INTERNAL_12 (1U<<27)
#define UDI_TREVENT_INTERNAL_13 (1U<<28)
#define UDI_TREVENT_INTERNAL_14 (1U<<29)
#define UDI_TREVENT_INTERNAL_15 (1U<<30)
/* Logging Event */
#define UDI_TREVENT_LOG (1U<<31)


// ------------------------------------------
// UDI core Ch. 24 - Management Metalanguage
// ------------------------------------------

// Todo: UDI Core 24.*

// UDI core 2.24.4 24.10-11
typedef struct {
    udi_cb_t                        gcb;
    udi_trevent_t                   trace_mask;
    udi_index_t                     meta_idx;
} udi_usage_cb_t;

// UDI core 2.24.5 24.16-17
typedef struct {
    char                            attr_name[UDI_MAX_ATTR_NAMELEN];
    udi_ubit8_t                     attr_min[UDI_MAX_ATTR_SIZE];
    udi_ubit8_t                     attr_min_len;
    udi_ubit8_t                     attr_max[UDI_MAX_ATTR_SIZE];
    udi_ubit8_t                     attr_max_len;
    udi_instance_attr_type_t        attr_type;
    udi_ubit32_t                    attr_stride;
} udi_filter_element_t;

// UDI core 2.24.5 24.18-20
typedef struct {
    udi_cb_t                        gcb;
    udi_ubit32_t                    child_ID;
    void *                          child_data;
    udi_instance_attr_list_t *      attr_list;
    udi_ubit8_t                     attr_valid_length;
    const udi_filter_element_t *    filter_list;
    udi_ubit8_t                     filter_list_length;
    udi_ubit8_t                     parent_ID;
} udi_enumerate_cb_t;
/* Special parent_ID filter values */
#define UDI_ANY_PARENT_ID 0

// UDI core 2.24.4 24.10-11
//void udi_usage_ind (udi_usage_cb_t *cb,
//                    udi_ubit8_t resource_level);
typedef void udi_usage_ind_op_t(udi_usage_cb_t *, udi_ubit8_t);
udi_usage_ind_op_t udi_static_usage;
/* Values for resource_level */
#define UDI_RESOURCES_CRITICAL 1
#define UDI_RESOURCES_LOW 2
#define UDI_RESOURCES_NORMAL 3
#define UDI_RESOURCES_PLENTIFUL 4

// UDI core 2.24.5 24.21-23
//void udi_enumerate_req (
//    udi_enumerate_cb_t *cb,
//    udi_ubit8_t enumeration_level );
typedef void udi_enumerate_req_op_t(udi_enumerate_cb_t *, udi_ubit8_t);
// Values for enumeration_level
#define UDI_ENUMERATE_START 1
#define UDI_ENUMERATE_START_RESCAN 2
#define UDI_ENUMERATE_NEXT 3
#define UDI_ENUMERATE_NEW 4
#define UDI_ENUMERATE_DIRECTED 5
#define UDI_ENUMERATE_RELEASE 6
// Wrapper function
udi_enumerate_req_op_t udi_enumerate_no_children;

// UDI core 2.24.5 24.24-26
void udi_enumerate_ack (udi_enumerate_cb_t *cb, udi_ubit8_t enumeration_result, udi_index_t ops_idx );
/* Values for enumeration_result */
#define UDI_ENUMERATE_OK 0
#define UDI_ENUMERATE_LEAF 1
#define UDI_ENUMERATE_DONE 2
#define UDI_ENUMERATE_RESCAN 3
#define UDI_ENUMERATE_REMOVED 4
#define UDI_ENUMERATE_REMOVED_SELF 5
#define UDI_ENUMERATE_RELEASED 6
#define UDI_ENUMERATE_FAILED 255

// UDI core 2.24.4 24.8
typedef struct {
    udi_cb_t gcb;
} udi_mgmt_cb_t;

// UDI core 2.24.6  24.30-31
//void udi_devmgmt_req (
//    udi_mgmt_cb_t *cb,
//    udi_ubit8_t mgmt_op,
//    udi_ubit8_t parent_ID );
typedef void udi_devmgmt_req_op_t (udi_mgmt_cb_t *, udi_ubit8_t, udi_ubit8_t);
/* Values for mgmt_op */
#define UDI_DMGMT_PREPARE_TO_SUSPEND 1
#define UDI_DMGMT_SUSPEND 2
#define UDI_DMGMT_SHUTDOWN 3
#define UDI_DMGMT_PARENT_SUSPENDED 4
#define UDI_DMGMT_RESUME 5
#define UDI_DMGMT_UNBIND 6

// UDI core 2.24.6  24.32-33
void udi_devmgmt_ack (udi_mgmt_cb_t *cb, udi_ubit8_t flags, udi_status_t status);
/* Values for flags */
#define UDI_DMGMT_NONTRANSPARENT (1U<<0)
/* Meta-Specific Status Codes */
#define UDI_DMGMT_STAT_ROUTING_CHANGE (UDI_STAT_META_SPECIFIC | 1)

// UDI core 2.24.6 24.34
//void udi_final_cleanup_req (
//    udi_mgmt_cb_t *cb);
typedef void udi_final_cleanup_req_op_t(udi_mgmt_cb_t *);

// UDI core 2.24.6 24.35
void udi_final_cleanup_ack(udi_mgmt_cb_t *cb );

// UDI core 2.24.4 24.7
typedef const struct {
    udi_usage_ind_op_t *            usage_ind_op;
    udi_enumerate_req_op_t *        enumerate_req_op;
    udi_devmgmt_req_op_t *          devmgmt_req_op;
    udi_final_cleanup_req_op_t *    final_cleanup_req_op;
} udi_mgmt_ops_t;


// ---------------------------------
// UDI core ch. 10 - Initialisation
// --------------------------------- (inclusion complete)

// UDI core 1.10.2 10.9-10
typedef void udi_op_t(void);
typedef udi_op_t * const udi_ops_vector_t;
typedef const struct {
    udi_index_t             ops_idx;
    udi_index_t             meta_idx;
    udi_index_t             meta_ops_num;
    udi_size_t              chan_context_size;
    udi_ops_vector_t *      ops_vector;
    const udi_ubit8_t *     op_flags;
} udi_ops_init_t;

// UDI core 1.10.2 10.11-13
typedef const struct {
    udi_index_t         cb_idx;
    udi_index_t         meta_idx;
    udi_index_t         meta_cb_num;
    udi_size_t          scratch_requirement;
    udi_size_t          inline_size;
    udi_layout_t *      inline_layout;
} udi_cb_init_t;

// UDI core 1.10.2.14
typedef const struct {
    udi_index_t         ops_idx;
    udi_index_t         cb_idx;
} udi_cb_select_t;

// UDI core 1.10.2.15
typedef const struct {
    udi_index_t         cb_idx;
    udi_size_t          scratch_requirement;
} udi_gcb_init_t;

// UDI core 1.10.2 10.5-6
typedef const struct {
    udi_mgmt_ops_t *    mgmt_ops;
    const udi_ubit8_t * mgmt_op_flags;
    udi_size_t          mgmt_scratch_requirement;
    udi_ubit8_t         enumeration_attr_list_length;
    udi_size_t          rdata_size;
    udi_size_t          child_data_size;
    udi_ubit8_t         per_parent_paths;
} udi_primary_init_t;

// UDI core 1.10.2 10.7-8
typedef const struct {
    udi_index_t         region_idx;
    udi_size_t          rdata_size;
} udi_secondary_init_t;

/* Maximum Legal Scratch Requirement */
#define UDI_MAX_SCRATCH 4000
/* Operation Flags */
#define UDI_OP_LONG_EXEC (1U<<0)

// UDI core 1.10.2 10.3-4
typedef const struct {
    udi_primary_init_t *    primary_init_info;
    udi_secondary_init_t *  secondary_init_list;
    udi_ops_init_t *        ops_init_list;
    udi_cb_init_t *         cb_init_list;
    udi_gcb_init_t *        gcb_init_list;
    udi_cb_select_t *       cb_select_list;
} udi_init_t;

extern udi_init_t udi_init_info;

// UDI core 1.10.3 10.18
typedef struct {
    udi_size_t      max_legal_alloc;
    udi_size_t      max_safe_alloc;
    udi_size_t      max_trace_log_formatted_len;
    udi_size_t      max_instance_attr_len;
    udi_ubit32_t    min_curtime_res;
    udi_ubit32_t    min_timer_res;
} udi_limits_t;
/* architectural minimums */
#define UDI_MIN_ALLOC_LIMIT 4000
#define UDI_MIN_TRACE_LOG_LIMIT 200
#define UDI_MIN_INSTANCE_ATTR_LIMIT 64

// UDI core 1.10.3 10.17
typedef struct {
    udi_index_t     region_idx;
    udi_limits_t    limits;
} udi_init_context_t;

// ---------------------------------
// UDI core ch. 13 -
// --------------------------------- (partial inclusion)

// Todo: remainder of UDI Core 13.*

// UDI core 1.13.2 13.3
typedef struct {
    udi_size_t buf_size;
} udi_buf_t;

// UDI core 1.13.3 13.5-6
typedef struct {
    udi_ubit32_t udi_xfer_max;
    udi_ubit32_t udi_xfer_typical;
    udi_ubit32_t udi_xfer_granularity;
    udi_boolean_t udi_xfer_one_piece;
    udi_boolean_t udi_xfer_exact_size;
    udi_boolean_t udi_xfer_no_reorder;
} udi_xfer_constraints_t;


typedef void udi_buf_write_call_t (udi_cb_t *gcb, udi_buf_t *new_dst_buf );
typedef void udi_buf_write_op_t (
    udi_buf_write_call_t *callback,
    udi_cb_t *gcb,
    const void *src_mem,
    udi_size_t src_len,
    udi_buf_t *dst_buf,
    udi_size_t dst_off,
    udi_size_t dst_len,
    udi_buf_path_t path_handle );
udi_buf_write_op_t udi_buf_write;

// UDI core 1.13.5 13.19
void udi_buf_read (udi_buf_t *src_buf, udi_size_t src_off, udi_size_t src_len, void *dst_mem );

// UDI core 1.13.5 13.20
void udi_buf_free ( udi_buf_t *buf );

// ---------------------------------
// UDI core ch. 16 - Initialisation
// --------------------------------- (inclusion complete)

// UDI core 1.16.2 16.2-3
typedef void udi_channel_anchor_call_t (udi_cb_t *gcb, udi_channel_t anchored_channel );
void udi_channel_anchor (udi_channel_anchor_call_t *callback, udi_cb_t *gcb, udi_channel_t channel, udi_index_t ops_idx, void *channel_context );
typedef void udi_channel_anchor_op_t (udi_channel_anchor_call_t *callback, udi_cb_t *gcb, udi_channel_t channel, udi_index_t ops_idx, void *channel_context );

// UDI core 1.16.2 16.4-5
typedef void udi_channel_spawn_call_t (udi_cb_t *gcb, udi_channel_t new_channel );
void udi_channel_spawn (udi_channel_spawn_call_t *callback, udi_cb_t *gcb, udi_channel_t channel, udi_index_t spawn_idx, udi_index_t ops_idx, void *channel_context );
typedef void udi_channel_spawn_op_t (udi_channel_spawn_call_t *callback, udi_cb_t *gcb, udi_channel_t channel, udi_index_t spawn_idx, udi_index_t ops_idx, void *channel_context );

// UDI core 1.16.2 16.6
void udi_channel_set_context (udi_channel_t target_channel, void *channel_context );

// UDI core 1.16.2 16.7
void udi_channel_op_abort (udi_channel_t target_channel, udi_cb_t *orig_cb );

// UDI core 1.16.2 16.8
void udi_channel_close ( udi_channel_t channel );

// UDI core 1.16.3 16.10-12
typedef struct {
    udi_cb_t gcb;
    udi_ubit8_t event;
    union {
        struct {
            udi_cb_t *bind_cb;
        } internal_bound;
        struct {
            udi_cb_t *bind_cb;
            udi_ubit8_t parent_ID;
            udi_buf_path_t *path_handles;
        } parent_bound;
        udi_cb_t *orig_cb;
    } params;
} udi_channel_event_cb_t;
/* Channel event types */
#define UDI_CHANNEL_CLOSED 0
#define UDI_CHANNEL_BOUND 1
#define UDI_CHANNEL_OP_ABORTED 2

// UDI core 1.16.3 16.13
void udi_channel_event_ind (udi_channel_event_cb_t *cb);
typedef void udi_channel_event_ind_op_t(udi_channel_event_cb_t *cb );

// UDI core 1.16.3 16.14
void udi_channel_event_complete (udi_channel_event_cb_t *cb, udi_status_t status );
typedef void udi_channel_event_complete_op_t (udi_channel_event_cb_t *cb, udi_status_t status );

// ---------------------------------
// UDI core ch. 25 -
// --------------------------------- (partial inclusion)

// Todo: remainder of UDI Core 25.*

// UDI core 2.25.5 25.11
typedef struct {
    udi_cb_t gcb;
    udi_xfer_constraints_t xfer_constraints;
} udi_gio_bind_cb_t;
/* Control Block Group Number */
#define UDI_GIO_BIND_CB_NUM 1

// UDI core 2.25.5 25.12
typedef void udi_gio_bind_req_op_t (udi_gio_bind_cb_t *cb );

// UDI core 2.25.5 25.13
typedef void udi_gio_bind_ack_op_t (udi_gio_bind_cb_t *cb, udi_ubit32_t device_size_lo, udi_ubit32_t device_size_hi, udi_status_t status );
udi_gio_bind_ack_op_t udi_gio_bind_ack;

// UDI core 2.25.5 25.14
typedef void udi_gio_unbind_req_op_t (udi_gio_bind_cb_t *cb );
udi_gio_unbind_req_op_t udi_gio_unbind_req;

// UDI core 2.25.5 25.15
typedef void udi_gio_unbind_ack_op_t (udi_gio_bind_cb_t *cb );
udi_gio_unbind_ack_op_t udi_gio_unbind_ack;

// UDI core 2.25.6 25.18
typedef udi_ubit8_t udi_gio_op_t;
#define UDI_GIO_OP_CUSTOM 16
#define UDI_GIO_OP_MAX 64
#define UDI_GIO_DIR_READ (1U<<6)
#define UDI_GIO_DIR_WRITE (1U<<7)
#define UDI_GIO_OP_READ UDI_GIO_DIR_READ
#define UDI_GIO_OP_WRITE UDI_GIO_DIR_WRITE

// UDI core 2.25.6 25.17
typedef struct {
    udi_cb_t gcb;
    udi_gio_op_t op;
    void *tr_params; // udi_gio_rw_params_t
    udi_buf_t *data_buf;
} udi_gio_xfer_cb_t;
/* Control Block Group Number */
#define UDI_GIO_XFER_CB_NUM 2

// UDI core 2.25.6 25.20
typedef struct {
    udi_ubit32_t offset_lo;
    udi_ubit32_t offset_hi;
} udi_gio_rw_params_t;

// UDI core 2.25.6 25.21
typedef void udi_gio_xfer_req_op_t (udi_gio_xfer_cb_t *cb );

// UDI core 2.25.6 25.22
void udi_gio_xfer_ack (udi_gio_xfer_cb_t *cb );

// UDI core 2.25.6 25.23
void udi_gio_xfer_nak (udi_gio_xfer_cb_t *cb, udi_status_t status );

// UDI core 2.25.7 25.25
typedef struct {
    udi_cb_t gcb;
    udi_ubit8_t event_code;
    void *event_params;
} udi_gio_event_cb_t;
/* Control Block Group Number */
#define UDI_GIO_EVENT_CB_NUM 3

// UDI core 2.25.7 25.27
typedef void udi_gio_event_res_op_t (udi_gio_event_cb_t *cb );
udi_gio_event_res_op_t udi_gio_event_res_unused;

// UDI core 2.25.4 25.8
typedef const struct {
    udi_channel_event_ind_op_t *channel_event_ind_op;
    udi_gio_bind_req_op_t *gio_bind_req_op;
    udi_gio_unbind_req_op_t *gio_unbind_req_op;
    udi_gio_xfer_req_op_t *gio_xfer_req_op;
    udi_gio_event_res_op_t *gio_event_res_op;
} udi_gio_provider_ops_t;
/* Ops Vector Number */
#define UDI_GIO_PROVIDER_OPS_NUM 1

// Todo: UDI Core 8.*
// Todo: UDI Core 12.*
// Todo: UDI Core 14.*
// Todo: UDI Core 18.*
// UDI core 1.18.2 18.3
void udi_assert ( udi_boolean_t expr );

// Todo: UDI Core 19.*
// Todo: UDI Core 20.*

// ---------------------------------
// UDI core ch. 21 - Queue Management Utility Functions
// --------------------------------- (inclusion complete)

// UDI core 2.21.2.1 21.3

typedef struct udi_queue {
    struct udi_queue *next;
    struct udi_queue *prev;
} udi_queue_t;

// UDI core 2.21.2 21.5-6, 21.8-14

void udi_enqueue(udi_queue_t *new_el, udi_queue_t *old_el);
udi_queue_t * udi_dequeue(udi_queue_t *element);

#define UDI_QUEUE_INIT(listhead) ((listhead)->next = (listhead)-> prev = (listhead))
#define UDI_QUEUE_EMPTY(listhead) ((listhead)->next = (listhead))

#define UDI_ENQUEUE_HEAD(listhead, element) udi_enqueue(element, listhead)
#define UDI_ENQUEUE_TAIL(listhead, element) udi_enqueue(element, (listhead)->prev)
#define UDI_QUEUE_INSERT_AFTER(old_el, new_el) uni_enqueue(new_el, old_el)
#define UDI_QUEUE_INSERT_BEFORE(old_el, new_el) udi_enqueue(new_el, (old_el)-prev)

#define UDI_DEQUEUE_HEAD(listhead) udi_dequeue((listhead)->next)
#define UDI_DEQUEUE_TAIL(listhead) udi_dequeue((listhead)->prev)
#define UDI_QUEUE_REMOVE(element) ((void)udi_dequeue(element))

#define UDI_FIRST_ELEMENT(listhead) ((listhead)->next)
#define UDI_LAST_ELEMENT(listhead) ((listhead)-prev)
#define UDI_NEXT_ELEMENT(element) ((element)->next)
#define UDI_PREV_ELEMENT(element) ((element)->prev)

#define UDI_QUEUE_FOREACH(listhead, element, tmp) \
    for ((element) = (UDI_FIRST_ELEMENT(listhead); \
        ((tmp) = UDI_NEXT_ELEMENT(element)), \
        ((element) != (listhead)); \
        (element) = (tmp)))

#define UDI_BASE_STRUCT(memberp, struct_type, memeber_name) \
    ((struct_type *)((char *)(memberp) - offsetof(struct_type, member_name)))

// ---------------------------------
// UDI core ch. 22 - Endianness Management Utility Functions
// --------------------------------- (partial inclusion)

// Todo: remainder of UDI Core 22.*

// UDI core 2.22.2.2.1 22.5-6

#define UDI_BFMASK(p, len) (((1U << (len)) - 1) << (p))
#define UDI_BFGET(val, p, len) (((udi_ubit8_t)(val) >> (p)) & ((1U << (len)) - 1))
#define UDI_BFSET(val, p, len, dst) ((dst) = \
    ((dst) & ~UDI_BFMASK(p, len)) | (((udi_ubit8_t)(val) << (p)) & UDI_BFMASK(p, len)))

// Todo: UDI core 2.22.2.2 22.8-10

// UDI core 2.22.2.3 22.12-14

#define UDI_ENDIAN_SWAP_16(data16) ((((data16) & 0x00ff) << 8) | (((data16) >> 8) & 0x00ff))
#define UDI_ENDIAN_SWAP_32(data32) ((((data32) & 0x000000ff) << 24) | (((data32) & 0x0000ff00) << 8) | \
                                    (((data32) >> 8) & 0x0000ff00) | (((data32) >> 24) & 0x000000ff))

void udi_endian_swap(const void * src, void * dst, udi_ubit8_t swap_size, udi_ubit8_t stride, udi_ubit8_t rep_count);

#define UDI_ENDIAN_SWAP_ARRAY(src, element_size, count) \
    udi_endian_swap(src, src, element_size, element_size, count)

// Todo: UDI Core 24.*
// Todo: UDI Core 26.*
// Todo: UDI Core 27.*
// Todo: UDI Core 28.*
// Todo: UDI Core 29.*
// Todo: UDI Core 30.*
// Todo: UDI Core 31.*
// Todo: UDI Core 32.*
// Todo: UDI Core 33.*

#endif
