/**
* Summary: udi_block.h
* Contains the block device metalanguage interface details
*
* Author:
*     Marcel Sondaar
*
* License:
*     <Public Domain>
*/

#include <udi.h>

#ifndef UDI_BLOCK_VERSION
#error "UDI_BLOCK_VERSION not defined."
#elif UDI_BLOCK_VERSION != 0x101
#error "UDI_BLOCK_VERSION not supported."
#endif

// Type: udi_block_op_t
// an enumeration for reading or writing operations
typedef udi_ubit8_t udi_block_op_t;
#define UDI_BLOCK_DIR_READ (1U<<6)
#define UDI_BLOCK_DIR_WRITE (1U<<7)
#define UDI_BLOCK_OP_READ UDI_BLOCK_DIR_READ
#define UDI_BLOCK_OP_WRITE UDI_BLOCK_DIR_WRITE


// Structure: udi_block_bind_cb_t
// Contains the operations of a read/write transaction
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;    
} udi_block_bind_cb_t;

// Function: udi_block_bind_req
// function pointer prototype for connecting to a block device
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_block_bind_req_op_t (udi_block_bind_cb_t *cb );
udi_block_bind_req_op_t udi_block_bind_req;

// Function: udi_block_bind_ack
// function pointer prototype for acknowledging a connection request
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_block_bind_ack_op_t (udi_block_bind_cb_t *cb, udi_ubit32_t block_count_lo, udi_ubit32_t block_count_hi, udi_ubit32_t block_size, udi_status_t status );
udi_block_bind_ack_op_t udi_block_bind_ack;

// Function: udi_block_unbind_req
// function pointer prototype for disconnecting a block device
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_block_unbind_req_op_t (udi_block_bind_cb_t *cb );
udi_block_unbind_req_op_t udi_block_unbind_req;

// Function: udi_block_unbind_ack
// function pointer prototype for connecting to a block device
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_block_unbind_ack_op_t (udi_block_bind_cb_t *cb );
udi_block_unbind_ack_op_t udi_block_unbind_ack;


// Structure: udi_block_xfer_cb_t
// Contains the operations of a read/write transaction
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;
    // Variable: op
    // The operation to perform
    udi_block_op_t op;
    // Variable: start
    // the starting block
    udi_size_t start;
    // Variable: count
    // the number of blocks to read
    udi_size_t count;
    // Variable: data_buf
    // The buffer to pass data in
    udi_buf_t *data_buf;    
} udi_block_xfer_cb_t;

// Structure: udi_blockmap_xfer_cb_t
// Contains the operations of a read/write transaction on a mapped block device
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;
    // Variable: op
    // The operation to perform
    udi_block_op_t op;
    // Variable: start
    // the starting block
    udi_size_t start;
    // Variable: count
    // the number of blocks to read
    udi_size_t count;
    // Variable: map
    // the mapping to use
    udi_size_t map;
    // Variable: data_buf
    // The buffer to pass data in    
    udi_buf_t *data_buf;    
} udi_blockmap_xfer_cb_t;

// Function: udi_block_xfer_req
// function pointer prototype for the block transfer request
// 
// in:
//     cb - A pointer to a <udi_block_xfer_cb_t> with the arguments of the call
//
typedef void udi_block_xfer_req_op_t (udi_block_xfer_cb_t *cb );
udi_block_xfer_req_op_t udi_block_xfer_req;

// Function: udi_block_xfer_ack
// function pointer prototype for the block transfer success reply
// 
// in:
//     cb - A pointer to a <udi_block_xfer_cb_t> with the arguments of the call
//
typedef void udi_block_xfer_ack_op_t (udi_block_xfer_cb_t *cb );
udi_block_xfer_ack_op_t udi_block_xfer_ack;

// Function: udi_block_xfer_nak
// function pointer prototype for the block transfer fail reply
// 
// in:
//     cb - A pointer to a <udi_block_xfer_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_block_xfer_nak_op_t (udi_block_xfer_cb_t *cb, udi_status_t status );
udi_block_xfer_nak_op_t udi_block_xfer_nak;

// Function: udi_blockmap_xfer_req
// function pointer prototype for the block transfer request
// 
// in:
//     cb - A pointer to a <udi_blockmap_xfer_cb_t> with the arguments of the call
//
typedef void udi_blockmap_xfer_req_op_t (udi_blockmap_xfer_cb_t *cb );
udi_blockmap_xfer_req_op_t udi_blockmap_xfer_req;

// Function: udi_blockmap_xfer_ack
// function pointer prototype for the block transfer success reply
// 
// in:
//     cb - A pointer to a <udi_blockmap_xfer_cb_t> with the arguments of the call
//
typedef void udi_blockmap_xfer_ack_op_t (udi_blockmap_xfer_cb_t *cb );
udi_blockmap_xfer_ack_op_t udi_blockmap_xfer_ack;

// Function: udi_blockmap_xfer_nak
// function pointer prototype for the block transfer fail reply
// 
// in:
//     cb - A pointer to a <udi_blockmap_xfer_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_xfer_nak_op_t (udi_blockmap_xfer_cb_t *cb, udi_status_t status );
udi_blockmap_xfer_nak_op_t udi_blockmap_xfer_nak;

// Structure: udi_blockmap_map_cb_t
// Contains the requests of a mapping modification call
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;
    
    //40: <command> <request no> <(4) source map> <(8) source start> <(4) dest map> <(8) dest start> <(8) size>
        
    udi_ubit32_t source_map;
    udi_ubit32_t dest_map;
    udi_ubit32_t source_off_lo;
    udi_ubit32_t source_off_hi;
    udi_ubit32_t dest_off_lo;
    udi_ubit32_t dest_off_hi;
    udi_ubit32_t map_size_lo;
    udi_ubit32_t map_size_hi;
    udi_index_t map_privileges;
            
} udi_blockmap_map_cb_t;

// Function: udi_blockmap_map_req
// function pointer prototype for the block transfer fail reply
// 
// in:
//     cb - A pointer to a <udi_blockmap_map_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_map_req_op_t (udi_blockmap_map_cb_t *cb);
udi_blockmap_map_req_op_t udi_blockmap_map_req;

// Function: udi_blockmap_map_ack
// function pointer prototype for the block transfer fail reply
// 
// in:
//     cb - A pointer to a <udi_blockmap_map_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_map_ack_op_t (udi_blockmap_map_cb_t *cb);
udi_blockmap_map_ack_op_t udi_blockmap_map_ack;

// Function: udi_blockmap_map_ack
// function pointer prototype for the block transfer fail reply
// 
// in:
//     cb - A pointer to a <udi_blockmap_map_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_map_nak_op_t (udi_blockmap_map_cb_t *cb, udi_status_t status );
udi_blockmap_map_nak_op_t udi_blockmap_map_nak;

// Structure: udi_blockmap_mgmt_cb_t
// Contains the requests of a mapping management calls
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;    
    // Variable: map
    // the map index in question
    udi_ubit32_t map;
    udi_ubit32_t map_size_lo;
    udi_ubit32_t map_size_hi;    
    
} udi_blockmap_mgmt_cb_t;

// Function: udi_blockmap_new_req
// Create a new mapping
// 
// in:
//     cb - A pointer to a <udi_blockmap_mgmt_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_new_req_op_t (udi_blockmap_mgmt_cb_t *cb);
udi_blockmap_new_req_op_t udi_blockmap_new_req;

// Function: udi_blockmap_new_ack
// Confirm a new mapping
// 
// in:
//     cb - A pointer to a <udi_blockmap_mgmt_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_new_ack_op_t (udi_blockmap_mgmt_cb_t *cb);
udi_blockmap_new_ack_op_t udi_blockmap_new_ack;

// Function: udi_blockmap_del_req
// Create a new mapping
// 
// in:
//     cb - A pointer to a <udi_blockmap_mgmt_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_del_req_op_t (udi_blockmap_mgmt_cb_t *cb);
udi_blockmap_del_req_op_t udi_blockmap_del_req;

// Function: udi_blockmap_del_ack
// Confirm a new mapping
// 
// in:
//     cb - A pointer to a <udi_blockmap_mgmt_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_del_ack_op_t (udi_blockmap_mgmt_cb_t *cb);
udi_blockmap_del_ack_op_t udi_blockmap_del_ack;

// Structure: udi_blockmap_map_cb_t
// Contains the requests of a map information demand call
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;
        
    udi_ubit32_t map;
    udi_ubit32_t off_lo;
    udi_ubit32_t off_hi;
    udi_ubit32_t map_size_lo;
    udi_ubit32_t map_size_hi;        
} udi_blockmap_demand_cb_t;

// Function: udi_blockmap_demand_req
// function pointer prototype for mapping driver reqesting a fill-in
// 
// in:
//     cb - A pointer to a <udi_blockmap_demand_cb_t> with the arguments of the call
typedef void udi_blockmap_map_demand_op_t (udi_blockmap_demand_cb_t *cb);
udi_blockmap_map_demand_op_t udi_blockmap_demand_req;

// Function: udi_blockmap_demand_ack
// function pointer prototype for the fill-in complete message
// 
// in:
//     cb - A pointer to a <udi_blockmap_demand_cb_t> with the arguments of the call
//     status - the error message
typedef void udi_blockmap_demand_ack_op_t (udi_blockmap_demand_cb_t *cb);
udi_blockmap_demand_ack_op_t udi_blockmap_demand_ack;

/* Structure: udi_block_provider_ops_t
   The block metalanguage entry points (provider side)
 */
typedef const struct {
    udi_channel_event_ind_op_t  *channel_event_ind_op;
    udi_block_bind_req_op_t     *block_bind_req_op;
    udi_block_unbind_req_op_t   *block_unbind_req_op;
    udi_block_xfer_req_op_t     *block_xfer_req_op;
} udi_block_provider_ops_t;


/* Structure: udi_block_ops_t
   The block metalanguage entry points (client side)
*/
typedef const struct {
    udi_channel_event_ind_op_t  *channel_event_ind_op;
    udi_block_bind_ack_op_t     *block_bind_ack_op;
    udi_block_unbind_ack_op_t   *block_unbind_ack_op;
    udi_block_xfer_ack_op_t     *block_xfer_ack_op;
    udi_block_xfer_nak_op_t     *block_xfer_nak_op;
} udi_block_ops_t;

/* Structure: udi_blockmap_provider_ops_t
   The mapped block metalanguage entry points (provider side)
*/
typedef const struct {
    udi_channel_event_ind_op_t      *channel_event_ind_op;
    udi_block_bind_req_op_t         *block_bind_req_op;
    udi_block_unbind_req_op_t       *block_unbind_req_op;
    udi_blockmap_xfer_req_op_t      *blockmap_xfer_req_op;
    udi_blockmap_map_req_op_t       *blockmap_map_req_op;
    udi_blockmap_new_req_op_t       *blockmap_new_req_op;
    udi_blockmap_del_req_op_t       *blockmap_del_req_op;
} udi_blockmap_provider_ops_t;

/* Structure: udi_blockmap_ops_t
   The mapped block device metalanguage entry points (client side)
*/
typedef const struct {
    udi_channel_event_ind_op_t      *channel_event_ind_op;
    udi_block_bind_ack_op_t         *block_bind_ack_op;
    udi_block_unbind_ack_op_t       *block_unbind_ack_op;
    udi_blockmap_xfer_ack_op_t      *block_xfer_ack_op;
    udi_blockmap_xfer_nak_op_t      *block_xfer_nak_op;
    udi_blockmap_map_ack_op_t       *blockmap_map_ack_op;
    udi_blockmap_map_nak_op_t       *blockmap_map_nak_op;
    udi_blockmap_new_ack_op_t       *blockmap_new_ack_op;
    udi_blockmap_del_ack_op_t       *blockmap_del_ack_op;
} udi_blockmap_ops_t;

/* Ops Vector Number */
#define UDI_BLOCK_PROVIDER_OPS_NUM 4
#define UDI_BLOCK_CLIENT_OPS_NUM 5
#define UDI_BLOCKMAP_PROVIDER_OPS_NUM 6
#define UDI_BLOCKMAP_CLIENT_OPS_NUM 7
