/**
 * Summary: udi_env.h
 * contains environment-specific UDI definitions
 *
 * Author:
 *     Bogdan Barbu
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#ifndef __UDI_ENV_H__
#define __UDI_ENV_H__

#define UDI_VERSION 0x101

#include <udi.h>

#define MOS_CB_CALLBACK      (1 << 0)
#define	MOS_CB_ALLOC_PENDING (1 << 1)

typedef void (*mos_callback_t)(void);

typedef struct {
    udi_index_t         meta_idx;
    udi_index_t         group;
    udi_index_t         cb_idx;
    udi_size_t          size;
} mos_mei_cb_mold_t;

// Structure: mos_mei_cb_mold_t
// contains index to control-block-info mappings for a given region
typedef struct {
    mos_mei_cb_mold_t  *cb_mold[256];
} mos_driver_t;

// Structure: mos_region_t
// contains the backend properties of a region
typedef struct {
    mos_driver_t       *driver;
    void               *contextdata;
} mos_region_t;

typedef enum {
    ALLOC_STATE_IDLE,
    ALLOC_STATE_DONE,
    ALLOC_STATE_FAILED,
    ALLOC_STATE_CANCEL,
    ALLOC_STATE_PENDING
} mos_alloc_state_t;

typedef struct {
    mos_alloc_state_t   state;
} mos_alloc_marshal_t;

// Structure: mos_buf_impl_t
// Generic part of an actual storage buffer 
typedef struct mos_buf_impl
{
    // Variable: refcount
    // Holds the amount of references to this data. Once the refcount reaches zero, the buffer is unused and should be freed
    udi_index_t refcount;
    // Variable: free_op
    // a function pointer to the buffer deconstructor
    //
    // input:
    //     buf - the pointer to this structure
    //
    // output:
    //     none
    //
    void (*free_op)(struct mos_buf_impl * buf);
    // Variable: copy_op
    // a function pointer to a generic data copying function
    //
    // input:
    //     buf - the pointer to this structure
    //     dst - a pointer to a memory location where a copy should be made
    //     off - the offset into the buffer
    //     len - the amount of bytes to copy
    //
    // output:
    //     none
    //
    void (*copy_op)(struct mos_buf_impl * buf, void * dst, udi_size_t off, udi_size_t len);
    // Variable: lock_op
    // a function pointer that gives direct read-only access to the data, allowing to defer or avoid a copy operation
    // This function may be NULL when direct exposure is not available
    //
    // input:
    //     buf - the pointer to this structure
    //     off - the offset into the buffer
    //     len - the length of the buffer
    //
    // output:
    //     return - a pointer to immutable data storage starting at off, or NULL on failure
    //
    const void * (*lock_op)(struct mos_buf_impl * buf, udi_size_t off, udi_size_t len);
    // Variable: unlock_op
    // a function pointer to the unlock function
    // this function may be NULL, but only if <copy_op> is NULL as well
    //
    // input:
    //     buf - the pointer to this structure
    //     pointer - the previously acquired pointer
    //
    // output:
    //     none. The passed pointer may no longer be used.
    //
    void (*unlock_op)(struct mos_buf_impl * buf, const void * pointer);
} mos_buf_impl_t;

typedef struct mos_malloc_buf_impl
{
    mos_buf_impl_t buf;
    udi_ubit8_t * data;
} mos_malloc_buf_impl_t;

udi_buf_t *	_udi_buf_alloc(void);
void            _udi_null_buf_free_op(mos_buf_impl_t * buf);
void            _udi_generic_buf_copy_op(mos_buf_impl_t * buf, void * dst, udi_size_t off, udi_size_t len);
void            _udi_generic_buf_free_op(mos_buf_impl_t * buf);
void            _udi_malloc_buf_free_op(mos_buf_impl_t * buf);
void            _udi_malloc_buf_copy_op(mos_buf_impl_t * buf, void * dst, udi_size_t off, udi_size_t len);
const void *    _udi_malloc_buf_lock_op(mos_buf_impl_t * buf, udi_size_t off, udi_size_t len);
void            _udi_malloc_buf_unlock_op(mos_buf_impl_t * buf, const void * pointer);


// Structure: mos_buf_part_t
// Describes a segment of data
typedef struct mos_buf_part
{
    // Variable: data
    // contains a reference to the actual data storage in an <mos_buf_impl_t>
    mos_buf_impl_t * data;
    // Variable: next
    // Contains a reference to the next buffer part in the sequence
    struct mos_buf_part * next;
    // Variable: prev
    // Contains a reference to the previous buffer part in the sequence
    struct mos_buf_part * prev;
    // Variable: size
    // amount of bytes referenced by this buffer part
    udi_size_t size;
    // Variable: offset
    // offset into the actual data that stores the data
    udi_size_t offset;
} mos_buf_part_t;

// Structure: mos_buf_tag_t
// A linked list of tags
typedef struct mos_buf_tag
{
    // Variable: tag
    // contains the tag info
    udi_buf_tag_t tag;
    // Variable: next
    // points to the next tag, if any
    struct mos_buf_tag * next;
} mos_buf_tag_t;

// Structure: mos_buf_t
// Contains the hidden portion of a buffer object. It is located directly in front of the udi_buf_t structure
// Buffers are composed of segments of immutable data, stored in a linked-list fashion.
// This gives O(1) access to the most expected operation (concatenations), but suffers on repeated random mutations.
// However, buffer calls should be minimal compared to the actual amount of data stored due to general overhead.
typedef struct mos_buf
{    
    // Variable: first
    // Contains a pointer to the first buffer part in the list, which is a <mos_buf_part_t>.
    mos_buf_part_t * first;
    // Variable: last
    // Contains a pointer to the last buffer part in the list, which is a <mos_buf_part_t>.
    mos_buf_part_t * last;
    // Variable: tags
    // Contains a pointer to the first tag in the list of associated tags as <mos_buf_tag_t>
    mos_buf_tag_t * tags;
    // Variable: allocated
    // Contains the total size of the buffer, and therefore indirectly the offset of <last>
    // allowing for reverse searches
    udi_size_t allocated;
} mos_buf_t;

// Function: _udi_buf_inject
// Helper function for modifying buffer segments
//
// in:
//     buf  - the internal buffer object to modify
//     part - the segment to inject
//     off  - the offset into the buffer where this segment belongs
//     len  - the amount of bytes to overwrite (0 to just insert)
//
// out:
//     none
//
void _udi_buf_inject(mos_buf_t * buf, mos_buf_part_t * part, udi_size_t off, udi_size_t len);

udi_buf_t * _udi_buf_enter(void * pointer, udi_size_t length);
void _udi_buf_exit(udi_buf_t * buffer, void * destination);

// Structure: mos_pio_impl_t
// Contains the functions to perform PIO instructions
typedef struct mos_pio_impl
{    
    void (*write_8)  (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit8_t data);
    void (*write_16) (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit16_t data);
    void (*write_32) (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit32_t data);
    udi_ubit8_t  (*read_8)  (struct mos_pio_impl * impl, udi_size_t offset);
    udi_ubit16_t (*read_16) (struct mos_pio_impl * impl, udi_size_t offset);
    udi_ubit32_t (*read_32) (struct mos_pio_impl * impl, udi_size_t offset);
    
    // Variable: limit
    // The size of this array (for safety checking)
    udi_size_t limit;
} mos_pio_impl_t;

// Structure: mos_mmio_pio_impl_t
// Contains the functions to perform PIO instructions
typedef struct mos_mmio_pio_impl
{
    // Variable: pio_impl
    // Pointer to the internal implementation functions    
    mos_pio_impl_t pio_impl;
    
    // Variable: base
    // The starting virtual address where this area is mapped
    void * base;        
} mos_mmio_pio_impl_t;

typedef struct mos_port_pio_impl
{
    mos_pio_impl_t pio_impl;
    udi_ubit16_t base;    
} mos_port_pio_impl_t;

mos_pio_impl_t * _udi_create_pio_mmio(void * base, udi_size_t length);
mos_pio_impl_t * _udi_create_pio_port(udi_ubit16_t base, udi_size_t length);

// Structure: mos_pio_handle_list
// List of available PIO regsets
typedef struct mos_pio_handle_list
{
    udi_ubit32_t regset_idx;
    mos_pio_impl_t * handlers;
    struct mos_pio_handle_list * next;
} mos_pio_handle_list_t;

extern mos_pio_handle_list_t * mos_pio_handle_list_head;

void _udi_tail_pio_handle(udi_ubit32_t regset_idx, mos_pio_impl_t * impl);
void _udi_init_pci(udi_ubit32_t devid);

// Structure: mos_system_cb_t
// contains the info the message handling backends needs to convert between system and UDI conventions.
typedef struct mos_system_cb
{
    // Variable: origin
    // contains the sender address of a message
    udi_ubit32_t        origin;
    // Variable: requestindex
    // contains the request number for pipelined requests
    udi_ubit32_t        requestindex;
} mos_system_cb_t;


// Structure: mos_cb_t
// contains the private part of a UDI control block - it can be found directly in front of the udi_cb_t structure
typedef struct mos_cb {
    udi_index_t         op_idx;
    mos_region_t       *region;
    mos_callback_t      func;
    mos_alloc_marshal_t allocm;
    mos_system_cb_t     system;
    udi_ubit8_t         flags;    
    struct mos_cb      *Old;
    struct mos_cb      *New;
} mos_cb_t;

// Structure mos_system_channel_t
// Contains the platform-dependent part of channel handles;
typedef struct mos_system_channel
{
    udi_ubit32_t * fn_offset;
    udi_ubit32_t destination_address;
} mos_system_channel_t;

// Structure: mos_channel_t
// contains the hidden part of the channel type. 
typedef struct mos_channel {
    udi_mei_ops_vec_template_t *channel_funcs;
    mos_system_channel_t        system;
} mos_channel_t;

void mos_alloc_queue(mos_cb_t * cb);

//int _udi_setregioncount(int count);
//void * _udi_getregiondata(int index);
//void _udi_cleanregiondata(void);
mos_region_t * _mos_region_enter(void);                         // new region
void _mos_region_exit(mos_region_t * region);                   // cleanup region

udi_cb_t * _udi_cb_enter(mos_region_t * region, int cbsize);    // new CB
udi_cb_t * _udi_cb_unpark(udi_ubit32_t ticket);                 // continue with existing CB
udi_ubit32_t _udi_cb_park(udi_cb_t * cb);                       // suspend CB
void _udi_cb_exit(udi_cb_t * cb);                               // done with CB
mos_system_cb_t * _udi_cb_system(udi_cb_t * cb);                // access system part of CB

typedef struct udi_hidden_meta_init_t
{    
    udi_mei_init_t * metalanguage;
    udi_ubit32_t * native_function_base;
    udi_ubit32_t * native_proto_indices;
    udi_ubit32_t bound_index;
} udi_hidden_meta_init_t;
typedef void udi_meta_initproc_t(int region, int meta);

extern udi_hidden_meta_init_t * udi_props_meta_list;

// Structure: udi_pio_local_trans_t
// local storage of opcodes
typedef struct udi_pio_local_trans_t {
    udi_ubit8_t pio_op;
    udi_ubit8_t tran_size;
    udi_ubit16_t operand;
} udi_pio_local_trans_t;

typedef struct udi_pio_trans_internal_t
{
    udi_pio_local_trans_t * opcodes;
    udi_size_t opcodecount;
    udi_size_t regoffset;
    udi_size_t regsize;
    int index;
    mos_pio_impl_t * ioaccess;
    udi_size_t labels[8];
} udi_pio_trans_internal_t;

#endif
