/**
* Summary: udi_fs.h
* Contains the filesystem driver interface details
*
* Author:
*     Marcel Sondaar
*
* License:
*     <Public Domain>
* 
* 
* The filesystem metalanguage is split in several separate metalanguages, 
* each covering a portion of potential features the filesystem might provide
* 
*/
#ifndef UDI_FS_H
#define UDI_FS_H

#include <udi.h>

#ifndef UDI_FS_VERSION
#error "UDI_FS_VERSION not defined."
#elif UDI_FS_VERSION != 0x101
#error "UDI_FS_VERSION not supported."
#endif

#include "fs_provider_gen.h"
#include "fs_client_gen.h"

/* Group: udi_fs_blockread
 * 
 * This metalanguage supports opening files by name and returning blocklists of 
 * the file's locations on disk.
 * 
 */

// Structure: udi_fs_blockread_bind_cb_t
// Contains a mostly empty control block to initialize the read functionality
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;    
} udi_fs_blockread_bind_cb_t;

// Function: udi_fs_blockread_bind_req
// function pointer prototype for connecting to a block device
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_bind_cb_t>
//
typedef void udi_fs_blockread_bind_req_op_t (udi_fs_blockread_bind_cb_t *cb );
udi_fs_blockread_bind_req_op_t udi_fs_blockread_bind_req;

// Function: udi_fs_blockread_bind_ack
// function pointer prototype for acknowledging a connection request
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_bind_cb_t>
//     status - A status value, usually to indicate if a filesystem was correctly mounted.
//
typedef void udi_fs_blockread_bind_ack_op_t (udi_fs_blockread_bind_cb_t *cb, udi_status_t status);
udi_fs_blockread_bind_ack_op_t udi_fs_blockread_bind_ack;

// Function: udi_fs_blockread_unbind_req
// function pointer prototype for disconnecting a filesystem driver
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_fs_blockread_unbind_req_op_t (udi_fs_blockread_bind_cb_t *cb );
udi_fs_blockread_unbind_req_op_t udi_fs_blockread_unbind_req;

// Function: udi_fs_blockread_unbind_ack
// function pointer prototype for disconnecting to a filesystem driver
// 
// in:
//     cb - A pointer to a <udi_block_bind_cb_t>
//
typedef void udi_fs_blockread_unbind_ack_op_t (udi_fs_blockread_bind_cb_t *cb );
udi_fs_blockread_unbind_ack_op_t udi_fs_blockread_unbind_ack;

// Structure: udi_fs_blockread_filename_cb_t
// Contains a cb used for filename transfers
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;   
    // Variable: buf
    // buffer holding the filename
    udi_buf_t * buf;
} udi_fs_blockread_filename_cb_t;

// Function: udi_fs_blockread_stat_req
// function pointer prototype for requesting file presence and size
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//
typedef void udi_fs_blockread_stat_req_op_t (udi_fs_blockread_filename_cb_t *cb);
udi_fs_blockread_stat_req_op_t udi_fs_blockread_stat_req;

// Function: udi_fs_blockread_stat_ack
// function pointer prototype for requesting file presence and size
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//     size_lo - bottom 32 bits of filesize
//     size_hi - top 32 bits of filesize
//
typedef void udi_fs_blockread_stat_ack_op_t (udi_fs_blockread_filename_cb_t *cb, udi_ubit32_t size_lo, udi_ubit32_t size_hi);
udi_fs_blockread_stat_ack_op_t udi_fs_blockread_stat_ack;

// Function: udi_fs_blockread_stat_nak
// function pointer prototype for returning a read error
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//     status - cause of the failure
//
typedef void udi_fs_blockread_stat_nak_op_t (udi_fs_blockread_filename_cb_t *cb, udi_status_t status);
udi_fs_blockread_stat_nak_op_t udi_fs_blockread_stat_nak;

// Function: udi_fs_blockread_open_req
// function pointer prototype for opening a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//
typedef void udi_fs_blockread_open_req_op_t (udi_fs_blockread_filename_cb_t *cb);
udi_fs_blockread_open_req_op_t udi_fs_blockread_open_req;

// Function: udi_fs_blockread_open_ack
// function pointer prototype for returning success on opening a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//     handle - a reference for the file for later use
//
typedef void udi_fs_blockread_open_ack_op_t (udi_fs_blockread_filename_cb_t *cb, udi_index_t handle);
udi_fs_blockread_open_ack_op_t udi_fs_blockread_open_ack;

// Function: udi_fs_blockread_open_nak
// function pointer prototype for file opening failure
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//     handle - a reference for the file for later use
//
typedef void udi_fs_blockread_open_nak_op_t (udi_fs_blockread_filename_cb_t *cb, udi_status_t handle);
udi_fs_blockread_open_nak_op_t udi_fs_blockread_open_nak;

// Structure: udi_fs_blockread_close_cb_t
// Contains a cb used for file closing
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;   
    // Variable: handle
    // the file handle
    udi_index_t handle;
} udi_fs_blockread_close_cb_t;

// Function: udi_fs_blockread_close_req
// function pointer prototype for closing a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//
typedef void udi_fs_blockread_close_req_op_t (udi_fs_blockread_close_cb_t *cb);
udi_fs_blockread_close_req_op_t udi_fs_blockread_close_req;

// Function: udi_fs_blockread_close_ack
// function pointer prototype for closing a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_filename_cb_t>
//
typedef void udi_fs_blockread_close_ack_op_t (udi_fs_blockread_close_cb_t *cb);
udi_fs_blockread_close_ack_op_t udi_fs_blockread_close_ack;

// Structure: udi_fs_blockread_map_cb_t
// Contains a cb used for file closing
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;   
    // Variable: handle
    // the file handle
    udi_index_t handle;
    // Variable: buf
    // the buffer to place mappings in
    udi_buf_t * buf;
    // Variable: off_lo
    // offset into the file to map
    udi_ubit32_t off_lo;
    udi_ubit32_t off_hi;
    // Variable: len_lo
    // length of the data to map
    udi_ubit32_t len_lo;
    udi_ubit32_t len_hi;
} udi_fs_blockread_map_cb_t;

// Function: udi_fs_blockread_map_req
// function pointer prototype for closing a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_map_cb_t>
//
typedef void udi_fs_blockread_map_req_op_t (udi_fs_blockread_map_cb_t *cb);
udi_fs_blockread_map_req_op_t udi_fs_blockread_map_req;

// Function: udi_fs_blockread_map_ack
// function pointer prototype for closing a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_map_cb_t>
//
typedef void udi_fs_blockread_map_ack_op_t (udi_fs_blockread_map_cb_t *cb);
udi_fs_blockread_map_ack_op_t udi_fs_blockread_map_ack;

// Function: udi_fs_blockread_map_nak
// function pointer prototype for closing a file
// 
// in:
//     cb - A pointer to a <udi_fs_blockread_map_cb_t>
//     status - cause of the failure
//
typedef void udi_fs_blockread_map_nak_op_t (udi_fs_blockread_map_cb_t *cb, udi_status_t status);
udi_fs_blockread_map_nak_op_t udi_fs_blockread_map_nak;

/* Structure: udi_fs_blockread_provider_ops_t
   The mapped filesystem mapped read entry points (provider side)
*/
typedef const struct {
    udi_channel_event_ind_op_t          *channel_event_ind_op;
    udi_fs_blockread_bind_req_op_t      *fs_blockread_bind_req_op;
    udi_fs_blockread_unbind_req_op_t    *fs_blockread_unbind_req_op;
    udi_fs_blockread_stat_req_op_t      *fs_blockread_stat_req_op;
    udi_fs_blockread_open_req_op_t      *fs_blockread_open_req_op;
    udi_fs_blockread_close_req_op_t     *fs_blockread_close_req_op;
    udi_fs_blockread_map_req_op_t       *fs_blockread_map_req_op;
} udi_fs_blockread_provider_ops_t;

/* Structure: udi_fs_blockread_ops_t
   The mapped filesystem mapped read metalanguage entry points (client side)
*/
typedef const struct {
    udi_channel_event_ind_op_t          *channel_event_ind_op;
    udi_fs_blockread_bind_ack_op_t      *fs_blockread_bind_ack_op;
    udi_fs_blockread_unbind_ack_op_t    *fs_blockread_unbind_ack_op;
    udi_fs_blockread_stat_ack_op_t      *fs_blockread_stat_ack_op;
    udi_fs_blockread_stat_nak_op_t      *fs_blockread_stat_nak_op;
    udi_fs_blockread_open_ack_op_t      *fs_blockread_open_ack_op;
    udi_fs_blockread_open_nak_op_t      *fs_blockread_open_nak_op;
    udi_fs_blockread_close_ack_op_t     *fs_blockread_close_ack_op;
    udi_fs_blockread_map_ack_op_t       *fs_blockread_map_ack_op;
    udi_fs_blockread_map_nak_op_t       *fs_blockread_map_nak_op;
} udi_fs_blockread_ops_t;





/* Group: udi_fs_blockrw
 * 
 * This metalanguage supports adding, modifying and reading files by name, returing 
 * blocklists corresponding to the locations on disk.
 * 
 */

// Structure: udi_fs_blockrw_bind_cb_t
// Contains a mostly empty control block to initialize the read-write functionality
typedef struct {
    // Variable: gcb
    // The main control block
    udi_cb_t gcb;    
} udi_fs_blockrw_bind_cb_t;

#define UDI_FS_MOUNT_READ (1U<<0)
#define UDI_FS_MOUNT_WRITE (1U<<1)
#define UDI_FS_MOUNT_CLEAN (1U<<2)

// Function: udi_fs_blockrw_bind_req
// function pointer prototype for connecting to a block device
// 
// in:
//     cb - A pointer to a <udi_fs_blockrw_bind_cb_t>
//
typedef void udi_fs_blockrw_bind_req_op_t (udi_fs_blockrw_bind_cb_t *cb );
udi_fs_blockrw_bind_req_op_t udi_fs_blockrw_bind_req;

// Function: udi_fs_blockrw_bind_ack
// function pointer prototype for acknowledging a connection request
// 
// in:
//     cb - A pointer to a <udi_fs_blockrw_bind_cb_t>
//     access - A series of UDI_FS_MOUNT_* flags indicating the operations possible.
//     status - A status value, usually to indicate if a filesystem was correctly mounted.
//
typedef void udi_fs_blockrw_bind_ack_op_t (udi_fs_blockrw_bind_cb_t *cb, udi_ubit32_t access, udi_status_t status);
udi_fs_blockrw_bind_ack_op_t udi_fs_blockrw_bind_ack;

#endif
