/**
 * Summary: udi_physio.h
 * contains the uniform driver interface header governing physical I/O
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#ifndef __UDI_PHYSIO_H__
#define __UDI_PHYSIO_H__

#ifndef UDI_PHYSIO_VERSION
#error "UDI_PHYSIO_VERSION not defined."
#elif UDI_PHYSIO_VERSION != 0x101
#error "UDI_PHYSIO_VERSION not supported."
#endif

// Todo: remainder of UDI physical IO Ch. 3

// UDI physical IO 3.3 p3.14
typedef _PDCLIB_uintptr_t udi_dma_constraints_t;
/* NULL dma constraints constant */
#define UDI_NULL_DMA_CONSTRAINTS 0

// UDI physical IO 3.3 p3.15-23
typedef udi_ubit8_t udi_dma_constraints_attr_t;
/* DMA Convenience Attribute Codes */
#define UDI_DMA_ADDRESSABLE_BITS 100
#define UDI_DMA_ALIGNMENT_BITS 101
/* DMA Constraints on the Entire Transfer */
#define UDI_DMA_DATA_ADDRESSABLE_BITS 110
#define UDI_DMA_NO_PARTIAL 111
/* DMA Constraints on the Scatter/Gather List */
#define UDI_DMA_SCGTH_MAX_ELEMENTS 120
#define UDI_DMA_SCGTH_FORMAT 121
#define UDI_DMA_SCGTH_ENDIANNESS 122
#define UDI_DMA_SCGTH_ADDRESSABLE_BITS 123
#define UDI_DMA_SCGTH_MAX_SEGMENTS 124
/* DMA Constraints on Scatter/Gather Segments */
#define UDI_DMA_SCGTH_ALIGNMENT_BITS 130
#define UDI_DMA_SCGTH_MAX_EL_PER_SEG 131
#define UDI_DMA_SCGTH_PREFIX_BYTES 132
/* DMA Constraints on Scatter/Gather Elements */
#define UDI_DMA_ELEMENT_ALIGNMENT_BITS 140
#define UDI_DMA_ELEMENT_LENGTH_BITS 141
#define UDI_DMA_ELEMENT_GRANULARITY_BITS 142
/* DMA Constraints for Special Addressing */
#define UDI_DMA_ADDR_FIXED_BITS 150
#define UDI_DMA_ADDR_FIXED_TYPE 151
#define UDI_DMA_ADDR_FIXED_VALUE_LO 152
#define UDI_DMA_ADDR_FIXED_VALUE_HI 153
/* DMA Constraints on DMA Access Behavior */
#define UDI_DMA_SEQUENTIAL 160
#define UDI_DMA_SLOP_IN_BITS 161
#define UDI_DMA_SLOP_OUT_BITS 162
#define UDI_DMA_SLOP_OUT_EXTRA 163
#define UDI_DMA_SLOP_BARRIER_BITS 164
/* Values for UDI_DMA_SCGTH_ENDIANNESS */
#define UDI_DMA_LITTLE_ENDIAN (1U<<6)
#define UDI_DMA_BIG_ENDIAN (1U<<5)
/* Values for UDI_DMA_ADDR_FIXED_TYPE */
#define UDI_DMA_FIXED_ELEMENT 1
#define UDI_DMA_FIXED_LIST 2
#define UDI_DMA_FIXED_VALUE 3

// ---------------------------------------------------------
// UDI core ch. 2 - DMA Constraints Management
// --------------------------------- (inclusion complete) --

// UDI physical IO 2.2 p2.3
typedef struct {
    udi_dma_constraints_attr_t attr_type;
    udi_ubit32_t attr_value;
} udi_dma_constraints_attr_spec_t;

// UDI physical IO 2.2 p2.4-6
typedef void udi_dma_constraints_attr_set_call_t (udi_cb_t *gcb, udi_dma_constraints_t new_constraints, udi_status_t status );
typedef void udi_dma_constraints_attr_set_op_t (
    udi_dma_constraints_attr_set_call_t *callback,
    udi_cb_t *gcb,
    udi_dma_constraints_t src_constraints,
    const udi_dma_constraints_attr_spec_t *attr_list,
    udi_ubit16_t list_length,
    udi_ubit8_t flags );
udi_dma_constraints_attr_set_op_t udi_dma_constraints_attr_set;
/* Constraints Flags */
#define UDI_DMA_CONSTRAINTS_COPY (1U<<0)

// UDI physical IO 2.2 p2.7
void udi_dma_constraints_attr_reset (udi_dma_constraints_t constraints, udi_dma_constraints_attr_t attr_type );

// UDI physical IO 2.2 p2.8
void udi_dma_constraints_free (udi_dma_constraints_t constraints );



// Todo: remainder of UDI physical IO Ch. 4



// UDI physical IO 4.2 p4.4
typedef _PDCLIB_intptr_t udi_pio_handle_t;
/* Null handle value for udi_pio_handle_t */
#define UDI_NULL_PIO_HANDLE 0

// UDI physical IO 4.3 p4.15-25
typedef const struct {
    udi_ubit8_t pio_op;
    udi_ubit8_t tran_size;
    udi_ubit16_t operand;
} udi_pio_trans_t;
/* Values for tran_size */
#define UDI_PIO_1BYTE 0
#define UDI_PIO_2BYTE 1
#define UDI_PIO_4BYTE 2
#define UDI_PIO_8BYTE 3
#define UDI_PIO_16BYTE 4
#define UDI_PIO_32BYTE 5
/* Values for register numbers in pio_op */
#define UDI_PIO_R0 0
#define UDI_PIO_R1 1
#define UDI_PIO_R2 2
#define UDI_PIO_R3 3
#define UDI_PIO_R4 4
#define UDI_PIO_R5 5
#define UDI_PIO_R6 6
#define UDI_PIO_R7 7
/* Values for addressing modes in pio_op */
#define UDI_PIO_DIRECT 0x00
#define UDI_PIO_SCRATCH 0x08
#define UDI_PIO_BUF 0x10
#define UDI_PIO_MEM 0x18
/* Values for Class A opcodes in pio_op */
#define UDI_PIO_IN 0x00
#define UDI_PIO_OUT 0x20
#define UDI_PIO_LOAD 0x40
#define UDI_PIO_STORE 0x60
/* Values for Class B opcodes in pio_op */
#define UDI_PIO_LOAD_IMM 0x80
#define UDI_PIO_CSKIP 0x88
#define  UDI_PIO_Z 0
#define  UDI_PIO_NZ 1
#define  UDI_PIO_NEG 2
#define  UDI_PIO_NNEG 3
#define UDI_PIO_IN_IND 0x90
#define UDI_PIO_OUT_IND 0x98
#define UDI_PIO_SHIFT_LEFT 0xA0
#define UDI_PIO_SHIFT_RIGHT 0xA8
#define UDI_PIO_AND 0xB0
#define UDI_PIO_AND_IMM 0xB8
#define UDI_PIO_OR 0xC0
#define UDI_PIO_OR_IMM 0xC8
#define UDI_PIO_XOR 0xD0
#define UDI_PIO_ADD 0xD8
#define UDI_PIO_ADD_IMM 0xE0
#define UDI_PIO_SUB 0xE8
/* Values for Class C opcodes in pio_op */
#define UDI_PIO_BRANCH 0xF0
#define UDI_PIO_LABEL 0xF1
#define UDI_PIO_REP_IN_IND 0xF2
#define UDI_PIO_REP_OUT_IND 0xF3
#define UDI_PIO_DELAY 0xF4
#define UDI_PIO_BARRIER 0xF5
#define UDI_PIO_SYNC 0xF6
#define UDI_PIO_SYNC_OUT 0xF7
#define UDI_PIO_DEBUG 0xF8
#define UDI_PIO_END 0xFE
#define UDI_PIO_END_IMM 0xFF
/* Values for UDI_PIO_DEBUG operand */
#define UDI_PIO_TRACE_OPS_NONE 0
#define UDI_PIO_TRACE_OPS1 1
#define UDI_PIO_TRACE_OPS2 2
#define UDI_PIO_TRACE_OPS3 3
#define UDI_PIO_TRACE_REGS_NONE (0U<<2)
#define UDI_PIO_TRACE_REGS1 (1U<<2)
#define UDI_PIO_TRACE_REGS2 (2U<<2)
#define UDI_PIO_TRACE_REGS3 (3U<<2)
#define UDI_PIO_TRACE_DEV_NONE (0U<<4)
#define UDI_PIO_TRACE_DEV1 (1U<<4)
#define UDI_PIO_TRACE_DEV2 (2U<<4)
#define UDI_PIO_TRACE_DEV3 (3U<<4)

// UDI physical IO 4.3 p4.28-29
typedef void udi_pio_trans_call_t (
    udi_cb_t *gcb,
    udi_buf_t *new_buf,
    udi_status_t status,
    udi_ubit16_t result );
typedef void udi_pio_trans_op_t (
    udi_pio_trans_call_t *callback,
    udi_cb_t *gcb,
    udi_pio_handle_t pio_handle,
    udi_index_t start_label,
    udi_buf_t *buf,
    void *mem_ptr );
udi_pio_trans_op_t udi_pio_trans;

// UDI physical IO 4.2 p4.5-8
typedef void udi_pio_map_call_t (udi_cb_t *gcb, udi_pio_handle_t new_pio_handle );
void udi_pio_map (
    udi_pio_map_call_t *callback,
    udi_cb_t *gcb,
    udi_ubit32_t regset_idx,
    udi_ubit32_t base_offset,
    udi_ubit32_t length,
    udi_pio_trans_t *trans_list,
    udi_ubit16_t list_length,
    udi_ubit16_t pio_attributes,
    udi_ubit32_t pace,
    udi_index_t serialization_domain );
/* Values for pio_attributes */
#define UDI_PIO_STRICTORDER (1U<<0)
#define UDI_PIO_UNORDERED_OK (1U<<1)
#define UDI_PIO_MERGING_OK (1U<<2)
#define UDI_PIO_LOADCACHING_OK (1U<<3)
#define UDI_PIO_STORECACHING_OK (1U<<4)
#define UDI_PIO_BIG_ENDIAN (1U<<5)
#define UDI_PIO_LITTLE_ENDIAN (1U<<6)
#define UDI_PIO_NEVERSWAP (1U<<7)
#define UDI_PIO_UNALIGNED (1U<<8)

// UDI physical IO 4.2 p4.9
void udi_pio_unmap (udi_pio_handle_t pio_handle );



// Todo: remainder of UDI physical IO Ch. 5

// UDI physical IO 5.2 p5.6
typedef struct {
    udi_cb_t gcb;
} udi_bus_bind_cb_t;
#define UDI_BUS_BIND_CB_NUM 1

// UDI physical IO 5.2 p5.7
void udi_bus_bind_req (udi_bus_bind_cb_t *cb );
typedef void udi_bus_bind_req_op_t (udi_bus_bind_cb_t *cb );

// UDI physical IO 5.2 p5.8-9
typedef void udi_bus_bind_ack_op_t (
    udi_bus_bind_cb_t *cb,
    udi_dma_constraints_t dma_constraints,
    udi_ubit8_t preferred_endianness,
    udi_status_t status );
/* Values for preferred_endianness */
#define UDI_DMA_BIG_ENDIAN (1U<<5)
#define UDI_DMA_LITTLE_ENDIAN (1U<<6)
#define UDI_DMA_ANY_ENDIAN (1U<<0)

// UDI physical IO 5.2 p5.10
typedef void udi_bus_unbind_req_op_t (udi_bus_bind_cb_t *cb );
udi_bus_unbind_req_op_t udi_bus_unbind_req;

// UDI physical IO 5.2 p5.11
typedef void udi_bus_unbind_ack_op_t (udi_bus_bind_cb_t *cb );
udi_bus_unbind_ack_op_t udi_bus_unbind_ack;

// UDI physical IO 5.3 p5.13-14
typedef struct {
    udi_cb_t gcb;
    udi_index_t interrupt_idx;
    udi_ubit8_t min_event_pend;
    udi_pio_handle_t preprocessing_handle;
} udi_intr_attach_cb_t;
/* Bridge Attach Control Block Group Number */
#define UDI_BUS_INTR_ATTACH_CB_NUM 2

// UDI physical IO 5.3 p5.15-18
typedef void udi_intr_attach_req_op_t (udi_intr_attach_cb_t *intr_attach_cb );

// UDI physical IO 5.3 p5.19-20
typedef void udi_intr_attach_ack_op_t (udi_intr_attach_cb_t *intr_attach_cb, udi_status_t status);
udi_intr_attach_ack_op_t udi_intr_attach_ack_unused;

// UDI physical IO 5.3 p5.21
typedef struct {
    udi_cb_t gcb;
    udi_index_t interrupt_idx;
} udi_intr_detach_cb_t;
/* Bridge Detach Control Block Group Number */
#define UDI_BUS_INTR_DETACH_CB_NUM 3

// UDI physical IO 5.3 p5.22
typedef void udi_intr_detach_req_op_t (udi_intr_detach_cb_t *intr_detach_cb );

// UDI physical IO 5.3 p5.23
typedef void udi_intr_detach_ack_op_t (udi_intr_detach_cb_t *intr_detach_cb );
udi_intr_detach_ack_op_t udi_intr_detach_ack_unused;

// UDI physical IO 5.2 p5.4
typedef const struct {
    udi_channel_event_ind_op_t *channel_event_ind_op;
    udi_bus_bind_ack_op_t *bus_bind_ack_op;
    udi_bus_unbind_ack_op_t *bus_unbind_ack_op;
    udi_intr_attach_ack_op_t *intr_attach_ack_op;
    udi_intr_detach_ack_op_t *intr_detach_ack_op;
} udi_bus_device_ops_t;
/* Bus Device Ops Vector Number */
#define UDI_BUS_DEVICE_OPS_NUM 1

#endif
