/*
    Summary: math.h
    *The libc mathematics header*

    Author:
        Marcel Sondaar

    License:
        Public Domain
        
    Defines:
        <cos> <cosf> <cosl> <ceil> <ceilf> <ceill> <floor> <floorf> 
        <floorl> <sin> <sinf> <sinl> <tan> <tanf> <tanl> <trunc> 
        <truncf> 

    Linker parameter:
        -lc

 */

/*  7.12 Mathematics <math.h>
    Added to PDCLib
 */

typedef float float_t;
typedef double double_t;

#define HUGE_VAL 1.7976931348623157e+308
#define HUGE_VALF 3.40282347e+38F
#define HUGE_VALL 1.18973149535723176508575932662800702e+4932L

#define NAN (0.0F/0.0F)
#define INFINITY (1.0F/0.0F)

#define FP_NAN		    0x0100
#define FP_NORMAL       0x0400
#define FP_INFINITE	    (FP_NAN | FP_NORMAL)
#define FP_ZERO	        0x4000
#define FP_SUBNORMAL	(FP_NORMAL | FP_ZERO)

// todo: math.h section 17.12 point 8

#define MATH_ERRNO 1
#define MATH_ERREXCEPT 2

// we set errno on range (overflow) or domain errors
#define math_errhandling (MATH_ERRNO)

int _PDCLIB_fpclassifyl(long double x);
int _PDCLIB_fpclassifyd(double x);
int _PDCLIB_fpclassifyf(float x);

#define fpclassify(x) (sizeof (x) == sizeof (float) ? _PDCLIB_fpclassifyf (x)	  \
		       : sizeof (x) == sizeof (double) ? _PDCLIB_fpclassify (x) \
		       : _PDCLIB_fpclassifyl (x))

#define isfinite(x) ((fpclassify(x) & FP_NAN) == 0)

#define isinf(x) (fpclassify(x) == FP_INFINITE)

int _PDCLIB_isnanl(long double x);
int _PDCLIB_isnan(double x);
int _PDCLIB_isnanf(float x);

#define isnan(x) (sizeof (x) == sizeof (float) ? __isnanf (x)	\
		  : sizeof (x) == sizeof (double) ? __isnan (x)	\
		  : __isnanl (x))

#define isnormal(x) (fpclassify(x) == FP_NORMAL)

int _PDCLIB_signbitl(long double x);
int _PDCLIB_signbit(double x);
int _PDCLIB_signbitf(float x);

#define signbit(x) (sizeof (x) == sizeof (float) ? __signbitf (x)	\
		    : sizeof (x) == sizeof (double) ? __signbit (x)	\
		    : __signbitl (x))


//todo: write trigoniometry functions

float acosf(float x);
double acos(double x);
long double acosl(long double x);

float asinf(float x);
double asin(double x);
long double asinl(long double x);

float atanf(float x);
double atan(double x);
long double atanl(long double x);

float atan2f(float y, float x);
double atan2(double y, double x);
long double atan2l(long double y, long double x);

float cosf(float x);
double cos(double x);
long double cosl(long double x);

float sinf(float x);
double sin(double x);
long double sinl(long double x);

float tanf(float x);
double tan(double x);
long double tanl(long double x);


//todo: write hyperbolic functions

float acosfh(float x);
double acosh(double x);
long double acoslh(long double x);

float asinhf(float x);
double asinh(double x);
long double asinhl(long double x);

float atanhf(float x);
double atanh(double x);
long double atanhl(long double x);

float coshf(float x);
double cosh(double x);
long double coshl(long double x);

float sinhf(float x);
double sinh(double x);
long double sinhl(long double x);

float tanhf(float x);
double tanh(double x);
long double tanhl(long double x);


// todo: write exponent functions

float expf(float x);
double exp(double x);
long double expl(long double x);

float exp2f(float x);
double exp2(double x);
long double exp2l(long double x);

float expm1f(float x);
double expm1(double x);
long double expm1l(long double x);

float frexpf(float value, int *e);
double frexp(double value, int *e);
long double frexpl(long double value, int *e);

int ilogbf(float x);
int ilogb(double x);
int ilogbl(long double x);

float ldexpf(float x, int e);
double ldexp(double x, int e);
long double ldexpl(long double x, int e);

float logf(float x);
double log(double x);
long double logl(long double x);

float log10f(float x);
double log10(double x);
long double log10l(long double x);

float log1pf(float x);
double log1p(double x);
long double log1pl(long double x);

float log2f(float x);
double log2(double x);
long double log2l(long double x);

float logbf(float x);
double logb(double x);
long double logbl(long double x);

float modff(float value, float *iptr);
double modf(double value, double *iptr);
long double modfl(long double value, long double *iptr);

float scalbnf(float x, int n);
double scalbn(double x, int n);
long double scalbnl(long double x, int n);

float scalblnf(float x, long int n);
double scalbln(double x, long int n);
long double scalblnl(long double x, long int n);


// todo: power and abs functions

float cbrtf(float x);
double cbrt(double x);
long double cbrtl(long double x);

float fabsf(float x);
double fabs(double x);
long double fabsl(long double x);

float hypotf(float x, float y);
double hypot(double x, double y);
long double hypotl(long double x, long double y);

float powf(float x, float y);
double pow(double x, double y);
long double powl(long double x, long double y);

float sqrtf(float x);
double sqrt(double x);
long double sqrtl(long double x);


// todo: error and gamma functions

float erff(float x);
double erf(double x);
long double erfl(long double x);

float erfcf(float x);
double erfc(double x);
long double erfcl(long double x);

float lgammaf(float x);
double lgamma(double x);
long double lgammal(long double x);

float tgammaf(float x);
double tgamma(double x);
long double tgammal(long double x);


// todo: nearest int functions

float ceilf(float x);
double ceil(double x);
long double ceill(long double x);

float floorf(float x);
double floor(double x);
long double floorl(long double x);

float nearbyintf(float x);
double nearbyint(double x);
long double nearbyintl(long double x);

float rintf(float x);
double rint(double x);
long double rintl(long double x);

long int lrintf(float x);
long int lrint(double x);
long int lrintl(long double x);

long long int llrintf(float x);
long long int llrint(double x);
long long int llrintl(long double x);

float roundf(float x);
double round(double x);
long double roundl(long double x);

long int lroundf(float x);
long int lround(double x);
long int lroundl(long double x);

long long int llroundf(float x);
long long int llround(double x);
long long int llroundl(long double x);

float truncf(float x);
double trunc(double x);
long double truncl(long double x);


// todo: remainder functions

float fmodf(float x, float y);
double fmod(double x, double y);
long double fmodl(long double x, long double y);

float remainderf(float x, float y);
double remainder(double x, double y);
long double remainderl(long double x, long double y);

float remquof(float x, float y, int *quo);
double remquo(double x, double y, int *quo);
long double remquol(long double x, long double y, int *quo);


// todo: manipulation functions

double copysign(double x, double y);
float copysignf(float x, float y);
long double copysignl(long double x, long double y);

float nanf(const char *tagp);
double nan(const char *tagp);
long double nanl(const char *tagp);

float nextafterf(float x, float y);
double nextafter(double x, double y);
long double nextafterl(long double x, long double y);


// todo: min max and difference

float fdimf(float x, float y);
double fdim(double x, double y);
long double fdiml(long double x, long double y);

float fmaxf(float x, float y);
double fmax(double x, double y);
long double fmaxl(long double x, long double y);

float fminf(float x, float y);
double fmin(double x, double y);
long double fminl(long double x, long double y);


// todo: muladd

float fmaf(float x, float y, float z);
double fma(double x, double y, double z);
long double fmal(long double x, long double y, long double z);


// comparison

#define isgreater(x, y) ((__fp_unordered_compare(x, y) \
			   & 0x4500) == 0)
#define isless(x, y) ((__fp_unordered_compare (y, x) \
                       & 0x4500) == 0)
#define isgreaterequal(x, y) ((__fp_unordered_compare (x, y) \
                               & FP_INFINITE) == 0)
#define islessequal(x, y) ((__fp_unordered_compare(y, x) \
			    & FP_INFINITE) == 0)
#define islessgreater(x, y) ((__fp_unordered_compare(x, y) \
			      & FP_SUBNORMAL) == 0)
#define isunordered(x, y) ((__fp_unordered_compare(x, y) \
			    & 0x4500) == 0x4500)

// helper functions

float _PDCLIB_clampradf(float x);
double _PDCLIB_clampradd(double x);
long double _PDCLIB_clampradl(long double x);

