ORG 100000h

BITS 32
CPU 386

%include "inc_ia.asm"
%include "inc_kernel.asm"
%include "inc_chips.asm"

; kernel signature - stage2 needs it for verification
; kernel is called with [EBX]=bootloader struct, [EDX]=ramdisk

;
; Summary: stage3_ia32_pc.asm
; *The stage 3 / main kernel for My Operating System (MOS)*
;
; This module is responsible for managing and enabling the running of user applications
;
; For software programming, check the <System calls> section
;
; Author:
;     Marcel Sondaar
;
; License:
;     Educational purposes
;

kernelimagestart:   times 4 NOP
                    MOV ESI, 0x80000
                    MOV EDI, 0x100000
                    MOV ECX, kernelimagesize
                    CLD
                    REP MOVSB
                    MOV EAX, kernelentry
                    JMP EAX

times 32-$+kernelimagestart DB 0
                    ; DD _k_startprocfound ; cheap bochs debug stub
                    DD scheduler
                    DD AllocateV8086Taskset

mbmagic             EQU 0x1BADB002
mbflags             EQU 0x00010003
multiboot:          DD mbmagic
                    DD mbflags
                    DD 0-mbmagic-mbflags
                    DD multiboot
                    DD $$
                    DD kernelimageend
                    DD 0x400000
                    DD multibootentry

multibootentry:     MOV EBX, 0xB8000

                    MOV word [EBX], 0x0700 + 'M'
                    ADD EBX, 2
                    MOV word [EBX], 0x0700 + 'B'
                    ADD EBX, 2
                    MOV word [EBX], 0x0700 + '/'
                    ADD EBX, 2


                    MOV word [EBX], 0x0720
                    ADD EBX, 2

                    JMP trapentry


kernelentry:        ; copy over boot data structure
                    MOV ESI, EBX
                    MOV EDI, mosSeg
                    MOV ECX, mosSegSize
                    REP MOVSB
                    ; store ramdisk location
                    MOV [lpgRamdisk], EDX

                    ; couldnt get brendan's bios to work, so, manual work it is
                    ; fill lower 4m of memory with CCCCCCCCs
                    MOV ECX, (0x300000 - kernelimagesize)
                    MOV EDI, kernelimageend
                    MOV AL, 0xCC
                    REP STOSB


                    MOV EBX, [mosTextBase]
                    XOR EAX, EAX
                    MOV AH, 80
                    MOV AL, [mosTextChar+1]
                    MUL AH
                    XOR ECX, ECX
                    MOV CL, [mosTextChar]
                    ADD EAX, ECX
                    SHL EAX, 1
                    ADD EBX, EAX

trapentry:          MOV word [EBX], 0x0750
                    ADD EBX, 2
                    MOV word [EBX], 0x074d
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    ; set up a temporary stack
                    MOV ESP, 0x1fffc

                    ; initialize memory allocation table
                    CALL InitializePaging

                    MOV EAX, CR0
                    OR EAX, 0x80000000
                    MOV CR0, EAX

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0750
                    ADD EBX, 2
                    JMP dword pagingonline         ; flush pipeline
pagingonline:       MOV EAX, [lpgKernelStack]
                    ADD EAX, 0xffc
                    MOV ESP, EAX
                    MOV word [EBX], 0x0747
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    ; paging is online, now we can use the memory allocating functions
                    ; initialize descriptor table
                    CALL InitializeGDT

                    ; prepare and enable interrupts
                    CALL InitializePIC
                    CALL InitializeIDT

                    ; establish kernel task
                    Call InitializeIO
                    Call InitializeTDT
                    CALL InitializeTSS

                    ; Run kernel patch schemes
                    Call GetProcessorInfo
                    Call GetFPUInfo
                    Call InstallPatches

                    ; initialize the scheduler
                    CALL InitializePIT
                    CALL InitializeScheduler

                    ; initialize more parts of the system

                    Call InitializeIPC

                    ; todo: AP detection and bootstrap
                    ; CALL InitializeAPIC ; todo
                    ; CALL Multiprocessor ; needs fixing

                    ; todo: Parse memtable

                    ; bootstrap first process
                    CALL BootstrapUserspace

                    ; ready, set, go, uhm...:p
                    ;JMP $

                    ;Todo: fix debug stub
                    CPU 386 FPU
.lp:                FINIT
                    JMP .lp
                    CPU 386

; --------------------------------------------------------------------------------------------------
; Group: Kernel subfunctions
; --------------------------------------------------------------------------------------------------


                    ; Function: AddGate
                    ; allocates a gate and assigns a handler to it
                    ;
                    ; in:
                    ;     EBX - CR3 of the gate's handler
                    ;     EDI - the handler's function
                    ;
                    ; out:
                    ;     EDX - Gate number allocated
                    ;
                    ; destroyed:
                    ;     ECX
                    ;
AddGate:            MOV EDX, [lpgTaskHandlers]          ; get the gate jump table
                    ADD EDX, 8                          ; skip entry 0 (thats the kernel)
                    MOV ECX, (PAGE_SIZE / 8) - 1        ; set a counter to the size of the jump table
.loop:              CMP dword [EDX+4], 0                ; check if its empty
                    JE .tryacquire                      ; if so, try claiming it
.loopnow:           DEC ECX                             ; otherwise, decrement counter
                    ADD EDX, 8                          ; go to next entry
                    JNZ .loop                           ; and return
                    XOR EDX, EDX                        ; end of table, return zero
                    RET                                 ; and return

.tryacquire:        LOCK BTS dword [EDX], 0             ; take mutex encoded in table
                    JC .loop                            ; if taken, return to the loop
                    CMP dword [EDX+4], 0                ; check condition properly
                    JNE .bail                           ; bail if it was set
                    MOV [EDX+4], EDI                    ; if not, load the target address
                    INC EBX                             ; set mutex bit
                    MOV [EDX], EBX                      ; load target address space as well
                    DEC EBX                             ; restore EBX
                    LOCK BTR dword [EDX], 0             ; release mutex
                    SUB EDX, [lpgTaskHandlers]          ; compute port number
                    RET                                 ; return answer

.bail:              LOCK BTR dword [EDX], 0             ; release mutex
                    JMP .loopnow                        ; and continue with the loop (we can skip the check)



                    ; Function: AddThread
                    ; adds a new thread to the scheduler's list
                    ;
                    ; in:
                    ;     ESI - CR3 of the thread to be added
                    ;     EDI - TDT location
                    ;     EAX - processor to schedule to
                    ;
                    ; out:
                    ;     CF - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     ECX
                    ;
AddThread:          LOCK BTR dword [semScheduler], EAX      ; TAC on the scheduler page
                    JNC AddThread                           ; keep trying on failure
                    PUSH EAX                                ; store eax
                    PUSH EDI                                ; store edi
                    MOV EDI, [scheduler_table+4*EAX]        ; get the table we want
                    XOR EAX, EAX                            ; eax = 0
                    MOV ECX, PAGE_SIZE / 4 - 8              ; size of the page minus the last one
                    REPNE SCASD                             ; find an null entry
                    JNE .bail                               ; check if the list is full
                    MOV ECX, EDI                            ; store a local copy of the address
                    POP EDI                                 ; restore edi
                    MOV [ECX+4], EAX                        ; write the new null pointer (eax is still 0)
                    MOV [ECX+8], EAX                        ; another dword of it
                    POP EAX                                 ; restore eax
                    MOV [ECX], ESI                          ; write the CR3
                    MOV [ECX-4], EDI                        ; write the TDT
                    LOCK BTS dword [semScheduler], EAX      ; release lock (cf is old bit, which is clear)
                    RET                                     ; return
.bail:              POP EDI                                 ; restore registers
                    POP EAX                                 ; this one too
                    LOCK BTS dword [semScheduler], EAX      ; release lock (cf is old bit, which is clear)
                    STC                                     ; set failure status
                    RET                                     ; return


                    ; Function: AddBlockedThread
                    ; adds a new thread to the blocked list
                    ;
                    ; in:
                    ;     ESI - CR3 of the thread to be added
                    ;     EDI - TDT location
                    ;
                    ; out:
                    ;     CF - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     EAX ECX
                    ;
AddBlockedThread:   LOCK BTR dword [semBlockedTask], 0      ; TAC on the scheduler page
                    JNC AddThread                           ; keep trying on failure
                    PUSH EDI                                ; store edi
                    MOV EDI, [scheduler_sleeptask]          ; get the table we want
                    XOR EAX, EAX                            ; eax = 0
                    MOV ECX, PAGE_SIZE / 4                  ; size of the page minus the last one
                    REPNE SCASD                             ; find an null entry
                    JNE .bail                               ; check if the list is full
                    MOV ECX, EDI                            ; store a local copy of the address
                    POP EDI                                 ; restore edi
                    MOV [ECX], ESI                          ; write the CR3
                    MOV [ECX-4], EDI                        ; write the TDT
                    LOCK BTS dword [semBlockedTask], 0      ; release lock (cf is old bit, which is clear)
                    RET                                     ; return
.bail:              POP EDI                                 ; restore register
                    LOCK BTS dword [semBlockedTask], 0      ; release lock (cf is old bit, which is clear)
                    STC                                     ; set failure status
                    RET                                     ; return


; Fixme: Starvation in Route functions
; add cli/sti blocks to prevent this

                    ; Function: AddRoute
                    ; adds a name to port lookup to the table
                    ;
                    ; in:
                    ;     EAX - port number
                    ;     EDX - route name
                    ;
                    ; out:
                    ;     CF - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     none
AddRoute:           PUSH EDI
                    PUSH ECX
                    PUSH EAX

                    CLI
.mutex:             LOCK BTR dword [semTaskRoutes], 0
                    JNC .mutex
                    MOV AX, [ticTaskRoutes]
                    INC word [ticTaskRoutes]
                    LOCK BTS dword [semTaskRoutes], 0
                    STI
.queueloop:         CMP AX, [trnTaskRoutes]
                    JNE .queueloop

                    MOV EDI, [lpgTaskRoutes]
                    MOV ECX, PAGE_SIZE / 8
.loop:              MOV EAX, [EDI]
                    OR EAX, EAX
                    JZ .fill
                    CMP EAX, EDX
                    JE .bail
                    ADD EDI, 8
                    DEC ECX
                    JNZ .loop
.bail:              LOCK INC word [trnTaskRoutes]
                    STC
                    POP EAX
                    POP ECX
                    POP EDI
                    RET
.fill:              POP EAX
                    POP ECX
                    MOV [EDI], EDX
                    MOV [EDI+4], EAX
                    POP EDI
                    LOCK INC word [trnTaskRoutes]
                    RET


                    ; Function: AllocateInterrupt
                    ; attempts to allocate an IDT entry of choice
                    ;
                    ; in:
                    ;     - AL = interrupt number
                    ;
                    ; out:
                    ;     - CF = clear on success
                    ;
                    ; destroyed:
                    ;     EBX EDX
                    ;

AllocateInterrupt:  PUSH EDI
                    PUSH EAX
                    MOV EDI, [lpgIdtAllocators]
                    AND EAX, 0xff
                    SHL EAX, 3
                    LEA EDI, [EDI+EAX]
                    POP EAX
                    MOV EDX, [scheduler_curtask]
                    MOV EBX, [scheduler_curtask+4]
                    MOV EDX, [EDX+TDT_KERNELSTACK]
                    CMP dword [EDI], 0
                    JNE .fail
.retry:             CMP byte [semIDT], 1
                    JNE .retry
                    LOCK DEC byte [semIDT]
                    JNZ .abort
                    CMP dword [EDI], 0
                    JNE .fail2
                    MOV [EDI], EDX
                    MOV [EDI+4], EBX
                    LOCK INC byte [semIDT]
                    CLC
                    POP EDI
                    RET
.fail2              LOCK INC byte [semIDT]
.fail               POP EDI
                    STC
                    RET
.abort:             LOCK INC byte [semIDT]
                    JMP .retry


                    ; Function: AllocateGDTEntry
                    ; Allocates a free slot in the global descriptor table and loads
                    ; it with the given parameters
                    ;
                    ; in:
                    ;    - EBX = lo dword
                    ;    - EDX = hi dword
                    ;
                    ; out:
                    ;    - EAX = descriptor
                    ;    - EDX = linear offset of the current descriptor
                    ;
                    ; destroyed:
                    ;     EAX EBX ECX EDX
                    ;
AllocateGDTEntry:   MOV EAX, [lpgGdtOffset]
                    ADD EAX, 8
                    MOV ECX, 8192 - 1
.try:               CMP dword [EAX], 0
                    JNE .next
                    CMP dword [EAX + 4], 0
                    JNE .next
                    ; this descriptor is empty, but more than one task might get here
                    ; currently the synchronizing code may cause starvation, but based on usage its unlikely
                    CMP byte [semGDT], 1                 ; wait until semaphore is up
                    JNE .try                             ; loop until it is
                    LOCK DEC byte [semGDT]               ; atomic decrement on semaphore
                    JZ .allow                            ; check if we were first
                    LOCK INC byte [semGDT]               ; rollback the action
                    JMP .try                             ; and try again
.allow:             MOV [EAX], EBX                       ; write lo descriptor dword
                    MOV [EAX+4], EDX                     ; write hi descriptor dword
                    LOCK INC byte [semGDT]               ; release semaphore
                    MOV EBX, [lpgGdtOffset]
                    MOV EDX, EAX
                    SUB EAX, EBX                         ; calculate segment index
                    RET
.next:              ADD EAX, 8                           ; next index
                    LOOP .try
                    XOR EAX, EAX                         ; since when do we need all 8192 descs?!
                    XOR EDX, EDX
                    RET


                    ; Function: AllocateGDTEntry
                    ; Allocates and writes a TSS entry in the GDT
                    ;
                    ; in:
                    ;     - EDI = address of free memory
                    ;
                    ; out:
                    ;     - EAX = descriptor
                    ;
                    ; destroyed:
                    ;     EAX EBX ECX EDX
AllocateGDTTSS:     MOV EBX, EDI                         ; EBX = address
                    SHL EBX, 16                          ; EBX = addr[0-15],empty
                    OR EBX, 1                            ; EBX = addr[0-15],size
                    MOV EDX, EDI
                    AND EDX, 0xff000000
                    MOV EAX, EDI
                    SHR EAX, 16
                    MOV DL, AL
                    XOR EAX, EAX
                    MOV AL, GDT_FLAG_PRESENT | GDT_FLAG_RING0 | GDT_TYPE_TSS32
                    MOV AH, GDT_FLAG_PAGES
                    SHL EAX, 8
                    OR EDX, EAX
                    CALL AllocateGDTEntry
                    RET


                    ; Function: AllocateMemoryLo 
                    ; allocates memory from the table, preferrably from the beginning
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     - EAX = physical address
                    ;     - CF = clear on success
                    ;
                    ; destroyed:
                    ;     none
                    ;
AllocateMemoryLo:   PUSH ECX
                    PUSH EDI
                    MOV EDI, [lpgMemTable]
                    MOV ECX, 1024
.l1:                MOV EAX, [EDI]
                    AND EAX, PAGE_SYS_FREEMASK
                    JZ .skip
                    MOV EAX, [EDI]
                    AND EAX, PAGE_ADDRMASK
                    CALL AllocateMemoryUnit
                    JC .skip
                    LOCK DEC dword [EDI]
                    AND EDI, PAGE_PARAMMASK
                    SHL EDI, 22-2
                    SHL EAX, 12
                    ADD EAX, EDI
                    POP EDI
                    POP ECX
                    RET
.skip:              ADD EDI, 4
                    DEC ECX
                    JNZ .l1
                    POP EDI
                    POP ECX
                    STC
                    RET

                    ; Function: AllocateMemoryHi
                    ; allocates memory from the table, preferrably from the end
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     - EAX = physical address
                    ;     - CF = clear on success
                    ;
                    ; destroyed:
                    ;     none
                    ;
AllocateMemoryHi:   PUSH ECX
                    PUSH EDI
                    MOV EDI, [lpgMemTable]
                    ADD EDI, PAGE_SIZE - 4
                    MOV ECX, 1024
.l1:                MOV EAX, [EDI]
                    AND EAX, PAGE_SYS_FREEMASK
                    JZ .skip
                    MOV EAX, [EDI]
                    AND EAX, PAGE_ADDRMASK
                    CALL AllocateMemoryUnit
                    JC .skip
                    LOCK DEC dword [EDI]
                    AND EDI, PAGE_PARAMMASK
                    SHL EDI, 22-2
                    SHL EAX, 12
                    ADD EAX, EDI
                    POP EDI
                    POP ECX
                    RET
.skip:              SUB EDI, 4
                    DEC ECX
                    JNZ .l1
                    POP EDI
                    POP ECX
                    STC
                    RET

                    ; Function: AllocateMemoryUnit
                    ; Allocate one unit of free memory from a given table
                    ;
                    ; in:
                    ;     - EAX = Subtable Address
                    ;
                    ; out:
                    ;     - EAX = page relative to this table
                    ;     - CF = set on failure
                    ;
                    ; destroyed:
                    ;     none
                    ;
AllocateMemoryUnit: PUSH ECX
                    PUSH EDX
                    MOV ECX, 1024
.l1:                MOV EDX, [EAX]
                    CMP EDX, PAGE_SYS_MEMORY | PAGE_SYS_NONEALLOC
                    JE .try
.nope:              ADD EAX, 4
                    DEC ECX
                    JNZ .l1
                    POP EDX
                    POP ECX
                    XOR EAX, EAX
                    STC
                    RET
.retry              LOCK INC byte [semMemTable]
.try:               CMP byte [semMemTable], 1
                    JNE .try
                    LOCK DEC byte [semMemTable]
                    JNZ .retry
                    MOV EDX, [EAX]
                    CMP EDX, PAGE_SYS_MEMORY | PAGE_SYS_NONEALLOC
                    JE .doalloc
                    LOCK INC byte [semMemTable]
.doalloc            MOV dword [EAX], PAGE_SYS_ONEALLOC | PAGE_SYS_MEMORY
                    LOCK INC byte [semMemTable]
                    AND EAX, PAGE_PARAMMASK
                    SHR EAX, 2
                    POP EDX
                    POP ECX
                    CLC
                    RET


                    ; Function: AllocatePortLo
                    ; Attempts to allocate a port in the lower 2k area
                    ;
                    ; in:
                    ;     EDX = requested port
                    ;
                    ; out:
                    ;     CF = clear on success
                    ;
                    ; destroyed:
                    ;     none
                    ;
AllocatePortLo:     PUSH EDI
                    MOV EDI, EDX
                    SHL EDI, 1
                    ADD EDI, [lpgIOMapOffset]
.retry:             CMP word [EDI], 0
                    JNE .fail
                    CMP byte [semPortLo], 1
                    JNE .retry
                    LOCK DEC byte [semPortLo]
                    JNZ .abort
                    CMP word [EDI], 0
                    JNE .abort
                    MOV word [EDI], 1
                    LOCK INC byte [semPortLo]
                    POP EDI
                    CLC
                    RET
.fail:              POP EDI
                    STC
                    RET
.abort:             LOCK INC byte [semPortLo]
                    JMP .retry


                    ; Function: AllocatePortHi
                    ; Attempts to allocate a port (4) in the higher area
                    ;
                    ; in:
                    ;     EDX = requested port
                    ;
                    ; out:
                    ;     CF = clear on success
                    ;
                    ; destroyed:
                    ;     none
                    ;
AllocatePortHi:     PUSH EDI
                    MOV EDI, EDX
                    SHR EDI, 2
                    SHL EDI, 1
                    ADD EDI, [lpgIOMapOffset]
                    ADD EDI, PAGE_SIZE
.retry:             CMP word [EDI], 0
                    JNE .fail
                    CMP byte [semPortLo], 1
                    JNE .retry
                    LOCK DEC byte [semPortHi]
                    JNZ .abort
                    CMP word [EDI], 0
                    JNE .abort
                    MOV word [EDI], 1
                    LOCK INC byte [semPortHi]
                    POP EDI
                    CLC
                    RET
.fail:              POP EDI
                    STC
                    RET
.abort:             LOCK INC byte [semPortHi]
                    JMP .retry
                    STC
                    RET


                    ; Function: AllocateTaskset
                    ; allocates a task set (descriptor + kernel stack page)
                    ;
                    ; in:
                    ;     - EDI = virtual address to allocate at
                    ;     - EBX = virtual address to start execution
                    ;
                    ; out:
                    ;     - CF = clear on success
                    ;     - EAX = thread handle
                    ;     - EDI = virual address of task descriptor
                    ;
                    ; destroyed:
                    ;     ECX, ESI
                    ;
AllocateTaskset:    PUSH EDX
                    PUSH EBX
                    Call AllocateMemoryHi                ; get one page of memory
                    JNC .ok1
                    JMP .fail
.ok1:               PUSH EDI                             ; push it
                    MOV ESI, EAX
                    Call InsertPage
                    Call AllocateMemoryHi
                    JNC .ok2
                    JMP .fail
.ok2:               MOV ESI, EAX
                    ADD EDI, PAGE_SIZE
                    PUSH EDI
                    CALL InsertPage
                    POP EBX
                    POP EAX
                    ADD EAX, PAGE_SIZE-4
                    POP EDX                              ; virtual address
                    PUSH EDX
                    PUSH EDI
                    PUSH EAX
                    MOV EDI, EBX
                    XOR EAX, EAX
                    MOV ECX, PAGE_SIZE/4
                    CLD
                    REP STOSD
                    POP EAX
                    POP EDI
                    MOV [EBX+TDT_KERNELSTACKTOP], EAX
                    MOV dword [EAX], 0x20 + 3
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], EFLAGS_IF
                    SUB EAX, 4
                    MOV dword [EAX], 0x18 + 3
                    SUB EAX, 4
                    MOV [EAX], EDX
                    SUB EAX, 4
                    MOV dword [EAX-5*4], EAX
                    SUB EAX, 8*4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23

                    MOV [EBX+TDT_KERNELSTACK], EAX       ; [EBX] becomes TDT, [EAX] becomes stack
                    MOV [EBX+TDT_PHYSADDR], ESI
                    MOV EAX, EBX
                    POP EBX
                    POP EDX
                    CLC
                    RET
.fail:              POP EBX
                    POP EDX
                    STC
                    RET


                    ; Function: AllocateV8086Taskset
                    ; allocates a task set (descriptor + kernel stack page)
                    ;
                    ; in:
                    ;     - EDI = virtual address to allocate at
                    ;     - EBX = packed cs:ip to start execution at
                    ;
                    ; out:
                    ;     - CF = clear on success
                    ;     - EAX = thread handle
                    ;     - EDI = virual address of task descriptor
                    ;
                    ; destroyed:
                    ;     ECX, ESI
                    ;
AllocateV8086Taskset:
                    PUSH EDX
                    PUSH EBX
                    Call AllocateMemoryHi                ; get one page of memory
                    JNC .ok1
                    JMP .fail
.ok1:               PUSH EDI                             ; push it
                    MOV ESI, EAX
                    Call InsertPage
                    Call AllocateMemoryHi
                    JNC .ok2
                    JMP .fail
.ok2:               MOV ESI, EAX
                    ADD EDI, PAGE_SIZE
                    PUSH EDI
                    CALL InsertPage
                    POP EBX
                    POP EAX
                    ADD EAX, PAGE_SIZE-4
                    POP EDX                              ; virtual address
                    PUSH EDX
                    PUSH EDI
                    PUSH EAX
                    MOV EDI, EBX
                    XOR EAX, EAX
                    MOV ECX, PAGE_SIZE/4
                    CLD
                    REP STOSD
                    POP EAX
                    POP EDI
                    MOV [EBX+TDT_KERNELSTACKTOP], EAX

                    ; v8086 segregs
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4

                    MOV dword [EAX], 0x20 + 3
                    SUB EAX, 4
                    MOV dword [EAX], 0
                    SUB EAX, 4
                    MOV dword [EAX], EFLAGS_IF | EFLAGS_VM | EFLAGS_IOPL3
                    SUB EAX, 4
                    ; CS
                    MOV dword [EAX - 2], EDX
                    MOV word [EAX + 2], 0
                    SUB EAX, 4
                    ; IP
                    MOV [EAX], DX
                    MOV word [EAX+2], 0
                    SUB EAX, 4

                    MOV dword [EAX-5*4], EAX
                    SUB EAX, 8*4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23
                    SUB EAX, 4
                    MOV dword [EAX], 0x23

                    MOV [EBX+TDT_KERNELSTACK], EAX       ; [EBX] becomes TDT, [EAX] becomes stack
                    MOV [EBX+TDT_PHYSADDR], ESI
                    MOV EAX, EBX
                    POP EBX
                    POP EDX
                    CLC
                    RET
.fail:              POP EBX
                    POP EDX
                    STC
                    RET


                    ; Function: BootstrapUserspace
                    ; load first userspace module and schedule it for execution
                    ;
                    ; the file is named 'STAGE4' and is considered a flat binary
                    ; and is loaded and executed at the 4mb mark
                    ;
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out: 
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;
BootstrapUserspace: MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0700 + 'O'
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    MOV EAX, [lpgRamdisk]
.findentry:         CMP dword [EAX], 'STAG'
                    JNE .nextentry
                    CMP dword [EAX+4], 'E4  '
                    JNE .nextentry
                    MOV ECX, [EAX+8]
                    MOV EDX, [EAX+12]
                    ADD EDX, [lpgRamdisk]
                    MOV [.imagesize], ECX
                    MOV [.imageoffs], EDX
                    JMP .foundentry
.nextentry:         CMP dword [EAX], 0
                    JE .foundnone
                    ADD EAX, 16
                    JMP .findentry

.foundnone:         MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0700 + '?'
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    HLT

.foundentry:        PUSH ECX
                    SHR ECX, 12
                    MOV EDI, 0x400000
                    INC ECX

.allocloop:         CALL AllocateMemoryHi
                    MOV ESI, EAX
                    CALL InsertPageUser
                    ADD EDI, PAGE_SIZE
                    DEC ECX
                    JNZ .allocloop

                    POP ECX

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0700 + '.'
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    SHR ECX, 2
                    INC ECX
                    MOV ESI, [.imageoffs]
                    MOV EDI, 0x400000
                    REP MOVSD
.test:
                    MOV ESI, [lpgKernelHole]
                    MOV EDI, ESI
                    ADD ESI, 2 * PAGE_SIZE
                    MOV [lpgKernelHole], ESI
                    MOV EBX, 0x400000
                    Call AllocateTaskset

                    PUSH EAX
                    MOV EAX, [lpgRamdisk]
.test3:             Call SetTaskRegisters
                    POP EAX

                    ; Fixme: subroutine for this
.test2:             MOV EDX, [scheduler_table]
                    MOV EBX, CR3
                    MOV dword [EDX+16], 0
                    MOV dword [EDX+20], 0
                    MOV [EDX+12], EBX
                    MOV [EDX+8], EAX


                    RET
ALIGN 4
.imagesize          DD 0
.imageoffs          DD 0


                    ; Function: GetGateAddress
                    ; Looks up a gate's values
                    ;
                    ; in:
                    ;     EAX - gate number
                    ;     
                    ; out:
                    ;     EDX - CR3
                    ;     EBX - Address
                    ;
                    ; destroyed:
                    ;     none
                    ;
GetGateAddress:     PUSH EAX
                    ADD EAX, [lpgTaskHandlers]
.getmutex:          LOCK BTS dword [EAX], 0
                    JC .getmutex
                    MOV EDX, [EAX]
                    MOV EBX, [EAX+4]
                    LOCK BTR dword [EAX], 0
                    POP EAX
                    RET


                    ; Function: GetPageEntry
                    ; Loads a page table with a given entry
                    ;
                    ; in:
                    ;     - EDI = location to check
                    ;
                    ; out:
                    ;     - ESI = page table entry
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
GetPageEntry:       MOV EBX, [lpgPagemap]
                    MOV EAX, EDI
                    SHR EAX, 22
                    SHL EAX, 2
                    MOV EDX, [EBX+EAX]
                    AND EDX, PAGE_ADDRMASK
                    JZ .checkpagedir
.continue:          MOV EBX, EDI
                    AND EBX, PAGE_TABLEMASK
                    SHR EBX, 10
                    MOV ESI, [EDX+EBX]
                    RET
.checkpagedir:      MOV EBX, [lpgPagedir]
                    MOV EDX, [EBX+EAX]
                    AND EDX, PAGE_ADDRMASK
                    JNZ .maptable
.requirenew:        XOR ESI, ESI                ; hehe, the page table does not exist. Let's be lazy :)
                    RET

.maptable           PUSH ESI
                    PUSH EDI
                    MOV ESI, EDX
                    AND ESI, PAGE_ADDRMASK
                    MOV EDI, [lpgPageResolver]
                    Call InvalidatePage
                    CALL InsertPage
                    MOV EDX, EDI
                    POP EDI
                    POP ESI
                    JMP .continue
                    ; Fixme: GetPageEntry race condition
                    ; mutex on resolver use


                    ; Function: GetFPUInfo
                    ; Get FPU prescence, detect its type
                    ; then configure the kernel accordingly
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX
                    ;
                    CPU 386 FPU
GetFPUInfo:         MOV EAX, [lpgProcessorCaps]            ; CPUID stores info about onboard FPU
                    AND EAX, CPUID_FPU                     ; test the FPU bit
                    JNZ .hasfpu                            ; FPU present if set
                    MOV EAX, CR0                           ; CR0 has a bit on the matter as well
                    AND EAX, CR0_ET                        ; Check Extension Type
                    JZ .nofpu                              ; The processor supports no FPU
                    MOV EAX, CR0                           ; Start probe, get CR0
                    AND EAX, (-1) - (CR0_TS + CR0_EM)      ; clear TS and EM to force fpu access
                    OR EAX, CR0_NE                         ; set NE (workaround no-wait bug)
                    MOV CR0, EAX                           ; store control word
                    FNINIT                                 ; load defaults to FPU
                    FNSTSW [.testword]                     ; store status word
                    CMP word [.testword], 0                ; compare the written status
                    JNE .nofpu

                    ;we have an FPU
.hasfpu:            MOV EAX, CR0                           ; pop cr0
                    AND EAX, (-1) - (CR0_EM)               ; clear emulation bit
                    OR EAX, CR0_MP + CR0_NE                ; set monitor and native exceptions bits
                    MOV CR0, EAX                           ; write control register
                    OR dword [lpgProcessorCaps], CPUID_FPU ; set FPU bit in info for cpuid-less procs
                    RET

                    ;we have no usable FPU
.nofpu:             MOV EAX, CR0                           ; pop cr0
                    OR EAX, CR0_EM                         ; set FPU emulation
                    AND EAX, (-1) - (CR0_MP + CR0_NE)      ; clear monitor and native exceptions
                    MOV CR0, EAX                           ; write cr0
                    Call ProcessorNoFPpatch                ; patch the kernel for No-FP operation
                    RET
                    CPU 386

.testword:          DW 0x55AA




                    ; Function: GetProcessorInfo
                    ; Gets processor info and configures it accordingly
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;

                    ; Todo: expand detection for sse3, 3dnow, syscall
GetProcessorInfo:   PUSHFD
                    POP EAX
                    OR EAX, EFLAGS_ID
                    PUSH EAX
                    POPFD
                    PUSHFD
                    POP EBX
                    CMP EBX, EAX
                    JE .cancpuid

                    ; Todo: AC capability detection

                    ; Todo: Cyrix detection

                    MOV byte [lpgProcessorType], PROC_UNKNOWN

                    RET

CPU 686             ; lots of architecture-specific code here

.cancpuid:          XOR EAX, EAX
                    CPUID
                    MOV [lpgProcessorName+0], EBX        ; store processor name
                    MOV [lpgProcessorName+4], EDX        ;
                    MOV [lpgProcessorName+8], ECX        ;
                    XOR EBX, EDX                         ; create a quick hash
                    XOR EBX, ECX                         ;
                    CMP EBX, 0x506e7f40                  ; hash for intels
                    JNE .notintel
                    JMP .intel
.notintel:          CMP EBX, 0x454d5a47                  ; hash for amd's
                    JNE .noamd
                    JMP .amd
.noamd:             RET

.intel:             MOV byte [lpgProcessorType], PROC_I486
                    Call ProcessorTLBpatch
                    CMP EAX, 1
                    JGE .hasfeat
                    RET
.hasfeat:           MOV EAX, 1
                    CPUID
                    MOV [lpgProcessorCaps], EDX
                    RET


.amd:               MOV byte [lpgProcessorType], PROC_AK5
                    Call ProcessorTLBpatch
                    CMP EAX, 1
                    JGE .hasfeat
                    RET
.hasamdfeat:        MOV EAX, 1
                    CPUID
                    MOV [lpgProcessorCaps], EDX
                    RET

CPU 386


                    ; Function: GetRoute
                    ; Looks up a route in the route table and returns
                    ;
                    ; in:
                    ;     EDX - route name
                    ;
                    ; out:
                    ;     EAX - route port, or 0 on failure
                    ;
                    ; destroyed:
                    ;     none
                    ;
GetRoute:           PUSH EDI
                    PUSH ECX

                    CLI
.mutex:             LOCK BTR dword [semTaskRoutes], 0
                    JNC .mutex
                    MOV CX, [ticTaskRoutes]
                    INC word [ticTaskRoutes]
                    LOCK BTS dword [semTaskRoutes], 0
                    STI
.queueloop:         CMP CX, [trnTaskRoutes]
                    JNE .queueloop

                    MOV EDI, [lpgTaskRoutes]
                    MOV ECX, PAGE_SIZE / 8
.loop:              CMP [EDI], EDX
                    JE .done
                    ADD EDI, 8
                    DEC ECX
                    JNZ .loop
                    POP ECX
                    POP EDI
                    LOCK INC word [trnTaskRoutes]
                    XOR EAX, EAX
                    RET
.done:              MOV EAX, [EDI+4]
                    POP ECX
                    POP EDI
                    LOCK INC word [trnTaskRoutes]
                    RET



                    ; Function: HardwareDelay
                    ; Delays for a short period of time.
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
HardwareDelay:      PUSH ECX
                    MOV ECX, 32
.loop:              XOR ECX, EAX    ; some pipeline hostile NOP
                    XOR ECX, EAX
                    LOOP .loop      ; an iteration takes 3 clocks average
                    POP ECX
                    RET



                    ; Function: InitializeIDT
                    ; Sets up the IDT and loads it
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;
InitializeIDT:      MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0749
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    CALL AllocateMemoryLo
                    MOV ESI, EAX
                    MOV EDI, [lpgKernelHole]
                    MOV EBX, EDI
                    MOV EAX, EDI
                    ADD EBX, 256 * 8
                    MOV [lpgIdtOffset], EAX
                    MOV [lpgIdtAllocators], EBX
                    CALL InsertPageGlobal
                    ADD EDI, 4096
                    MOV [lpgKernelHole], EDI

                    MOV ECX, 256 * 8 / 4
                    MOV EDI, [lpgIdtOffset]
                    XOR EAX, EAX
                    REP STOSD
                    LIDT [lpgIdtLimit]

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0744
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    MOV EDI, [lpgIdtOffset]
                    MOV ESI, .int_list
                    MOV ECX, 0x20
.loop_ia:           LODSD
                    MOV EBX, EAX
                    AND EAX, 0x0000ffff
                    SHR EBX, 16
                    MOV [EDI+IDT_OFFSET_LO], AX
                    MOV [EDI+IDT_OFFSET_HI], BX
                    MOV word [EDI+IDT_SELECTOR_OFFS], 0x08
                    MOV byte [EDI+IDT_FLAG_OFFS], GDT_FLAG_RING0 | IDT_TYPE_TRAP32 | GDT_FLAG_PRESENT
                    ADD EDI, 8
                    DEC ECX
                    JNZ .loop_ia

                    MOV ECX, 0x10
.loop_irq:          LODSD
                    MOV EBX, EAX
                    AND EAX, 0x0000ffff
                    SHR EBX, 16
                    MOV [EDI+IDT_OFFSET_LO], AX
                    MOV [EDI+IDT_OFFSET_HI], BX
                    MOV word [EDI+IDT_SELECTOR_OFFS], 0x08
                    MOV byte [EDI+IDT_FLAG_OFFS], GDT_FLAG_RING0 | IDT_TYPE_INT32 | GDT_FLAG_PRESENT
                    ADD EDI, 8
                    DEC ECX
                    JNZ .loop_irq

                    LODSD
                    MOV EBX, EAX
                    AND EAX, 0x0000ffff
                    SHR EBX, 16
                    MOV [EDI+IDT_OFFSET_LO], AX
                    MOV [EDI+IDT_OFFSET_HI], BX
                    MOV word [EDI+IDT_SELECTOR_OFFS], 0x08
                    MOV byte [EDI+IDT_FLAG_OFFS], GDT_FLAG_RING3 | IDT_TYPE_TRAP32 | GDT_FLAG_PRESENT

                    XOR EAX, EAX
                    INT 0x30

                    ; fill in allocation table
                    MOV EDI, [lpgIdtAllocators]
                    MOV EAX, [lpgPagedir]                ; cr3
                    MOV EBX, [lpgKernelStack]            ; sp
                    MOV ECX, 32                          ; the first 32 ints belong to the kernel
.loop_at1           STOSD                                ; write cr3
                    MOV [EDI], EBX                       ; write sp0
                    ADD EDI, 4
                    DEC ECX
                    JNZ .loop_at1                        ; and loop
                    XOR EAX, EAX
                    MOV ECX, 16 * 2
                    REP STOSD
                    MOV EAX, [lpgPagedir]
                    STOSD
                    MOV [EDI], EBX

.trap:              STI                                  ; we can do interrupts now

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0754
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET

.int_list:          DD int_fault ; Int00 #DF,  Division by zero
                    DD int_fault ; Int01 #DB,  Debug interrupt
                    DD int_abort ; Int02 NMI,  Non-maskable interrupt
                    DD int_fault ; Int03 #BP,  Breakpoint
                    DD int_fault ; Int04 #OF,  Overflow
                    DD int_fault ; Int05 #BR,  Bound exceeded
                    DD int_fault ; Int06 #UD,  Undefined Instruction
                    DD int_nm    ; Int07 #NM,  Co-Processor failure
                    DD int_abort ; Int08 DF!,  Double-fault
                    DD int_fault ; Int09 #SO,  Co-Processor segment overrun
                    DD int_fault ; Int0a #TS,  Invalid TSS
                    DD int_fault ; Int0b #NP,  Segment not present
                    DD int_fault ; Int0c #SS,  Stack Segment Violation
                    DD int_gpf   ; Int0d #GP,  General Protection Violation
                    DD int_pgf   ; Int0e #PF,  Pagefault
                    DD 0         ;             Intel Reserved
                    DD int_fault ; Int10 #MF,  Math fault
                    DD int_fault ; Int11 #AC,  Alignment fault
                    DD int_abort ; Int12 MC!,  Machine Check / Hardware Failure
                    DD int_fault ; Int13 #XF,
TIMES 0x20-0x14     DD 0         ;             Intel Reserved
                    DD schedulertemp  ; Int20 IRQ0, PIT
                    DD int_pic1  ; Int21 IRQ1, KBC/RTC/PS2
                    DD int_pic1  ; Int22 IRQ2, unknown
                    DD int_pic1  ; Int23 IRQ3, COM2
                    DD int_pic1  ; Int24 IRQ4, COM1
                    DD int_pic1  ; Int25 IRQ5, IDE controller
                    DD int_pic1  ; Int26 IRQ6, FDC
                    DD int_pic1  ; Int27 IRQ7, LPT
                    DD int_pic2  ; Int28 IRQ8, RTC
                    DD int_pic2  ; Int29 IRQ9, redirect
                    DD int_pic2  ; Int2a IRQ10,unknown
                    DD int_pic2  ; Int2b IRQ11,unknown
                    DD int_pic2  ; Int2c IRQ12,mouse
                    DD int_pic2  ; Int2d IRQ13,Co-Processor
                    DD int_pic2  ; Int2e IRQ14,IDE controller
                    DD int_pic2  ; Int2f IRQ15,unknown
                    DD sys_int   ; Int30 SYS   System Callgate


                    ; Function: InitializeIO
                    ; Allocates and fills an IO handout table
                    ;
                    ; Only 16k ports can be accessed.
                    ; (Some ISA cards dont decode the top bits anyway)
                    ;
                    ; The first 2k ports are fine grained to one address (0x0000-0x0800).
                    ; The remaining 14k ports are handed out in doubleword accesses (4 consecutive addresses).
                    ; A short integer is used to count the amount of allocations for the given port.
                    ; This requires 1x 4k page for the low addresses and 2x 4k pages for the high addresses,
                    ; adding to 12k of memory (with some redundancy)
                    ;
                    ; PIT, PIC1 and PIC2 are claimed by default
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;
InitializeIO:       CALL AllocateMemoryLo           ; Allocate one 4k page. EAX=phys
                    MOV ESI, EAX                    ; ESI = physical address
                    MOV EDI, [lpgKernelHole]        ; EDI = virtual address
                    MOV [lpgIOMapOffset], EDI       ; Write the io map offset
                    MOV ECX, EDI                    ; ECX = io map base
                    CALL InsertPageGlobal           ; Map ESI to EDI. EAX EBX EDX destroyed
                    CALL AllocateMemoryLo           ; Allocate another 4k page. EAX=phys
                    ADD ESI, PAGE_SIZE              ; ESI = new physical address
                    ADD EDI, PAGE_SIZE              ; EDI = new virtual address
                    CALL InsertPageGlobal           ; Map ESI to EDI. EAX EBX EDX destroyed
                    CALL AllocateMemoryLo           ; Allocate another 4k page. EAX=phys
                    ADD ESI, PAGE_SIZE              ; ESI = new physical address
                    ADD EDI, PAGE_SIZE              ; EDI = new virtual address
                    CALL InsertPageGlobal           ; Map ESI to EDI. EAX EBX EDX destroyed
                    ADD EDI, PAGE_SIZE              ; EDI is next free address
                    MOV [lpgKernelHole], EDI        ; write new top-of-heap


                    MOV EDI, ECX                    ; load io map base
                    XOR EAX, EAX                    ; zero eax
                    MOV ECX, 3 * PAGE_SIZE / 4      ; amount of dwords to write
                    PUSH EDI                        ; store address for later
                    CLD                             ;
                    REP STOSD                       ; zero the table
                    POP EDI                         ; reload address

                    INC EAX                         ; EAX = 1
                    MOV [EDI + 2 * PIC1_COMMAND], AX; Claim PIC1.c
                    MOV [EDI + 2 * PIC1_DATA], AX   ; Claim PIC1.d
                    MOV [EDI + 2 * PIC2_COMMAND], AX; Claim PIC2.c
                    MOV [EDI + 2 * PIC2_DATA], AX   ; Claim PIC2.d
                    MOV [EDI + 2 * PIT_CHAN0], AX   ; Claim PIT.0
                    MOV [EDI + 2 * PIT_CHAN1], AX   ; Claim PIT.1
                    MOV [EDI + 2 * PIT_CHAN2], AX   ; Claim PIT.2
                    MOV [EDI + 2 * PIT_COMMAND], AX ; Claim PIT.cmd

                    RET                             ; done


                    ; Function: InitializeGDT
                    ; Sets up the GDT and loads it
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;
InitializeGDT:      MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0747
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    CALL AllocateMemoryLo
                    MOV ESI, EAX
                    MOV EDI, [lpgKernelHole]
                    MOV [lpgGdtOffset], EDI
                    CALL InsertPageGlobal

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0744
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    MOV ECX, (64/4) - 1
.activategdt_more:  PUSH ECX
                    ADD EDI, 4096
                    PUSH EDI
                    CALL AllocateMemoryLo
                    POP EDI
                    MOV ESI, EAX
                    CALL InsertPageGlobal
                    POP ECX
                    DEC ECX
                    JNZ .activategdt_more
                    ADD EDI, 4096
                    MOV [lpgKernelHole], EDI
                    MOV ECX, 64*1024/4
                    MOV EDI, [lpgGdtOffset]
                    XOR EAX, EAX
                    REP STOSD

                    MOV EDI, [lpgGdtOffset]
                    MOV word [EDI+1*8+GDT_LIMIT_LO], 0xffff
                    MOV byte [EDI+1*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                    MOV byte [EDI+1*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING0 | GDT_TYPE_CODE

                    MOV word [EDI+2*8+GDT_LIMIT_LO], 0xffff
                    MOV byte [EDI+2*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                    MOV byte [EDI+2*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING0 | GDT_TYPE_DATA  | GDT_TYPE_DATARW

                    MOV word [EDI+3*8+GDT_LIMIT_LO], 0xffff
                    MOV byte [EDI+3*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                    MOV byte [EDI+3*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_CODE

                    MOV word [EDI+4*8+GDT_LIMIT_LO], 0xffff
                    MOV byte [EDI+4*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                    MOV byte [EDI+4*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_DATA  | GDT_TYPE_DATARW

.trap:
                    LGDT [lpgGdtLimit]
                    JMP dword 0x08:.activategdt_swap
.activategdt_swap:  MOV EAX, 0x10
                    MOV DS, AX
                    MOV ES, AX
                    MOV FS, AX
                    MOV GS, AX
                    MOV SS, AX

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0754
                    ADD EBX, 2

                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET


                    ; Function: InitializeIPC
                    ; prepares the data structures needed for
                    ; inter-process communication
                    ;
                    ; right now that only involves allocating two pages and zeroing them
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-task registers
                    ;
InitializeIPC:      Call AllocateMemoryHi
                    MOV EDI, [lpgKernelHole]
                    MOV ESI, EAX
                    MOV [lpgTaskHandlers], EDI
                    Call InsertPageGlobal
                    ADD EDI, 4096
                    Call AllocateMemoryHi
                    MOV ESI, EAX
                    MOV [lpgTaskRoutes], EDI
                    Call InsertPageGlobal
                    ADD EDI, 4096
                    MOV [lpgKernelHole], EDI
                    
                    MOV EDI, [lpgTaskHandlers]
                    MOV ECX, 2 * PAGE_SIZE / 4
                    XOR EAX, EAX
                    REP STOSD

                    RET


                    ; Function: InitializePaging
                    ; constructs the page and allocation tables needed for a paged system
                    ;
                    ; fixed pages after the kernel:
                    ;    memory
                    ;    - (+0) Memory Allocation Directory
                    ;    - (+1) Memory Allocation Page (for the first 4mb)
                    ;    - (+2) Kernel Page Directory
                    ;    - (+3) Kernel Linear-to-physical map
                    ;    - (+4) Kernel Page Table (first 4mb)
                    ;    - (+5) Stack
                    ;    address space
                    ;    - (+0) Memory Allocation Directory
                    ;    - (+1) Memory Allocation Page (for the first 4mb)
                    ;    - (+2) Kernel Page Directory
                    ;    - (+3) Physical-linear mapping of entries
                    ;    - (+4) Kernel Page Table (first 4mb)
                    ;    - (+5) Stack
                    ;    - (+6) Page directory resolve area
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-stack registers
                    ;

InitializePaging:   LEA EAX, [kernelimageend]   ; end of image
                    AND EAX, 0xfffff000         ; page align
                    ADD EAX, 4096               ; set to next page boundary
                    MOV [lpgKernelEndPage], EAX ; write end of kernel

                    ; write addresses
                    MOV [lpgMemTable], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [.l_memtable1], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [lpgPagedir], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [lpgPagemap], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [.l_pagetable1], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [lpgKernelStack], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [lpgPageResolver], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [lpgKernelHole], EAX

                    ; empty memory
                    MOV ECX, 6 * PAGE_SIZE / 4  ; empty 6 pages, in groups of four (dword)
                    MOV EDI, [lpgKernelEndPage]
                    PUSH EAX
                    XOR EAX, EAX
                    CLD
                    REP STOSD
                    POP EAX

                    ; fill in memory table
                    MOV EDX, [lpgMemTable]      ; memory table
                    MOV EBX, [.l_memtable1]     ; second address
                    SHR EAX, 12                 ; get the end of the kernel in pages
                    MOV ECX, 1024               ; pages contained in table
                    SUB ECX, EAX                ; get remaining pages
                    OR EBX, ECX                 ; store the free memory count
                    MOV [EDX], EBX              ; write the page table entry

                    ; fill in page directories
                    MOV EDX, [.l_pagetable1]    ; get pagetable address
                    MOV EAX, [lpgPagedir]       ; get directory address
                    MOV EBX, [lpgPagemap]       ; get map address
                    OR EDX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USERSPACE
                                                ; mark both as currently in memory and accessible
                    MOV [EAX], EDX              ; write pagetable
                    MOV [EBX], EDX              ; write page directory

                    ; fill in page table
                    MOV EDI, [.l_pagetable1]
                    MOV EAX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_UNCACHEABLE
                    MOV ECX, 256
.l1                 STOSD
                    ADD EAX, PAGE_SIZE
                    DEC ECX
                    JNZ .l1
                    AND EAX, PAGE_ADDRMASK
                    OR EAX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_GLOBAL
                    MOV ECX, [lpgKernelEndPage]
                    SUB ECX, kernelimagestart
                    SHR ECX, 12
                    ADD ECX, 2
.l2                 STOSD
                    ADD EAX, PAGE_SIZE
                    DEC ECX
                    JNZ .l2
                    MOV ECX, 4
                    XOR EAX, PAGE_GLOBAL
.l3                 STOSD
                    ADD EAX, PAGE_SIZE
                    DEC ECX
                    JNZ .l3

                    ; fill in memory table
                    MOV EDI, [.l_memtable1]
                    MOV EAX, PAGE_SYS_UNDEF | PAGE_SYS_NONEALLOC
                    MOV ECX, 256
                    REP STOSD
                    MOV EAX, PAGE_SYS_MEMORY | PAGE_SYS_ONEALLOC
                    MOV ECX, [lpgKernelEndPage]
                    SUB ECX, kernelimagestart
                    SHR ECX, 12
                    ADD ECX, 6
                    REP STOSD
                    MOV EDX, [lpgKernelHole]
                    SHR EDX, 12
                    MOV ECX, 1024
                    SUB ECX, EDX
                    MOV EAX, PAGE_SYS_MEMORY | PAGE_SYS_NONEALLOC
                    REP STOSD

                    ; load control register
                    MOV EAX, [lpgPagedir]       ; get kernel pagetable entry
                    MOV CR3, EAX                ; set the paging entry
                    RET

                    ALIGN 4
.l_pagetable1       DD 0
.l_memtable1        DD 0


                    ; Function: InitializePIC
                    ; Brings the PIC into a known state
                    ;
                    ; maps the irq's 0-15 to the non-intel area
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroys:
                    ;     EAX, EBX
                    ;
InitializePIC:      MOV AL, PIC_ICW1_INIT | PIC_ICW1_ICW4
                    OUT PIC1_COMMAND, AL  ; program chip: PIC1
                    Call HardwareDelay

                    ; i have interleaved the instructions as the linux kernel seems to do it,
                    ; and so far this is broken on socket a systems

                    MOV AL, PIC_ICW1_INIT | PIC_ICW1_ICW4
                    OUT PIC2_COMMAND, AL  ; program chip: PIC2
                    Call HardwareDelay

                    MOV AL, 0x20          ; ICW 2 = base vector
                    OUT PIC1_DATA, AL
                    Call HardwareDelay

                    MOV AL, 0x28          ; ICW 2 = base vector
                    OUT PIC2_DATA, AL
                    Call HardwareDelay

                    MOV AL, 1 << 2        ; ICW 3 = slaves connected
                    OUT PIC1_DATA, AL
                    Call HardwareDelay

                    MOV AL, 0             ; ICW 3 = slaves connected
                    OUT PIC2_DATA, AL
                    Call HardwareDelay

                    MOV AL, PIC_ICW4_8086 ; ICW 4 = operation mode
                    OUT PIC1_DATA, AL
                    Call HardwareDelay

                    MOV AL, PIC_ICW4_8086 ; ICW 4 = operation mode
                    OUT PIC2_DATA, AL
                    Call HardwareDelay

                    MOV AL, 0x0           ; OCW 1 = mask
                    OUT PIC1_DATA, AL
                    Call HardwareDelay

                    MOV AL, 0x0           ; OCW 1 = mask
                    OUT PIC2_DATA, AL
                    Call HardwareDelay

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0750
                    ADD EBX, 2
                    MOV word [EBX], 0x0749
                    ADD EBX, 2
                    MOV word [EBX], 0x0743
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET

                    ; Function: InitializePIT
                    ; Brings the PIT into a known state
                    ;
                    ; programs channel 0 to a ~100hz rate generator
                    ; some bioses do not pre-load the pit to 18hz rate
                    ; making this step necessary to get a regular interrupt
                    ; issued on those systems
                    ;
                    ; todo: needs the exact needed frequency calculated
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroys:
                    ;     EAX, EBX
                    ;
InitializePIT:      MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0700 + 'P'    ; P: starting procedure
                    ADD EBX, 2

                    MOV AL, PIT_CW_CHAN0 | PIT_CW_LOHI | PIT_CW_RATE | PIT_CW_BINARY
                    OUT PIT_COMMAND, AL         ; write the mode to the PIT
                    Call HardwareDelay
                    MOV AX, 10000
                    OUT PIT_CHAN0, AL
                    Call HardwareDelay
                    MOV AL, AH
                    OUT PIT_CHAN0, AL
                    Call HardwareDelay

                    ; test: see if the amd XP likes this better
                    MOV AL,20h
                    OUT 20h,AL

                    ; test if the PIT works
                    MOV dword [lpgTickCount], 0
                    MOV word [EBX], 0x0700 + '0'    ; P0: entering test
.check1:            CMP dword [lpgTickCount], 0
                    JE .check1
                    MOV word [EBX], 0x0700 + '1'    ; P1: PIT fired once
.check2:            CMP dword [lpgTickCount], 1
                    JE .check2
                    MOV word [EBX], 0x0700 + '2'    ; P2: PIT fired twice
.check3:            CMP dword [lpgTickCount], 2
                    JE .check3
                    ; once we get here, the PIT is pretty much stable:

                    ; display completion
                    MOV word [EBX], 0x0700 + 'I'
                    ADD EBX, 2
                    MOV word [EBX], 0x0700 + 'T'    ; PIT: initialization ok
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET


                    ; Function: InitializeScheduler
                    ; prepares and starts the scheduler
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-task registers
                    ;
InitializeScheduler:MOV EBX, [lpgKernelTDT]
                    MOV EDX, [lpgPagedir]
                    MOV [scheduler_curtask], EBX
                    MOV [scheduler_curtask+4], EDX

                    Call AllocateMemoryHi
                    MOV EDI, [lpgKernelHole]
                    MOV ESI, EAX
                    MOV [scheduler_table], EDI
                    MOV [scheduler_offset], EDI
                    Call InsertPageGlobal
                    ADD EDI, 4096
                    MOV [lpgKernelHole], EDI
                    SUB EDI, 4096
                    MOV ECX, 4096/4
                    XOR EAX, EAX
                    REP STOSD
                    MOV EDX, [scheduler_table]
                    MOV EBX, [lpgKernelTDT]
                    MOV EAX, [lpgPagedir]
                    MOV [EDX], EBX
                    MOV [EDX+4], EAX
                    MOV [scheduler_curtask], EBX

                    MOV EAX, CR0
                    OR EAX, CR0_TS
                    MOV [scheduler_cr0_ts], EAX

                    MOV EAX, [lpgKernelHole]
                    MOV EDX, [EBX+TDT_PHYSADDR]
                    MOV [scheduler_fpusolv], EAX
                    ADD EAX, PAGE_SIZE
                    MOV [scheduler_curfpu], EDX
                    MOV [lpgKernelHole], EAX

                    ; take IRQ 0
                    MOV AL, 0x20
                    CALL AllocateInterrupt

                    ; fill in IRQ 0
                    MOV AL, 0x20
                    LEA EDI, [scheduler]
                    Call SetInterruptVector

                    ; create blocked threads section
                    Call AllocateMemoryHi
                    MOV EDI, [lpgKernelHole]
                    MOV ESI, EAX
                    MOV [scheduler_sleeptask], EDI
                    Call InsertPageGlobal
                    XOR EAX, EAX
                    MOV ECX, PAGE_SIZE / 4
                    CLD
                    REP STOSD
                    MOV [lpgKernelHole], EDI

                    RET


                    ; Function: InitializeTSS
                    ; build the kernel tss from scratch
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-task registers
                    ;
InitializeTSS:      MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0754
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    CALL AllocateMemoryHi            ; allocate memory
                    MOV ESI, EAX
                    MOV EDI, [lpgKernelHole]         ; get heap offset
                    MOV [lpgTSSaddress], EDI         ; store address
                    Call InsertPageGlobal            ; load the gdt into memory

                    MOV EDX, CR3
                    MOV [EDI+TSS_CR3], EDX           ; this one cost me over one hour to figure:
                                                     ; a task switch does not store LDT or CR3
                                                     ; and a few more - thus on return from scheduler
                                                     ; the page directory becomes corrupted

                    MOV DX, SS
                    MOV [EDI+TSS_SS0], DX
                    MOV [EDI+TSS_SS0+2], DX
                    MOV word [EDI+TSS_IOBASE], 2048

                    ADD EDI, PAGE_SIZE
                    MOV [lpgKernelHole], EDI

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0753
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX

                    XOR EAX, EAX                    ; EAX = 0
                    SUB EDI, 2048                   ; EDI = pointer to iopl bitmap
                    DEC EAX                         ; EAX = 0xffffffff
                    MOV ECX, 2048 / 4               ; ECX = dwords to write
                    CLD                             ;
                    REP STOSD                       ; fill iopl with ones.
                                                    ; EDI = lpgKernelHole

                    SUB EDI, PAGE_SIZE

                    CALL AllocateGDTTSS              ; allocate apropriate descriptor
                    MOV [lpgTSSDescriptor], AX       ; store descriptor
                    LTR AX                           ; load kernel task

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0753
                    ADD EBX, 2
                    MOV word [EBX], 0x0720
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET

                    ; Function: InitializeTDT
                    ; Builds a Task Description Table for the kernel
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     all non-task registers
                    ;
InitializeTDT:      Call AllocateMemoryHi
                    MOV EDI, [lpgKernelHole]
                    MOV ESI, EAX
                    MOV [lpgKernelTDT], EDI
                    CALL InsertPage
                    MOV EDX, [lpgKernelStack]
                    MOV EBX, [lpgPagedir]
                    MOV [EDI+TDT_KERNELSTACK], EDX
                    MOV [EDI+TDT_PHYSADDR], ESI
                    ADD EDI, 4096
                    MOV [lpgKernelHole], EDI

                    MOV EBX, [lpgTextPtr]
                    MOV word [EBX], 0x0700 + 'T'
                    ADD EBX, 2
                    MOV [lpgTextPtr], EBX
                    RET

                    ; Function: InsertPage
                    ; Loads a page table with a given entry
                    ;
                    ; in:
                    ;     - ESI = page to address
                    ;     - EDI = location to insert to
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
InsertPage:         PUSH ESI
                    OR ESI, PAGE_PRESENT | PAGE_WRITABLE
                    CALL SetPageEntry
                    POP ESI
                    RET

                    
                    ; Function: InsertPageGlobal
                    ; loads a page table with a given entry, marking it global
                    ;
                    ; in:
                    ;     - ESI = page to address
                    ;     - EDI = location to insert to
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
InsertPageGlobal:   PUSH ESI
                    OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGE_GLOBAL
                    CALL SetPageEntry
                    POP ESI
                    RET


                    ; Function: InsertPageUser
                    ; loads a page table with a given entry, marking it for userspace
                    ;
                    ; in:
                    ;     - ESI = page to address
                    ;     - EDI = location to insert to
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
InsertPageUser:     PUSH ESI
                    OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USERSPACE
                    CALL SetPageEntry
                    POP ESI
                    RET


                    ; Function: InsertPageDriver
                    ; loads a page table with a given entry, marking it both uncacheable and for userspace
                    ;
                    ; in:
                    ;     - ESI = page to address
                    ;     - EDI = location to insert to
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
InsertPageDriver:   PUSH ESI
                    OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USERSPACE | PAGE_UNCACHEABLE
                    CALL SetPageEntry
                    POP ESI
                    RET


                    ; Function: InvalidatePage
                    ; flushes a page from the TLB.
                    ; On 386's it has to flush the entire TLB
                    ;
                    ; patched by <ProcessorTLBPatch>
                    ;
                    ; in:
                    ;     - EDI = address to be invalidated
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
InvalidatePage:     PUSH EDI                         ; we should retain EDI
                    MOV EDI, CR3                     ; pop page directory out
                    MOV CR3, EDI                     ; and back in to flush the TLB
                    POP EDI                          ; restore EDI
                    RET                              ; and return


                    ; Function: InstallPatches
                    ; checks and runs the patches
InstallPatches:     MOV EAX, [lpgProcessorCaps]
                    AND EAX, CPUID_FXSR
                    JZ .skip1
                    Call ProcessorSIMDpatch
.skip1              RET

; Todo: ProcessorCOMApatch
; for fixing the cyrix 6x86 coma bug


                    ; Function: ProcessorF00Fpatch
                    ; for installing the workaround for the intel LOCK CMPXCHG8B bug
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
ProcessorF00Fpatch: PUSHAD
                    MOV EDI, [lpgIdtOffset]
                    CALL GetPageEntry
                    OR ESI, PAGE_WRITETHROUGH
                    CALL SetPageEntry
                    POPAD


                    ; Function: ProcessorNoFPpatch
                    ; changes the #NM interrupt handler to fault (using self-modifying code)
                    ; instead of switching fpu contexts
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
ProcessorNoFPpatch: PUSH ESI
                    PUSH EDI
                    PUSH ECX
                    MOV ECX, .patchend - .patchcode
                    MOV ESI, .patchcode
                    MOV EDI, int_nm
                    REP MOVSB
                    POP ECX
                    POP EDI
                    POP ESI
                    RET
.patchcode:         PUSH 0
                    LEA EAX, [int_oops]
                    JMP EAX
.patchend:


                    ; ProcessorSIMDpatch
                    ; changes the #NM interrupt handler to save SSE context as well.
                    ; configures the PC to allow SSE instructions
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
ProcessorSIMDpatch: PUSH ESI
                    PUSH EDI
                    PUSH ECX
                    MOV ECX, .patchend - .patchcode ; patch #nm handler
                    MOV ESI, .patchcode             ; with patchcode
                    MOV EDI, int_nm.smclocate       ; to patch area
                    REP MOVSB                       ; apply patch

                    MOV ECX, CR4                    ; patch processor config
                    OR ECX, CR4_OSFXSR              ; enable OS Fast save/restore support
                    MOV CR4, ECX                    ; write CR4
                    ; todo: write SIMD exception handler
                    POP ECX
                    POP EDI
                    POP ESI
                    RET

                    CPU 686
.patchcode:         FXSAVE  [EDI+TDT_FPUSTATE]      ; save state
                    FXRSTOR [ECX+TDT_FPUSTATE]      ; load state
                    STI                             ; enable ints again
                    POPAD                           ; restore GPRS
                    IRET                            ; resume execution
.patchend:          CPU 386

                    ; Function: ProcessorSYSEpatch
                    ; enables the use of SYSENTER/SYSEXIT instructions
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
ProcessorSYSEpatch: PUSH EAX
                    PUSH ECX
                    PUSH EDX

                    XOR EDX, EDX

                    CPU 686
                        LEA EAX, [sys_sysenter]
                        MOV ECX, MSR_SYSENTER_EIP
                        WRMSR

                        MOV EAX, [lpgKernelStack]
                        ADD EAX, PAGE_SIZE-4
                        MOV ECX, MSR_SYSENTER_ESP
                        WRMSR

                        MOV EAX, 0x08
                        MOV ECX, MSR_SYSENTER_CS
                        WRMSR
                    CPU 386

                    POP EDX
                    POP ECX
                    POP EAX
                    RET
                    ; todo: Test SYSENTER/SYSEXIT - bochs has no clue


; Todo: ProcessorSYSCpatch
; for allowing SYSCALL/SYSRET


                    ; Function: ProcessorTLBpatch
                    ; replaces InvalidatePage with a more efficient version
                    ;
                    ; patch highly recommended on 486 or better
                    ; patch breaks 386s
                    ;
                    ; in:
                    ;     none
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
ProcessorTLBpatch:  PUSH ESI
                    PUSH EDI
                    PUSH ECX
                    LEA EDI, [InvalidatePage]
                    LEA ESI, [.alternative]
                    MOV ECX, .alternativeend - .alternative
                    REP MOVSB
                    POP ECX
                    POP EDI
                    POP ESI
                    RET
                    ; this piece is shorter and much faster
                    ; will replace InvalidatePage on 486+ using smc
.alternative:       CPU 486                          ; 486 or better required
                    INVLPG [EDI]                     ; flush page
                    RET                              ; and return
.alternativeend:    CPU 386                          ; end of replacement stub




                    ; Function: SetInterruptEntry
                    ; sets and IDT entry
                    ;
                    ; in:
                    ;     - AL = interrupt number
                    ;     - EBX = hi dword
                    ;     - EDX = lo dword
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
                    ;
SetInterruptEntry:  PUSH EDI
                    PUSH EAX
                    MOV EDI, [lpgIdtOffset]
                    AND EAX, 0xff
                    SHL EAX, 3
                    LEA EDI, [EDI+EAX]
                    CLI
                    MOV [EDI], EDX
                    MOV [EDI+4], EBX
                    STI
                    POP EAX
                    POP EDI
                    RET


                    ; Function: SetInterruptVector
                    ; sets the vector for interrupt handling
                    ;
                    ; in:
                    ;     - AL = interrupt number
                    ;     - EDI = location of interrupt handler
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     unknown
                    ;
SetInterruptVector: PUSH EBX
                    PUSH EDX
                    MOV EBX, EDI
                    MOV EDX, EDI
                    AND EBX, 0xffff0000
                    AND EDX, 0x0000ffff
                    OR BH, GDT_FLAG_PRESENT | GDT_FLAG_RING0 | IDT_TYPE_INT32
                    OR EDX, 0x00080000
                    Call SetInterruptEntry
                    POP EDX
                    POP EBX
                    RET

                    ; Function: SetPageEntry
                    ; Loads a page table with a given entry
                    ;
                    ; in:
                    ;     - ESI = page table entry
                    ;     - EDI = location to insert to
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     EAX EBX EDX
                    ;
SetPageEntry:       MOV EBX, [lpgPagemap]
                    MOV EAX, EDI
                    SHR EAX, 22
                    SHL EAX, 2
                    MOV EDX, [EBX+EAX]
                    AND EDX, PAGE_ADDRMASK
                    JZ .checkpagedir
.continue:          MOV EBX, EDI
                    AND EBX, PAGE_TABLEMASK
                    SHR EBX, 10
                    MOV EAX, ESI
                    OR EAX, PAGE_PRESENT | PAGE_WRITABLE
                    MOV [EDX+EBX], EAX
                    RET
.checkpagedir:      MOV EBX, [lpgPagedir]
                    MOV EDX, [EBX+EAX]
                    AND EDX, PAGE_ADDRMASK
                    JNZ .maptable
.requirenew:        PUSH EAX
                    CALL AllocateMemoryHi                   ; request a new page of memory
                    MOV EDX, EAX
                    POP EAX
                    OR EDX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USERSPACE
                    PUSHAD
.mutex2:            LOCK BTR word [semPageresolve], 0
                    JNC .mutex2
                    MOV ESI, EDX
                    MOV EDI, [lpgPageResolver]
                    Call InvalidatePage
                    Call InsertPage
                    MOV ECX, PAGE_SIZE / 4
                    XOR EAX, EAX
                    CLD
                    REP STOSD
                    LOCK BTS word [semPageresolve], 0
                    POPAD

                    MOV [EBX+EAX], EDX
.maptable           PUSH ESI
                    PUSH EDI
                    MOV ESI, EDX
                    AND ESI, PAGE_ADDRMASK
                    MOV EDI, [lpgPageResolver]
.mutex:             LOCK BTR word [semPageresolve], 0
                    JNC .mutex
                    Call InvalidatePage
                    CALL InsertPage
                    MOV EDX, EDI
                    POP EDI
                    POP ESI
                    MOV EBX, EDI
                    AND EBX, PAGE_TABLEMASK
                    SHR EBX, 10
                    MOV EAX, ESI
                    OR EAX, PAGE_PRESENT | PAGE_WRITABLE
                    MOV [EDX+EBX], EAX
                    LOCK BTS word [semPageresolve], 0
                    RET


                    ; Function: SetTaskRegisters
                    ; set GPRs for a given task
                    ;
                    ; in:
                    ;     - EDI = TDT to modify, must not be scheduled
                    ;     - EAX = task's new EAX
                    ;     - EBX = task's new EBX
                    ;     - ECX = task's new ECX
                    ;     - EDX = task's new EDX
                    ;     - ESI = task's new EBP
                    ;
                    ; out:
                    ;     none
                    ;
                    ; destroyed:
                    ;     none
SetTaskRegisters:   PUSH EDI
                    MOV EDI, [EDI+TDT_KERNELSTACK]
                    MOV [EDI+7*4], ESI
                    MOV [EDI+8*4], EBX
                    MOV [EDI+9*4], EDX
                    MOV [EDI+10*4], ECX
                    MOV [EDI+11*4], EAX
                    POP EDI
                    RET

; --------------------------------------------------------------------------------------------------
; Group: Interrupt Handlers
; --------------------------------------------------------------------------------------------------


schedulertemp:      PUSH EBX
                    PUSH EAX
                    XOR EAX, EAX
                    MOV al, [scheduler_index]
                    INC AL
                    AND AL, 0x3
                    MOV [scheduler_index], AL
                    MOV BL, [scheduler_chars+EAX]
                    MOV BH, 0x07
                    MOV [0xB8000 + 160 * 24 + 79 * 2], BX
                    INC dword [lpgTickCount]
                    MOV AL, PIC_EOI
                    OUT PIC1_COMMAND, AL
                    POP EAX
                    POP EBX
                    IRETD


scheduler:          PUSHAD
                    PUSH DS
                    PUSH ES
                    PUSH FS 
                    PUSH GS

                    ; task switch
                    MOV EBX, [scheduler_offset]          ; load current task
                    MOV EAX, [scheduler_curtask]         ; load current TDT
                    ADD EBX, 8                           ; get next tasks address
                    MOV [EAX+TDT_KERNELSTACK], ESP       ; store ESP
                    MOV EDX, [EBX]                       ; load next task
                    CMP EDX, 0                           ; if we are at the end of the list
                    JNE .nowrap                          ; no wraparound
                    MOV EBX, [scheduler_table]           ; jump to start of list
                    MOV EDX, [EBX]                       ; load next task
.nowrap:            MOV [scheduler_offset], EBX          ; set next task pointer
                    MOV [scheduler_curtask], EDX         ; write current task
                    MOV EDI, [lpgTSSaddress]             ; get tss address
                    MOV ECX, [EDX+TDT_KERNELSTACKTOP]    ; get kernel stack start
                    MOV ESP, [EDX+TDT_KERNELSTACK]       ; get kernel stack address
                    MOV [EDI + TSS_ESP0], ECX            ; write ESP0

                    ; loopy counter
                    XOR EBX, EBX                         ; empty EBX
                    MOV BL, [scheduler_index]            ; load current counter
                    INC BL                               ; add one to bl
                    AND BL, 0x3                          ; and wrap around 4
                    MOV [scheduler_index], BL            ; write new counter
                    MOV AL, [scheduler_chars+EBX]        ; get character
                    MOV AH, 0x07                         ; set color
                    MOV [0xB8000 + 24 * 160 + 158], AX   ; display character on display

                    ; other functions
                    MOV EDX, [scheduler_cr0_ts]          ; get default CR0 (save a few clocks)
                    INC dword [scheduler_ticks]          ; increase amount of ticks
                    MOV CR0, EDX                         ; Write CR0 with TS set

                    ; EOI to PIC
                    MOV AL, PIC_EOI
                    OUT PIC1_COMMAND, AL

                    POP GS
                    POP FS
                    POP ES
                    POP DS
                    POPAD
                    IRETD


                    ; exception handlers
int_pgf:            MOV dword [scheduler_chars], 'PGF!'
                    JMP int_oops
int_gpf:            MOV dword [scheduler_chars], 'SGV!'
                    JMP int_oops
int_fault:          MOV dword [scheduler_chars], '!-!-'
                    PUSH 0
                    JMP int_oops

                    ; todo: write proper abort handlers
int_abort:          PUSH -1
                    JMP int_oops
                    HLT
                    JMP $

                    ; dump registers
int_oops:           PUSH EAX
                    PUSH EBX
                    MOV EBX, 0xB8000 + 4
                    MOV EAX, ESI
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17491753

                    ADD EBX, 160
                    MOV EAX, EDI
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17491744

                    ADD EBX, 160
                    MOV EAX, ESP
                    ADD EAX, 8
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17501753

                    ADD EBX, 160
                    MOV EAX, EBP
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17501742

                    ADD EBX, 160
                    MOV EAX, ECX
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17581743

                    ADD EBX, 160
                    MOV EAX, EDX
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17581744

                    POP EBX
                    POP EAX

                    MOV EDX, EAX
                    MOV ECX, EBX

                    MOV EBX, 0xB8000 + 4 + 6 * 160
                    MOV EAX, EDX
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17581741

                    ADD EBX, 160
                    MOV EAX, ECX
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17581742

                    ADD EBX, 320
                    MOV EAX, [ESP]
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17431745

                    ADD EBX, 160
                    MOV EAX, [ESP+4]
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17501749

                    ADD EBX, 160
                    MOV EAX, [ESP+8]
                    CALL int_printhex
                    MOV dword [EBX-4], 0x17531743

.halt               HLT
                    JMP .halt

                    ; print a 32-bit value
int_printhex:       PUSH ECX
                    PUSH EDX
                    MOV ECX, 8
.loop:              XOR EDX, EDX
                    MOV DL, AL
                    AND DL, 0xF
                    MOV DL, [.hexchars+EDX]
                    MOV DH, 0x17
                    MOV [EBX+2*ECX], DX
                    SHR EAX, 4
                    LOOP .loop
                    POP EDX
                    POP ECX
                    RET

.hexchars       DB  "0123456789ABCDEF"


                    ; Function: int_nm
                    ; handles #NM and switches context for FPU state
                    ;
                    ; Patched by <processorNoFPpatch>
                    ;
                    ; *interrupt handler*
                    ;
                    CPU 386 FPU
int_nm:             PUSHAD                          ; interrupt, push GPRs
                    MOV ESI, [scheduler_curfpu]     ; get last TDTs physical address
                    MOV EDI, [scheduler_fpusolv]    ; get local address to map
                    MOV ECX, [scheduler_curtask]    ; get current task info
                    CALL InsertPage                 ; map old TDT to reserved area
                    CALL InvalidatePage             ; flush (possibly present) old page
                    MOV EAX, [ECX+TDT_PHYSADDR]     ; get physical address
                    CLI                             ; no INTS, really
                    CLTS                            ; clear TS flag
                    MOV [scheduler_curfpu], EAX     ; set new fpu owner
.smclocate:         FNSAVE [EDI+TDT_FPUSTATE]       ; save state
                    FRSTOR [ECX+TDT_FPUSTATE]       ; load state
                    STI                             ; enable ints again
                    POPAD                           ; restore GPRS
                    IRET                            ; resume execution

                    NOP                             ; two extra bytes so the SIMD patch
                    NOP                             ; can replace FNSAVE/FRSTOR with FXSAVE/FXRSTOR
                    CPU 386
                    

                    ; todo: fninit bug in 486 and Pentium I series
                    ; no-wait instructions can cause an exception in compatibility mode (NE=0)
                    ; not sure wether we need this as native exceptions seem to be supported in 386+
                    ; (it requires mobo circuitry though, might need testing)

int_pic1:           PUSH EAX
                    INC dword [lpgTickCount]
                    MOV AL, PIC_EOI
                    OUT PIC1_COMMAND, AL
                    POP EAX
                    IRETD
                    JMP int_pic1

int_pic2:           PUSH EAX
                    MOV AL, PIC_EOI
                    OUT PIC2_COMMAND, AL
                    MOV AL, PIC_EOI
                    OUT PIC1_COMMAND, AL
                    POP EAX
                    IRETD
                    JMP int_pic2


sys_int:            CALL sys_main
                    JC .setcf
                    AND dword [ESP+8], 0xFFFFFFFF - EFLAGS_CF
                    IRETD
.setcf              OR dword [ESP+8], EFLAGS_CF
                    IRETD

sys_sysenter:       PUSH ECX
                    PUSH EDX
                    CALL sys_main
                    POP ECX
                    POP EDX
                    CPU 686
                    SYSEXIT
                    CPU 386


sys_main:           OR AX, AX
                    JZ sys_entrypoint
                    STC
                    RET

; Todo: thread status checking

                    ; This is the os main system call handler.
                    ; in EAX/LO = task, an 0 meaning kernel
                    ; in EAX/HI = function number
sys_missingcall     STC
                    RET

sys_entrypoint:     SHR EAX, 16
                    CMP EAX, syscalllimit
                    JGE sys_missingcall
                    JMP sys_docall

                    ;PUSH EAX
                    ;MOV AL, [sys_checkprivileges + EAX]
                    ;AND EAX, 0xff
                    ;JMP [sys_privilegeswitch + 4 * EAX]

sys_docall:         ;POP EAX
                    JMP [sys_kernelswitch + 4 * EAX]

sys_privilegeuser:  ;MOV EAX, TSS_SYS_LEVEL_U
                    ;JMP sys_privilegecheck
sys_privilegeshell: ;MOV EAX, TSS_SYS_LEVEL_M
                    ;JMP sys_privilegecheck
sys_privilegedriver:;MOV EAX, TSS_SYS_LEVEL_D
                    ;JMP sys_privilegecheck
sys_privilegekernel:;MOV EAX, TSS_SYS_LEVEL_K
sys_privilegecheck: ;PUSH EDX
;                    PUSH EAX
;                    ;CALL FindCurrentTSS
;                    MOV EDX, EAX
;                    POP EAX
;                    MOV AL, [EDX+EAX]
;                    POP EDX
;                    CMP AL, PRIVILEGE_ALLOW
;                    JE sys_docall
;                    CMP AL, PRIVILEGE_DISALLOW
;                    JE sys_missingcall
                    ; todo: kernel call monitoring
;                    JMP sys_missingcall

; --------------------------------------------------------------------------------------------------
; Group: Scheduler
; --------------------------------------------------------------------------------------------------

                    ; Variable: scheduler_table
                    ; points to the schedule tables for each processor
scheduler_table:    TIMES 8 DD 0

                    ; Variable: scheduler_offset
                    ; current offsets into the task table
scheduler_offset:   TIMES 8 DD 0

                    ; Variable: scheduler_curtask
                    ; contains the current threads for each processor
                    ; in TDT/CR3
scheduler_curtask:  TIMES 8 * 2 DD 0

                    ; Variable: scheduler_curfpu
                    ; Contains the physical address for the TDT
                    ; currently using the FPU state
scheduler_curfpu:   TIMES 8 DD 0

                    ; Variable: scheduler_fpusolv
                    ; contains the address of a page that is to contain
                    ; the TDT when mapped into memory
scheduler_fpusolv:  TIMES 8 DD 0

                    ; Variable: scheduler_sleeptask
                    ; Points to a page containing not-scheduled (blocked) tasks
scheduler_sleeptask:DD 0

                    ; Variable: scheduler_ticks
                    ; contains the amount of ticks generated since system startup
scheduler_ticks:    DD 0

                    ; Variable: scheduler_cr0_ts
                    ; local copy of CR0 with TS set
scheduler_cr0_ts:   DD 0

scheduler_index:    DB 0
scheduler_chars:    DB "|/-\"

; --------------------------------------------------------------------------------------------------
; Group: System calls
; --------------------------------------------------------------------------------------------------



                    ; Function: KernelVersion
                    ; Returns the kernel version
                    ;
                    ; Privilege level: 
                    ;     Informational
                    ;
                    ; in:
                    ;     - EAX = 0x00000000
                    ;
                    ; out:
                    ;     - EAX = version: bits 0-15 = revision, 16-23 = minor, 24-31 = major
                    ;
                    ; destroyed:
                    ;     none
                    ;
SYSPL_KERNELVERSION EQU PRIVILEGE_INFO
KernelVersion:      MOV EAX, 0x01030001
                    RET

                    ; Function: MemMap
                    ; Maps a certain region in physical space to a region in kernel space
                    ;
                    ; Use with caution as this function does not perform any sanity checks
                    ;
                    ; Privilege level:
                    ;     Kernel
                    ;
                    ; in:
                    ;     - EAX = 0x00010000
                    ;     - EBX = amount of pages to map
                    ;     - ESI = starting page in physical memory
                    ;     - EDI = starting page in current address space
                    ;
                    ; out:
                    ;     - EBX = 0
                    ;
                    ; destroyed:
                    ;     EAX EDX
                    ;
SYSPL_MEMMAP        EQU PRIVILEGE_KERNEL
MemMap:             OR EBX, EBX
                    JZ .b00ring
                    PUSH ESI
                    PUSH EDI
                    AND ESI, PAGE_ADDRMASK
                    AND EDI, PAGE_ADDRMASK
.more:              PUSH EBX
                    Call InsertPageDriver
                    Call InvalidatePage
                    POP EBX
                    ADD ESI, PAGE_SIZE
                    ADD EDI, PAGE_SIZE
                    DEC EBX
                    JNZ .more
                    POP EDI
                    POP ESI
.b00ring:           RET

                    ; Function: BlockAlloc
                    ; Allocates memory and maps it to userspace
                    ;
                    ; Returns the amount of memory actually mapped
                    ;
                    ; Privilege level:
                    ;     User
                    ;
                    ; in:
                    ;     - EAX = 0x00020000
                    ;     - EBX = amount of memory to map (bytes)
                    ;     - EDI = starting page in current address space
                    ;
                    ; out:
                    ;     - EBX = amount of memory actually mapped
                    ;
                    ; destroyed:
                    ;     EAX EDX
                    ;
SYSPL_BLOCKALLOC    EQU PRIVILEGE_USER
BlockAlloc:         OR EBX, 0
                    JZ .b00ring
                    PUSH EDI
                    PUSH ESI
                    XOR EAX, EAX
.loop:              PUSH EAX
                    PUSH EBX
                    Call AllocateMemoryHi
                    OR AL, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USERSPACE
                    PUSH EAX
                    Call GetPageEntry   ; check if somebody beat us here
                    OR ESI, ESI
                    JNZ .bail           ; oops, memory was already allocated
                    POP ESI
                    Call SetPageEntry
                    POP EBX
                    POP EAX
                    ADD EAX, PAGE_SIZE
                    ADD EDI, PAGE_SIZE
                    SUB EBX, PAGE_SIZE
                    JNLE .loop
                    MOV EBX, EAX
                    POP ESI
                    POP EDI
                    CLC
                    RET
.b00ring:           CLC
                    RET
.bail:              POP ESI
                    POP EBX
                    POP EAX
                    MOV EBX, EAX
                    POP ESI
                    POP EDI
                    CLC
                    RET
; Fixme: race condition on address space allocation
; pagetable r/w should be altered so that we can allocate pages for use.
; Right now its either checked inconsistently or blindly written



                    ; Function: GetTimerTicks
                    ; Returns the amount of interrupts the kernel timer has generated
                    ;
                    ; Privilege leel:
                    ;     User
                    ;
                    ; in:
                    ;     EAX = 0x00030000
                    ;
                    ; out:
                    ;     EAX = number of ticks occurred
                    ;
                    ; destroyed:
                    ;     none
                    ;
SYSPL_GETTIMERTICKS EQU PRIVILEGE_USER
GetTimerTicks:      MOV EAX, [scheduler_ticks]
                    RET

                    ; Function: PortAlloc
                    ; Requests access to a range of ports
                    ;
                    ; Privilege Level:
                    ;    Driver
                    ;
                    ; in:
                    ;     EAX - 0x00040000
                    ;     EDI - First port to be opened
                    ;     EBX - Amount of ports to be opened
                    ;
                    ; out:
                    ;     EAX - Amount of ports allocated. Success if EAX >= EBX
                    ;
                    ; destroyed:
                    ;     unknown
                    ;
SYSPL_PORTALLOC     EQU PRIVILEGE_DRIVER
PortAlloc:          XOR EAX, EAX            ; EAX = 0
                    OR EBX, EBX             ; compare ebx to zero
                    JE .b00ring             ; b00ring
.dostuff:           PUSH EDX                ; Save EDX
                    PUSH ESI                ; Save ESI
                    PUSH ECX                ; Save ECX
                    MOV ESI, [lpgTSSaddress]; ESI = TSS base
                    ADD ESI, 2048           ; ESI = IOPL base
                    MOV EDX, EDI            ; EDX = loop counter
.loop:              CMP EDX, 2048           ; compare for range
                    JL .lowrange            ; 0-2047 is the low range
.hirange:           CMP EDX, 16 * 1024      ; 2048-16383 is the mid range
                    JGE .done               ; we wont grant port 16384+
                    CALL AllocatePortHi     ; Allocate a port in the high range
                    JC .done                ; bail on error
                    
                    PUSH EBX
                    PUSH EAX
                    MOV EBX, EDX
                    MOV ECX, EDX
                    SHR EBX, 3
                    AND CL, 4
                    MOV AL, 0xf
                    ADD ESI, EBX
                    SHL EAX, CL
                    XOR [ESI], AL
                    SUB ESI, EBX
                    POP EAX
                    POP EBX

                    ADD EDX, 4              ; next port is 4 away
                    ADD EAX, 4              ; and we opened 4 ports
                    JMP .next               ; next port
.lowrange:          CALL AllocatePortLo     ; Allocate a port in the low range
                    JC .done                ; bail on error

                    PUSH EBX
                    PUSH EAX
                    MOV EBX, EDX
                    MOV ECX, EDX
                    SHR EBX, 3
                    AND CL, 7
                    MOV AL, 1
                    ADD ESI, EBX
                    SHL EAX, CL
                    XOR [ESI], AL
                    SUB ESI, EBX
                    POP EAX
                    POP EBX

                    INC EAX                 ; next port is one away
                    INC EDX                 ; and we opened one
.next:              CMP EAX, EBX            ; compare opened ports to total ports
                    JL .loop                ; loop if not done
.done:              POP ECX                 ; Restore ECX
                    POP ESI                 ; Restore ESI
                    POP EDX                 ; Restore EDX
.b00ring            RET                     ; and return

            
                    ; Function: GetCpuInfo
                    ; Returns processor name, version and capabilities,
                    ; independent of CPUID and known cpu bugs.
                    ;
                    ; Data is generated by <GetProcessorInfo>
                    ;
                    ; Privilege Level:
                    ;     Informational
                    ;
                    ; in:
                    ;     EAX - 0x00050000
                    ;     EBX - CPU number (zero based)
                    ;
                    ; out:
                    ;     EBX - name 0-3
                    ;     EDX - name 4-7
                    ;     ECX - name 8-11
                    ;     EAX - first set of capabilities
                    ;     ESI - second set of capabilities
                    ;     EDI - packed processor version
                    ;           24-31: model shortcut
                    ;           15-12: type
                    ;           11-08: family
                    ;           07-04: model
                    ;           03-00: revision
                    ;     CF  - clear if present, set if not present
                    ;           if set, the rest of the output is not changed
                    ;
                    ; destroyed:
                    ;     none
                    ;
SYSPL_GETCPUINFO    EQU PRIVILEGE_INFO
GetCpuInfo:         OR EBX, EBX
                    JZ .fill
                    STC
                    RET
.fill:              MOV EBX, [lpgProcessorName]
                    MOV EDX, [lpgProcessorName+4]
                    MOV ECX, [lpgProcessorName+8]
                    MOV EAX, [lpgProcessorCaps]
                    MOV ESI, [lpgProcessorCaps+4]
                    MOV EDI, [lpgProcessorType]
                    CLC
                    RET


                    ; Function: CreateThread
                    ; Creates, Allocates and starts a new thread
                    ;
                    ; Privilege Level:
                    ;     User level
                    ;
                    ; in:
                    ;     EAX - 0x000
                    ;     EBX - Address of the thread to be started
                    ;     EDI - Linear Address where the information is to be stored60000
                    ;
                    ; out:
                    ;     EAX - amount of bytes allocated
                    ;     CF  - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     ECX ESI
                    ;
SYSPL_CREATETHREAD  EQU PRIVILEGE_USER
CreateThread:       ; Todo: fix race conditions by pre-allocating the memory
                    PUSH EDI
                    Call AllocateTaskset                ; EDI = task handle
                    MOV ESI, CR3                        ; ESI = CR3
                    XOR EAX, EAX                        ; EAX = 0
                    Call AddThread                      ; Assign TDT/CR3 to processor #0
                    POP EDI
                    MOV EAX, PAGE_SIZE * 2
                    RET


                    ; Function: RouteAlloc
                    ; Requests a name to be bound to a port
                    ;
                    ; Privilege Level:
                    ;     Driver
                    ;
                    ; in:
                    ;     EAX - 0x00070000
                    ;     EBX - Route to be allocated
                    ;     EDI - Port to point at
                    ;
                    ; out:
                    ;     CF - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     EAX EDX
                    ;
SYSPL_ROUTEALLOC:   EQU PRIVILEGE_DRIVER
RouteAlloc:         MOV EDX, EBX
                    MOV EAX, EDI
                    Call AddRoute
                    RET


                    ; Function: RouteFind
                    ; Requests to find a gate
                    ;
                    ; Privilege Level:
                    ;     User
                    ;
                    ; in:
                    ;     EAX - 0x00080000
                    ;     EBX - Name to check
                    ;
                    ; out:
                    ;     EAX - gate associated with the requested name, or 0 if it wasnt found
                    ;
                    ; destroyed:
                    ;     EDX
                    ;
SYSPL_ROUTEFIND:    EQU PRIVILEGE_USER
RouteFind:          MOV EDX, EBX
                    Call GetRoute
                    RET


                    ; Function: GateAlloc
                    ; Requests a new gate to be given
                    ;
                    ; Privilege Level:
                    ;     User
                    ;
                    ; in:
                    ;     EAX - 0x00090000
                    ;     EDI - Pointer to the handler function
                    ;
                    ; out:
                    ;     EAX - number of the acquired gate
                    ;
                    ; destroyed:
                    ;     ECX EDX
                    ;
SYSPL_GATEALLOC:    EQU PRIVILEGE_USER
GateAlloc:          PUSH EBX
                    MOV EBX, CR3
                    Call AddGate
                    MOV EAX, EDX
                    POP EBX
                    RET



                    ; Function: GateLookup
                    ; Looks up the information of the handler behind a gate
                    ;
                    ; Privilege Level:
                    ;     User
                    ;
                    ; in:
                    ;     EAX - 0x000A0000
                    ;     EBX - gate number
                    ;
                    ; out:
                    ;     EAX - Address Space Handle of the handler
                    ;     EBX - Pointer of the handler function
                    ;
                    ; destroyed:
                    ;     EDX
                    ;
SYSPL_GATELOOKUP:   EQU PRIVILEGE_USER
GateLookup:         MOV EAX, EBX
                    AND EAX, 0xff8
                    CMP EAX, EBX
                    JNE .abort
                    Call GetGateAddress
                    MOV EAX, EDX
                    RET
.abort:             XOR EAX, EAX
                    XOR EBX, EBX
                    RET
                    
                    
                    ; Function: CreateV8086Thread
                    ; Creates, Allocates and starts a new thread
                    ;
                    ; Privilege Level:
                    ;     User level
                    ;
                    ; in:
                    ;     EAX - 0x000B0000
                    ;     EBX - CS:IP value to start execution
                    ;     EDI - Linear Address where the information is to be stored
                    ;
                    ; out:
                    ;     EAX - amount of bytes allocated
                    ;     CF  - clear on success, set on failure
                    ;
                    ; destroyed:
                    ;     ECX ESI
                    ;
SYSPL_CREATEVTHREAD EQU PRIVILEGE_USER
CreateVThread:      ; Todo: fix race conditions by pre-allocating the memory
                    PUSH EDI
                    Call AllocateV8086Taskset           ; EDI = task handle
                    MOV ESI, CR3                        ; ESI = CR3
                    XOR EAX, EAX                        ; EAX = 0
                    Call AddThread                      ; Assign TDT/CR3 to processor #0
                    POP EDI
                    MOV EAX, PAGE_SIZE * 2
                    RET

ALIGN 4
                    ; Constant: sys_kernelswitch
                    ; Jump table for the kernel functions
sys_kernelswitch:   DD KernelVersion
                    DD MemMap
                    DD BlockAlloc
                    DD GetTimerTicks
                    DD PortAlloc
                    DD GetCpuInfo
                    DD CreateThread
                    DD RouteAlloc
                    DD RouteFind
                    DD GateAlloc
                    DD GateLookup
                    DD CreateVThread
syscalllimit        EQU ($-sys_kernelswitch) / 4

                    ; Constant: sys_checkprivileges
                    ; lookup table for required privileges
sys_checkprivileges:DB SYSPL_KERNELVERSION
                    DB SYSPL_MEMMAP
                    DB SYSPL_BLOCKALLOC
                    DB SYSPL_GETTIMERTICKS
                    DB SYSPL_PORTALLOC
                    DB SYSPL_GETCPUINFO
                    DB SYSPL_CREATETHREAD
                    DB SYSPL_ROUTEALLOC
                    DB SYSPL_ROUTEFIND
                    DB SYSPL_GATEALLOC
                    DB SYSPL_GATELOOKUP
                    DB SYSPL_CREATEVTHREAD

ALIGN 4
                    ; Constant: sys_privilegeswitch
                    ; jump table for kernel calls
sys_privilegeswitch:DD sys_docall
                    DD sys_privilegeuser
                    DD sys_privilegeshell
                    DD sys_privilegedriver
                    DD sys_privilegekernel

; --------------------------------------------------------------------------------------------------
; Group: Locks
; --------------------------------------------------------------------------------------------------

                    ;Variable: semPageTable
                    ;Taken for write lock on page tables
semPageTable:       DB 1
semMemTable:        DB 1

                    ;Variable: semPortLo
                    ;Taken for write lock on lower range port table
semPortLo:          DB 1
                    ;Variable: semPortHi
                    ;Taken for write lock on upper range port table
semPortHi:          DB 1

semGDT:             DB 1
semIDT:             DB 1

semPageresolve:     DB 1

                    ; Variable: semBlockedTask
                    ; Taken for write access on the blocked task list
semBlockedTask:     DB 1
                    ; Variable: semScheduler
                    ; Taken for write access to a scheduler's list
semScheduler:       DB 0xff

                    ; Variable: semTaskRoutes
                    ; taken for read or write access to the task route map
semTaskRoutes:      DB 1
                    ; Variable: ticTaskRoutes
                    ; ticket counter for the task route map
ticTaskRoutes:      DW 0
                    ; Variable: trnTaskRoutes
                    ; turn counter for the task route map
trnTaskRoutes:      DW 0

; --------------------------------------------------------------------------------------------------
; Group: Globals
; --------------------------------------------------------------------------------------------------
ALIGN 4
                    ; Variable: lpgRamdisk
                    ; Ramdisk address
lpgRamdisk:         DD 0
                    ; Variable: lpgTextPtr
                    ; Text video memory address
lpgTextPtr:         DD 0
                    ; Variable: lpgKernelEndPage
                    ; First page after the kernel section
lpgKernelEndPage:   DD 0

                    ; Variable: lpgMemtable
                    ; Location of memory allocation table in memory
lpgMemTable:        DD 0
                    ; Variable: lpgLocalPagedir
                    ; Page Directory Mirror address
lpgPagedir:         DD 0
                    ; Variable: lpgLocalPagemap
                    ; Physical to linear map of the page directory
lpgPagemap:         DD 0
                    ; Variable: lpgPageResolver
                    ; Temporary location for unpositioned page directories
lpgPageResolver:    DD 0

                    ; Variable: lpgKernelpager
                    ; Location of kernel page table
lpgKernelStack:     DD 0
                    ; Variable: lpgKernelHole
                    ; location of free paging space after kernel stuff
lpgKernelHole:      DD 0
                    ; Variable: lpgKernelTDT
                    ; location of kernel TDT
lpgKernelTDT:       DD 0


                    ; Variable: lpgTaskState
                    ; TSS Address
lpgTaskState:       DD 0

                    ; Variable: lpgGdtLimit
                    ; Global Descriptor Table limit
lpgGdtLimit:        DW 0xffff
                    ; Variable: lpgGdtOffset
                    ; Global Descriptor Table offset
lpgGdtOffset:       DD 0

                    ; Variable: lpgIdtLimit
                    ; Interrupt Descriptor Table limit
lpgIdtLimit:        DW 0x7ff
                    ; Variable: lpgIdtOffset
                    ; Interrupt Descriptor Table offset
lpgIdtOffset:       DD 0
                    ; Variable: lpgIdtAllocators
                    ; Location of interrupt owners
lpgIdtAllocators:   DD 0

                    ; Variable: lpgIOMapOffset
                    ; Location of port allocation tables
lpgIOMapOffset:     DD 0

                    ; Variable: lpgTSSDescriptor
                    ; Kernel thread TSS descriptor
lpgTSSDescriptor:   DW 0
                    ; Variable: lpgTSSaddress
                    ; Kernel thread TSS address
lpgTSSaddress:      DD 0

                    ; Variable: lpgSchedulerIndex
                    ; Scheduler index page
lpgSchedulerIndex:  DD 0
                    ; Variable: lpgTickCount
                    ; Amount of milliseconds since kernel initialisation
lpgTickCount:       DD 0, 0


                    ; Variable: lpgLAPICBase
                    ; Base address for the onboard local APIC
lpgLAPICBase:       DD 0
                    ; Variable: lpgProcessorcount
                    ; amount of Processors in the system
lpgProcessorcount:  DB 1
                    ; Variable: lpgBootstrapApic
                    ; the APIC ID of the bootstrap processor
lpgBootstrapApic:   DB 0


                    ; Variable: lpgProcessorType
                    ; the installed processor type
lpgProcessorType:   DB 0
                    DB 0
                    ; Variable: lpgProcessorVersion
                    ; the packed version word, 4 bits a number (x.x.x.x)
lpgProcessorVersion:DW 0
                    ; Variable: lpgProcessorName
                    ; processor type
lpgProcessorName:   DB "UndefinedCPU"
                    ; Variable: lpgProcessorCaps
                    ; the installed processor's capabilities
lpgProcessorCaps:   DD 0
                    DD 0


                    ; Variable: lpgTaskRoutes
                    ; pointer to an name-to-port table
lpgTaskRoutes:      DD 0
                    ; Variable: lpgTaskHandlers
                    ; pointer to a task-lookup table
lpgTaskHandlers:    DD 0

%include "inc_info.asm"

kernelimageend:
kernelimagesize EQU kernelimageend - kernelimagestart
