; Summary: debug.asm
; Debugging interface independent kernel debugger code
;
; Depends on <Debugger_Write> and <Debugger_Poll> to handle input and output.
;
; Author:
;     Marcel Sondaar
;
; License:
;     Public Domain
;

; Current interaction driver
%include "debug_vga.asm"


Debugger_WriteInt:  PUSH EAX
                    PUSH EDX
                    PUSH ECX
                    PUSH ESI

                    XOR EDX, EDX
                    MOV ECX, 8
                    MOV ESI, .writee
.loop:              MOV DL, AL
                    SHR EAX, 4
                    AND DL, 0x0F
                    MOV DL, [.hexdigits+EDX]
                    DEC ESI
                    MOV [ESI], DL
                    DEC ECX
                    JNZ .loop

                    MOV ESI, .writep
                    Call Debugger_Write

                    POP ESI
                    POP ECX
                    POP EDX
                    POP EAX
                    RET
.writep:            DB "0x"
.writea:            DB "00000000"
.writee:            DB 32, 0
.hexdigits:         DB "0123456789ABCDEF"




Debugger_Trap:      PUSHAD

                    LEA EDX, [ESP]
                    Call Debugger_Regdump

                    POPAD
                    IRETD

Debugger_Enter:     MOV byte [debugger_cont], 0
                    MOV ESI, .welcome
                    Call Debugger_Write
.loop:              Call Debugger_Poll
                    CMP byte [debugger_cont], 0
                    JE .loop
                    RET
.welcome:	        DB 'Entering guru meditation mode', 13, 10, 0


debugger_cont:      DB 0


Debugger_Regdump:   PUSH EAX
                    PUSH EDX
                    PUSH ESI
                    PUSH EBX
                    PUSH ECX

                    MOV EBX, .mi
                    MOV ECX, 8

.loop:
                    MOV ESI, EBX
                    Call Debugger_Write
                    MOV EAX, [EDX]
                    Call Debugger_WriteInt

                    ADD EBX, 8
                    ADD EDX, 4

                    SUB ECX, 1
                    TEST ECX, 1
                    JNZ .loop
                    MOV ESI, .crlf
                    Call Debugger_Write
                    OR ECX, ECX
                    JNZ .loop


                    POP ECX
                    POP EBX
                    POP ESI
                    POP EDX
                    POP EAX
                    RET
.mi                 DB "EDI = ", 0, 0
                    DB " ESI = ", 0
                    DB "EBP = ", 0, 0
                    DB " ESP = ", 0
                    DB "EBX = ", 0, 0
                    DB " EDX = ", 0
                    DB "ECX = ", 0, 0
                    DB " EAX = ", 0
.crlf               DB 10, 13, 0

                    ALIGN 4
Debugger_cmd_off:   DD 0
Debugger_cmdline:   TIMES 16 DB 0

; Function: Debugger_ParseHex
; Converts a hex string to a numerical value
; Will discard whitespace, then parse any number of hex digits (ignore x) 
; until null or space is found (success) or another character (failure)
;
; in:
;     ESI - address of string
;
; out:
;     EAX - parsed number
;     ESI - next character to parse
;     CF - set for an incomplete parse
;
; destroyed:
;     ESI ECX
;
Debugger_ParseHex:	    CMP byte [ESI], 32
                        JNE .beginparse
                        INC ESI
                        JMP Debugger_ParseHex
.beginparse:            XOR ECX, ECX
                        XOR EAX, EAX
                        CMP byte [ESI], 0
                        JE .fail
.parse:                 LODSB
                        CMP AL, 0
                        JE .done
                        CMP AL, 32
                        JE .done
                        CMP AL, '0'
                        JL .fail
                        CMP AL, '9'
                        JLE .isnum
                        CMP AL, 'A'
                        JL .fail
                        CMP AL, 'F'
                        JLE .isupper
                        CMP AL, 'X'
                        JE .parse
                        CMP AL, 'a'
                        JL .fail
                        CMP AL, 'f'
                        JLE .islower
                        CMP AL, 'x'
                        JE .parse
                        JMP .fail

.done:                  MOV EAX, ECX
                        CLC
                        RET
.fail:                  XOR EAX, EAX
                        STC
                        RET
.isnum:                 SUB AL, '0'
                        JMP .rotate
.isupper:               ADD AL, 10 - 'A'
                        JMP .rotate
.islower:               ADD AL, 10 - 'a'
.rotate:                TEST ECX, 0xf0000000
                        JNZ .fail               ; overflow
                        SHL ECX, 4
                        OR ECX, EAX
                        JMP .parse


; Function: Debugger_inchar
; inserts a character into the command buffer
;
; in:
;     AL - the ascii character to write
;
; out:
;     none
;
; destroyed:
;     unknown
;
Debugger_inchar:            CMP AL, 0xD
                            JE .execute
                            CMP AL, 0xA
                            JE .execute
                            CMP AL, 0x8
                            JE .backspace
.newchar:	                MOV ECX, [Debugger_cmd_off]
                            CMP ECX, 15
                            JE .full
                            MOV [Debugger_cmdline + ECX], AL
                            LEA ESI, [ECX + Debugger_cmdline]
                            INC ECX
                            MOV byte [Debugger_cmdline + ECX], 0
                            MOV [Debugger_cmd_off], ECX
                            Call Debugger_Write
.full:		                RET
.backspace:	                MOV ECX, [Debugger_cmd_off]
                            CMP ECX, 0
                            JE .full
                            DEC ECX
                            MOV byte [Debugger_cmdline + ECX], 0
                            MOV [Debugger_cmd_off], ECX
                            MOV ESI, .bsptext
                            Call Debugger_Write
                            RET
.bsptext:	                DB 0x8, 0
.rettext:                   DB 0xD, 0xA, 0
.execute:	                MOV ESI, .rettext
                            Call Debugger_Write
                            MOV dword [Debugger_cmd_off], 0
                            ; big parsing switch
                            MOV EDX, 4 ; 3 commands known
                            MOV EBX, .switches
                            CLD
.loop:                      XOR ECX, ECX
                            MOV ESI, Debugger_cmdline
                            MOV CL, [EBX+4]             ; length of compare
                            LEA EDI, [EBX+5]            ; text
                            MOV EAX, [EBX]              ; match function
                            LEA EBX, [EBX+ECX+5]        ; next entry
                            REPE CMPSB
                            JNZ .next
                            JMP EAX 
.next:                      DEC EDX
                            JNZ .loop
                            MOV ESI, .failtext
                            Call Debugger_Write
                            RET

.failtext:	                DB 'Unknown command', 0xD, 0xA, 0
.switches	                DD .cmd_t
                            DB 2, 't', 32
                            DD .cmd_x
                            DB 2, 'x', 32
                            DD .cmd_xp
                            DB 3, 'xp', 32
                            DD Debugger_PTDump
                            DB 4, 'tab', 0
.parsetext:                 DB 'Parse error', 0xD, 0xA, 0
.parseerror:                MOV ESI, .parsetext
                            Call Debugger_Write
                            RET

.cmd_t:                     MOV ESI, Debugger_cmdline + 2
                            Call Debugger_ParseHex
                            JC .parseerror
                            JMP Debugger_ShowPagewalk                    
.cmd_x:
.cmd_xp: 	                MOV ESI, Debugger_cmdline + 2
                            Call Debugger_ParseHex
                            JC .parseerror                            
                            JMP Debugger_DumpPhysRam


                            ; Function: Pagewalk32
                            ; Converts a linear address to the corresponding physical address, using the current value of CR3
                            ;
                            ; Paging should be disabled when this function is called.
                            ;
                            ; in:
                            ;     EAX - address
                            ;
                            ; out:
                            ;     EAX - physical address
                            ;     EBX - physical address containing the final mapping
                            ;     CF - set if the address was not currently mapped
                            ;
                            ; destroyed:
                            ;     none
                            ;
Pagewalk32:                 PUSH EDX
                            MOV EDX, CR3
                            Call PagewalkEx32
                            POP EDX
                            RET
                            



                            ; Function: PagewalkEx32
                            ; Converts a linear address to the corresponding physical address, using the current value of CR3
                            ;
                            ; Paging should be disabled when this function is called.
                            ;
                            ; in:
                            ;     EAX - address
                            ;     EDX - physical address of the page directory
                            ;
                            ; out:
                            ;     EAX - physical address
                            ;     EBX - physical address containing the final mapping
                            ;     CF - set if the address was not currently mapped
                            ;
                            ; destroyed:
                            ;     EDX
                            ;
PagewalkEx32:               MOV EBX, EAX
                            AND EAX, PAGE_TABLEMASK
                            SHR EBX, 22
                            SHR EAX, 10
                            SHL EBX, 2
                            ADD EBX, EDX
                            MOV EDX, [EBX]
                            TEST EDX, 1
                            JZ .nomap                            
                            TEST EDX, PAGE_GRANULAR
                            JNZ .mapped4
                            AND EDX, PAGE_ADDRMASK
                            LEA EBX, [EAX+EDX]
                            MOV EAX, [EAX+EDX]
                            TEST EAX, 1                        
                            JZ .nomap                            
.mapped:                    AND EAX, PAGE_TABLEMASK 
                            CLC
                            RET

.mapped4:                   MOV EAX, EDX
                            AND EAX, PAGE_TABLEMASK 
                            CLC
                            RET

.nomap:                     STC
                            RET
                            




Debugger_ShowPagewalk:      PUSH EAX
                            Call Pagewalk32
                            JC .nomap

                            MOV EDX, EAX
                            POP EAX
                            Call Debugger_WriteInt
                            MOV ESI, .colon                        
                            Call Debugger_Write
                            MOV EAX, EDX
                            Call Debugger_WriteInt
                            MOV ESI, .newline                        
                            Call Debugger_Write                            
                            RET

.nomap:                     POP EAX
                            Call Debugger_WriteInt
                            MOV ESI, .fail
                            Call Debugger_Write
                            RET

.colon:                     DB ':', 32, 0
.fail:                      DB ': not mapped'   ; overflow into newline
.newline:                   DB 0xA, 0xD, 0





Debugger_PTDump:            MOV EDI, 0              ; pagedir base
.pdloop:                    MOV EAX, EDI
                            Call Pagewalk32
                            JNC .dotab
                            MOV EDX, CR3
                            AND EDX, PAGE_ADDRMASK
                            AND EBX, PAGE_ADDRMASK
                            CMP EBX, EDX
                            JNE .dotab
.nexttab:                   ADD EDI, 4*1024*1024
                            JNZ .pdloop
                            RET

                            ; loop per table
.dotab:                     MOV ECX, EDI
.loopnomap:                 MOV EAX, ECX
                            Call Pagewalk32
                            JNC .startmap
                            ADD ECX, PAGE_SIZE
                            TEST ECX, PAGE_TABLEMASK
                            JZ .nexttab
                            JMP .loopnomap
.startmap:                  MOV [.startv], ECX
                            MOV [.startp], EAX
                            MOV EDX, EAX                    ; EDX expected next address
                            ADD ECX, PAGE_SIZE
                            ADD EDX, PAGE_SIZE
                            TEST ECX, PAGE_TABLEMASK
                            JZ .printdump                   ; new address is out of range
.loopmap:                   MOV EAX, ECX
                            Call Pagewalk32
                            JC .printdump                   ; new address is not mapped
                            CMP EDX, EAX
                            JNE .printdump                  ; new address is not sequential
                            ADD ECX, PAGE_SIZE
                            ADD EDX, PAGE_SIZE
                            TEST ECX, PAGE_TABLEMASK
                            JZ .printdump                   ; new address is out of range
                            JMP .loopmap
.printdump:                 MOV EBX, ECX
                            DEC EBX
                            MOV EAX, [.startv]
                            Call Debugger_WriteInt          ; start address
                            MOV ESI, .dash
                            Call Debugger_Write
                            MOV EAX, EBX
                            Call Debugger_WriteInt          ; end address
                            MOV ESI, .arrow
                            Call Debugger_Write
                            MOV EAX, [.startp]
                            Call Debugger_WriteInt          ; start physical
                            MOV ESI, .dash
                            Call Debugger_Write
                            MOV EAX, [.startp]              ; p_end = p_start + (v_end-v_start)
                            ADD EAX, EBX
                            SUB EAX, [.startv]
                            Call Debugger_WriteInt          ; end physical
                            MOV ESI, .newline
                            Call Debugger_Write

                            TEST ECX, PAGE_TABLEMASK        ; test for two break reasons
                            JZ .nexttab                     
                            JMP .loopnomap                  ; otherwise continue trying
.startv                     DD 0
.startp                     DD 0
.dash:                      DB '-', 32, 0
.arrow:                     DB '->', 32, 0
.newline:                   DB 0xD, 0xA, 0 



Debugger_DumpPhysRam:       PUSH EAX
                            Call Debugger_WriteInt
                            MOV ESI, .spacer
                            Call Debugger_Write
                            POP ESI
                            Call Debugger_DumpRam
                            RET
                            MOV ESI, .newline
                            Call Debugger_Write                            
.spacer:                    DB ':', 32, 0
.newline:                   DB 0xA, 0xD, 0


Debugger_DumpRam:           MOV CL, 16
                            XOR EDX, EDX
.loop:                      MOV DL, [ESI]
                            INC ESI
                            MOV CH, DL
                            SHR EDX, 4
                            MOV DL, [EDX+Debugger_WriteInt.hexdigits]
                            AND CH, 0x0F
                            MOV [.textout], DL
                            MOV DL, CH
                            MOV DL, [EDX+Debugger_WriteInt.hexdigits]
                            MOV [.textout + 1], DL
                            PUSH ESI
                            PUSH ECX
                            MOV ESI, .textout
                            Call Debugger_Write
                            POP ECX
                            POP ESI
                            DEC CL
                            JNZ .loop                                                        
                            RET

.textout:                   DB 32, 32, 32, 0      
                     