;
; Summary: stage3.asm [ia-common]
; *The shared stage 3 / main kernel for My Operating System (MOS)*
;
; This module is responsible for managing and enabling the running of user applications
;
; For software programming, check the <System calls> section
;
; Author:
;     Marcel Sondaar
;
; License:
;     Educational purposes
;



%macro OOPS 0
    PUSH EAX
    MOV EAX, __LINE__
    CALL Oops
%endmacro




                        ; Function: kernelentry
                        ; contains the entry point of the kernel
kernelentry:            MOV ESP, kernelimageend+0x2000  ; make room for stack
                        AND ESP, 0xfffff000             ; align on page boundary

                        Call FixInterrupts              ; clean up PICs so that we do not get bothered by spare ints

                        MOV EDI, ESP
                        MOV EAX, 0xCCCCCCCC
                        MOV ECX, 0x00400000
                        SUB ECX, ESP
                        SHR ECX, 2
                        CLD
                        ;REP STOSD
                        
                        ;Call Debugger_Enter

                        Call ClearScreen                ; clean up screen so that we can list information
                        Call FindRSDP                   ; locate one useful ACPI table
                        Call FindMPTable
                        Call DetectProcessors           ; determine processor configuration

                        MOV EAX, 9
                        MOV EBX, [qwProcessorSharedCaps]
                        MOV EDX, [qwProcessorSharedCaps+4]
                        Call PrintCPUCaps

                        XOR ECX, ECX
                        Call SetCPUBits                 ; Set bits
debugpos EQU 0xB8000 + 158
                        ; boot kernel structures
                        Call PrintInitLine              ; Show progress message
			MOV WORD [debugpos], 0x0940
                        Call InitializeKernelHeap       ; Prepare memory management structures
			MOV WORD [debugpos], 0x0941
                        Call InitializeGDT32            ; Initialize GDT
			MOV WORD [debugpos], 0x0942
                        Call SwitchGDT32                ; and take it
			MOV WORD [debugpos], 0x0943
                        Call InitializeIDT32            ; Initialize IDT
			MOV WORD [debugpos], 0x0944
                        Call SwitchIDT32                ; and take it
			MOV WORD [debugpos], 0x0945
                        Call InitializeAllocator        ; Initialize the memory allocator
			MOV WORD [debugpos], 0x0946
                        Call InitializeTSSBSP           ; Initialize the TSS
			MOV WORD [debugpos], 0x0947
                        XOR EAX, EAX
                        XOR ECX, ECX
                        Call CreateInitialTDS           ; Create a kernel TDS for this processor
			MOV WORD [debugpos], 0x0948
                        Call InitializeScheduler        ; Initialise Scheduling
			MOV WORD [debugpos], 0x0949

                        ; test basic scheduling operation
                        Call Schedule
                        MOV WORD [debugpos], 0x094A
                        Call Schedule
			MOV WORD [debugpos], 0x094B

                        ; initialize peripherals needed for operation
                        Call InitializePIC
			MOV WORD [debugpos], 0x094C
                        Call InitializePIT
			MOV WORD [debugpos], 0x094D

                        ;Call InitializeRTC ; warning: do not uncomment - may break the Ariel box (possibly more)
                        Call ComputeApicFrequency
			MOV WORD [debugpos], 0x094E
                        Call PrintAPICLine
			MOV WORD [debugpos], 0x094F
                        Call InitializeSyscalls
			MOV WORD [debugpos], 0x0950

                        ;Call Debugger_Init

                        ; finalize kernel setup
                        Call CheckpointBSP
			MOV WORD [debugpos], 0x0951
                        Call BuildKernelPageMap
			MOV WORD [debugpos], 0x0952
                        Call BootstrapUserspace
			MOV WORD [debugpos], 0x0953

                        ; go live
                        JMP kernelruntime
			MOV WORD [debugpos], 0x0954

                        ; old test code
                        MOV EAX, 0
                        Call EnableIRQ

                        MOV EAX, 0x20
                        MOV EDX, .inttest
                        Call SetIDTEntry32



                        ; Page allocator test
;.stub:
                        ;xchg bx, bx ; magic breakpoint
                        ;MOV EAX, 0x1E0000
                        ;call RequestMemory32
                        ;MOV EAX, 0x1E0000
                        ;call RequestMemory32
                        ;MOV EAX, 0x1E0000
                        ;call RemoveMemoryReference32

                        ;MOV EAX, 0x2E0000
                        ;call RequestMemory32
                        ;MOV EAX, 0x2E0000
                        ;call RequestAddress32
                        ;MOV EAX, 0x2E0000
                        ;call RequestAddress32
                        ;MOV EAX, 0x2E0000
                        ;call RemoveMemoryReference32



.stoploop:              HLT                             ; loop forever
                        JMP .stoploop

.inttest:               MOV word [0xB8000 + 79 * 2], 0x0700 + 'X'
                        PUSH EAX
                        MOV AL, 0x20
                        OUT PIC1_COMMAND, AL
                        POP EAX
                        IRETD

                        ; Function: mpentry
                        ; the common entry code for multiprocessor systems
                        ;
                        ; in:
                        ;     ECX - processor number
                        ;
                        ; out:
                        ;     does not return
                        ;
mpentry:                PUSH ECX                        ; save processor number
                        MOV EBP, ESP                    ; create stack frame

                        MOV EAX, [EBP]                  ; load processor number
                        CALL ProcessorDiagnostic        ; run processor diagnostics
                        MOV ECX, [EBP]
                        Call SetCPUBits                 ; load CRx configuration
                        Call InitializeAPRegisters      ; load GDTR, IDTR
                        Call InitializeTSSAP            ; create a TSS
                        MOV EAX, [EBP]
                        MOV ECX, 1
                        Call CreateInitialTDS           ; Create a kernel TDS for this processor
                        MOV ECX, [EBP]
                        Call InitializeScheduler
                        Call Schedule

                        Call CheckpointAP

.stoploop:              HLT                             ; loop forever
                        JMP .stoploop


; ====================================================================================================================
; Group: Initializers
; These functions constitute to the initialisation of the kernel. Once the kernel is running, these can be eliminated
; ====================================================================================================================


                        ; Function: AllocateFromHeap
                        ; Allocates several bytes at the end of heap.
                        ; Only multiples of PAGE_SIZE should be used to guarantee page-alignment
                        ;
                        ; in:
                        ;     EAX - bytes required
                        ;
                        ; out:
                        ;     EAX - location
                        ;
                        ; destroyed:
                        ;     none
                        ;
AllocateFromHeap:       LOCK BTR word [.lock], 0
                        JNC AllocateFromHeap
                        PUSH EDX
                        MOV EDX, [lpHeapTop]
                        ADD [lpHeapTop], EAX
                        MOV EAX, EDX
                        POP EDX
                        MOV byte [.lock], 1
                        RET

.lock:                  DB 1



                        ; Function: ApicBasedDelay
                        ; delay x cycles based on bus speed
ApicBasedDelay:         PUSH EDI
                        MOV EDI, [LAPIC_REG_DCR]    ; get divide control register
                        OR EDI, 1011b               ; choose divide by one
                        MOV [LAPIC_REG_DCR], EDI    ; write divide control register

                        MOV EDI, [LAPIC_REG_LVT0]   ; get timer LVT register
                        OR EDI,  0x00010000         ; write bit 16: mask
                        AND EDI, 0xfffdffff         ; clear bit 17: periodic timer
                        MOV [LAPIC_REG_LVT0], EDI   ; write LVT register

                        POP EDI                     ; load EDI
                        MOV [LAPIC_REG_ICOUNT], EDI ; write EDI to the timer register to start countdown
.loop:                  CMP dword [LAPIC_REG_CCOUNT], 0
                        JNE .loop                   ; delay until timer runs out
                        RET



                        ; Function: BootstrapAP586
                        ; attempts to boot application processors in the system
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EBX ECX EDX ESI EDI
                        ;
BootstrapAP586:         TEST dword [dwaProcessorCaps1], KCPUID_APIC
                        JZ .nomp                        ; no MP without onboard APIC

                        ; test for MSR
                        TEST dword [dwaProcessorCaps1], KCPUID_MSR
                        JZ .noapicmsr                   ; cant force APIC on without MSRs

                        ; Write APIC base address
                        CPU 686
                        MOV EDX, 0
                        MOV EAX, 0
                        MOV ECX, MSR_APICBASE
                        RDMSR
                        AND EAX, 0x00000FFF
                        OR EAX, LAPIC_BASE | MSR_APICBASE_ENABLE
                        WRMSR
                        CPU 386
                        TEST EAX, MSR_APICBASE_BSP
                        JZ .nomp
                        ; Either we just set the APIC base, or it was hardwired to
                        ; a known value (LAPIC_BASE)

.noapicmsr:             LEA ESI, [.mpbootstub]          ; copy stub
                        MOV EDI, 0x8000                 ; to MP start location
                        CLD                             ; prepare block move
                        MOV ECX, .mpbootstubend - .mpbootstub
                        REP MOVSB                       ; copy multiprocessor stub

                        ; disable caches - this saves us from setting up paging
                        ; and memory manager stuff
                        MOV EAX, CR0
                        PUSH EAX
                        OR EAX, CR0_CD
                        MOV CR0, EAX

                        ; enable APIC
                        MOV EAX, [LAPIC_REG_SVR]
                        OR EAX, LAPIC_ENABLED           ; set the enable bit
                        MOV [LAPIC_REG_SVR], EAX

                        ; check apic version
                        MOV EAX, [LAPIC_REG_VERSION]

                        ; mask APIC error interrupt
                        MOV EAX, [LAPIC_REG_LVT3]
                        OR EAX, 0x00010000
                        MOV [LAPIC_REG_LVT3], EAX

                        ; clear error flags
                        MOV EAX, [LAPIC_REG_ESR]        ; read error register
                        AND EAX, 0xFFFFFF10             ; clear error bits
                        MOV [LAPIC_REG_ESR], EAX        ; empty error status


                        ; prepare APIC writes
                        MOV EBX, [LAPIC_REG_APIC_ID]    ; EBX = APIC register
                        AND EBX, 0x0F000000             ; EBX = this APIC's ID

                        XOR ECX, ECX                    ; reset counter

                        ; Send INIT IPI
                        MOV EAX, [LAPIC_REG_ICR_HI]
                        MOV EAX, [LAPIC_REG_ICR_LOW]
                        AND EAX, LAPIC_ICR_RESERVED
                        OR EAX, LAPIC_ICR_SH_OTHERS | LAPIC_ICR_INT_FIX | LAPIC_ICR_DEST_PHYS | LAPIC_ICR_INT_INIT | LAPIC_ICR_LVL_ASSERT | LAPIC_ICR_T_EDGE
                        MOV [LAPIC_REG_ICR_LOW], EAX

                        ; delay
                        MOV EDI, 0x500000
                        Call ApicBasedDelay

                        ; Send Startup IPI
                        MOV DL, [.mpprocessorid]
                        MOV EAX, [LAPIC_REG_ICR_HI]
                        MOV EAX, [LAPIC_REG_ICR_LOW]
                        AND EAX, LAPIC_ICR_RESERVED
                        OR EAX, LAPIC_ICR_SH_OTHERS | LAPIC_ICR_INT_FIX | LAPIC_ICR_DEST_PHYS | LAPIC_ICR_INT_SIPI | LAPIC_ICR_LVL_ASSERT | LAPIC_ICR_T_EDGE | 0x8
                        MOV [LAPIC_REG_ICR_LOW], EAX

                        ; delay
                        MOV EDI, 0x500000
                        Call ApicBasedDelay
.nomp:
                        ; re-enable caches
                        POP EAX
                        MOV CR0, EAX

                        ; store processor count
                        XOR EAX, EAX
                        MOV AL, [.mpprocessorid]
                        MOV [dwProcessorCount], EAX


                        RET

.mpprocessorid:         DB 1

                        ; This is the Application Processor Bootstub
                        ; execution starts here, and the processor is quickly launched into protected mode
                        BITS 16
                        CPU 486
.mpbootstub:            CLI                             ; disable ints
                        MOV AX, 0x0800                  ; duplicate bootstrap segment
                        MOV DS, AX                      ; load DS to match
                        MOV ES, AX                      ; load ES to match
                        LGDT [0x40]                     ; load GDT from stub (hardcoded address)
                        MOV EAX, CR0
                        OR AL, CR0_PE
                        MOV CR0, EAX                    ; enter protected mode
                        MOV AX, 0x8                     ; pick data selector
                        MOV ES, AX                      ; load segments with data selector
                        MOV DS, AX
                        MOV SS, AX
                        MOV FS, AX
                        MOV GS, AX
                        JMP dword 0x10:.mpbootstub2     ; jump to PM entry point
                        BITS 32
                        TIMES 64 - ($-.mpbootstub) DB 0 ; pad to align GDTR
                        ; GDTR
                        DW 3 * 8 - 1
                        DD 0x8048
                        DW 0
                        ; GDT, 00
                        DD 0
                        DD 0
                        ; GDT, 08 (data)
                        DW 0xFFFF
                        DW 0
                        DB 0
                        DB 0x92
                        DB 0xCF
                        DB 0
                        ; GDT, 10 (code)
                        DW 0xFFFF
                        DW 0
                        DB 0
                        DB 0x9a
                        DB 0xCF
                        DB 0

                        ; PM startup code for 486+
.mpbootstub2:           XOR ECX, ECX
                        MOV CL, 1
                        LOCK XADD [.mpprocessorid], CL  ; fetch and increment, CL being processor ID
                        MOV ESP, kernelimageend+0x1000  ; make room for stack
                        AND ESP, 0xfffff000             ; align on first page boundary after kernel
                        MOV EAX, 0x100                  ; alloc 256 bytes of stack
                        MUL ECX                         ; index into stack space
                        ADD ESP, EAX                    ; each processor gets 256 bytes in its own space
                        STI                             ; re-enable ints
                        JMP mpentry                     ; jump to entry point

.mpbootstubend:         CPU 386



                        ; Function: BootstrapUserspace
                        ; Locates the fourth stage binary, map it into memory and create an
                        ; initial task for that binary.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EBX ECX EDX ESI EDI
                        ;
                        ; Stage4 is loaded in different locations by grub and the custom stage2
                        ;
BootstrapUserspace:     MOV EAX, [lpRamdisk]
                        CMP EAX, 0
                        JNE .loadstage2
                        LEA ESI, [.msg3]
                        MOV EDX, PRINTINITLINE_STAGE
                        Call PrintLine
                        RET

.loadstage2             CMP dword [EAX], 'STAG'
                        JE .word2
                        CMP dword [EAX], 0
                        JE .fail
.next:                  ADD EAX, 16
                        JMP .loadstage2
.fail:                  LEA ESI, [.msg3]
                        MOV EDX, PRINTINITLINE_STAGE
                        Call PrintLine
                        RET
.word2:                 CMP dword [EAX+4], 'E4  '
                        JNE .next

                        ; we located stage 4 - now prepare the data area. Create a new page directory
                        PUSH EAX
                        MOV EAX, PAGE_SIZE
                        CALL AllocateFromHeap           ; Allocate page table for the kernel range
                        MOV EBX, EAX
                        MOV EAX, PAGE_SIZE
                        CALL AllocateFromHeap           ; Allocate page directory
                        MOV EDI, EAX
                        MOV ESI, EAX
                        MOV EDX, EBX
                        Call FillKernelPageDirectory

                        MOV EAX, PAGE_SIZE
                        Call AllocateFromHeap           ; Allocate a new page table for the user range
                        MOV EDX, 0x400000
                        Call InsertPageTable32          ; EAX = table, EBX = directory, EDX = 4M


                        MOV EDI, EAX                    ; EDI = page table
                        POP EAX
                        MOV ECX, [EAX+8]                ; ECX = number of bytes
                        MOV EDX, 1024
                        PUSH EAX
                        PUSH EDI

                        ; allocate and write pages with the stage 4 code
.allocloop:             MOV EAX, PAGE_SIZE
                        Call AllocateFromHeap
                        OR EAX, PAGE_WRITABLE | PAGE_USERSPACE | PAGE_PRESENT
                        STOSD
                        SUB ECX, PAGE_SIZE
                        DEC EDX
                        JNC .allocloop

                        ; zero the remainder
                        XOR EAX, EAX
                        MOV ECX, EDX
                        REP STOSD

                        POP EDI
                        POP EAX

.debugstub:
                        MOV ESI, [EAX+12]       ; EBX = offset
                        ADD ESI, [lpRamdisk]
                        MOV ECX, [EAX+8]        ; ECX = number of bytes


                        ; copy image
                        MOV EDI, [EDI]
                        AND EDI, PAGE_ADDRMASK
                        PUSH ECX
                        REP MOVSB
                        POP ECX

                        ; debug
                        LEA ESI, [.msg1d-1]
                        MOV EDX, [EAX+8]
                        Call PrintHex4
                        LEA ESI, [.msg1d+8]
                        MOV EDX, EBX
                        Call PrintHex4
                        LEA ESI, [.msg1]
                        MOV EDX, PRINTINITLINE_STAGE
                        Call PrintLine


                        ; create a task descriptor structure
                        MOV EAX, PAGE_SIZE
                        Call AllocateFromHeap       ; allocate memory for this


                        MOV EDX, [EBX]
                        AND EDX, PAGE_ADDRMASK
                        MOV EDI, [lpFreePagingOffset]
                        MOV ESI, EAX
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE
                        Call InsertPage32           ; add the entry to the table

.hitme:
                        MOV EDX, EBX                ; EDX = page directory
                        MOV ESI, EDI
                        MOV EDI, EAX
                        MOV EBX, 0x400000
                        PUSH EDX
                        MOV EDX, ESI
                        Call FillTaskDescriptor32   ; fill the descriptor with everything and the kitchen sink
                        MOV EDX, [lpRamdisk]
                        MOV [EBX+7*4], EDX
                        POP EDX

                        OR DL, KM_STDPAGING

                        MOV EAX, [lpaSchedulerBase]
                        MOV dword [EAX + 32], 0
                        MOV dword [EAX + 24], EDX
                        MOV dword [EAX + 16], ESI

                        ;crashes when returned to (and init section is discarded)
                        ;Call Schedule

                        RET

.msg1:               DB "Bootstrap: Loading next stage: (Stage2-Stage4) "
.msg1d:              DB "00000000@00000000", 0
.msg2:               DB "Bootstrap: Loading next stage: (Multiboot) "
.msg2d:              DB "00000000@00000000", 0
.msg3:               DB "Bootstrap: Next stage could not be found. Check your setup", 0



                        ; Function: BuildKernelPageMap
                        ; Creates a default page table used when creating a new
                        ; virtual address space.
                        ;
                        ; After calling this, <AllocateFromHeap> will no longer
                        ; allocate memory that is global.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX ECX EDI ESI
                        ;
BuildKernelPageMap:     MOV EAX, PAGE_SIZE
                        Call AllocateFromHeap
                        PUSH EAX
                        MOV [lpPageTableSource], EAX

                        ; zero memory
                        MOV EDI, EAX
                        MOV ECX, PAGE_SIZE / 4
                        XOR EAX, EAX
                        REP STOSD

                        POP EAX
                        MOV EDI, EAX
                        MOV ESI, kernelruntime
                        AND ESI, PAGE_ADDRMASK
                        MOV EDI, ESI                ; ESI = address passed
                        SHR EDI, 12 - 2
                        ADD EDI, EAX                ; EDI = location to start writing
                        MOV EAX, ESI                ; EAX = starting value to write
                        OR EAX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_ACCESSED | PAGE_DIRTY
                        TEST dword [qwProcessorMaxCaps], KCPUID_PGE
                        JZ .skip
                        OR EAX, PAGE_GLOBAL

                        ; build the map for the kernel
.skip:                  STOSD
                        ADD EAX, PAGE_SIZE          ; write a page table entry to the page
                        ADD ESI, PAGE_SIZE
                        CMP ESI, [lpHeapTop]        ; compare address with allocated data
                        JNGE .skip

                        ; create holes
                        MOV [lpPagingHoles], ESI    ; store the starting address of the hole
                        ADD ESI, 4 * PAGE_SIZE      ; reserve 4 pages
                        MOV [lpMMTableHoles], ESI   ; store the starting address of the hole
                        ADD ESI, 3 * PAGE_SIZE      ; reserve 4 pages
                        MOV [lpPagingKernelTables], ESI ; store where the kernel page tables can be found
                        ADD ESI, 4 * PAGE_SIZE      ; reserve 4 pages
                        MOV [lpZeroHole], ESI       ; store the location for kernel structure building
                        ADD ESI, PAGE_SIZE          ; reserve one page
                        MOV [lpFPUHole], ESI        ; store the location for storing FPU state
                        ADD ESI, 8*PAGE_SIZE        ; reserve one page
                        MOV [lpAddressHole], ESI    ; store the location for storing FPU state
                        ADD ESI, 2*PAGE_SIZE        ; reserve one page


                        ; write the holes to the page map
                        MOV ECX, 8+3+1+8+2
                        MOV EAX, PAGE_ACCESSED      ; set some bits so that the page cannot be allocated
                        REP STOSD

                        ; write the limit
                        MOV [lpFreePagingOffset], ESI

                        RET



                        ; Function: CheckDisabledApic
                        ; Checks if a software-disabled APIC is present
                        ;
                        ; in:
                        ;     EDX = CPUID output
                        ;
                        ; out:
                        ;     EDX = corrected CPUID output
                        ;
                        ; destroyed:
                        ;     none
CheckDisabledAPIC:

                        RET


                        ; Function: CheckpointBSP
                        ; Synchronizes progress across CPU's
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
CheckpointBSP:          PUSH EAX
                        MOV EAX, [dwProcessorCount]
                        DEC EAX
                        LOCK AND dword [.desync], 0
                        MOV [.sync], EAX
.loop:                  CMP dword [.sync], 0
                        JNE .loop
.loop2:                 CMP dword [.desync], EAX
                        JNE .loop2
                        POP EAX
                        RET

                        ALIGN 4
.sync:                  DD 0
.desync:                DD 0



                        ; Function: CheckpointAP
                        ; Synchronizes progress across CPU's
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
CheckpointAP:           CMP dword [CheckpointBSP.sync], 0
                        JE CheckpointAP
                        LOCK DEC dword [CheckpointBSP.sync]
.loop:                  CMP dword [CheckpointBSP.sync], 0
                        JNE .loop
                        LOCK INC dword [CheckpointBSP.desync]
                        RET




                        ; Function: ComputeApicFrequency
                        ; Compares the local APIC with the PIT timer to calculate
                        ; its operating frequency
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX
                        ;
ComputeApicFrequency:   TEST dword [dwaProcessorCaps1], CPUID_APIC
                        JNZ .continue
                        STC
                        RET

.continue:              XOR EBX, EBX
                        MOV EAX, [LAPIC_REG_DCR]
                        AND AL, 0xF0
                        OR AL, 0x0B
                        MOV [LAPIC_REG_DCR], EAX
                        MOV dword [LAPIC_REG_ICOUNT], 0x10000000
                        MOV EDX, 1
                        MOV AL, 0x0
                        OUT PIT_COMMAND, AL
                        Call HardwareDelay
                        IN AL, PIT_CHAN0
                        Call HardwareDelay
                        MOV AH, AL
                        IN AL, PIT_CHAN0
                        XCHG AL, AH
                        Call HardwareDelay
                        MOV BX, AX
.preloop:               MOV ECX, EDX
                        SHL EDX, 1
                        LOOP $
                        MOV AL, 0x0
                        OUT PIT_COMMAND, AL
                        Call HardwareDelay
                        IN AL, PIT_CHAN0
                        Call HardwareDelay
                        MOV AH, AL
                        IN AL, PIT_CHAN0
                        XCHG AL, AH
                        Call HardwareDelay
                        SUB AX, BX
                        NEG AX
                        CMP AX, 0x4000
                        JBE .preloop
                        MOV EAX, 0x10000000
                        SUB EAX, [LAPIC_REG_CCOUNT]
                        MOV EDX, EAX
                        MOV [LAPIC_REG_ICOUNT], EAX
                        MOV AL, 0x0
                        OUT PIT_COMMAND, AL
                        Call HardwareDelay
                        IN AL, PIT_CHAN0
                        Call HardwareDelay
                        MOV AH, AL
                        IN AL, PIT_CHAN0
                        XCHG AL, AH
                        Call HardwareDelay
                        MOV BX, AX
.postloop:              MOV ECX, [LAPIC_REG_CCOUNT]
                        CMP ECX, 0
                        JNE .postloop
                        MOV AL, 0x0
                        OUT PIT_COMMAND, AL
                        Call HardwareDelay
                        IN AL, PIT_CHAN0
                        Call HardwareDelay
                        MOV AH, AL
                        IN AL, PIT_CHAN0
                        XCHG AL, AH
                        Call HardwareDelay
                        SUB BX, AX
                        ; (E)BX = PIT cycles
                        ; EDX = APIC cycles
                        MOV [.pitcycles], EBX
                        MOV [.apiccycles], EDX
.compute:               MOV EAX, 0x00013174
                        MUL EDX
                        DIV EBX
                        SHR EBX, 1
                        CMP EDX, EBX
                        JNAE .noround
                        INC EAX
.noround:               MOV [.apicfreq], EAX

.finish:                RET
.pitcycles:             DD 0
.apiccycles:            DD 0
.apicfreq:              DD 0




                        ; Function: ComputeProcessorUnion
                        ; calculates <qwProcessorSharedCaps>
                        ; by takes all processor's capabilities, and keeping
                        ; only the bits common to all processors.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EBX ECX ESI EDI
                        ;
ComputeProcessorUnion:  MOV ECX, [dwProcessorCount]
                        STC
                        SBB EAX, EAX
                        SBB EBX, EBX
                        LEA ESI, [dwaProcessorCaps1]
                        LEA EDI, [dwaProcessorCaps2]
.loop:                  AND EAX, [ESI]
                        AND EBX, [EDI]
                        ADD ESI, 4
                        ADD EDI, 4
                        DEC ECX
                        JNZ .loop
                        MOV [qwProcessorSharedCaps], EAX
                        MOV [qwProcessorSharedCaps+4], EBX

                        MOV ECX, [dwProcessorCount]
                        XOR EAX, EAX
                        XOR EBX, EBX
                        LEA ESI, [dwaProcessorCaps1]
                        LEA EDI, [dwaProcessorCaps2]
.loop2:                 OR  EAX, [ESI]
                        OR  EBX, [EDI]
                        ADD ESI, 4
                        ADD EDI, 4
                        DEC ECX
                        JNZ .loop2
                        MOV [qwProcessorMaxCaps], EAX
                        MOV [qwProcessorMaxCaps+4], EBX

                        RET



                        ; Function: CreateInitialTDS
                        ; Creates the initial task description state and
                        ; loads it for the current processor
                        ;
                        ; in:
                        ;     EAX - processor number
                        ;     ECX - entries on the stack to copy
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ESI, EDX, ECX, EDI
                        ;
CreateInitialTDS:       PUSH EAX
                        MOV EAX, PAGE_SIZE
                        CALL AllocateFromHeap

                        ; copy stack contents
                        ADD ECX, 2
                        LEA ESI, [ESP]
                        LEA EDI, [EAX+PAGE_SIZE]
                        MOV EDX, ECX
                        SHL EDX, 2
                        SUB EDI, EDX
                        MOV EDX, EDI
                        REP MOVSD

                        ; switch to new stack, which is now 3k instead of the temporary one
                        MOV ESP, EDX

                        ; load the remainder of the TDT structure
                        MOV EDI, EAX
                        MOV [EAX+TDS_PHYSICAL], EAX
                        POP EAX
                        MOV dword [EDI+TDS_DESCHEDULE], Base_ScheduleIn
                        LEA EDX, [EDI+TDS_FPUSTATE]
                        MOV [lpaCurrentProcess+8*EAX], EDI
                        MOV dword [lpaCurrentScheduleProc+8*EAX], Base_ScheduleOut
                        MOV [lpaLastFPUState+8*EAX], EDX

                        RET



                        ; Function: DetectCoprocessor
                        ; Probes the current processor for an math coprocessor
                        ; and sets <dwaProcessorCaps1> accordingly
                        ;
                        ; in:
                        ;     EAX - processor number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
DetectCoprocessor:      CPU 386 FPU
                        PUSH EDX                                ; Save edx
                        MOV EDX, CR0                            ; CR0 has a bit on the matter as well
                        AND EDX, CR0_ET                         ; Check Extension Type
                        JZ .nofpu                               ; The processor supports no FPU
                        MOV EDX, CR0                            ; Start probe, get CR0
                        AND EDX, (-1) - (CR0_TS + CR0_EM)       ; clear TS and EM to force fpu access
                        ; Fixme: CR0_NE does not exist on a 386
                        OR EDX, CR0_NE                          ; set NE (workaround no-wait bug)
                        MOV CR0, EDX                            ; store control word
                        FNINIT                                  ; load defaults to FPU
                        FNSTSW [.testword]                      ; store status word
                        CMP word [.testword], 0                 ; compare the written status
                        JNE .nofpu

                        ;we have an FPU
.hasfpu:                OR dword [dwaProcessorCaps1 + 4 * EAX], KCPUID_FPU
.nofpu:                 POP EDX                                 ; restore EDX
                        RET                                     ; and return

                        CPU 386

.testword:              DW 0x55AA



                        ; Function: DetectProcessors
                        ; Retrieves the information from the BSP,
                        ; looks for APs and boots them
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EBX ECX EDX ESI EDI
                        ;

                        ; Fixme: Lots of reboots/lockups on Pentium 1 boxes

DetectProcessors:       LEA ESI, [.msg1]                ; get message
                        MOV EDX, 0                      ; line number: #1
                        Call PrintLine                  ; print "Checking BSP"

                        ; check BSP
                        XOR EAX, EAX                    ; select processor id 0
                        Call ProcessorDiagnostic        ; run diagnostic

                        ; check if multiprocessor testing makes sense
                        CMP byte [baProcessorArch], 0   ; get the architectural family
                        JE .done                        ; if unknown, dont look for APs
.trymp:                 LEA ESI, [.msg2]                ; get message
                        MOV EDX, 0                      ; line number: #1
                        Call PrintLine                  ; print "Finding APs"

                        ; check for Pentium-style systems
                        CMP byte [baProcessorArch], KARCH_PENTIUM
                        JNE .mparch2                    ; if processor type = pentium
                        ;Call BootstrapAP586             ; find APs using local APIC
                        JMP .done                       ; skip rest of comparisons

                        ; check for other systems

.mparch2:               ; TODO: Opteron, Athlon64x2, K7: SMP (HTT?)

                        ; TODO: HyperThreading

                        ; all processors were loaded, initialize them
.done:                  Call ComputeProcessorUnion      ; find common denominator

                        ; print completion message
                        LEA ESI, [.msg3]
                        MOV EDX, 0
                        Call PrintLine
                        ; done
                        RET

.msg1:               DB "Processor Diagnostic: <Checking BSP>", 0
.msg2:               DB "Processor Diagnostic: <Finding APs>", 0
.msg3:               DB "Processor Diagnostic: <Processor configuration complete>", 0



                        ; Function: FindMPTable
                        ; Locates the Intel multiprocessor table
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX, ECX, EDI
                        ;
FindMPTable:            PUSH ECX
                        MOV ECX, 0x10000 / 4
                        MOV EDI, 0xF0000
                        CALL .locateloop
                         CLC
                        JC .found

                        XOR ESI, ESI
                        XOR ECX, ECX
                        MOV SI, [0x400]             ; it may also be in the EBDA
                        MOV CX, 0xA000
                        SUB CX, SI
                        CALL .locateloop
                         ;CLC
                        JC .found
                        MOV word [0xB8000 + 8 * 160 - 4], 0x0700 + '-'
                        POP ECX
                        RET

.found:                 MOV [lpMPTablePointer], EDI
                        MOV word [0xB8000 + 8 * 160 - 4], 0x0700 + 'M'
                        POP ECX
                        RET



.locateloop:            MOV EAX, '_MP_'
                        REPNE SCASD
                        JE .check
                        CLC
                        RET

.check:                 PUSH EDI
                        SUB EDI, 4
                        XOR EAX, EAX
                        XOR EBX, EBX
                        MOV BL, [EDI+8]
                        SHL EBX, 4
                        DEC EBX
.sumloop:               ADD AL, [EDI+EBX]
                        DEC EBX
                        JNS .sumloop
                        POP EDI
                        CMP AL, 0
                        JNE .locateloop
                        SUB EDI, 4
                        STC
                        RET




                        ; Function: FindRSDP
                        ; Locates the RSDP table, if present.
                        ; The location is stored for further reference
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ESI
                        ;
FindRSDP:               PUSH ECX
                        MOV ECX, 0x00020000 / 16
                        MOV ESI, 0xE0000            ; the rsdp may be within the bios image
                        CALL .locateloop
                        JC .found
                        XOR ESI, ESI
                        XOR ECX, ECX
                        MOV SI, [0x400]             ; it may also be in the EBDA
                        MOV CX, 0xA000
                        SUB CX, SI
                        CALL .locateloop
                        JC .found
                        MOV word [0xB8000 + 8 * 160 - 2], 0x0700 + '-'
                        POP ECX
                        RET

.found:                 MOV [lpRSDPBase], ESI
                        MOV word [0xB8000 + 8 * 160 - 2], 0x0700 + 'A'
                        POP ECX
                        RET


.locateloop:            CMP dword [ESI], 'RSD '
                        JE .check
.resumeloop:            DEC ECX
                        JZ .endloop
                        ADD ESI, 16
                        JMP .locateloop
.check:                 CMP dword [ESI+4], 'PTR '
                        JNE .resumeloop
                        STC
                        RET
.endloop:               CLC
                        RET




                        ; Function: FixInterrupts
                        ; Fix PIC issues: mask all ints at the PIC, then enable interrupts
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
FixInterrupts:          MOV AL, 0xff
                        OUT 0x21, AL
                        OUT 0xA1, AL
                        ;STI
                        RET




                        ; Function: InitializeAllocator
                        ; Initializes the memory allocator tables
                        ;
                        ; The memory allocator basically is a reference counter with a 4-level tree
                        ; covering the entire canonical address space.
                        ; The tree is formatted as follows:
                        ;
                        ; each entry consists of the top 3 levels consists of one Qword, the 4th layer
                        ; only uses one Dword as it does not need to contain 64-bit pointers.
                        ; each entry can be either a pointer or a reference count.
                        ; The two most significant bits indicate the usage.
                        ;  00 - non-memory area, use count
                        ;  01 - memory area, use count
                        ;  10 - reference, mapped into physical space
                        ;  11 - reference, mapped into linear space
                        ; For level 4, only the most significant bit is considered.
                        ;  1  - memory with use count
                        ;  0  - non-memory with use count
                        ; References contain the address in the entry, which is aligned to a page boundary,
                        ; and a count of references/areas that have free memory in the least significant bits.
                        ; Non-referenced areas simply consist of a reference count.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     all non-stack GPRs
                        ;
InitializeAllocator:    XOR EAX, EAX                    ; Zero out EAX
                        MOV EDI, [lpL1MemoryTable]
                        MOV ECX, PAGE_SIZE/4
                        REP STOSD                       ; Clear L1 table
                        MOV EDI, [lpL2MemoryTable]
                        MOV ECX, PAGE_SIZE/4
                        REP STOSD                       ; Clear L2 table
                        MOV EDI, [lpL3MemoryTable]
                        MOV ECX, PAGE_SIZE/4
                        REP STOSD                       ; Clear L3 table
                        MOV EDI, [lpL4MemoryTable]
                        MOV ECX, PAGE_SIZE/4
                        REP STOSD                       ; Clear L4 table

                        MOV EDI, [lpL4MemoryTable]      ; EDI = address of L4 table
                        MOV ESI, [lpL3MemoryTable]      ; ESI = address of L3 table
                        MOV EAX, ESI                    ; EAX = address of L3 table
                        INC EAX                         ; set bit 0, to indicate free memory is present
                        MOV [EDI], EAX                  ; write first entry to L4 table
                        MOV byte [EDI+7], PMAB_POINTER_VIRT

                        MOV EDI, [lpL2MemoryTable]      ; EDI = address of L2 table
                        MOV EAX, EDI                    ; EAX = address of L2 table
                        INC EAX                         ; set bit 0, to indicate free memory is present
                        MOV [ESI], EAX                  ; Write first entry to L3 table
                        MOV byte [ESI+7], PMAB_POINTER_VIRT

                        MOV ESI, [lpL1MemoryTable]      ; ESI = address of L1 table
                        MOV EAX, ESI                    ; EAX = address of L1 table
                        MOV [EDI], EAX                  ; Write entry to L2 table
                        MOV byte [EDI+7], PMAB_POINTER_VIRT

                        ADD EAX, 0x800                  ; make use of the second half of the table
                        MOV [EDI+8], EAX                ; load the second entry in L2 table
                        MOV byte [EDI+15], PMAB_POINTER_VIRT
                        MOV EBX, EDI                    ; keep the pointer until we have computed the remaining BSS size

                        ;MOV ECX, [lpHeapTop]            ; ECX = end of heap
                        ;SUB ECX, 0x100000               ; ECX = kernel size
                        ;SHR ECX, 12                     ; ECX = kernel size in pages
                        ;MOV EDX, 256                    ;
                        ;SUB EDX, ECX                    ; EDX = entries in L1 table unoccupied

                        ;OR [EDI], EDX                   ; emit the count value

                        ; todo: allocation of kernel pages
                        ; these are the amount of occupied/unoccupied pages
                        ; they should later be set to the actual pages that are used.
                        MOV ECX, 256
                        MOV EDX, 512

                        MOV EAX, 0x1 | PMAD_L1COUNT_MEM
                        LEA EDI, [ESI + 4 * 0x100]      ; EDI = halfway through the first page table
                        REP STOSD

                        MOV EAX, 0x0 | PMAD_L1COUNT_MEM
                        MOV ECX, EDX
                        REP STOSD

                        RET



                        ; Function: InitializeAPRegisters
                        ; loads the AP's special registers with the values generated by the BSP
                        ; this also waits for the BSP to present the data
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
InitializeAPRegisters:  PUSH EAX
                        MOV EAX, [lpVideoMemoryBase]
                        ; Sync to GDT
.syncgdt:               CMP dword [SwitchGDT32.gdtr.offset], 0
                        JE .syncgdt
                        Call SwitchGDT32
                        LOCK INC byte [EAX + PRINTINTLINE_GDT + 8]
                        ; Sync to IDT
.syncidt:               CMP dword [SwitchIDT32.idtr.offset], 0
                        JE .syncidt
                        Call SwitchIDT32
                        LOCK INC byte [EAX + PRINTINTLINE_IDT + 8]

                        ; restore EAX
                        POP EAX
                        RET
                        ; Fixme: this is detrimental to speed on P4/HTT processors




                        ; Function: InitializeKernelHeap
                        ; Loads pointers for all the needed structures
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
InitializeKernelHeap:   MOV EAX, kernelimageend + 0x2000
                        AND EAX, 0xfffff000

                        ; these structures are needed
                        MOV [lpKernelBasePageTable], EAX
                        ADD EAX, PAGE_SIZE
                        MOV [lpAddressSpaceMirror], EAX
                        ADD EAX, PAGE_SIZE
                        MOV [lp32BitGDT], EAX
                        ADD EAX, PAGE_SIZE / 2
                        MOV [lp32BitIDT], EAX
                        ADD EAX, PAGE_SIZE / 2
                        MOV [lpL4MemoryTable], EAX
                        ADD EAX, PAGE_SIZE
                        MOV [lpL3MemoryTable], EAX
                        ADD EAX, PAGE_SIZE
                        MOV [lpL2MemoryTable], EAX
                        ADD EAX, PAGE_SIZE
                        MOV [lpL1MemoryTable], EAX
                        ADD EAX, PAGE_SIZE

                        ; test for PAE
                        TEST dword [qwProcessorMaxCaps], KCPUID_PAE
                        JZ .nomoreextras
                        MOV [lpKernelBasePAETable], EAX
                        ADD EAX, PAGE_SIZE
                        ; test for longmode
                        TEST dword [qwProcessorMaxCaps+4], KCPUID_AMD64
                        JZ .nomoreextras
                        MOV [lp64BitIDT], EAX
                        ADD EAX, PAGE_SIZE
.nomoreextras:          ; store the end-of-heap. The rest can be loaded with the memory manager.
                        MOV [lpHeapTop], EAX

                        ; print progress message
                        MOV EAX, [lpVideoMemoryBase]
                        ADD EAX, PRINTINTLINE_HEAP
                        MOV dword [EAX], 0x07000700 + 'H' + 'E' * 65536
                        MOV dword [EAX+4], 0x07000700 + 'A' + 'P' * 65536
                        RET



                        ; Function: InitializeGDT32
                        ; initializes the 32 bit GDT
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX ECX EDI
                        ;
InitializeGDT32:        MOV EDI, [lp32BitGDT]
                        MOV ECX, 0x800 / 4
                        XOR EAX, EAX
                        REP STOSD

                        ; fill the GDT Structure
                        MOV EDI, [lp32BitGDT]
                        ; 32-bit kernel code
                        MOV word [EDI+1*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+1*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                        MOV byte [EDI+1*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING0 | GDT_TYPE_CODE
                        ; 32-bit kernel data
                        MOV word [EDI+2*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+2*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                        MOV byte [EDI+2*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING0 | GDT_TYPE_DATA  | GDT_TYPE_DATARW
                        ; 32-bit user code
                        MOV word [EDI+3*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+3*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                        MOV byte [EDI+3*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_CODE
                        ; 32-bit user data
                        MOV word [EDI+4*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+4*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_32BIT
                        MOV byte [EDI+4*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_DATA  | GDT_TYPE_DATARW
                        ; 16-bit user code
                        MOV word [EDI+5*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+5*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_16BIT
                        MOV byte [EDI+5*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_CODE
                        ; 16-bit user data
                        MOV word [EDI+6*8+GDT_LIMIT_LO], 0xffff
                        MOV byte [EDI+6*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_16BIT
                        MOV byte [EDI+6*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_FLAG_RING3 | GDT_TYPE_DATA  | GDT_TYPE_DATARW
			; 16-bit kernel code
			MOV word [EDI+7*8+GDT_LIMIT_LO], 0xffff
			MOV byte [EDI+7*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_16BIT
			MOV byte [EDI+7*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_TYPE_CODE
			; 16-bit kernel data
			MOV word [EDI+8*8+GDT_LIMIT_LO], 0xffff
			MOV byte [EDI+8*8+GDT_FLAGS_HI], GDT_LIMIT_HIMASK | GDT_FLAG_PAGES | GDT_FLAG_16BIT
			MOV byte [EDI+8*8+GDT_FLAGS_LO], GDT_FLAG_PRESENT | GDT_TYPE_DATA  | GDT_TYPE_DATARW


                        ; print completion message
                        MOV EAX, [lpVideoMemoryBase]
                        ADD EAX, PRINTINTLINE_GDT
                        MOV dword [EAX], 0x07000700 + 'G' + 'D' * 65536
                        MOV dword [EAX+4], 0x07000700 + 'T' + '(' * 65536
                        MOV dword [EAX+8], 0x07000700 + '1' + ')' * 65536

                        ; fill the switch statement
                        MOV word [SwitchGDT32.gdtr.length], 0x800 - 1
                        MOV dword [SwitchGDT32.gdtr.offset], EDI

                        RET



                        ; Function: InitializeIDT32
                        ; Initializes the 32-bit IDT
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX ECX ESI EDI
                        ;
InitializeIDT32:        MOV EDI, [lp32BitIDT]
                        MOV ECX, 0x800 / 4
                        XOR EAX, EAX
                        REP STOSD

                        MOV EDI, [lp32BitIDT]
                        MOV ESI, .intfunctions
                        MOV ECX, 32
.loop:                  LODSD
                        MOV EBX, EAX
                        AND EAX, 0x0000ffff
                        AND EBX, 0xffff0000
                        OR EAX, 0x00080000
                        OR EBX, (GDT_FLAG_RING0 | IDT_TYPE_TRAP32 | GDT_FLAG_PRESENT) << 8
                        STOSD
                        MOV EAX, EBX
                        STOSD
                        DEC ECX
                        JNZ .loop
                        MOV ECX, 16
.loop2:                 LODSD
                        MOV EBX, EAX
                        AND EAX, 0x0000ffff
                        AND EBX, 0xffff0000
                        OR EAX, 0x00080000
                        OR EBX, (GDT_FLAG_RING0 | IDT_TYPE_INT32 | GDT_FLAG_PRESENT) << 8
                        STOSD
                        MOV EAX, EBX
                        STOSD
                        DEC ECX
                        JNZ .loop2

                        MOV EAX, .padfunction
                        MOV EBX, EAX
                        AND EAX, 0x0000ffff
                        AND EBX, 0xffff0000
                        OR EAX, 0x00080000
                        OR EBX, (GDT_FLAG_RING0 | IDT_TYPE_TRAP32 | GDT_FLAG_PRESENT) << 8
                        MOV ECX, 256 - (32 + 16)
.loop3:                 STOSD
                        XCHG EAX, EBX
                        STOSD
                        XCHG EAX, EBX
                        DEC ECX
                        JNZ .loop3

                        ; print progress message
                        MOV EAX, [lpVideoMemoryBase]
                        ADD EAX, PRINTINTLINE_IDT
                        MOV dword [EAX], 0x07000700 + 'I' + 'D' * 65536
                        MOV dword [EAX+4], 0x07000700 + 'T' + '(' * 65536
                        MOV dword [EAX+8], 0x07000700 + '1' + ')' * 65536


                        ; fill the idtr values
                        MOV EDI, [lp32BitIDT]
                        MOV word [SwitchIDT32.idtr.length], 0x800 - 1
                        MOV dword [SwitchIDT32.idtr.offset], EDI
                        RET

                        ; Todo: replace with real handlers
.intfunctions:          DD IntHandler.errstub           ; #00  #DF,  Division by zero
                        DD IntHandler.errstub           ; #01  #DB,  Debug interrupt
                        DD IntHandler.errstub           ; #02  NMI,  Non-maskable interrupt
                        DD IntHandler.errstub           ; #03  #BP,  Breakpoint
                        DD IntHandler.errstub           ; #04  #OF,  Overflow
                        DD IntHandler.errstub           ; #05  #BR,  Bound exceeded
                        DD IntHandler.errstub           ; #06  #UD,  Undefined Instruction
                        DD IsrNMHandler32               ; #07  #NM,  Co-Processor failure
                        DD IntHandler.errstub           ; #08  #DF,  Double-fault
                        DD IntHandler.errstub           ; #09  #SO,  Co-Processor segment overrun
                        DD IntHandler.errstub           ; #10  #TS,  Invalid TSS
                        DD IntHandler.errstub           ; #11  #NP,  Segment not present
                        DD IntHandler.errstub           ; #12  #SS,  Stack Segment Violation
                        DD IsrGPFHandler32              ; #13  #GP,  General Protection Violation
                        DD IsrPFHandler32               ; #14  #PF,  Pagefault
                        DD IntHandler.errstub           ; #15
                        DD IntHandler.errstub           ; #16  #MF,  Math fault
                        DD IntHandler.errstub           ; #17  #AC,  Alignment Check Exception
                        DD IntHandler.errstub           ; #18  #MC,  Machine Check Exception
                        DD IntHandler.errstub           ; #19  #XF,  Streaming SIMD Exception
                        DD IntHandler.errstub           ; #20
                        DD IntHandler.errstub           ; #21
                        DD IntHandler.errstub           ; #22
                        DD IntHandler.errstub           ; #23
                        DD IntHandler.errstub           ; #24
                        DD IntHandler.errstub           ; #25
                        DD IntHandler.errstub           ; #26
                        DD IntHandler.errstub           ; #27
                        DD IntHandler.errstub           ; #28
                        DD IntHandler.errstub           ; #29
                        DD IntHandler.errstub           ; #30
                        DD IntHandler.errstub           ; #31

                        DD IntHandler.masterstub        ; #32 IRQ0
                        DD IntHandler.masterstub        ; #33 IRQ1
                        DD IntHandler.masterstub        ; #34 IRQ2
                        DD IntHandler.masterstub        ; #35 IRQ3
                        DD IntHandler.masterstub        ; #36 IRQ4
                        DD IntHandler.masterstub        ; #37 IRQ5
                        DD IntHandler.masterstub        ; #38 IRQ6
                        DD IntHandler.masterstub        ; #39 IRQ7
                        DD IntHandler.slavestub         ; #40 IRQ8
                        DD IntHandler.slavestub         ; #41 IRQ9
                        DD IntHandler.slavestub         ; #42 IRQ10
                        DD IntHandler.slavestub         ; #43 IRQ11
                        DD IntHandler.slavestub         ; #44 IRQ12
                        DD IntHandler.slavestub         ; #45 IRQ13
                        DD IntHandler.slavestub         ; #46 IRQ14
                        DD IntHandler.slavestub         ; #47 IRQ15
.intcount               EQU ($ - .intfunctions) / 4
.padfunction            EQU IntHandler.unimplstub




                        ; Function: InitializePIC
                        ; Brings the PIC into a known state
                        ;
                        ; maps the master PIC's irq's 0-7 to the non-intel area
                        ; maps the slave PIC's irq's 8-15 after the masters'
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroys:
                        ;     EAX, EBX
                        ;

                        ; FIXME: Socket-A Interrupt bug
                        ;
                        ; Copied from the previous kernel generation. However
                        ; there was an issue regarding PIT interrupts not firing.
                        ; Masking interrupts should instead of disabling interrupts
                        ; should fix this (according to brendan). PIC/PIT should be
                        ; tested with Socket-A systems before removal of fixme
                        ;

InitializePIC:          CLI                     ; Interrupts should be disabled
                                                ; to prevent ints firing during config sequence

                        MOV AL, PIC_ICW1_INIT | PIC_ICW1_ICW4
                        OUT PIC1_COMMAND, AL    ; program chip: PIC1
                        Call HardwareDelay

                        MOV AL, PIC_ICW1_INIT | PIC_ICW1_ICW4
                        OUT PIC2_COMMAND, AL    ; program chip: PIC2
                        Call HardwareDelay

                        MOV AL, 0x20            ; ICW 2 = base vector
                        OUT PIC1_DATA, AL
                        Call HardwareDelay

                        MOV AL, 0x28            ; ICW 2 = base vector
                        OUT PIC2_DATA, AL
                        Call HardwareDelay

                        MOV AL, 1 << 2          ; ICW 3 = slaves connected
                        OUT PIC1_DATA, AL
                        Call HardwareDelay

                        MOV AL, 0               ; ICW 3 = slaves connected
                        OUT PIC2_DATA, AL
                        Call HardwareDelay

                        MOV AL, PIC_ICW4_8086   ; ICW 4 = operation mode
                        OUT PIC1_DATA, AL
                        Call HardwareDelay

                        MOV AL, PIC_ICW4_8086   ; ICW 4 = operation mode
                        OUT PIC2_DATA, AL
                        Call HardwareDelay

                        MOV AL, 0xff            ; OCW 1 = mask
                        OUT PIC1_DATA, AL
                        Call HardwareDelay

                        MOV AL, 0xff            ; OCW 1 = mask
                        OUT PIC2_DATA, AL
                        Call HardwareDelay

                        STI                     ; reenable ints - the system should be safe
                                                ; from stray ones occurring

                        ; enable cascade bit
                        MOV EAX, 0x02
                        Call EnableIRQ


                        RET



                        ; Function: InitializePIT
                        ;
                        ; Configure the PIT to a rate generator
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
InitializePIT:          MOV AL, PIT_CW_RATE | PIT_CW_CHAN0 | PIT_CW_LOHI
                        OUT PIT_COMMAND, AL
                        CALL HardwareDelay

                        MOV AL, 0xFF
                        OUT PIT_CHAN0, AL
                        CALL HardwareDelay

                        MOV AL, 0xFF
                        OUT PIT_CHAN0, AL
                        CALL HardwareDelay

                        RET



                        ; Function: InitializeRTC
                        ; Initializes the RTC as an interrupt source
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;

                        ; Fixme: InitializeRTC crashes some computers
                        ;
                        ; possible cause: 
                        ;   - NMI's get enabled, 
                        ;   - register renaming
                        ;
                        ; both fixed in current version, needs testing
                        ;
                        ; in all cases, the code is currently not used 
                        ; and can safely be kept disabled

InitializeRTC:          PUSH EAX

.waitfortime:
                        MOV AL, 0x0A | 0x80    ; 0x80 to keep the NMI disabled
                        OUT 0x70, AL
                        IN AL, 0x71
                        TEST AL, 0x80
                        JNZ .waitfortime

                        ; program the main RTC register with our dividers
                        MOV AL, 0x0A | 0x80
                        OUT 0x70, AL
                        MOV AL, 0x2C
                        OUT 0x71, AL

                        ; enable periodic interrupt
                        MOV AL, 0x0B | 0x80
                        OUT 0x70, AL
                        MOV AH, AL
                        IN AL, 0x71
                        OR AL, 0x40
                        XCHG AH, AL                 ; all accesses to 0x71 reset the index
                        OUT 0x70, AL
                        XCHG AH, AL
                        OUT 0x71, AL

                        ; register appropriate interrupt handler
                        MOV EAX, 0x28
                        MOV EDX, IrqRTCHandler
                        Call SetIDTEntry32

                        ; allow the RTC interrupt at the PIC
                        MOV EAX, 8
                        Call EnableIRQ

                        ; clear interrupt pending status, in case it was set and the EOI went missing
                        MOV AL, 0x0C | 0x80
                        OUT 0x70, AL
                        IN AL, 0x71

                        POP EAX

                        RET




InitializeScheduler:    ; Function: InitializeScheduler
                        ; Initializes the scheduling arrays for this processor
                        ;
                        ; in:
                        ;     ECX - processor number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EDX
                        ;
                        MOV EAX, PAGE_SIZE
                        CALL AllocateFromHeap

                        ; add the current (idle) task to the scheduler list
                        MOV [lpaSchedulerBase + 8 * ECX], EAX
                        MOV [lpaSchedulerOffset + 8 * ECX], EAX

                        ; fill in the scheduler list
                        MOV EDX, [lpaCurrentProcess + 8 * ECX]
                        MOV [EAX], EDX
                        MOV EDX, [lpaCurrentProcess + 8 * ECX + 4]
                        MOV [EAX + 4], EDX
                        MOV EDX, [lpaCurrentAddressSpace + 8 * ECX]
                        MOV [EAX + 8], EDX
                        MOV EDX, [lpaCurrentAddressSpace + 8 * ECX + 4]
                        MOV [EAX + 12], EDX

                        ; add the list terminator
                        XOR EDX, EDX
                        MOV [EAX + 0x10], EDX
                        MOV [EAX + 0x14], EDX
                        MOV [EAX + 0x18], EDX
                        MOV [EAX + 0x1C], EDX

                        RET



                        ; Function: InitializeSyscalls
                        ; Registers the system call handlers
                        ;
                        ;
InitializeSyscalls:     MOV EAX, 0x81
                        MOV EDX, Entrypoint_Int
                        Call AllocateIDTEntry
                        Call SetIDTEntry32Public

                        RET




                        ; Function: InitializeTSSAP
                        ; Creates an initial TSS for the current processor.
                        ; All TSSes will share the IO permission bitmap of the
                        ; other TSSes. The offset hereto has to be set
                        ; bits.
                        ;
                        ; in:
                        ;     ECX - the processor number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EDX EDI
                        ;
InitializeTSSAP:        CMP byte [InitializeTSSBSP.done], 0
                        JE InitializeTSSAP

                        MOV EDX, [lpaTSS + 8*ECX]   ; Get reserved TSS location
                        PUSH ECX

                        MOV EDI, EDX
                        MOV ECX, 128 / 4
                        XOR EAX, EAX
                        REP STOSD                   ; Zero the TSS

                        POP ECX                     ; restore ECX

                        MOV EDI, [lpaIoBitmap]
                        SUB EDI, EDX
                        ; ADD EDI, 256 / 8

                        PUSH ESI
                        MOV ESI, EDI

                        MOV dword [EDX + TSS_IOBASE], EDI
                        MOV word [EDX + TSS_SS0], 0x10

                        MOV EAX, ECX
                        ADD EAX, 0x10
                        Call RequestGDT32Entry      ; allocate a GDT entry.
                        JC .panic

                        SHL EAX, 3                  ; generate selector from index
                        MOV EDI, EAX                ; copy selector
                        ADD EAX, [lp32BitGDT]       ; get offset of descriptor

                        ; write the TSS selector to the GDT
                        MOV [EAX + GDT_BASE_LO], DX
                        SHR EDX, 16
                        ADD SI, 0x1fff
                        MOV word [EAX + GDT_LIMIT_LO], SI
                        MOV byte [EAX + GDT_BASE_HA], DL
                        MOV byte [EAX + GDT_BASE_HI], DH
                        MOV byte [EAX + GDT_FLAGS_HI], 0x00
                        MOV byte [EAX + GDT_FLAGS_LO], GDT_TYPE_TSS32 | GDT_FLAG_RING0 | GDT_FLAG_PRESENT
                        POP ESI

                        LTR DI                      ; load the task register

                        ; show status message
                        LOCK INC dword [.count]
                        MOV dword [0xB8000 + PRINTINTLINE_TSS + 0], 0x07000700 + 'T' + 65536 * 'S'
                        MOV dword [0xB8000 + PRINTINTLINE_TSS + 4], 0x07000700 + 'S' + 65536 * '('
                        MOV EAX, [.count]
                        MOV dword [0xB8000 + PRINTINTLINE_TSS + 8], EAX
                        RET

.panic:                 MOV EAX, [lpVideoMemoryBase]
                        MOV dword [EAX], 0x0c000c00 + 'G' + 'E' * 65536
                        CLI
                        OOPS

.count:                 DD 0x07000700 + ')' * 65536 + '0'




                        ; Function: InitializeTSSBSP
                        ; Initializes the TSS for the BSP, and all the setup
                        ; required for the TSSes on other processors.
                        ;
                        ; The TSSes are located after one another, and they share
                        ; an IRB/IOPB. the topmost 256 ports of the address space
                        ; are not supported to save a page per instance of the
                        ; bitmap. This method allows that the version of a bitmap
                        ; in any address space can simply be overridden by paging
                        ; in different memory. The amount of CPUs that are supported
                        ; by this method is at 511 (512*128 overflows the 16 bit
                        ; offset to the bitmap) This is well above the hardcoded
                        ; limit of 8 CPUs currently in place.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX ECX EDX EDI
InitializeTSSBSP:       MOV EAX, [dwProcessorCount]     ; get the processor count
                        MOV ECX, EAX
                        SHL EAX, 7                      ; multiply by 128
                        DEC EAX
                        OR EAX, PAGE_PARAMMASK          ; round to the next page size
                        INC EAX
                        MOV EDX, EAX                    ; copy the offset
                        ADD EAX, PAGE_SIZE * 3
                        Call AllocateFromHeap           ; allocate space for all TSSs and the IO bitmap
                        MOV EDI, lpaTSS
                        ADD EDX, EAX                    ; EDX = io bitmap offset
.fillloop:              MOV [EDI], EAX                  ; write TSS addresses
                        ADD EAX, 128
                        ADD EDI, 8
                        DEC ECX
                        JNZ .fillloop
                        PUSH EDX
                        ADD EDX, PAGE_SIZE
                        MOV [lpaIoBitmap], EDX          ; write IO bitmap address


                        ; fill the Interrupt redirection bitmap with all-ones (no redirection)
                        ; fill the I/O permission bitmap with all-ones
                        POP EDI
                        MOV ECX, 3 * PAGE_SIZE / 4
                        STC
                        SBB EAX, EAX
                        CLD
                        REP STOSD

                        LOCK INC byte [.done]

                        MOV ECX, 0
                        Call InitializeTSSAP
                        RET

.done                   DB 0




                        ; Function: PrintCPUCaps
                        ;
                        ; in:
                        ;     EAX - line number
                        ;     EBX - caps 1
                        ;     EDX - caps 2
                        ;
PrintCPUCaps:           PUSH EAX

                        XOR ECX, ECX
                        MOV CL, 18
.loop1:                 BT EBX, ECX
                        JNC .skip1
                        MOV EDI, [.caps1 + 4 * ECX]
                        MOV [.msg + 4 + 4 * ECX], EDI
                        JMP .next1
.skip1:                 MOV dword [.msg + 4 + 4 * ECX], 0x20202020
.next1:                 DEC ECX
                        JGE .loop1

                        LEA ESI, [.msg]
                        PUSH EDX
                        MOV EDX, EAX
                        Call PrintLine
                        POP EDX

                        SHR EBX, 19
                        PUSH EDX
                        SHL EDX, 32-19
                        OR EBX, EDX
                        POP EDX

                        XOR ECX, ECX
                        MOV CL, 18
.loop2:                 BT EBX, ECX
                        JNC .skip2
                        MOV EDI, [.caps2 + 4 * ECX]
                        MOV [.msg + 4 + 4 * ECX], EDI
                        JMP .next2
.skip2:                 MOV dword [.msg + 4 + 4 * ECX], 0x20202020
.next2:                 DEC ECX
                        JGE .loop2

                        PUSH EDX
                        MOV EDX, EAX
                        INC EDX
                        LEA ESI, [.msg]
                        Call PrintLine
                        POP EDX

                        MOV EBX, EDX
                        SHR EBX, 6

                        XOR ECX, ECX
                        MOV CL, 18
.loop3:                 BT EBX, ECX
                        JNC .skip3
                        MOV EDI, [.caps3 + 4 * ECX]
                        MOV [.msg + 4 + 4 * ECX], EDI
                        JMP .next3
.skip3:                 MOV dword [.msg + 4 + 4 * ECX], 0x20202020
.next3:                 DEC ECX
                        JGE .loop3

                        PUSH EDX
                        MOV EDX, EAX
                        INC EDX
                        INC EDX
                        LEA ESI, [.msg]
                        Call PrintLine
                        POP EDX


                        POP EAX

                        RET


.msg:                   DB 'Cap:'
                        TIMES 76 DB 32
                        DB 0

.caps1:                 DB 'FPU VME DE  PSE TSC MSR PAE MCE CX8 AIC 010 SE  MTR PGE MCA CMV PAT P36 PSN '
.caps2:                 DB 'CLF 020 DTS API MMX FXS SSE SE2 SS  HTT TM1 E64 PBE SE3 SE4 3DN 3DE 486 MXE '
.caps3:                 DB 'CYR A64 008 009 010 011 012 013 014 015 016 017 018 019 020 021 022 023 024 '




                        ; Function: PrintAPICLine
                        ; Prints the APIC Bus frequency
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX ECX EDX
                        ;
PrintAPICLine:          MOV dword [0xB8000 + PRINTINITLINE_APIC], 0x07500741
                        MOV dword [0xB8000 + PRINTINITLINE_APIC + 4], 0x07430749
                        MOV word  [0xB8000 + PRINTINITLINE_APIC + 8], 0x0700 + '('
                        MOV word  [0xB8000 + PRINTINITLINE_APIC + 18], 0x0700 + '.'
                        MOV dword [0xB8000 + PRINTINITLINE_APIC + 24], 0x07000700 + 'M' + 'H' * 65536
                        MOV dword [0xB8000 + PRINTINITLINE_APIC + 28], 0x07000700 + 'z' + ')' * 65536

                        MOV EAX, [ComputeApicFrequency.apicfreq]
                        SHR EAX, 16
                        XOR EDX, EDX
                        MOV ECX, 10
                        DIV ECX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 16], DX
                        XOR EDX, EDX
                        DIV ECX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 14], DX
                        XOR EDX, EDX
                        DIV ECX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 12], DX
                        XOR EDX, EDX
                        DIV ECX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 10], DX

                        MOV AX, [ComputeApicFrequency.apicfreq]
                        MOV CX, 10
                        MUL CX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 20], DX
                        MUL CX
                        ADD DX, 0x0700 + '0'
                        MOV word [0xB8000 + PRINTINITLINE_APIC + 22], DX

                        RET


                        ; Function: PrintHex4
                        ; Converts a dword value to hex and emit it to a string
                        ;
                        ; in:
                        ;     ESI - location to write
                        ;     EDX - value to convert
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ESI
                        ;
PrintHex4:              PUSH EDX
                        PUSH EAX
                        PUSH ECX
                        ADD ESI, 8
                        XOR EAX, EAX
                        MOV ECX, 8
.loop:                  MOV AL, DL
                        AND AL, 0xf
                        MOV AL, [EAX+.hexvalues]
                        MOV [ESI], AL
                        DEC ESI
                        SHR EDX, 4
                        DEC ECX
                        JNZ .loop
                        POP ECX
                        POP EAX
                        POP EDX
                        RET
.hexvalues          DB  "0123456789ABCDEF"



                        ; Function: PrintInitLine
                        ; prints the initalizer line
PrintInitLine:          MOV EDX, PRINTINITLINE_CLN
                        MOV ESI, .msg1
                        Call PrintLine
                        MOV EDX, PRINTINITLINE_PLN
                        MOV ESI, .msg2
                        Call PrintLine
                        RET

.msg1:                  DB "Initializing System Structures: ", 0
.msg2:                  DB "Initializing Peripherals: ", 0

PRINTINITLINE_CLN       EQU 12
PRINTINITLINE_PLN       EQU 13
PRINTINITLINE_STAGE     EQU 14

PRINTINTLINE_HEAP       EQU PRINTINITLINE_CLN * 160 + 2 * 32
PRINTINTLINE_GDT        EQU PRINTINITLINE_CLN * 160 + 2 * 37
PRINTINTLINE_IDT        EQU PRINTINITLINE_CLN * 160 + 2 * 44
PRINTINTLINE_TSS        EQU PRINTINITLINE_CLN * 160 + 2 * 51

PRINTINITLINE_APIC      EQU PRINTINITLINE_PLN * 160 + 2 * 26




                        ; Function: ProcessorDiagnostic
                        ; Gets the CPU, VM and related information
                        ;
                        ; in:
                        ;     EAX - index to use
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX EBX ECX EDX ESI
                        ;
                        ; The procedure is as follows:
                        ; - test eflags for ID and AC bits
                        ; - run the divtest
                        ; - use the information to classify the processor as
                        ;   386, 486+ w/o cpuid or 486+ w/ cpuid
                        ;
                        ; - for 386s, check for FPU, and set cpu info accordingly
                        ; - for 486+ w/o cpuid, check for cyiricies with cpuid disabled
                        ;   and turn it on if passed, then treat it as 486+ with cpuid
                        ; - for remaining 486+s, check for FPU presence
                        ;   and enable 486 instruction set in info
                        ; - for 486+s with cpuid, check for original signature,
                        ;   if known branch to that cpu's subroutine and configure it further
                        ; - for all unknown processors remaining, blindly load first cpuid
                        ;   capabilities set
ProcessorDiagnostic:    LOCK BTR dword [.mutex], 0  ; lock diagnostic function
                        JNC ProcessorDiagnostic     ; spin until successful

                        PUSH EAX
                        ADD AL, '0'
                        MOV [.procid], AL

                        ; perform EFLAGS test
                        PUSHFD                      ; push flags
                        POP EBX                     ; EBX = eflags
                        MOV EDX, EBX                ; EDX = original eflags
                        XOR EBX, EFLAGS_AC | EFLAGS_ID | EFLAGS_VIF
                        PUSH EBX                    ; store new eflags value
                        POPFD                       ; write eflags and mask results
                        PUSHFD                      ; read eflags
                        POP EBX                     ; load ebx with modified eflags
                        PUSH EDX                    ; push original
                        POPFD                       ; restore eflags
                        XOR EDX, EBX                ; EDX = bits that can be changed
                        MOV [.eflagtest], EDX       ; store test result
                        LEA ESI, [.eflagstoggle]    ; set up target
                        Call PrintHex4              ; print test result to display

                        ; perform DIV test
.divtestflags       EQU (EFLAGS_CF + EFLAGS_OF + EFLAGS_SF + EFLAGS_ZF + EFLAGS_PF + EFLAGS_AF)
                        PUSHFD
                        POP EBX
                        AND EBX, -1 - .divtestflags ; clear meaningful flags
                        PUSH EBX
                        MOV AX, 5
                        MOV CL, 2
                        POPFD
                        DIV CL                      ; perform the divide
                        PUSHFD
                        POP EBX
                        AND EBX, .divtestflags      ; filter meaningful flags
                        MOV [.divtest1], BX         ; write result of divtest #1

                        PUSHFD
                        POP EBX
                        OR EBX, .divtestflags       ; set meaningful flags
                        PUSH EBX
                        MOV AX, 5
                        MOV CL, 2
                        POPFD
                        DIV CL                      ; perform the divide again
                        PUSHFD
                        POP EBX
                        AND EBX, .divtestflags      ; filter meaningful flags
                        MOV [.divtest2], BX         ; write result of divtest #2

                        MOV EDX, [.divtest1]        ; load complete divtest resultset
                        LEA ESI, [.divtest]         ; load pointer to substring
                        Call PrintHex4              ; write divtest result to the ID string

                        CLD
                        LEA EDI, [.procname]

                        ; determine which detection routine to run
                        TEST dword [.eflagtest], EFLAGS_AC
                        JZ .386diagnostic           ; No AC, its a 386 style processor
                        TEST dword [.eflagtest], EFLAGS_ID
                        JNZ .cpuiddiagnostic_1      ; With CPUID, its a 2nd generation 486 or better
                        JMP .486diagnostic          ; Without CPUID, its an 1st generation 486, cyrix or the like

                        ; this is a 386. Boring
.386diagnostic:         LEA ESI, [.cpu_386compat]
                        POP EAX
                        PUSH EAX
                        MOV byte [baProcessorArch+4*EAX], 0
                        MOV dword [dwaProcessorCaps1+4*EAX], 0
                        MOV dword [dwaProcessorCaps2+4*EAX], 0
                        Call DetectCoprocessor
                        JMP .fillanddisplay

                        ; this is a 486 or better without cpuid.
.486diagnostic:         POP EAX
                        PUSH EAX
                        CMP dword [.divtest1], 0x08010000
                        JE .check_cyrix

.486diagnostic_nocyrix:
                        MOV byte [baProcessorArch+4*EAX], 0
                        MOV dword [dwaProcessorCaps1+4*EAX], 0
                        MOV dword [dwaProcessorCaps2+4*EAX], KCPUID_486
                        Call DetectCoprocessor
                        LEA ESI, [.cpu_486compat]
                        JMP .fillanddisplay

                        CPU 486
.cpuiddiagnostic_1:     MOV EAX, 0x80860000         ; Try Transmeta CPUID
                        CPUID
                        CMP EBX, [.cpuid_transmeta]
                        JNE .cpuiddiagnostic_2
                        CMP EDX, [.cpuid_transmeta+4]
                        JNE .cpuiddiagnostic_2
                        CMP ECX, [.cpuid_transmeta+8]
                        JNE .cpuiddiagnostic_2
                        JMP .check_transmeta

.cpuiddiagnostic_2:     MOV EAX, 0x80000000         ; Try AMD extended CPUID
                        CPUID
                        CMP EBX, [.cpuid_amd]
                        JNE .cpuiddiagnostic_2b
                        CMP EDX, [.cpuid_amd+4]
                        JNE .cpuiddiagnostic_2b
                        CMP ECX, [.cpuid_amd+8]
                        JNE .cpuiddiagnostic_2b
                        JMP .check_amd

.cpuiddiagnostic_2b:    XOR EAX, EAX                ; Try AMD CPUID, try 2
                        CPUID
                        CMP EBX, [.cpuid_amd]
                        JNE .cpuiddiagnostic_3
                        CMP EDX, [.cpuid_amd+4]
                        JNE .cpuiddiagnostic_3
                        CMP ECX, [.cpuid_amd+8]
                        JNE .cpuiddiagnostic_3
                        JMP .check_amd

.cpuiddiagnostic_3:     XOR EAX, EAX                ; Try Intel CPUID
                        CPUID
                        CMP EBX, [.cpuid_intel]
                        JNE .cpuiddiagnostic_4
                        CMP EDX, [.cpuid_intel+4]
                        JNE .cpuiddiagnostic_4
                        CMP ECX, [.cpuid_intel+8]
                        JNE .cpuiddiagnostic_4
                        JMP .check_intel

.cpuiddiagnostic_4:     CMP EBX, [.cpuid_cyrix]     ; Try Cyrix CPUID
                        JNE .cpuiddiagnostic_unk
                        CMP EDX, [.cpuid_cyrix+4]
                        JNE .cpuiddiagnostic_unk
                        CMP ECX, [.cpuid_cyrix+8]
                        JNE .cpuiddiagnostic_unk
                        JMP .check_cyrix

                        ; Things to do when the processor is unknown
.cpuiddiagnostic_unk:   LEA ESI, [.cpu_486compatid]
                        JMP .fillanddisplay

                        ; AMD Initialisation
.check_amd:
                        AND AH, 0x0f
                        CMP AH, 0x04
                        JE .amd_gen4
                        JG .amd_gen5
                        LEA ESI, [.cpu_amdcompat]
                        JMP .amd_caps
.amd_gen4:              LEA ESI, [.cpu_Am5x86]
                        JMP .amd_caps
.amd_gen5:              LEA ESI, [.cpu_AmdK]

.amd_caps:              ;PUSH EDX
                        MOV EAX, 1
                        CPUID
                        AND EDX, 0xffffffff - (CPUID_APIC)
                        PUSH EDX
                        MOV EAX, 0x80000001
                        CPUID
                        POP ECX
                        XOR EBX, EBX
                        TEST EDX, CPUID_LM
                        JZ .amd_lmcheckend
                        OR EBX, KCPUID_AMD64
                        CMP dword [.divtest1], 0x08D50000
                        JNE .amd_lmcheckend
                        LEA ESI, [.cpu_bochs_64]
.amd_lmcheckend:        OR EBX, KCPUID_486
                        POP EAX
                        PUSH EAX
                        MOV byte [baProcessorArch+EAX], KARCH_AMD
                        MOV dword [dwaProcessorCaps1+4*EAX], ECX
                        MOV dword [dwaProcessorCaps2+4*EAX], EBX
                        JMP .fillanddisplay

                        ; Cyrix Initialisation
.check_cyrix:           CLI                             ; force msr writes to be atomic and sequential

                        ; read device identification
                        MOV AL, CYRIX_DIR0              ; select device identification register 0
                        OUT 0x22, AL                    ; select MSR
                        IN AL, 0x23                     ; read MSR
                        MOV [.randombits], AL           ; store DIR0
                        MOV AL, CYRIX_DIR1              ; select device identification register 1
                        OUT 0x22, AL                    ; write MSR index
                        IN AL, 0x23                     ; read MSR
                        MOV [.randombits + 1], AL       ; store DIR1

                        STI

                        CMP byte [.randombits], 0x30
                        JB .check_cyrix_nocpuid

                        CLI

                        ; enable MAPEN bit
                        MOV AL, CYRIX_CCR3              ; select Configuration control register 3
                        MOV AH, AL                      ; copy
                        OUT 0x22, AL                    ; write MSR index
                        IN AL, 0x23                     ; read MSR
                        AND AL, 0x0f                    ; mask low bits
                        OR AL, 0x10                     ; set MAPEN to 1
                        XCHG AH, AL
                        OUT 0x22, AL                    ; write MSR index
                        XCHG AH, AL
                        OUT 0x23, AL                    ; write new CCR3

                        ; enable CPUID instruction
                        MOV AL, CYRIX_CCR4              ; select configuration control register 4
                        MOV AH, AL
                        OUT 0x22, AL                    ; write index
                        IN AL, 0x23                     ; read MSR
                        OR AL, 0x80                     ; set CPUID bit
                        XCHG AH, AL
                        OUT 0x22, AL                    ; write index
                        XCHG AH, AL
                        OUT 0x23, AL                    ; write new CCR4 to enable CPUID

                        STI                             ; reenable interrupts

.check_cyrix_wcpuid:    ; CPUID capable Cyrix (6x86+, MediaGX+)

                        MOV BL, [.randombits]

                        ; probe stepping in order
                        CMP BL, 0x40
                        JAE .check_cyrix_wcpuid_2
                        LEA ESI, [.cpu_Cx6x86]
                        JMP .check_cyrix_wcpuid_f
.check_cyrix_wcpuid_2:  CMP BL, 0x42
                        JNE .check_cyrix_wcpuid_3
                        LEA ESI, [.cpu_CxGXM]
                        JMP .check_cyrix_wcpuid_f
.check_cyrix_wcpuid_3:  CMP BL, 0x50
                        JAE .check_cyrix_wcpuid_4
                        LEA ESI, [.cpu_CxGX1]
                        JMP .check_cyrix_wcpuid_f
.check_cyrix_wcpuid_4:  CMP byte [.randombits + 1], 0x07
                        JB .check_cyrix_wcpuid_5
                        CMP byte [.randombits + 1], 0x80
                        JAE .check_cyrix_wcpuid_5
                        LEA ESI, [.cpu_CxMII]
                        JMP .check_cyrix_wcpuid_f
.check_cyrix_wcpuid_5:
                        LEA ESI, [.cpu_Cx6x86MX]
.check_cyrix_wcpuid_f:

                        ; TODO: fix MMX and EMMI detection

                        ; TODO: Extended CPUID

                        MOV EAX, 1
                        CPUID

                        MOV ECX, KCPUID_486

                        POP EAX                         ; get CPU number
                        PUSH EAX                        ; restore CPU number
                        MOV byte [baProcessorArch+EAX], KARCH_CYRIX
                        MOV dword [dwaProcessorCaps1+4*EAX], EDX
                        MOV dword [dwaProcessorCaps2+4*EAX], ECX
                        JMP .fillanddisplay

.check_cyrix_nocpuid:   ; Older cyrix (486, 5x86)

                        ; boring chips by default
                        XOR EDX, EDX
                        MOV ECX, KCPUID_486
                        MOV BL, [.randombits]

                        ; probe stepping in order
                        CMP BL, 0x10
                        JAE .check_cyrix_nocpuid_2
                        LEA ESI, [.cpu_Cx486]
                        JMP .check_cyrix_nocpuid_f
.check_cyrix_nocpuid_2: CMP BL, 0x14
                        JAE .check_cyrix_nocpuid_3
                        LEA ESI, [.cpu_Cx486S]
                        JMP .check_cyrix_nocpuid_f
.check_cyrix_nocpuid_3: CMP BL, 0x20
                        JAE .check_cyrix_nocpuid_4
                        LEA ESI, [.cpu_Cx486DX]
                        JMP .check_cyrix_nocpuid_f
.check_cyrix_nocpuid_4: LEA ESI, [.cpu_Cx5x86]
.check_cyrix_nocpuid_f:

                        POP EAX
                        PUSH EAX
                        MOV byte [baProcessorArch+EAX], KARCH_CYRIX
                        MOV dword [dwaProcessorCaps1+4*EAX], EDX
                        MOV dword [dwaProcessorCaps2+4*EAX], ECX
                        JMP .fillanddisplay


                        ; Intel Initialisation
.check_intel:           MOV EAX, 1
                        CPUID
                        ; Todo: fix cpuid bugs for intel class chips
                        ;AND EDX, 0xffffffff - (CPUID_APIC | CPUID_SYSENTER)
                        AND AH, 0x0f
                        CMP AH, 0x04
                        JE .intel_486
                        MOV CL, KARCH_PENTIUM
                        CMP AH, 0x05
                        JE .intel_586
.intel_686:             LEA ESI, [.cpu_i686]
                        Call CheckDisabledAPIC
                        JMP .intel_write
.intel_486:             MOV CL, KARCH_486
                        LEA ESI, [.cpu_i486]
                        JMP .intel_write
.intel_586:             LEA ESI, [.cpu_i586]
                        JMP .intel_write

.intel_write:           POP EAX
                        PUSH EAX
                        MOV byte [baProcessorArch+EAX], CL
                        MOV dword [dwaProcessorCaps1+4*EAX], EDX
                        MOV dword [dwaProcessorCaps2+4*EAX], KCPUID_486

                        JMP .fillanddisplay

                        ; Transmeta Initialisation
.check_transmeta:       LEA ESI, [.cpu_tmcompat]
                        JMP .fillanddisplay
                        CPU 386

                        ; fill processor name
.fillanddisplay:        MOV ECX, 4
                        REP MOVSD
.display:               POP EAX                         ; recall processor number
                        MOV EDX, [dwaProcessorCaps1+4*EAX]
                        LEA ESI, [.cpuid1]
                        Call PrintHex4                  ; print CPU features #1
                        MOV EDX, [dwaProcessorCaps2+4*EAX]
                        LEA ESI, [.cpuid2]
                        Call PrintHex4                  ; print CPU features #2

                        INC EAX
                        LEA ESI, [.infostring]
                        MOV EDX, EAX
                        Call PrintLine                  ; print CPU diagnostic line
                        LOCK BTS dword [.mutex], 0      ; unlock diagnostic function
                        RET

.mutex:                 DD 1


.infostring:            DB "Proc #"
.procid:                DB 32, ": "
.procname:              TIMES 17 DB 32
.cpuid1:                TIMES 9 DB 32
.cpuid2:                TIMES 9 DB 32
.eflagstoggle:          TIMES 9 DB 32
.divtest:               TIMES 9 DB 32
.termzero:              DB 0

; local test results
.eflagtest:
.randombits:            DW 0
                        DW 0
.divtest1:              DW 0
.divtest2:              DW 0

; flags allowed
.systemsupport1:        DD KCPUID_CMPXCHG8B | KCPUID_TSC | KCPUID_CMOV | KCPUID_APIC | KCPUID_PAE
.systemsupport2:        DD KCPUID_486 | KCPUID_AMD64

; cpuid return strings
.cpuid_amd:             DB "AuthenticAMD"
.cpuid_cyrix:           DB "CyrixInstead"
.cpuid_intel:           DB "GenuineIntel"
.cpuid_transmeta:       DB "TransmetaCPU"

; cpu names
.cpu_amdcompat          DB "Unknown AMD     "
.cpu_cxcompat           DB "Unknown Cyrix   "
.cpu_tmcompat           DB "Unkn. Transmeta "

.cpu_i386:              DB "Intel 386       "
.cpu_i486:              DB "Intel 486       "
.cpu_i586:              DB "Intel 586       "
.cpu_i686:              DB "Intel 686+      "

.cpu_Am5x86             DB "Am486 or Am5x86 "
.cpu_AmdK               DB "AMD K6 or better"
.cpu_Amd64              DB "AMD64 or better "

.cpu_Cx486              DB "Cyrix Cx486     "
.cpu_Cx486S             DB "Cyrix Cx486S    "
.cpu_Cx486DX            DB "Cyrix Cx486DX   "
.cpu_Cx5x86             DB "Cyrix Cx5x86    "
.cpu_Cx6x86             DB "Cyrix 6x86      "
.cpu_Cx6x86MX           DB "Cyrix 6x86MX    "
.cpu_CxMII              DB "Cyrix MII       "
.cpu_CxGX1              DB "Cyrix MediaGX   "
.cpu_CxGXM              DB "Cyrix GXm       "

.cpu_386compat:         DB "Unknown 386     "
.cpu_486compat:         DB "Unknown 486+    "
.cpu_486compatid:       DB "Unknown 486+ID  "
.cpu_unk_x86_64:        DB "Unknown 64bit   "

.cpu_bochs_64:          DB "Bochs x86-64    "





                        ; Function: SetCPUBits
                        ; loads CR0/CR4 with the appropriate bits
                        ;
                        ; note: cyrices (and possibly others) do not support cr4
                        ; so do no write it when we don't need it
                        ;
                        ; in:
                        ;     ECX - processor number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
SetCPUBits:             MOV EAX, [dwaProcessorCaps1+4*ECX]
                        PUSH EDX
                        NOT EAX ; make test-all-bit comparisons easier
                        XOR EDX, EDX

                        ; VME and PVI
                        TEST EAX, KCPUID_VME
                        JNZ .novme
                        OR EDX, CR4_VME | CR4_PVI
.novme:
                        ; Debugging Extentions
                        TEST EAX, KCPUID_DE
                        JNZ .node
                        ; do something
.node:
                        ; Page Size Extentions
                        TEST EAX, KCPUID_PSE
                        JNZ .nopse
                        OR EDX, CR4_PSE
.nopse
                        ; Global Pages
                        TEST EAX, KCPUID_PGE
                        JNZ .nogp
                        OR EDX, CR4_PGE
.nogp:
                        ; SSE sets several cr4 bits
                        TEST EAX, KCPUID_SSE | KCPUID_FXSR
                        JNZ .nosse
                        OR EDX, CR4_OSFXSR | CR4_OSXMMEXCPT
.nosse:
                        ; Write to CR4
                        OR EDX, EDX
                        JZ .dontwrite
                        MOV CR4, EDX
.dontwrite:             POP EDX

                        RET


; todo: set SSE bits in CR4


; ====================================================================================================================
; Group: Runtime components
; Common routines during kernel operation.
; ====================================================================================================================
kernelruntime:          Call Schedule
                        JMP kernelruntime



                        ; Function: AddMemoryReference32
                        ; Ups the reference count to a memory location.
                        ;
                        ; in:
                        ;     EAX - memory address to reference
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ECX, EDX
                        ;
AddMemoryReference32:   Call FindL3MemoryTable32
                        LOCK INC dword [EDX]
                        JMP ReleaseMemoryLocks




                        ; Function: AllocateGDT32Entry
                        ; finds an empty GDT entry and allocates it.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     EAX - GDT segment selector
                        ;     CF - clear on success, set if the GDT is fully occupied
                        ;
                        ; destroyed:
                        ;     none
AllocateGDT32Entry:     PUSH EDX
                        MOV EAX, [.cache]
                        MOV EDX, EAX
.retry:                 Call RequestGDT32Entry
                        JNC .done
                        INC EAX
                        AND EAX, 0x000001FF
                        CMP EAX, EDX
                        JNE .retry
                        XOR EAX, EAX
                        STC
                        RET
.done:                  MOV [.cache], EAX
                        POP EDX
                        CLC
                        RET

.cache:                 DD 0x00000007




                        ; Function: AllocateIDTEntry
                        ; Tries to allocate one entry in the IDT
                        ;
                        ; in:
                        ;     EAX - interrupt number
                        ;
                        ; out:
                        ;     CF - clear on success, set if the entry has been taken
                        ;
                        ; destroyed:
                        ;     none
                        ;

                        ; Todo: fail stub
AllocateIDTEntry:       CMP AL, 0x20
                        STC
                        RET




                        ; Function: AllocateMemory32
                        ; Tries to allocate a piece of memory
                        ; To allow for full userspace control of memory
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     EAX - physical address of allocated page of memory
                        ;     CF - set if no memory could be allocated
                        ;
                        ; destroyed:
                        ;     ECX, EDX
                        ;
AllocateMemory32:       PUSH ESI
                        MOV ESI, [.cachepos]
                        MOV EAX, ESI
                        CMP EAX, [.end]
                        JB .check2
                        MOV EAX, [.end]
.check2:                CMP EAX, [.start]
                        JAE .loop
                        MOV EAX, [.start]
                        CMP EAX, [.end]
                        JB .loop
                        STC
                        RET
.loop:                  PUSH EAX
                        CALL RequestMemory32
                        OR EAX, EAX
                        JZ .checkquit
                        POP EAX
                        ADD EAX, 0x1000
                        CMP EAX, [.end]
                        JBE .norotate
                        MOV EAX, [.start]
.norotate:              CMP EAX, ESI
                        JNE .loop
                        XOR EAX, EAX
                        POP ESI
                        STC
                        RET
.checkquit:             CMP CL, 1
                        JNE .crashburn
                        POP EAX
                        POP ESI
                        CLC
                        RET

                        ; fork the page
.crashburn:             OOPS



ALIGN 4
.start:                 DD 0x00200000
.end:                   DD 0x00400000
.cachepos:              DD 0x00200000



                        ; Function: ClearPageTableEntry32Hole
                        ; Clears a page table entry if it is mapped and not used by the kernel.
                        ;
                        ; This version uses the paging holes to temporarily map in the tables
                        ; for write access. Other write methods may be faster
                        ;
                        ; This assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     EDX - the physical address of the page directory
                        ;     EDI - the virtual address to unmap
                        ;
                        ; out:
                        ;     EDX - the page table entry
                        ;     CF - clear if the entry was unmapped, set if the write could not be performed
                        ;
                        ; destroyed:
                        ;     EAX, EDX, EDI
                        ;
ClearPageTableEntry32Hole:
                        MOV EAX, [lpPagingHoles]        ; get the location where we can allocate stuff
                        PUSH EBX
                        AND EAX, PAGE_TABLEMASK
                        MOV EBX, [lpPagingKernelTables] ; get the kernel area page table
                        SHR EAX, 10
                        ADD EAX, EBX                    ; location of the entry to alter
                        STC
                        SBB EBX, EBX

                        ; lock the holes
.lock1:                 LOCK BTS dword [lpPagingLocks], 0
                        JC .lock1
.lock2:                 LOCK BTS dword [lpPagingLocks], 1
                        JC .lock2

                        ; map in page directory
                        OR EDX, PAGE_PRESENT | PAGE_WRITABLE
                        MOV [EAX], EDX
                        ; fixme: improve speed here.
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; look up table address
                        PUSH EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 20
                        MOV EDX, [EDX+EDI]

                        ; test if we can write here (i.e. there's a table present)
                        TEST EDX, PAGE_PRESENT
                        JZ .crashburnpop
                        TEST EDX, PAGE_GRANULAR
                        JNZ .crashburnpop

                        AND EDX, ~(PAGE_USERSPACE | PAGE_WRITETHROUGH | PAGE_UNCACHEABLE)
                        OR EDX, PAGE_WRITABLE

                        ; map in page table
                        MOV [EAX+4], EDX
                        ; fixme: improve speed
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; check page table entry
                        POP EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_TABLEMASK
                        SHR EDI, 10
                        LEA EDX, [EDX+EDI+PAGE_SIZE]
                        MOV EBX, [EDX]
                        TEST EBX, PAGE_USERSPACE
                        JZ .crashburn
                        TEST EBX, PAGE_PRESENT
                        JZ .crashburn

                        ; write entry
                        MOV dword [EDX], 0
                        MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        MOV EDX, EBX
                        CLC
                        POP EBX
                        RET

.crashburnpop:          POP EDI
.crashburn:             MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        MOV EDX, EBX
                        STC
                        POP EBX
                        RET
                        
                        
                        
                        ; Function: DumpCRState
                        ; prints a crashdump with the CRx registers
                        ;
                        ; in: 
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
DumpCRState:            PUSHAD
                        MOV DWORD [DumpRegState.string], 'CR0 '
                        MOV EAX, CR0
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 16
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'CR2 '
                        MOV EAX, CR2
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 17
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'CR3 '
                        MOV EAX, CR3
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 18
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'CR4 '
                        MOV EAX, CR4
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 19
                        Call PrintLine
                                                
                        POPAD
                        RET
                        
                        
                        
                        
                        ; Function: DumpIntState
                        ; prints a crashdump from an interrupt stackframe
                        ;
                        ; in: 
                        ;     EBP - pointer to top of interrupt frame
                        ;     EAX - interrupt/exception number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
DumpIntState:           PUSHAD
.remainder:             MOV DWORD [DumpRegState.string], 'INT '                        
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 10
                        Call PrintLine

                        MOV DWORD [DumpRegState.string], 'EIP '
                        MOV EAX, [EBP]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 11
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'CS  '
                        MOV EAX, [EBP + 4]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 12
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'EFL '
                        MOV EAX, [EBP + 8]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 13
                        Call PrintLine
                        
                        CMP DWORD [EBP + 4], 8
                        JE .nouserstack
                        
                        MOV DWORD [DumpRegState.string], 'USP '
                        MOV EAX, [EBP + 12]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 14
                        Call PrintLine
                        
                        MOV DWORD [DumpRegState.string], 'USS '
                        MOV EAX, [EBP + 16]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 15
                        Call PrintLine
.nouserstack                        
                        POPAD
                        RET
                        
                        
                        
                        
                        ; Function DumpIntErrorState
                        ; prints a crashdump from an interrupt stackframe with error code
                        ;
                        ; in:
                        ;     EBP - pointer to top of interrupt frame
                        ;     EAX - interrupt/exception number                        
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;                        
DumpIntErrorState:      PUSHAD
                        PUSH EAX
                        MOV DWORD [DumpRegState.string], 'EC  '
                        MOV EAX, [EBP]
                        CALL DumpRegState.hexconv
                        MOV ESI, DumpRegState.string
                        MOV EDX, 9
                        Call PrintLine
                        
                        POP EAX
                        ADD EBP, 4
                        JMP DumpIntState.remainder


                        
                        ; Function: DumpRegState
                        ; Prints a crashdump from a PUSHAD image
                        ;
                        ; in:
                        ;     EBP - pointer to PUSHAD image
                        ;
                        ; out:
                        ;     none
                        ; 
                        ; destroyed:
                        ;     none
DumpRegState:           PUSHAD
                        XOR EDX, EDX
.regloop:               MOV EAX, [.regdigits + 4*EDX]
                        MOV [.string], EAX
                        MOV EAX, [EBP + 4*EDX]
                        INC EDX
                        CALL .hexconv
                        MOV ESI, .string
                        PUSH EDX
                        CALL PrintLine
                        POP EDX
                        CMP EDX, 8
                        JNE .regloop
                        
                        POPAD
                        RET
             
.hexconv:		PUSHAD
                        MOV ESI, .string2 + 7
                        MOV ECX, 8
                        XOR EDX, EDX
.hexloop:               MOV DL, AL
                        AND DL, 0x0F
                        MOV DL, [.hexdigits + EDX]
                        MOV [ESI], DL
                        DEC ESI
                        SHR EAX, 4
                        LOOP .hexloop
                        POPAD
                        RET
                        
.hexdigits:             DB "0123456789ABCDEF"
.regdigits:             DB "EDI ESI EBP ESP EBX EDX ECX EAX "
.string:                DB "XXX 0x"
.string2:               DB "00000000"
                        DB 32, 0
                        
                        
                        
                        ; Function: EnableIRQ
                        ; Enables one IRQ at the PIC
                        ;
                        ; in:
                        ;     EAX - interrupt number
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;

EnableIRQ:              PUSH EAX
                        PUSH ECX
                        CMP AL, 8       ; check which pic is involved
                        JGE .slave      ; jump if its the slave PIC
                        MOV CL, AL
                        MOV CH, 0xfe
                        STC
                        ROL CH, CL
                        LOCK AND [bMasterMask], CH
                        MOV AL, [bMasterMask]
                        OUT PIC1_DATA, AL
                        POP ECX
                        POP EAX
                        RET

.slave:                 MOV CH, 0xfe
                        MOV CL, AL
                        SUB CL, 8
                        STC
                        ROL CH, CL
                        LOCK AND [bSlaveMask], CH
                        MOV AL, [bSlaveMask]
                        OUT PIC2_DATA, AL
                        POP ECX
                        POP EAX
                        RET




                        ; Function: FillKernelPageDirectory
                        ; Fills a page directory with default values
                        ;
                        ; in:
                        ;      EDI - virtual address of the first page table
                        ;      EBX - virtual address of the page directory
                        ;      ESI - physical address of the first page table
                        ;      EDX - physical address of the page directory
                        ;
                        ; out:
                        ;      none
                        ;
                        ; destroyed:
                        ;      EDI, ESI, ECX
                        ;
FillKernelPageDirectory:MOV ECX, ESI
                        OR ECX, PAGE_USERSPACE | PAGE_WRITABLE | PAGE_PRESENT
                        MOV [EBX], ECX

                        CALL FillKernelPageTable

                        PUSH EAX
                        LEA EDI, [EBX+4]
                        MOV ECX, (PAGE_SIZE / 4) - 1
                        XOR EAX, EAX
                        REP STOSD

                        POP EAX
                        RET




                        ; Function: FillKernelPageTable
                        ; Fills a page table with all the kernel pages
                        ;
                        ; in:
                        ;      EDI - virtual address to write to
                        ;      ESI - physical address of the page table
                        ;      EDX - physical address of the page directory
                        ;
                        ; out:
                        ;      none
                        ;
                        ; destroyed:
                        ;      EDI ECX
                        ;
FillKernelPageTable:    PUSH ESI
                        MOV ESI, [lpPageTableSource]
                        MOV ECX, PAGE_SIZE / 4
                        REP MOVSD                           ; copy the base page table over
                        POP ESI

                        MOV ECX, [lpPagingKernelTables]
                        PUSH ESI
                        AND ECX, PAGE_TABLEMASK
                        PUSH EDX
                        SHR ECX, 10
                        OR EDX, PAGE_PRESENT | PAGE_WRITABLE
                        LEA ECX, [ECX+EDI-PAGE_SIZE]
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE
                        MOV [ECX], ESI
                        POP EDX
                        MOV ESI, EDX
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE
                        MOV [ECX+4], ESI
                        POP ESI

                        RET




                        ; Function: FillPage32
                        ; Fills a page in physical memory with all ones
                        ;
                        ; Maps the page into a predefined memory hole, then
                        ; clears it.
                        ;
                        ; Assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     EAX - offset of the page
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
FillPage32:             PUSH ECX
                        PUSH EAX
                        MOV ECX, [lpZeroHole]           ; get the location of the hole used
                        AND EAX, PAGE_ADDRMASK
                        SHR ECX, 12-2
                        OR EAX, PAGE_PRESENT | PAGE_WRITABLE
                        PUSH EDI
                        ADD ECX, [lpPagingKernelTables] ; ECX = page table entry to alter

                        ; grab the lock
.check                  CMP word [ZeroPage32.lock], 0
                        JNE .check
                        LOCK BTS word [ZeroPage32.lock], 0
                        JC .check

                        ; write the page table and flush TLB
                        MOV [ECX], EAX
                        MOV ECX, CR3
                        MOV CR3, ECX

                        ; zero memory
                        MOV EDI, [lpZeroHole]
                        MOV ECX, PAGE_SIZE / 4
                        STC
                        SBB EAX, EAX
                        REP STOSD

                        ; release the lock and exit
                        LOCK BTC word [ZeroPage32.lock], 0
                        POP EDI
                        POP EAX
                        POP ECX
                        RET



                        
                        ; Function: FillTaskDescriptor32
                        ; Fills a page with a task descriptor and a schedulable stack
                        ; This descriptor is designed to run 32-bit applications
                        ;
                        ; in:
                        ;     EDI - location to write
                        ;     ESI - location of page in target virtual memory
                        ;     EDX - location of page in physical memory
                        ;     EBX - starting EIP
                        ;
                        ; out:
                        ;     EBX - pointer to GPR contents
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
FillTaskDescriptor32:   PUSH EDI
                        XOR EAX, EAX
                        MOV ECX, PAGE_SIZE / 4
                        CLD
                        REP STOSD
                        POP EDI
                        MOV dword [EDI+TDS_DESCHEDULE], Base_ScheduleIn
                        ; write something sensible to that XMM control word
                        MOV dword [EDI+TDS_FPUSTATE+24], 0x00001f80
                        MOV dword [EDI+TDS_FPUSTATE+28], 0x0000ffff
                        PUSH EDI
                        PUSH ESI
                        STD
                        ADD EDI, 0xFFC
                        MOV ECX, (.stackdataend - .stackdata) / 4
                        MOV ESI, .stackdataend - 4
                        REP MOVSD
                        MOV EAX, EDI
                        ADD EAX, 4
                        POP ESI
                        POP EDI
                        CLD

                        MOV [EDI + 0x1000 - 5 * 4], EBX
                        MOV EBX, EAX
                        SUB EAX, EDI
                        ADD EAX, ESI
                        MOV [EDI + TDS_BASE_ESP], EAX
                        LEA EAX, [ESI+0x1000]
                        MOV [EDI + TDS_BASE_ESP0], EAX
                        MOV [EDI + TDS_PHYSICAL], EDX
                        RET




.stackdata:             DD 0                    ; ^
                        DD 0                    ; |
                        DD 0                    ; |
                        DD 0                    ; | PUSHAD
                        DD 0                    ; |
                        DD 0                    ; |
                        DD 0                    ; |
                        DD 0                    ; v
                        DD 0                    ; POP        EBX
                        DD Scheduler_Launchuser ; RET
                        DD 0x20 | 3             ; POP        DS
                        DD 0x20 | 3             ; POP        ES
                        DD 0                    ; ^          EIP
                        DD 0x18 | 3             ; |          CS
                        DD EFLAGS_IF            ; | IRETD    EFLAGS
                        DD 0                    ; |          ESP3
                        DD 0x20 | 3             ; v          SS3
.stackdataend:




                        ; Function: FillTaskDescriptorRemote32
                        ; Fills a task structure in physical memory
                        ;
                        ; Maps the page into a predefined memory hole, then
                        ; clears it.
                        ;
                        ; Assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     ESI - location of page in target virtual memory
                        ;     EAX - location of page in physical memory
                        ;     EBX - starting EIP
                        ;     EDX - starting ESP
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EBX, ESI, EDI
                        ;
FillTaskDescriptorRemote32:
                        PUSH ECX
                        PUSH EDX
                        PUSH EAX
                        MOV ECX, [lpZeroHole]           ; get the location of the hole used
                        AND EAX, PAGE_ADDRMASK
                        SHR ECX, 12-2
                        OR EAX, PAGE_PRESENT | PAGE_WRITABLE
                        ADD ECX, [lpPagingKernelTables] ; ECX = page table entry to alter

                        ; grab the lock
.check                  CMP word [ZeroPage32.lock], 0
                        JNE .check
                        LOCK BTS word [ZeroPage32.lock], 0
                        JC .check

                        ; write the page table and flush TLB
                        MOV [ECX], EAX
                        MOV ECX, CR3
                        MOV CR3, ECX

                        ; fill page
                        POP EDX
                        PUSH EDX
                        MOV EDI, [lpZeroHole]
                        Call FillTaskDescriptor32
                        POP EAX
                        POP EDX
                        MOV [EBX+15*4], EDX

                        ; release the lock and exit
                        LOCK BTC word [ZeroPage32.lock], 0
                        POP ECX
                        RET




                        ; Function: FindL3MemoryTable32
                        ; Locks onto the memory address controlling this entry
                        ;
                        ; in:
                        ;     EAX - address to add reference to
                        ;
                        ; out:
                        ;     CL  - level of this entry
                        ;     CH  - locks taken
                        ;     EDX - memory location governing this entry (points to dword containing reference count)
                        ;
                        ; destroyed:
                        ;     ECX
FindL3MemoryTable32:    MOV EDX, EAX
                        PUSH ESI
                        XOR ECX, ECX
                        SHR EDX, 30-3
                        AND DL, 0xf8
                        ADD EDX, [lpL3MemoryTable]
                        MOV ESI, [EDX]
                        AND ESI, PAGE_ADDRMASK
                        CMP byte [EDX+7], PMAB_POINTER_VIRT
                        JE FindL2MemoryTable32.recurse
                        CMP byte [EDX+7], PMAB_POINTER_PHYS
                        JE .map
                        MOV ECX, 3
                        POP ESI
                        RET
                        ;todo: resolve virtual pointers
.map:                   CMP word [.lock], 0
                        JNE .map
                        LOCK BTS word [.lock], 0
                        JC .map

                        PUSH EAX
                        MOV EAX, ESI
                        MOV EDX, [lpMMTableHoles]           ; get the location of the hole used
                        AND EAX, PAGE_ADDRMASK
                        ADD EDX, 0x1000
                        MOV ESI, EDX
                        SHR EDX, 12-2
                        ADD EAX, PAGE_PRESENT | PAGE_WRITABLE
                        ADD EDX, [lpPagingKernelTables] ; ECX = page table entry to alter
                        MOV [EDX], EAX
                        MOV EAX, CR3
                        MOV CR3, EAX
                        OR CH, 2
                        POP EAX
                        JMP FindL2MemoryTable32.recurse


                        ;LOCK BTC word [.lock], 0
                          OOPS


.lock:                  DW 0




                        ; Function: FindL2MemoryTable32
                        ; Increases the reference count on one memory unit
                        ;
                        ; in:
                        ;     EAX - address to add reference to
                        ;     ESI - address of the corresponding L2 memory table
                        ;
                        ; out:
                        ;     CL  - level of this entry
                        ;     CH  - locks taken
                        ;     EDX - memory location governing this entry
                        ;
                        ; destroyed:
                        ;     ECX
FindL2MemoryTable32:    PUSH ESI
                        XOR ECX, ECX
.recurse:               MOV EDX, EAX
                        SHR EDX, 21-3
                        AND EDX, 0x00000ff8
                        ADD EDX, ESI
                        MOV ESI, [EDX]
                        AND ESI, PAGE_ADDRMASK
                        CMP byte [EDX+7], PMAB_POINTER_VIRT
                        JE FindL1MemoryTable32.recurse
                        CMP byte [EDX+7], PMAB_POINTER_PHYS
                        JE .map
                        MOV CL, 2
                        POP ESI
                        RET

                        ; todo: resolve virtual pointers

.map:                   CMP word [.lock], 0
                        JNE .map
                        LOCK BTS word [.lock], 0
                        JC .map

                        PUSH EAX
                        MOV EAX, ESI
                        MOV EDX, [lpMMTableHoles]           ; get the location of the hole used
                        AND EAX, PAGE_ADDRMASK
                        ADD EDX, 0x2000
                        MOV ESI, EDX
                        SHR EDX, 12-2
                        ADD EAX, PAGE_PRESENT | PAGE_WRITABLE
                        ADD EDX, [lpPagingKernelTables] ; ECX = page table entry to alter
                        MOV [EDX], EAX
                        MOV EAX, CR3
                        MOV CR3, EAX
                        OR CH, 4
                        POP EAX
                        JMP FindL1MemoryTable32.recurse

                        ALIGN 4
.lock:                  DD 0


                        ; Function: FindL1MemoryTable32
                        ; Increases the reference count on one memory unit
                        ;
                        ; in:
                        ;     EAX - address to add reference to
                        ;     ESI - address of the corresponding L1 memory table
                        ;
                        ; out:
                        ;     CL  - level of this entry
                        ;     CH  - locks taken
                        ;     EDX - memory location governing this entry
                        ;
                        ; destroyed:
                        ;     ECX(16:31)
                        ;
FindL1MemoryTable32:    PUSH ESI
                        XOR ECX, ECX
.recurse:               MOV EDX, EAX
                        AND EDX, PAGE_TABLEMASK
                        SHR EDX, 10
                        ADD EDX, ESI
                        MOV CL, 1
                        POP ESI
                        RET




                        ; Function: GetTssBitmapStatus32
                        ; Checks whether a page in the TSS has been mapped
                        ;
                        ; in:
                        ;     EDI - the page's index
                        ;
                        ; out:
                        ;     CF - set if the page can be written to, clear if unmapped
                        ;
                        ; destroyed:
                        ;     EAX
GetTssBitmapStatus32:   MOV EAX, [lpaIoBitmap]              ; get the address of the Io permission bitmap
                        AND EAX, PAGE_TABLEMASK
                        SHR EAX, 12-2                       ; get the page number, multiplied by 4
                        LEA EAX, [EAX+4*EDI]                ; get the offset into the page table
                        ADD EAX, [lpPagingKernelTables]     ; index into the kernel area page table
                        BT dword [EAX], PAGE_FREE1_BIT
                        RET




                        ; Function: HardwareDelay
                        ; Delays for a short period of time.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
HardwareDelay:          PUSH ECX
                        MOV ECX, 32
.loop:                  XOR ECX, EAX    ; some pipeline hostile NOP
                        XOR ECX, EAX
                        LOOP .loop      ; an iteration takes 3 clocks average
                        POP ECX
                        RET




                        ; Function: InsertLargePage32
                        ; adds a given large page to a page directory
                        ;
                        ; in:
                        ;     EAX - 4M Page address
                        ;     EBX - Directory address
                        ;     EDX - address to map to
                        ;
                        ; out:
                        ;     CF - success if clear, error if set
                        ;
                        ; destroyed:
                        ;     none
                        ;
InsertLargePage32:      PUSH EDI
                        PUSH ESI
                        MOV EDI, EDX
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 22 - 2
                        ADD EDI, EBX
.wait:                  CMP dword [EDI], 0
                        JNE .bail
                        LOCK BTS word [InsertPageTable32.lock], 0
                        JC .wait
                        CMP dword [EDI], 0
                        JNE .derefbail
                        MOV ESI, EAX
                        OR ESI, PAGE_PRESENT | PAGE_USERSPACE | PAGE_WRITABLE | PAGE_GRANULAR
                        MOV [EDI], ESI
                        LOCK BTC word [InsertPageTable32.lock], 0
                        POP ESI
                        POP EDI
                        CLC
                        RET

.derefbail:             LOCK BTC word [InsertPageTable32.lock], 0
.bail:                  POP ESI
                        POP EDI
                        STC
                        RET





                        ; Function: InsertPageTable32
                        ; adds a given page table to a page directory
                        ;
                        ; in:
                        ;     EAX - table address
                        ;     EBX - directory address
                        ;     EDX - address to map to
                        ;
                        ; out:
                        ;     CF - success if clear, error if set
                        ;
                        ; destroyed:
                        ;     none
                        ;
InsertPageTable32:      PUSH EDI
                        PUSH ESI
                        MOV EDI, EDX
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 22 - 2
                        ADD EDI, EBX
.wait:                  CMP dword [EDI], 0
                        JNE .bail
                        LOCK BTS word [.lock], 0
                        JC .wait
                        CMP dword [EDI], 0
                        JNE .derefbail
                        MOV ESI, EAX
                        OR ESI, PAGE_PRESENT | PAGE_USERSPACE | PAGE_WRITABLE
                        MOV [EDI], ESI
                        LOCK BTC word [.lock], 0
                        POP ESI
                        POP EDI
                        CLC
                        RET

.derefbail:             LOCK BTC word [.lock], 0
.bail:                  POP ESI
                        POP EDI
                        STC
                        RET

                        ALIGN 2
.lock                   DW 0



                        ; Function: InsertPage32
                        ; adds an entry to the page table.
                        ;
                        ; The entry may not be allocated before writing
                        ;
                        ; in:
                        ;     EDX - location of the page table
                        ;     ESI - physical address and access bytes of the page to map
                        ;     EDI - virtual address to map to
                        ;
                        ; out:
                        ;     CF - success if clear, error if set
                        ;
                        ; destroyed:
                        ;     none
                        ;
InsertPage32:           PUSH EAX
                        MOV EAX, EDI
                        AND EAX, PAGE_TABLEMASK
                        SHR EAX, 12 - 2
                        ADD EAX, EDX
.wait:                  CMP dword [EAX], 0
                        JNE .bail
                        LOCK BTS word [.lock], 0
                        JC .wait
                        CMP dword [EAX], 0
                        JNE .derefbail
                        MOV [EAX], ESI
                        LOCK BTC word [.lock], 0
                        POP EAX
                        CLC
                        RET

.derefbail:             LOCK BTC word [.lock], 0
.bail:                  POP EAX
                        STC
                        RET

                        ALIGN 2
.lock                   DW 0



                        ; Function: InsertTSSBitmapPage
                        ; Maps a page into the shared TSS area.
                        ; Checks if the map can be performed, then sets the map atomically.
                        ;
                        ; in:
                        ;     EAX - address of the page to map
                        ;     EDI - index of the page to map (0-1)
                        ;
                        ; out:
                        ;     CF - clear on success, set on failure
                        ;
                        ; destroyed:
                        ;     ECX
                        ;
InsertTSSBitmapPage:    MOV ECX, [lpaIoBitmap]          ; get the address of the bitmap
                        AND ECX, PAGE_TABLEMASK         ; mask out the page address
                        SHR ECX, 10                     ; shift to page number * 4
                        LEA ECX, [ECX+4*EDI]            ; make offset into table
                        ADD ECX, [lpPagingKernelTables]
                        ; get address of entry to alter

                        PUSH EAX
                        OR EAX, PAGE_PRESENT | PAGE_WRITABLE | PAGE_FREE1

.prelock:               BT word [ECX], PAGE_FREE1_BIT        ; check if the write is allowed
                        JC .fail
                        ; todo: write 486 overlay (cmpxchg)
                        CMP word [.lock], 0
                        JNE .prelock
                        LOCK BTS word [.lock], 0
                        JC .prelock
                        BT word [ECX], PAGE_FREE1_BIT
                        JC .lockedfail
                        MOV [ECX], EAX
                        LOCK BTC word [.lock], 0

                        ; todo: proper page flushing
                        MOV EAX, CR3
                        MOV CR3, EAX

                        POP EAX
                        CLC
                        RET




.lockedfail:            LOCK BTC word [.lock], 0
.fail:                  POP EAX
                        STC
                        RET

                        ALIGN 2
.lock                   DW 0




                        ; Function: IntHandler
                        ; Provides the default interrupt handlers.
                        ; These functions are called by the processor, and should not be accessed from applications
                        ;
                        ; in:
                        ;     _do not call_
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
IntHandler:

.errstub:               ;;OOPS
                        LIDT [0]
                        INT 0
                        JMP .errstub

.errstub2:              ;OOPS
                        MOV EAX, CR2
                        SHR EAX, 18
                        NOT EAX
                        AND AL, 0x7
                        Call .kbled
                        JMP $

.kbled:                 PUSH EAX
                        CALL .kbwait
                        MOV AL, 0xED
                        CALL .kbwrite
                        CALL .kbwait
                        POP EAX
                        CALL .kbwrite
                        RET

.kbwrite:               PUSH EAX
                        CALL .kbwait
                        POP EAX
                        OUT 0x60, AL
                        RET

.kbwait:                IN AL, 0x64
                        TEST AL, 0x02
                        JNZ .kbwait
                        RET

.unimplstub:            ;JMP IntHandler.errstub
                        IRETD

.masterstub:            PUSH EAX
                        MOV AL, PIC_EOI
                        OUT PIC1_COMMAND, AL
                        POP EAX
                        IRETD
.slavestub:             PUSH EAX
                        MOV AL, PIC_EOI
                        OUT PIC2_COMMAND, AL
                        OUT PIC1_COMMAND, AL
                        POP EAX
                        IRETD



                        ; Function: MapL1Table32
                        ; Inserts a new leaf into the memory allocator map
                        ;
                        ; in:
                        ;     EAX - memory address to reference
                        ;     EDI - memory location to point to
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ECX, EDX
                        ;
MapL1Table32:           Call FindL3MemoryTable32
                        CMP CL, 2
                        JNE .crashburn
                        MOV [EDX], EDI
                        ADD EDI, 0x800
                        MOV [EDX+8], EDI
                        LOCK XCHG CL, [.serialize]
                        MOV dword [EDX+4], 0x80000000
                        MOV dword [EDX+12], 0x80000000
                        LOCK XCHG CL, [.serialize]
                        JMP ReleaseMemoryLocks

.crashburn:             OOPS

.serialize:             DB 2



                        ; Function: MapL2Table32
                        ; Ups the reference count to a memory location.
                        ;
                        ; in:
                        ;     EAX - memory address to reference
                        ;     EDI - memory location to point to
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ECX, EDX
                        ;
MapL2Table32:           Call FindL3MemoryTable32
                        CMP CL, 3
                        JNE .crashburn
                        MOV [EDX], EDI
                        LOCK XCHG CL, [.serialize]
                        MOV dword [EDX+4], 0x80000000
                        LOCK XCHG CL, [.serialize]
                        JMP ReleaseMemoryLocks

.crashburn:             OOPS

.serialize:             DB 3




                        ; Function: Oops
                        ; Kills the thread and prints a report to port e9
                        ;
                        ; in:
                        ;     eax - line number
                        ;
                        ; out:
                        ;     -
                        ;
                        ; destroyed:
                        ;     does not return
                        ;
Oops:                   PUSHAD
                        CLD
                        MOV DX, 0xe9
                        MOV ESI, .message
                        MOV ECX, .message2 - .message
                        REP OUTSB

                        MOV EDI, .message_end
.loop:                  DEC EDI
                        MOV byte [EDI], '0'
                        MOV ECX, 10
                        XOR EDX, EDX
                        DIV ECX
                        ADD [EDI], DL
                        OR EAX, EAX
                        JNZ .loop

                        MOV DX, 0xe9
                        MOV ESI, .message2
                        MOV ECX, .message_end - .message2
                        REP OUTSB

                        MOV AL, 10
                        OUT DX, AL
                        MOV AL, 13
                        OUT DX, AL

                        MOV ESI, .message
                        ;CALL Debugger_Write

                        INT3

.lockup:                HLT
                        JMP .lockup

.message:               DB "Oops in kernel land on line "
.message2:              DB "0000000000"
.message_end:           DB 10, 13, 0



                        ; Function: ReadPageTableEntry32Hole
                        ; Sets a page table entry if allowed.
                        ;
                        ; This version uses the paging holes to temporarily map in the tables
                        ; for write access. Other write methods may be faster
                        ;
                        ; This assumes the processor is in 32-bit legacy paged mode
                        ;
                        ; in:
                        ;     EDX - the physical address of the page directory
                        ;     EDI - the virtual address to look up
                        ;
                        ; out:
                        ;     CF - clear for bottom level entries, set for higher-level entries
                        ;     EDX - the entry stored
                        ;
                        ; destroyed:
                        ;     EAX, EDX, EDI
                        ;
ReadPageTableEntry32Hole:
                        MOV EAX, [lpPagingHoles]        ; get the location where we can allocate stuff
                        PUSH EBX
                        AND EAX, PAGE_TABLEMASK
                        MOV EBX, [lpPagingKernelTables] ; get the kernel area page table
                        SHR EAX, 10
                        ADD EAX, EBX                    ; location of the entry to alter

                        ; lock the holes
.lock1:                 LOCK BTS dword [lpPagingLocks], 0
                        JC .lock1
.lock2:                 LOCK BTS dword [lpPagingLocks], 1
                        JC .lock2

                        ; map in page directory
                        OR EDX, PAGE_PRESENT
                        MOV [EAX], EDX
                        ; fixme: improve speed here.
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; look up table address
                        PUSH EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 20
                        MOV EDX, [EDX+EDI]

                        ; test if we can read into the next level (i.e. there's a table present)
                        TEST EDX, PAGE_PRESENT
                        JZ .toplevel
                        TEST EDX, PAGE_GRANULAR
                        JNZ .toplevel

                        AND EDX, ~(PAGE_USERSPACE | PAGE_WRITETHROUGH | PAGE_UNCACHEABLE)
                        OR EDX, PAGE_WRITABLE

                        ; map in page table
                        MOV [EAX+4], EDX
                        ; fixme: improve speed
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; check page table entry
                        POP EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_TABLEMASK
                        SHR EDI, 10
                        LEA EDX, [EDX+EDI+PAGE_SIZE]

                        ; read entry
                        MOV EDX, [EDX]
                        MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        CLC
                        POP EBX
                        RET

.toplevel:              MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        STC
                        POP EBX
                        RET




                        ; Function: ReleaseMemoryLocks
                        ; Releases memory locks as given by findmemory and relatives
                        ;
                        ; in:
                        ;     CH - bitmask of locks in need of releasemment
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     CH, EDX
ReleaseMemoryLocks:     OR CH, CH
                        JNZ .unlock3
                        RET

.unlock3:               TEST CH, 0x2
                        JZ .unlock2
                        LOCK BTC word [FindL3MemoryTable32.lock], 0
                        XOR CH, 0x02
.unlock2:               TEST CH, 0x4
                        JZ .unlock1
                        LOCK BTC word [FindL2MemoryTable32.lock], 0
                        XOR CH, 0x04
.unlock1:
                        OR CH, CH
                        JNZ .crashburn
                        RET
.crashburn:           ; todo: release virtual pointers
                        OOPS




                        ; Function: RemoveMemoryReference32
                        ; Decrements the reference count to a memory location.
                        ;
                        ; in:
                        ;     EAX - memory address to reference
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     ECX, EDX
                        ;
RemoveMemoryReference32:Call FindL3MemoryTable32
                        LOCK DEC dword [EDX]
                        JMP ReleaseMemoryLocks




                        ; Function: RequestAddress32
                        ; Requests ownership of one location in the physical address space
                        ;
                        ; in:
                        ;     EAX - memory address contained in the requested page
                        ;
                        ; out:
                        ;     EAX - zero on success, all ones on failure
                        ;     CL  - level of this entry
                        ;
                        ; destroyed:
                        ;     EDX
RequestAddress32:       Call FindL3MemoryTable32
                        TEST dword [EDX], 0x7fffffff
                        JNZ .bail
                        LOCK INC dword [EDX]
                        MOV EAX, [EDX]
                        AND EAX, 0x7FFFFFFF
                        CMP EAX, 1
                        JNE .derefbail
                        XOR EAX, EAX
                        JMP ReleaseMemoryLocks

.derefbail:             LOCK DEC dword [EDX]
.bail:                  STC
                        SBB EAX, EAX
                        JMP ReleaseMemoryLocks




                        ; Function: RequestGDT32Entry
                        ; Requests ownership of one specified GDT entry
                        ;
                        ; in:
                        ;     EAX - selector number
                        ;
                        ; out:
                        ;     CF - clear if the entry was allocated, set on failure
                        ;
                        ; destroyed:
                        ;     none
                        ;
RequestGDT32Entry:      PUSH EAX
                        PUSH ECX
                        MOV ECX, EAX
                        SHR EAX, 5
                        AND ECX, 0x0000001f
                        ADD EAX, baGDT32Bitmap
                        BTS dword [EAX], ECX
                        POP ECX
                        POP EAX
                        RET




                        ; Function: RequestMemory32
                        ; Requests ownership of one memory location
                        ;
                        ; This function will fail if the location is in use or not marked as memory
                        ;
                        ; in:
                        ;     EAX - memory address contained in the requested page
                        ;
                        ; out:
                        ;     EAX - zero on success, all ones on failure
                        ;     CL  - level of this entry
                        ;
                        ; destroyed:
                        ;     ECX, EDX
RequestMemory32:        Call FindL3MemoryTable32        ; get EDX -> entry
                        CMP CL, 1                       ; check CL for entry level
                        JNE .requestQword               ; if it is an L2/L3 entry, allocate it as Qword
                        CMP dword [EDX], 0x80000000     ; make sure its RAM, and its unallocated
                        JNZ .bail                       ; bail if not
                        LOCK INC dword [EDX]            ; attempt allocating it
                        CMP dword [EDX], 0x80000001     ; check if there were any concurrent reads
                        JNE .derefbail                  ; if so, release address and bail
                        XOR EAX, EAX                    ;
                        JMP ReleaseMemoryLocks
                        ; bail out
.derefbail:             LOCK DEC dword [EDX]
.bail:                  STC
                        SBB EAX, EAX
                        JMP ReleaseMemoryLocks
                        ; this is a high-level entry
.requestQword:          CMP byte [EDX+7], PMAB_COUNT_MEM
                        JNE .bail
                        CMP dword [EDX], 0
                        JNE .bail
                        LOCK INC dword [EDX]
                        CMP dword [EDX], 1
                        JNE .derefbail
                        CMP byte [EDX+7], PMAB_COUNT_MEM
                        JNE .derefbail
                        XOR EAX, EAX
                        JMP ReleaseMemoryLocks




                        ; Function: SetIDTEntry32
                        ; Points an IDT entry to a given function
                        ;
                        ; in:
                        ;     EAX - Interrupt number
                        ;     EDX - Address of the function
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
SetIDTEntry32:          SHL EAX, 3
                        PUSH EDX
                        ADD EAX, [lp32BitIDT]
                        CLI
                        MOV [EAX], DX
                        SHR EDX, 16
                        MOV word [EAX+2], 0x8
                        MOV word [EAX+4], (IDT_FLAG_PRESENT | IDT_FLAG_DPL0 | IDT_TYPE_TRAP32) << 8
                        MOV [EAX+6], DX
                        STI
                        POP EDX
                        RET




                        ; Function: SetIDTEntry32Public
                        ; Points an IDT entry to a given function, and allow it to be called from userland
                        ;
                        ; in:
                        ;     EAX - Interrupt number
                        ;     EDX - Address of the function
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     EAX
                        ;
SetIDTEntry32Public:    SHL EAX, 3
                        PUSH EDX
                        ADD EAX, [lp32BitIDT]
                        CLI
                        MOV [EAX], DX
                        SHR EDX, 16
                        MOV word [EAX+2], 0x8
                        MOV word [EAX+4], (IDT_FLAG_PRESENT | IDT_FLAG_DPL3 | IDT_TYPE_TRAP32) << 8
                        MOV [EAX+6], DX
                        STI
                        POP EDX
                        RET



                        ; Function: SetPageDirEntry32Hole
                        ; Sets a page directory entry if allowed.
                        ;
                        ; This version uses the paging holes to temporarily map in the tables
                        ; for write access. Other write methods may be faster
                        ;
                        ; This assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     EDX - the physical address of the page directory
                        ;     ESI - the physical address and bits of the entry to map
                        ;     EDI - the virtual address to map to
                        ;
                        ; out:
                        ;     CF - clear on success, set if the write could not be performed
                        ;
                        ; destroyed:
                        ;     EAX, EDX, EDI
                        ;
SetPageDirEntry32Hole:  MOV EAX, [lpPagingHoles]        ; get the location where we can allocate stuff
                        PUSH EBX
                        AND EAX, PAGE_TABLEMASK
                        MOV EBX, [lpPagingKernelTables] ; get the kernel area page table
                        SHR EAX, 10
                        ADD EAX, EBX                    ; location of the entry to alter

                        ; lock the holes
                        ; only lock the PD level hole
.lock2:                 LOCK BTS dword [lpPagingLocks], 1
                        JC .lock2

                        ; map in page directory
                        OR EDX, PAGE_PRESENT | PAGE_WRITABLE
                        MOV [EAX], EDX
                        ; fixme: improve speed here.
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; look up table address
                        PUSH EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 20                     ; dep stall
                        ADD EDX, EDI                    ; dep stall
                        POP EDI

                        CMP dword [EDX], 0              ; AGI stall
                        JNZ .crashburn

                        ; write entry
                        MOV [EDX], ESI
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        CLC
                        POP EBX
                        RET

.crashburn:             MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        STC
                        POP EBX
                        RET




                        ; Function: SetPageTableEntry32Hole
                        ; Sets a page table entry if allowed.
                        ;
                        ; This version uses the paging holes to temporarily map in the tables
                        ; for write access. Other write methods may be faster
                        ;
                        ; This assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     EDX - the physical address of the page directory
                        ;     ESI - the physical address and bits of the entry to map
                        ;     EDI - the virtual address to map to
                        ;
                        ; out:
                        ;     CF - clear on success, set if the write could not be performed
                        ;
                        ; destroyed:
                        ;     EAX, EDX, EDI
                        ;
SetPageTableEntry32Hole:MOV EAX, [lpPagingHoles]        ; get the location where we can allocate stuff
                        PUSH EBX
                        AND EAX, PAGE_TABLEMASK
                        MOV EBX, [lpPagingKernelTables] ; get the kernel area page table
                        SHR EAX, 10
                        ADD EAX, EBX                    ; location of the entry to alter

                        ; lock the holes
.lock1:                 LOCK BTS dword [lpPagingLocks], 0
                        JC .lock1
.lock2:                 LOCK BTS dword [lpPagingLocks], 1
                        JC .lock2

                        ; map in page directory
                        OR EDX, PAGE_PRESENT | PAGE_WRITABLE
                        MOV [EAX], EDX
                        ; fixme: improve speed here.
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; look up table address
                        PUSH EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_DIRECTORYMASK
                        SHR EDI, 20
                        MOV EDX, [EDX+EDI]

                        ; test if we can write here (i.e. there's a table present)
                        TEST EDX, PAGE_PRESENT
                        JZ .crashburnpop
                        TEST EDX, PAGE_GRANULAR
                        JNZ .crashburnpop

                        AND EDX, ~(PAGE_USERSPACE | PAGE_WRITETHROUGH | PAGE_UNCACHEABLE)
                        OR EDX, PAGE_WRITABLE

                        ; map in page table
                        MOV [EAX+4], EDX
                        ; fixme: improve speed
                        MOV EDX, CR3
                        MOV CR3, EDX

                        ; check page table entry
                        POP EDI
                        MOV EDX, [lpPagingHoles]
                        AND EDI, PAGE_TABLEMASK
                        SHR EDI, 10
                        LEA EDX, [EDX+EDI+PAGE_SIZE]
                        CMP dword [EDX], 0
                        JNZ .crashburn

                        ; write entry
                        MOV [EDX], ESI
                        MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        CLC
                        POP EBX
                        RET

.crashburnpop:          POP EDI
.crashburn:             MOV dword [EAX+4], PAGE_ACCESSED
                        MOV dword [EAX], PAGE_ACCESSED
                        LOCK BTC dword [lpPagingLocks], 1
                        LOCK BTC dword [lpPagingLocks], 0
                        STC
                        POP EBX
                        RET




                        ; Function: SwitchGDT32
                        ; Switches to the 32 bit GDT
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
SwitchGDT32:            LGDT [.gdtr]
                        PUSH EAX
                        MOV EAX, 0x10
                        MOV SS, AX
                        MOV DS, AX
                        MOV ES, AX
                        JMP dword 0x8:.flushcs
.flushcs                POP EAX
                        RET

.gdtr:                  ; GDT register
.gdtr.length:           DW 0
.gdtr.offset:           DD 0




                        ; Function: SwitchIDT32
                        ; Switches to the 32 bit IDT
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
SwitchIDT32:            LIDT [.idtr]
                        RET

.idtr:                  ; IDT register
.idtr.length:           DW 0
.idtr.offset:           DD 0




                        ; Function: QueueThread
                        ; adds a thread to the execution queue
                        ;
                        ; in:
                        ;     ECX - CPU number
                        ;     EDX - Address space field
                        ;     ESI - Task Description Structure field
                        ;
                        ; out:
                        ;     CF - set on failure, clear on success
                        ;
                        ;
                        ; destroyed:
                        ;     none
                        ;
QueueThread:            CMP ECX, [dwProcessorCount]
                        JAE .fail

.prelock:
                        BT dword [dwSchedulerLocks], ECX
                        JC .prelock
                        CLI
                        LOCK BTS dword [dwSchedulerLocks], ECX
                        JNC .ok
                        STI
                        JMP .prelock

.ok:                    PUSH EAX
                        PUSH ECX
                        MOV EAX, [lpaSchedulerBase + 8 * ECX]
                        MOV ECX, (PAGE_SIZE / 16) - 1
.loop:                  CMP dword [EAX], 0
                        JE .fill
                        ADD EAX, 16
                        DEC ECX
                        JNZ .loop
                        JMP .fail
.fill:                  XOR ECX, ECX
                        MOV dword [EAX+16], ECX
                        MOV dword [EAX+20], ECX
                        MOV dword [EAX+4], ECX
                        MOV dword [EAX+12], ECX
                        MOV dword [EAX+8], EDX
                        MOV dword [EAX], ESI

                        POP ECX
                        POP EAX
                        LOCK BTC dword [dwSchedulerLocks], ECX
                        STI
                        CLC
                        RET

.fail:
                        STC
                        RET




                        ; Function: WritePage32
                        ; adds an entry to the page table.
                        ;
                        ; This version does not perform any checking. Use <InsertPage32>
                        ; when necessary
                        ;
                        ; in:
                        ;     EDX - location of the page table
                        ;     ESI - physical address and access bytes of the page to map
                        ;     EDI - virtual address to map to
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
WritePage32:            PUSH EAX
                        MOV EAX, EDI
                        AND EAX, PAGE_TABLEMASK
                        SHR EAX, 12 - 2
                        ADD EAX, EDX
                        MOV [EAX], ESI
                        POP EAX
                        RET




                        ; Function: WriteAddressSpace32
                        ; Map and fill out a set of address space structures
                        ;
                        ; Maps a pagedirectory and pagetable, then load them
                        ; with the appropriate values.
                        ;
                        ; in:
                        ;     EAX - physical location of pagetable
                        ;     EBX - physical location of page directory
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
WriteAddressSpace32:    PUSH ECX
                        PUSH EDX
                        PUSH ESI
                        MOV EDX, EAX
                        MOV ESI, EBX
                        MOV ECX, [lpAddressHole]           ; get the location of the hole used
                        AND EDX, PAGE_ADDRMASK
                        AND ESI, PAGE_ADDRMASK
                        SHR ECX, 12-2
                        OR EDX, PAGE_PRESENT | PAGE_WRITABLE
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE
                        ADD ECX, [lpPagingKernelTables] ; ECX = page table entry to alter

                        ; grab the lock
.check                  CMP word [.lock], 0
                        JNE .check
                        LOCK BTS word [.lock], 0
                        JC .check

                        ; write entries
                        MOV [ECX], EDX
                        MOV [ECX+4], ESI

                        ; page flush
                        MOV ESI, CR3
                        MOV CR3, ESI

                        PUSH EBX
                        ;      EDI - virtual address of the first page table
                        ;      EBX - virtual address of the page directory
                        ;      ESI - physical address of the first page table
                        ;      EDX - physical address of the page directory
                        MOV EDI, [lpAddressHole]
                        MOV ESI, EAX
                        MOV EDX, EBX
                        LEA EBX, [EDI+PAGE_SIZE]
                        Call FillKernelPageDirectory
                        POP EBX

                        ; release the lock and exit
                        LOCK BTC word [.lock], 0
                        POP ESI
                        POP EDX
                        POP ECX
                        RET

ALIGN 2
.lock:                  DW 0



                        ; Function: ZeroPage32
                        ; Zeroes a page in physical memory
                        ;
                        ; Maps the page into a predefined memory hole, then
                        ; clears it.
                        ;
                        ; Assumes the processor is in 32-bit paged mode
                        ;
                        ; in:
                        ;     EAX - offset of the page
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
ZeroPage32:             PUSH ECX
                        PUSH EAX
                        MOV ECX, [lpZeroHole]           ; get the location of the hole used
                        AND EAX, PAGE_ADDRMASK
                        SHR ECX, 12-2
                        OR EAX, PAGE_PRESENT | PAGE_WRITABLE
                        PUSH EDI
                        ADD ECX, [lpPagingKernelTables] ; ECX = page table entry to alter

                        ; grab the lock
.check                  CMP word [.lock], 0
                        JNE .check
                        LOCK BTS word [.lock], 0
                        JC .check

                        ; write the page table and flush TLB
                        MOV [ECX], EAX
                        MOV ECX, CR3
                        MOV CR3, ECX

                        ; zero memory
                        MOV EDI, [lpZeroHole]
                        MOV ECX, PAGE_SIZE / 4
                        XOR EAX, EAX
                        REP STOSD

                        ; release the lock and exit
                        LOCK BTC word [.lock], 0
                        POP EDI
                        POP EAX
                        POP ECX
                        RET
align 2
.lock                   DW 0

%include "../ia_common/ucode.asm"
%include "../ia_common/debug.asm"

; ====================================================================================================================
; Group: Scheduler components
; These functions change between tasks.
; ====================================================================================================================


                        ; Function: Schedule
                        ; Causes execution to be suspended and the next thread to be readied.
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none - the task is suspended and the state is restored upon return
                        ;
Schedule:               PUSH EBX
                        STR BX                              ; retrieve the processor number
                        AND EBX, 0x00000078
                        JMP [lpaCurrentScheduleProc + EBX]



                        ; Function: Scheduler_Common
                        ; this function will browse through the scheduler list and pick a new task
                        ; to run.
                        ;
                        ; in:
                        ;     EBX - processor number * 8
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
Scheduler_Common:       MOV EAX, EBX
                        MOV ECX, [lpaSchedulerOffset + EBX]
                        SHL EAX, 6
                        MOV EDI, kernelimageend + 0x1000
                        ADD ECX, 16
                        ADD EAX, 0x200
                        AND EDI, 0xfffff000
                        MOV EDX, [ECX]
                        ADD EAX, EDI
                        OR EDX, EDX
                        MOV ESP, EAX                            ; stack must be valid in all address spaces
                        JNZ .nowrap
                        MOV ECX, [lpaSchedulerBase + EBX]
                        MOV EDX, [ECX]
.nowrap:                MOV EAX, [ECX+8]
                        MOV [lpaSchedulerOffset+EBX], ECX
                        OR EAX, EAX
                        JZ .noaschange
                        MOV ECX, [lpaCurrentAddressSpace+EBX]
                        CMP EAX, ECX
                        JE .noaschange

                        ; todo: Address space switches
                        CMP AL, CL
                        JNE .changecpumode
                        AND EAX, 0xFFFFFFF8
                        MOV CR3, EAX

.noaschange:            MOV [lpaCurrentProcess + EBX], EDX
                        JMP [EDX + TDS_DESCHEDULE]

.changecpumode:         XOR ECX, ECX
                        MOV CL, AL
                        AND CL, 0x07
                        JMP [.modes+4*ECX]

.modes:                 DD .nopaging        ; mode 0
                        DD .386paging       ; mode 1

.nopaging:              MOV ECX, CR0
                        AND ECX, 0x7fffffff
                        MOV CR0, EAX
                        JMP .noaschange

.386paging:             AND EAX, PAGE_ADDRMASK
                        MOV CR3, EAX
                        MOV ECX, CR0
                        OR ECX, 0x80000000
                        MOV CR0, ECX
                        JMP .noaschange

                        ; scheduling out a kernel task
Base_ScheduleOut:       PUSHAD
                        MOV EAX, [lpaCurrentProcess+EBX]
                        MOV [EAX+TDS_BASE_ESP], ESP
                        JMP Scheduler_Common


                        ; scheduling in a kernel task
Base_ScheduleIn:        MOV EAX, [lpaTSS+EBX]
                        MOV EDI, CR0
                        MOV ESP, [EDX+TDS_BASE_ESP]
                        MOV ECX, [EDX+TDS_BASE_ESP0]
                        MOV [EAX+TSS_ESP0], ECX
                        OR EDI, CR0_TS
                        MOV dword [lpaCurrentScheduleProc + EBX], Base_ScheduleOut
                        MOV CR0, EDI
                        POPAD
                        POP EBX
                        RET


                        ; Function: Scheduler_Launchuser
                        ; return stub that gets us into userland
Scheduler_Launchuser:   TEST dword [ESP+12], EFLAGS_VM
                        JNZ .skipregs
                        POP DS
                        POP ES
                        IRETD
.skipregs:              ADD ESP, 8
                        IRETD


; ====================================================================================================================
; Group: System calls
; These functions form the interface to all userspace applications
; ====================================================================================================================

                        ; Function: Entrypoint_Int
                        ; Contains the handler for system calls when accessed via the syscall interrupt.
                        ;
                        ; in:
                        ;     EAX - function number
                        ;     EBX - argument 1
                        ;     ESI - argument 2
                        ;     EDI - argument 3
                        ;
                        ; out:
                        ;     EAX - return value if applicable
                        ;     EBX - return value if applicable
                        ;     ESI - return value if applicable
                        ;     EDI - return value if applicable
                        ;     CF  - clear on success
                        ;
                        ; destroyed:
                        ;     ECX, EDX, depending on syscall: EAX, EBX, ESI, EDI
                        ;
Entrypoint_Int:         PUSH DS
                        PUSH ES
                        PUSH EAX
                        MOV EAX, 0x10
                        MOV DS, AX
                        MOV ES, AX
                        POP EAX
                        Call Entrypoint_Main
                        JC .setcarry
                        AND dword [ESP+16], ~ EFLAGS_CF
                        JMP .continue
.setcarry:              OR dword [ESP+16], EFLAGS_CF
.continue:              TEST dword [ESP+16], EFLAGS_VM
                        JNZ .skipregs
                        POP DS
                        POP ES
                        IRETD
.skipregs:              ADD ESP, 8
                        IRETD


                        ; Function: Entrypoint_Main
                        ; Does the actual decoding of the system call
                        ;
                        ; in:
                        ;     EAX - function number
                        ;     EBX - argument
                        ;     ESI - argument
                        ;     EDI - argument
                        ;
                        ; out:
                        ;     EAX - return value if applicable
                        ;     EBX - return value if applicable
                        ;     ESI - return value if applicable
                        ;     EDI - return value if applicable
                        ;     CF  - clear on success
                        ;
                        ; destroyed:
                        ;     ECX, EDX, depending on syscall: EAX, EBX, ESI, EDI
                        ;
Entrypoint_Main:        CMP EAX, (.syscalltableend - .syscalltable) / 4
                        JAE .notpresent

                        JMP [.syscalltable + 4 * EAX]

.notpresent:            STC
                        RET

.syscalltable:          DD KernelVersion
                        DD BlockAlloc
                        DD BlockAllocEx
                        DD BlockAllocPhys
                        DD EnterV8086
                        DD AllocateIoBitmap
                        DD PortAlloc
                        DD AllocatePageTable
                        DD BlockAllocExL
                        DD BlockAllocPhysL
                        DD AllocateIRBitmap
                        DD SetRedirectBits
                        DD ManageMemoryL1
                        DD ManageMemoryL2
                        DD Yield
                        DD CreateAddressSpace
                        DD CreateThread
                        DD CreateThreadRemote
                        DD TransferPage
                        DD AllocatePageTableRemote
                        DD BlockDealloc
			DD Yank
.syscalltableend:


                        ; Function: KernelVersion
                        ; Returns the kernel version
                        ;
                        ; Privilege level:
                        ;     Informational
                        ;
                        ; in:
                        ;     - EAX = 0x00000000
                        ;
                        ; out:
                        ;     - EAX = version: bits 0-15 = revision, 16-23 = minor, 24-31 = major
                        ;
                        ; destroyed:
                        ;     none
                        ;
SYSPL_KERNELVERSION     EQU PRIVILEGE_INFO
KernelVersion:          MOV EAX, 0x01040001
                        CLC
                        RET



                        ; Function: BlockAlloc
                        ; Allocates memory and maps it to userspace
                        ;
                        ; Returns the amount of memory actually mapped
                        ;
                        ; Privilege level:
                        ;     User
                        ;
                        ; in:
                        ;     - EAX = 0x00000001
                        ;     - EBX = amount of memory to map (pages)
                        ;     - EDI = starting page in current address space
                        ;
                        ; out:
                        ;     - EBX = amount of memory actually mapped
                        ;
                        ; destroyed:
                        ;     EAX EDX ESI
                        ;
SYSPL_BLOCKALLOC        EQU PRIVILEGE_USER
BlockAlloc:             PUSH EDI
                        PUSH ESI
                        OR EBX, EBX
                        MOV ESI, EBX
                        JZ .b00ring
                        XOR EBX, EBX
.loop:                  Call AllocateMemory32
                        JC .b00ring
                        ; we have a physical address in EAX, and its allocated

                        PUSH ESI
                        PUSH EDI
                        PUSH EAX
                        MOV EDX, CR3
                        MOV ESI, EAX
                        OR ESI, PAGE_PRESENT | PAGE_USERSPACE | PAGE_WRITABLE
                        Call SetPageTableEntry32Hole
                        ; todo: write address to page table if applicable
                        JC .fail
                        POP EAX
                        POP EDI
                        POP ESI

                        ADD EDI, PAGE_SIZE
                        INC EBX
                        DEC ESI
                        JNZ .loop

.b00ring:
                        POP ESI
                        POP EDI
                        CLC
                        RET

.fail:                  POP EAX

                        ; todo: release this address
                        ;OOPS
                        Call RemoveMemoryReference32

                        POP ESI
                        POP EDI
                        STC
                        RET




                        ; Function: BlockAllocEx
                        ; Allocates a specific piece memory and maps it to userspace
                        ;
                        ; Returns the amount of memory actually mapped
                        ;
                        ; Privilege level:
                        ;     User
                        ;
                        ; in:
                        ;     - EAX = 0x00000002
                        ;     - EBX = amount of memory to map (pages)
                        ;     - ESI = starting page in physical memory
                        ;     - EDI = starting page in current address space
                        ;
                        ; out:
                        ;     - EBX = amount of memory actually mapped
                        ;
                        ; destroyed:
                        ;     EAX EDX ESI EDI ECX
                        ;
SYSPL_BLOCKALLOCEX      EQU PRIVILEGE_USER
BlockAllocEx:           OR EBX, EBX
                        MOV ECX, EBX
                        JZ .b00ring
                        XOR EBX, EBX
                        AND ESI, PAGE_ADDRMASK
                        AND EDI, PAGE_ADDRMASK

.loop:                  PUSH ECX
                        MOV EAX, ESI
                        Call RequestMemory32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 1
                        JNE .fail
                        ; we have a physical address in ESI, and its allocated

                        PUSH ESI
                        PUSH EDI
                        PUSH EAX
                        MOV EDX, CR3
                        OR ESI, PAGE_PRESENT | PAGE_USERSPACE | PAGE_WRITABLE
                        Call SetPageTableEntry32Hole
                        ; todo: write address to page table if applicable
                        POP EAX
                        POP EDI
                        POP ESI
                        JC .fail

                        ADD EDI, PAGE_SIZE
                        ADD ESI, PAGE_SIZE

                        POP ECX
                        INC EBX
                        DEC ECX
                        JNZ .loop

                        CLC
                        RET

.b00ringpop:            POP ECX
.b00ring:               CLC
                        RET

.fail:                  POP ECX

                        ; todo: release this address
                        OOPS

                        STC
                        RET



                        ; Function: BlockAllocPhys
                        ; Allocates an area in the physical range and maps it to userspace
                        ;
                        ; Returns the amount of memory actually mapped
                        ;
                        ; Privilege level:
                        ;     Driver
                        ;
                        ; in:
                        ;     - EAX = 0x00000003
                        ;     - EBX = amount of memory to map (pages)
                        ;     - ESI = starting page in physical memory
                        ;     - EDI = starting page in current address space
                        ;
                        ; out:
                        ;     - EBX = amount of memory actually mapped
                        ;
                        ; destroyed:
                        ;     EAX EDX ESI EDI ECX
                        ;
SYSPL_BLOCKALLOCPHYS    EQU PRIVILEGE_DRIVER
BlockAllocPhys:         OR EBX, EBX
                        MOV ECX, EBX
                        JZ .b00ring
                        XOR EBX, EBX
                        AND ESI, PAGE_ADDRMASK
                        AND EDI, PAGE_ADDRMASK

.loop:                  PUSH ECX
                        MOV EAX, ESI
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 1
                        JNE .fail
                        ; we have a physical address in ESI, and its allocated

                        PUSH ESI
                        PUSH EDI
                        PUSH EAX
                        MOV EDX, CR3
                        OR ESI, PAGE_PRESENT | PAGE_USERSPACE | PAGE_WRITABLE
                        Call SetPageTableEntry32Hole
                        ; todo: write address to page table if applicable
                        POP EAX
                        POP EDI
                        POP ESI
                        JC .fail

                        ADD EDI, PAGE_SIZE
                        ADD ESI, PAGE_SIZE

                        POP ECX
                        INC EBX
                        DEC ECX
                        JNZ .loop

                        PUSH ECX
                        MOV ECX, CR3
                        MOV CR3, ECX
                        POP ECX

                        CLC
                        RET

.b00ringpop:            POP ECX
                        OOPS
.b00ring:               CLC
                        OOPS
                        RET

.fail:                  POP ECX

                        ; todo: release this address
                        OOPS

                        STC
                        RET


                        ; Function: EnterV8086
                        ; Makes the thread enter v8086 mode
                        ;
                        ; in:
                        ;     EAX - 0x00000004
                        ;     ESI - CS:IP
                        ;     EDI - EFLAGS
                        ;     EBX - monitor address
                        ;
SYSPL_ENTERV8086        EQU PRIVILEGE_USER
EnterV8086:             STR AX                              ; retrieve the processor number
                        AND EAX, 0x00000078
                        MOV EAX, [lpaCurrentProcess+EAX]
                        ; create iret stack frame (segment registers)
                        MOVZX EDX, word [EAX+TDS_V8086SEGS]
                        PUSH EDX
                        MOVZX EDX, word [EAX+TDS_V8086SEGS+2]
                        PUSH EDX
                        MOVZX EDX, word [EAX+TDS_V8086SEGS+4]
                        PUSH EDX
                        MOVZX EDX, word [EAX+TDS_V8086SEGS+6]
                        PUSH EDX
                        MOVZX EDX, word [EAX+TDS_V8086SEGS+8]
                        PUSH EDX
                        ; ESP
                        MOV EDX, [EAX+TDS_V8086REGS+28]
                        PUSH EDX
                        ; EFLAGS
                        MOV EDX, EDI
                        AND EDX, EFLAGS_CF | EFLAGS_AF | EFLAGS_ZF | EFLAGS_PF | EFLAGS_OF | EFLAGS_SF | EFLAGS_VIF | EFLAGS_ID | EFLAGS_AC | EFLAGS_TF | EFLAGS_DF
                        OR EDX, EFLAGS_IF | EFLAGS_VM
                        PUSH EDX
                        ; CS:EIP
                        MOV EDX, ESI
                        SHR ESI, 16
                        AND EDX, 0x0000FFFF
                        PUSH ESI
                        PUSH EDX
                        ; monitor
                        MOV [EAX+TDS_VMEXIT], EBX
                        MOV ECX, [EAX+TDS_V8086REGS+4]
                        MOV EDX, [EAX+TDS_V8086REGS+8]
                        MOV EBX, [EAX+TDS_V8086REGS+12]
                        MOV ESI, [EAX+TDS_V8086REGS+16]
                        MOV EDI, [EAX+TDS_V8086REGS+20]
                        MOV EBP, [EAX+TDS_V8086REGS+24]
                        MOV EAX, [EAX+TDS_V8086REGS]
                        IRETD


                        ; Function: AllocateIoBitmap
                        ; Allocates pages for the I/O permission bitmap.
                        ;
                        ; Due to the handling of the bitmaps in the TSS, the top 256 ports are
                        ; inaccessible.
                        ;
                        ; in:
                        ;     EAX - 0x00000005
                        ;     DI - allocate the part of the bitmap starting with this port number
                        ;     BX - the amount of ports that are requested
                        ;     ESI - the starting physical page to use.
                        ;           Supply all ones to let the kernel choose free pages
                        ;
                        ; out:
                        ;     CF - set if the area was not fully mapped
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI, EDI
                        ;
SYSPL_ALLOCATEIOBITMAP  EQU PRIVILEGE_USER
AllocateIoBitmap:       ;CMP DI, 65536           ; check for starting
                        ;JAE .bound
                        ;port limit disabled
                        DEC EBX
                        ;ADD EDI, 256           ; correct for the Interrupt redirection bitmap
                        ADD EBX, EDI            ; compute last address
                        AND EDI, 0xffff         ; clamp to 16 bits
                        AND EBX, 0xffff
                        CMP EDI, EBX            ; check for range
                        JA .bound

                        SHR EDI, 15             ; convert to page indices
                        SHR EBX, 15


.loop:                  Call GetTssBitmapStatus32
                        JC .mapped

                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr

                        MOV EAX, ESI
                        Call FillPage32

                        Call InsertTSSBitmapPage
                        JNC .mapped
                        Call RemoveMemoryReference32
                        JMP .bound

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

                        Call FillPage32

                        Call InsertTSSBitmapPage
                        JNC .mapped
                        Call RemoveMemoryReference32
                        JMP .bound

.mapped:                CMP EDI, EBX            ; check if we are done
                        JE .done
                        INC EDI                 ; go to next page and continue
                        JMP .loop
.done:                  CLC
                        RET

.nomem:
.bound:                 STC
                        RET

.memerr:                ; todo: fix crashburn condition
                        JMP .memerr




                        ; Function: PortAlloc
                        ; Allocates pages for the I/O permission bitmap.
                        ;
                        ; Due to the handling of the bitmaps in the TSS, the top 256 ports are
                        ; inaccessible.
                        ;
                        ; in:
                        ;     EAX - 0x00000006
                        ;     DI - allocate the part of the bitmap starting with this port number
                        ;     BX - the amount of ports that are requested
                        ;
                        ; out:
                        ;     CF - set if the area was not fully mapped
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI, EDI
                        ;

                        ; fixme: add port allocation management and safety stuff

SYSPL_PORTALLOC         EQU PRIVILEGE_DRIVER
PortAlloc:              MOVZX ECX, DI
                        CMP BX, 0
                        JE .done
                        ADD CX, BX
                        JC .fail                        ; test for overflow (port > 0xffff)
                        CMP DI, 32768
                        JAE .highhalf
                        MOVZX ECX, DI                   ; ecx = port
                        MOV EDI, 0
                        Call GetTssBitmapStatus32
                        JNC .fail
                        MOV EDI, [lpaIoBitmap]          ; edi = i/o bitmap address
                        MOV EDX, ECX
                        SHR EDX, 3
                        AND DX, 0xfffc
                        PUSH EDI
                        LEA EDI, [EDI+EDX]              ; edi = i/o bitmap address
                        POP EDX
                        AND ECX, 0x1f                   ; ecx = bit number
                        ADD EDX, PAGE_SIZE              ; edx = i/o bitmap boundary
.lowpre:                OR ECX, ECX
                        JZ .lowbyte
                        BTR dword [edi], ECX
                        INC CL
                        AND CL, 0x1f
                        DEC BX
                        JZ .done
                        OR ECX, ECX
                        JNZ .lowpre
                        ADD EDI, 4
                        CMP EDI, EDX
                        JE .hicont
.lowbyte:               CMP BX, 32
                        JL .lowpost
                        MOV dword [EDI], 0x00000000
                        SUB BX, 32
                        ADD EDI, 4
                        CMP EDI, EDX
                        JE .hicont
                        JMP .lowbyte
.lowpost:               OR BX, BX
                        JZ .done
                        BTR dword [edi], ECX
                        INC CL
                        DEC BX
                        JNZ .lowpost
                        OR BX, BX
                        JZ .done

                        ; fixme: step up from first page to second
.hicont:                OOPS

.highhalf:              MOVZX ECX, DI                   ; ecx = port
                        MOV EDI, 0
                        Call GetTssBitmapStatus32
                        JNC .fail
                        MOV EDI, [lpaIoBitmap]          ; edi = i/o bitmap address
                        MOV EDX, ECX
                        SHR EDX, 3
                        AND DX, 0xfffc                  ; edx = offset
                        PUSH EDI
                        LEA EDI, [EDI+EDX]              ; edi = i/o bitmap address
                        POP EDX
                        AND ECX, 0x1f                   ; ecx = bit number

                        OR ECX, ECX
                        JZ .hibyte
.hipre:                 BTR dword [edi], ECX
                        INC CL
                        AND CL, 0x1f
                        DEC BX
                        JZ .done
                        OR ECX, ECX
                        JNZ .hipre
                        ADD EDI, 4
.hibyte:                CMP BX, 32
                        JL .hipost
                        MOV dword [EDI], 0x00000000
                        SUB BX, 32
                        ADD EDI, 4
                        JMP .hibyte
.hipost:                OR BX, BX
                        JZ .done
                        BTR dword [edi], ECX
                        INC CL
                        DEC BX
                        JMP .hipost

.fail:                  STC
                        RET

.done:                  CLC
                        RET




                        ; Function: AllocatePageTable
                        ;
                        ; Maps a page table into the page directory
                        ; to allow for small pages to be used
                        ;
                        ; in:
                        ;     EAX - 0x00000007
                        ;     EDI - the virtual address that this page table needs to cover
                        ;     ESI - the starting physical page to use.
                        ;           Supply all ones to let the kernel choose a free page
                        ;
                        ; out:
                        ;     CF - set if the mapping could not be done
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI, EDI
                        ;

                        ; fixme: update destroyed regs
                        ; check
                        ; <RequestMemory32> <AllocateMemory32>
                        ; <InsertPageTable32> <ZeroPage32>
                        ; <RemoveMemoryReference32>

SYSPL_ALLOCATEPAGETABLE EQU PRIVILEGE_USER
AllocatePageTable:
                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr
                        JMP .applymem

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

.applymem:              Call ZeroPage32

                        ;     EBX - directory address
                        ;     EDX - address to map to
                        PUSH EDX
                        MOV EBX, [lpPagingKernelTables]
                        MOV EDX, EDI
                        ADD EBX, PAGE_SIZE
                        Call InsertPageTable32
                        POP EDX
                        JC .derefbail
                        CLC
                        RET

.derefbail:             Call RemoveMemoryReference32
.memerr:
.nomem:                 STC
                        RET


                        ; Function: BlockAllocExL
                        ; Allocates an area of memory in the physical range and maps it to userspace
                        ;
                        ; Returns the amount of memory actually mapped
                        ;
                        ; Privilege level:
                        ;     Driver
                        ;
                        ; in:
                        ;     - EAX = 0x00000008
                        ;     - EBX = amount of memory to map (large pages)
                        ;     - ESI = starting page in physical memory
                        ;     - EDI = starting page in current address space
                        ;
                        ; out:
                        ;     - EBX = amount of memory actually mapped
                        ;
                        ; destroyed:
                        ;     EAX EDX ESI EDI ECX
                        ;

                        ; todo: check destroyed regs
SYSPL_BLOCKALLOCEXL   EQU PRIVILEGE_USER
BlockAllocExL:          TEST dword [qwProcessorSharedCaps], KCPUID_PSE
                        JZ .fail
                        OR EBX, EBX
                        MOV ECX, EBX
                        JZ .b00ring
                        XOR EBX, EBX
                        AND ESI, PAGE_DIRECTORYMASK
                        AND EDI, PAGE_DIRECTORYMASK

.loop:                  PUSH ECX
                        PUSH EBX
                        MOV EAX, ESI
                        Call RequestMemory32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 2
                        JNE .fail1
                        ; we have a physical address in ESI, and its allocated

                        MOV EAX, ESI
                        ADD EAX, 2 * 1024 * 1024
                        Call RequestMemory32
                        OR EAX, EAX
                        JNZ .fail1
                        CMP CL, 2
                        JNE .fail2

                        ;     EAX - 4M page address
                        ;     EBX - directory address
                        ;     EDX - address to map to

                        MOV EBX, [lpPagingKernelTables]   ; location where page tables are
                        MOV EAX, ESI
                        ADD EBX, PAGE_SIZE                 ; 2nd order table: page directory
                        MOV EDX, EDI
                        Call InsertLargePage32
                        JC .fail2

                        ADD EDI, 4 * 1024 * 1024
                        ADD ESI, 4 * 1024 * 1024

                        POP EBX
                        POP ECX
                        INC EBX
                        DEC ECX
                        JNZ .loop

                        CLC
                        RET

.b00ringpop:            POP EBX
                        POP ECX
.b00ring:               CLC
                        RET

.fail2:
.fail1:                 POP EBX
                        POP ECX

                        ; todo: release this address
                        OOPS
.fail:
                        STC
                        RET




                        ; Function: BlockAllocPhysL
                        ; Allocates an area in the physical range and maps it to userspace
                        ;
                        ; Returns the amount of memory actually mapped
                        ;
                        ; Privilege level:
                        ;     Driver
                        ;
                        ; in:
                        ;     - EAX = 0x00000009
                        ;     - EBX = amount of memory to map (large pages)
                        ;     - ESI = starting page in physical memory
                        ;     - EDI = starting page in current address space
                        ;
                        ; out:
                        ;     - EBX = amount of memory actually mapped
                        ;
                        ; destroyed:
                        ;     EAX EDX ESI EDI ECX
                        ;

                        ; todo: check destroyed regs
SYSPL_BLOCKALLOCPHYSL   EQU PRIVILEGE_DRIVER
BlockAllocPhysL:        TEST dword [qwProcessorSharedCaps], KCPUID_PSE
                        JZ .fail
                        OR EBX, EBX
                        MOV ECX, EBX
                        JZ .b00ring
                        XOR EBX, EBX
                        AND ESI, PAGE_DIRECTORYMASK
                        AND EDI, PAGE_DIRECTORYMASK

.loop:                  PUSH ECX
                        PUSH EBX
                        MOV EAX, ESI
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 2
                        JNE .fail1
                        ; we have a physical address in ESI, and its allocated

                        MOV EAX, ESI
                        ADD EAX, 2 * 1024 * 1024
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .fail1
                        CMP CL, 2
                        JNE .fail2

                        ;     EAX - 4M page address
                        ;     EBX - directory address
                        ;     EDX - address to map to

                        MOV EBX, [lpPagingKernelTables]   ; location where page tables are
                        MOV EAX, ESI
                        ADD EBX, PAGE_SIZE                 ; 2nd order table: page directory
                        MOV EDX, EDI
                        Call InsertLargePage32
                        JC .fail2

                        ADD EDI, 4 * 1024 * 1024
                        ADD ESI, 4 * 1024 * 1024

                        POP EBX
                        POP ECX
                        INC EBX
                        DEC ECX
                        JNZ .loop

                        CLC
                        RET

.b00ringpop:            POP EBX
                        POP ECX
.b00ring:               CLC
                        RET

.fail2:
.fail1:                 POP EBX
                        POP ECX

                        ; todo: release this address
                        OOPS
.fail:
                        STC
                        RET




                        ; Function: AllocateIRBitmap
                        ; Allocates a page for the Interrupt Redirection Bitmap
                        ;
                        ;
                        ; in:
                        ;     EAX - 0x0000000A
                        ;     ESI - the starting physical page to use.
                        ;           Supply all ones to let the kernel choose free pages
                        ;
                        ; out:
                        ;     CF - set if the area was not fully mapped
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI
                        ;
SYSPL_ALLOCATEIRBITMAP  EQU PRIVILEGE_USER
AllocateIRBitmap:       STC
                        PUSH EDI
                        SBB EDI, EDI
                        Call GetTssBitmapStatus32
                        JC .mapped

                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr

                        MOV EAX, ESI
                        Call FillPage32

                        Call InsertTSSBitmapPage
                        JNC .mapped
                        Call RemoveMemoryReference32
                        JMP .bound

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

                        Call FillPage32

                        Call InsertTSSBitmapPage
                        JNC .mapped
                        Call RemoveMemoryReference32

.bound:
.mapped:                POP EDI
                        CLC
                        RET

.nomem:                 POP EDI
                        STC
                        RET

.memerr:                OOPS




                        ; Function: SetRedirectBits
                        ; Sets redirection bits in the Interrupt Redirection Bitmap
                        ;
                        ;
                        ; in:
                        ;     EAX - 0x0000000B
                        ;     ESI - the starting interrupt number
                        ;     EDI - the amount of entries to set
                        ;     EBX - zero to enable redirects, nonzero to disable
                        ;
                        ; out:
                        ;     CF - set if the area was not fully mapped
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI
                        ;
SYSPL_SETREDIRECTBITS   EQU PRIVILEGE_USER
SetRedirectBits:        OR EDI, EDI
                        JZ .boring

                        PUSH EDI
                        SBB EDI, EDI
                        Call GetTssBitmapStatus32
                        JNC .notmapped

                        POP EDI
                        MOV EDX, EDI
                        ADD EDI, ESI
                        CMP ESI, 255
                        JG .bounderr
                        CMP EDI, 256
                        JG .bounderr

                        MOV EDI, [lpaIoBitmap]
                        MOV EAX, ESI
                        SHR ESI, 5
                        MOV ECX, ESI
                        LEA EDI, [EDI+4*ESI-256/8]
                        AND ECX, 0x1f
                        OR EBX, EBX
                        JZ .clearloop
.setloop:               BTS [EDI], ECX
                        DEC EDX
                        JZ .done
                        INC ECX
                        AND ECX, 0x1f
                        JNZ .setloop
                        ADD EDI, 4
                        JMP .setloop
.clearloop:             BTC [EDI], ECX
                        DEC EDX
                        JZ .done
                        INC ECX
                        AND ECX, 0x1f
                        JNZ .clearloop
                        ADD EDI, 4
                        JMP .clearloop

.done:
.boring:                CLC
                        RET

.notmapped:
.bounderr:              STC
                        RET




                        ; Function: ManageMemoryL1
                        ; Insert a node into the memory manager tree at the 1st level
                        ; The node will describe 2M of memory in 4K units
                        ;
                        ; in:
                        ;     EAX - 0x0000000C
                        ;     ESI - the page in physical memory to be used. Supply all
                        ;           ones to let the kernel choose a free page
                        ;     EDI - the physical address that the node should contain
                        ;
                        ; out:
                        ;     CF - set if the node couldn't be mapped
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
SYSPL_MANAGEMEMORYL1    EQU PRIVILEGE_DRIVER
ManageMemoryL1:         MOV EAX, EDI
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 2
                        JNE .fail1

                        MOV EAX, EDI
                        ADD EAX, 2 * 1024 * 1024
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .fail1
                        CMP CL, 2
                        JNE .fail2


                        Call AllocateMemory32   ; find our own bit of memory
                        JC .fail
                        Call ZeroPage32         ; set the page to the desired values

                        XCHG EAX, EDI

                        Call MapL1Table32

                        CLC
                        RET


.b00ringpop:            STC
                        RET

                        ; release address
.fail2:                 OOPS
.fail1:                 OOPS
.fail:                  OOPS





                        ; Function: ManageMemoryL2
                        ; Insert a node into the memory manager tree at the 3rd level
                        ; The node will describe 1G of memory in 2M units
                        ;
                        ; in:
                        ;     EAX - 0x0000000D
                        ;     ESI - the page in physical memory to be used. Supply all
                        ;           ones to let the kernel choose a free page
                        ;     EDI - the physical address that the node should contain
                        ;
                        ; out:
                        ;     CF - set if the node couldn't be mapped
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
SYSPL_MANAGEMEMORYL2    EQU PRIVILEGE_DRIVER
ManageMemoryL2:
                        ; FIXME: ESI usage?

                        MOV EAX, EDI
                        Call RequestAddress32
                        OR EAX, EAX
                        JNZ .b00ringpop
                        CMP CL, 3
                        JNE .fail

                        Call AllocateMemory32   ; find our own bit of memory
                        JC .fail
                        Call ZeroPage32         ; set the page to the desired values

                        XCHG EAX, EDI

                        Call MapL2Table32

                        CLC
                        RET


.b00ringpop:            STC
                        RET

                        ; release address
.fail:                  MOV EAX, EDI
                        Call RemoveMemoryReference32
                        CLC
                        RET



                        ; Function: Yield
                        ; Yield the execution resource to another program
                        ;
                        ; in:
                        ;     EAX - 0x0000000E
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
SYSPL_YIELD         EQU PRIVILEGE_USER
Yield:                  CALL Schedule
                        RET



                        ; Function: CreateAddressSpace
                        ; Creates a new set of paging structures
                        ;
                        ; in:
                        ;     EAX - 0x0000000F
                        ;     ESI - the page in physical memory to be used. Supply all
                        ;           ones to let the kernel choose a free page
                        ;     EDI - the virtual address to map the page table
                        ;
                        ; out:
                        ;     CF - set if the set could not be created
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
SYSPL_CREATEADDRESSPACE EQU PRIVILEGE_USER
CreateAddressSpace:     CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem1
                        CMP CL, 1
                        JNE .memerr1
                        MOV EBX, EAX
                        ADD EAX, PAGE_SIZE      ; get next page in memory
                        Call RequestMemory32
                        OR EAX, EAX
                        JNZ .nomem2
                        CMP CL, 1
                        JNE .memerr2
                        JMP .gotmem

.automemory:            Call AllocateMemory32
                        JC .nomem1
                        MOV EBX, EAX
                        Call AllocateMemory32
                        JC .nomem2

                        ; EBX = pagedir
                        ; EAX = pagetable
.gotmem:                PUSH EDI
                        Call WriteAddressSpace32
                        POP EDI

                        PUSH EBX
                        MOV ESI, EBX
                        MOV EDX, CR3
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGETYPE_PD
                        Call SetPageTableEntry32Hole
                        JC .writeerr
                        POP EBX

.poke:
                        ;this is a crashme test, enable in case of debugging purposes
                        ;CLI
                        ;MOV EDX, CR3
                        ;MOV CR3, EBX
                        ;JMP 0x8:.peek
.peek:
                        ;MOV CR3, EDX
                        ;STI


                        CLC
                        RET

.nomem1:
.memerr1:               STC
                        RET
.nomem2:
.memerr2:

.writeerr:
                        OOPS



                        ; Function: CreateThread
                        ; Creates a new thread structure
                        ;
                        ; in:
                        ;     EAX - 0x00000010
                        ;     ESI - the page in physical memory to be used. Supply all
                        ;           ones to let the kernel choose a free page
                        ;     EDI - the virtual address to map the thread structure
                        ;     EBX - the entry point of the new thread
                        ;     EDX - the initial value for the new thread's stackpointer
                        ;
                        ; out:
                        ;     CF - set if the thread could not be created
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
SYSPL_CREATETHREAD  EQU PRIVILEGE_USER
CreateThread:           PUSH EDX
                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr

                        MOV EAX, ESI
                        AND EAX, PAGE_ADDRMASK
                        JMP .mapped

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

.mapped:                ; EAX = address of allocated mem

                        MOV ESI, EAX    ; ESI = pointer to TD
                        PUSH EBX
                        PUSH ESI
                        PUSH EDI        ; ESI = pointer to TD in virtual mem
                        MOV EDX, CR3
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGETYPE_TASK
                        Call SetPageTableEntry32Hole
                        JC .writeerr
                        POP EDI
                        POP ESI
                        POP EBX
                        PUSH EDI

                        MOV EDX, ESI
                        MOV ESI, EDI

                        ;     EDI - location to write
                        ;     ESI - location of page in target virtual memory
                        ;     EDX - location of page in physical memory
                        ;     EBX - starting EIP
                        Call FillTaskDescriptor32

                        ; EBX -> GPRs

                        ; set initial ESP
                        POP ESI
                        POP EDX
                        PUSH EDX
                        MOV [EBX+15*4], EDX

                        MOV EDX, CR3
                        MOV ECX, 0
                        ;MOV ESI, EDX
                        OR EDX, 0x1

                        Call QueueThread

                        POP EDX

                        STC
                        RET

.writeerr:              OOPS
.nomem:                 OOPS
.memerr:                OOPS




                        ; Function: CreateThreadRemote
                        ; Creates a new thread structure in a different address space
                        ;
                        ; in:
                        ;     EAX - 0x00000011
                        ;     ESI - the page in physical memory to be used. Supply all
                        ;           ones to let the kernel choose a free page
                        ;     EDI - the virtual address to map the thread structure
                        ;     EBX - the entry point of the new thread
                        ;     EDX - the initial value for the new thread's stackpointer
                        ;     ECX - the handle of the address space
                        ;
                        ; out:
                        ;     CF - set if the thread could not be created
                        ;
                        ; destroyed:
                        ;     unknown
                        ;
SYSPL_CREATETHREADREMOTE EQU PRIVILEGE_USER
CreateThreadRemote:     PUSH EDX
                        PUSH ECX
                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr

                        MOV EAX, ESI
                        AND EAX, PAGE_ADDRMASK
                        JMP .mapped

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

.mapped:                ; EAX = physical address of task table, ESI discardable, EDX+ECX pushed

                        ; test the reference
                        POP ECX

                        ; Fixme: Race condition. Read-addref-read pattern needs to be added

                        PUSH EAX
                        PUSH EDI
                        MOV EDI, ECX
                        MOV EDX, CR3
                        Call ReadPageTableEntry32Hole
                        JC .nodir
                        POP EDI
                        POP EAX
                        ; is it a PD?
                        MOV ESI, EDX
                        AND ESI, PAGE_FREE
                        AND EDX, PAGE_ADDRMASK
                        CMP ESI, PAGETYPE_PD
                        JNE .nodir

.step2:

                        ; Remote map the thread structure
                        PUSH EDX
                        MOV ESI, EAX    ; ESI = pointer to TD
                        PUSH EBX
                        PUSH ESI
                        PUSH EDI        ; ESI = pointer to TD in virtual mem
                        OR ESI, PAGE_PRESENT | PAGE_WRITABLE | PAGETYPE_TASK
                        Call SetPageTableEntry32Hole
                        JC .writeerr
                        POP EDI
                        POP ESI
                        POP EBX
                        POP ECX

                        MOV EAX, ESI
                        MOV ESI, EDI
                        POP EDX
                        PUSH EDX

                        ;     ESI - location of page in target virtual memory
                        ;     EAX - location of page in physical memory
                        ;     EBX - starting EIP
                        ;     EDX - starting ESP
                        Call FillTaskDescriptorRemote32

.step3:

                        MOV EDX, ECX ; move CR3 value where needed
                        MOV ECX, 0
                        OR EDX, 0x1
                        ; TD is still in ESI
                        Call QueueThread

                        POP EDX

                        STC
                        RET

.nodir:                 OOPS
.writeerr:              OOPS
.memerr:                OOPS
.nomem:                 OOPS




                        ; Function: TransferPage
                        ; Copies a virtual-to-physical mapping from the current address space to a remote address space
                        ;
                        ; in:
                        ;     EAX - 0x00000012
                        ;     ESI - location of the page in the current address space
                        ;     EDI - virtual address to copy the mapping to
                        ;     EBX - handle of the remote address space
                        ;
                        ; out:
                        ;     CF - clear on success, set on failure
SYSPL_TRANSFERPAGE      EQU PRIVILEGE_USER
TransferPage:

                        ; Fixme: Race condition. Read-addref-read pattern needs to be added

                        PUSH ESI
                        PUSH EDI
                        MOV EDI, EBX
                        MOV EDX, CR3
                        Call ReadPageTableEntry32Hole ; EDI = address, EDX = pagedir
                        JC .nodir
                        ; is it a PD?
                        MOV ESI, EDX
                        AND ESI, PAGE_FREE
                        AND EDX, PAGE_ADDRMASK
                        CMP ESI, PAGETYPE_PD
                        JNE .nodir
                        MOV ECX, EDX
                        POP EDI
                        POP ESI

                        ; ECX contains CR3 value for the other address space
.step2

                        PUSH EDI
                        MOV EDI, ESI
                        MOV EDX, CR3
                        Call ReadPageTableEntry32Hole ; EDI = address, EDX = pagedir
                        JC .nopage
                        MOV ESI, EDX
                        AND EDX, PAGE_USERSPACE | PAGE_PRESENT
                        POP EDI
                        CMP EDX, PAGE_USERSPACE | PAGE_PRESENT
                        JNE .nopage
                        ; EAX = PT entry


                        MOV EDX, ECX
                        ; EDI is restored previously
                        ; ESI is copied already
.step3:
                        Call SetPageTableEntry32Hole
                        JC .writefail


                        CLC
                        RET

.nodir:                 OOPS
.nopage:                OOPS
.writefail:             OOPS


                        ; Function: AllocatePageTableRemote
                        ;
                        ; Maps a page table into the page directory
                        ; to allow for small pages to be used
                        ;
                        ; in:
                        ;     EAX - 0x00000013
                        ;     EDI - the virtual address that this page table needs to cover
                        ;     ESI - the starting physical page to use.
                        ;           Supply all ones to let the kernel choose a free page
                        ;     EBX - handle of the remote address space
                        ;
                        ; out:
                        ;     CF - set if the mapping could not be done
                        ;
                        ; destroyed:
                        ;     EAX, EBX, ECX, EDX, ESI, EDI
                        ;

                        ; fixme: update destroyed regs
                        ; check
                        ; <RequestMemory32> <AllocateMemory32>
                        ; <InsertPageTable32> <ZeroPage32>
                        ; <RemoveMemoryReference32>

SYSPL_ALLOCATEPAGETABLEREMOTE EQU PRIVILEGE_USER
AllocatePageTableRemote:PUSH EBX
                        CMP ESI, -1
                        JE .automemory
                        MOV EAX, ESI
                        Call RequestMemory32    ; use selected bit of memory
                        OR EAX, EAX
                        JNZ .nomem
                        CMP CL, 1
                        JNE .memerr
                        JMP .applymem

.automemory:            Call AllocateMemory32   ; find our own bit of memory
                        JC .nomem

.applymem:              Call ZeroPage32
                        POP EBX
                        ; EAX = page location

                        ; Fixme: Race condition. Read-addref-read pattern needs to be added

                        PUSH EAX
                        PUSH EDI
                        MOV EDI, EBX
                        MOV EDX, CR3
                        Call ReadPageTableEntry32Hole ; EDI = address, EDX = pagedir
                        JC .nodir
                        ; is it a PD?
                        MOV ESI, EDX
                        AND ESI, PAGE_FREE
                        AND EDX, PAGE_ADDRMASK
                        CMP ESI, PAGETYPE_PD
                        JNE .nodir
                        MOV ECX, EDX
                        POP EDI
                        POP EAX

                        ; EAX = page location
                        ; ECX = CR3 value for the other address space
                        ; EDI = target address
                        ; ->
                        ;     EDX - the physical address of the page directory
                        ;     ESI - the physical address and bits of the entry to map
                        ;     EDI - the virtual address to map to
                        LEA ESI, [EAX + (PAGE_WRITABLE | PAGE_USERSPACE | PAGE_PRESENT)]
                        MOV EBX, ECX
                        Call SetPageDirEntry32Hole
                        JC .occupied

                        CLC
                        RET

                        OOPS

.nomem:                 OOPS
.memerr:                OOPS
.nodir:                 OOPS
.occupied:              OOPS





                        ; Function: BlockDealloc
                        ;
                        ; Removes references to a section of memory
                        ; to allow for small pages to be used
                        ;
                        ; in:
                        ;     EAX - 0x00000014
                        ;     ESI - The starting virtual address to deallocate
                        ;     EDI - The ending virtual address to deallocate
                        ;
                        ; out:
                        ;     CF - set if the mapping could not be completed
                        ;     EAX - the last virtual address unmapped.
                        ;
                        ; destroyed:
                        ;     ECX, EDX, EDI
                        ;
SYSPL_BLOCKDEALLOC  EQU PRIVILEGE_USER
BlockDealloc:           MOV EAX, ESI
                        PUSH EBX
                        MOV EBX, EDI
                        AND EAX, PAGE_ADDRMASK
                        AND EBX, PAGE_ADDRMASK

.loop:
                        MOV EDX, CR3
                        PUSH EAX
                        MOV EDI, EAX
                        Call ClearPageTableEntry32Hole
                        JC .fail
                        ; EBX contains PT value, edx, edi, eax destroyed
                        MOV EAX, EDX
                        AND EAX, PAGE_ADDRMASK
.test:                  Call RemoveMemoryReference32
                        ; edx, ecx destroyed

                        POP EAX
                        CMP EAX, EBX
                        JE .done
                        ADD EAX, PAGE_SIZE
                        JMP .loop

.done:                  POP EBX
                        ADD EAX, PAGE_SIZE - 1
                        CLC
                        RET

.fail:                  OOPS
                        POP EAX
                        DEC EAX
                        POP EBX
                        STC
                        RET


                        ; Function: Yank
                        ; Run privileged test code
                        ;
                        ; Will not stay here forever
                        ;
                        ; Currently uses a really ugly hack to change video modes
                        ;
                        ; in:
                        ;     EAX = 0x00000015
                        ;     EDX = mode number
                        ;
                        ; out:
                        ;     CF = CF value returned by the bios
                        ;     
                        ; destroyed:   
                        ;     EAX, ECX, EDX
                        
Yank:			        PUSH ESI
                        PUSH EDI
                        PUSH EBP
                        PUSH EBX
                        MOV EAX, CR0
                        AND EAX, 0x7FFFFFFF
                        CLI
                        MOV CR0, EAX		; yank off paging
                        MOV ESI, .payload
                        MOV EDI, 0x8000
                        MOV ECX, .payloadend - .payload
                        CLD
                        REP MOVSB
                        JMP FAR 8*7:0x8000 ; selector 7

.return:                MOV AX, 2 * 8
                        MOV DS, AX
                        MOV ES, AX
                        MOV SS, AX
                        MOV EAX, CR0
                        OR EAX, CR0_PG
                        MOV CR0, EAX
                        STI
                        POP EBX
                        POP EBP
                        POP EDI
                        POP ESI
                        SAL EDX, 1      ; EDX -> CF
                        RET
                        
.payload:
BITS 16			
                        MOV AX, 8*8
                        MOV SS, AX
                        MOV DS, AX
                        MOV ES, AX
                        MOV EAX, CR0
                        AND AL, 0xFE
                        MOV CR0, EAX
                        SIDT [0x8000+.oldidt-.payload]
                        LIDT [0x8000+.fixidt-.payload]

                        JMP FAR 0:0x8000 + .csflush - .payload
.csflush:               XOR AX, AX
                        MOV DS, AX
                        MOV [0x8000+.oldesp-.payload], ESP
                        MOV SS, AX
                        XOR SP, SP
                        
                        ; auto-choose vbe or legacy call
                        CMP DX, 0x100
                        JAE .vesa                        
                        MOV AX, DX
                        JMP .call
.vesa:                  MOV AX, 0x4F02
                        MOV BX, DX
.call:                  INT 0x10
                        SBB EDX, EDX
                        CLI                        
                        LIDT [0x8000+.oldidt-.payload]
                        
                        MOV EAX, CR0
                        OR AL, CR0_PE
                        MOV CR0, EAX
			MOV ESP, [0x8000+.oldesp-.payload]
                        JMP DWORD FAR 0x8:.return
                        
.fixidt:                DW 256*4-1
                        DD 0
.oldidt:                DW 0, 0, 0
.oldesp:                DD 0
BITS 32
.payloadend:

; ====================================================================================================================
; Group: Interrupt Handlers
; These functions assure a proper handling of exceptions and standard interrupts
; ====================================================================================================================




                        ; Function: IsrNMHandler
                        ; Handles the coprocessor exceptions
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;

                        ; fixme: segment register breakage
IsrNMHandler32:         PUSHAD
                        MOV AX, 0x23
                        MOV DS, AX
                        MOV ES, AX
                        STR BX
                        MOV EDX, [lpFPUHole]
                        AND EBX, 0x00000078
                        MOV EAX, [lpaCurrentProcess+EBX]
                        MOV ECX, [lpaLastFPUState+EBX]
                        MOV EDI, [EAX+TDS_PHYSICAL]
                        ADD EDI, TDS_FPUSTATE
                        CMP EDI, ECX
                        JE .done

;                       ptaddr = hole >> 10 + #cpu*4 + pagetable
;                       ptwrite = phys(31:12) + flags
;                       store = hole + #cpu*4096 + phys(11:0)
;                       EBX = #cpu*8
;                       ECX = current.phys
;                       EAX = requested.tds
;                       EDI = requested.phys
;                       EDX = hole
                        CLI
                        MOV ECX, [lpaLastFPUState+EBX]
                        MOV ESI, EBX    ; ESI = #CPU * 8
                        PUSH EBX
                        SHR EBX, 1      ; EBX = #CPU * 4
                        MOV EBP, EDX    ; EBP = hole
                        SHL ESI, 12-3   ; ESI = #CPU * 4096
                        ADD EBX, [lpPagingKernelTables] ; EBX = #CPU * 4 + pagetable
                        SHR EBP, 10     ; EBP = hole >> 10
                        ADD EDX, ESI    ; EDX = hole + #cpu*4096, ESI free
                        MOV ESI, ECX    ; ECX = phys
                        ADD EBX, EBP    ; EBX = ptaddr, EBP free
                        ADD EAX, TDS_FPUSTATE
                        AND ESI, PAGE_PARAMMASK     ; ESI = phys(11:0)
                        AND ECX, PAGE_ADDRMASK      ; ECX = phys(31:12)
                        POP EBP         ; EBP = #CPU * 8
                        ADD EDX, ESI    ; EDX = store, ESI free
                        OR ECX, PAGE_PRESENT | PAGE_WRITABLE ; ECX = ptwrite
                        MOV [lpaLastFPUState+EBP], EDI
                        MOV [EBX], ECX  ; point hole to appropriate piece of memory

                        MOV ECX, CR3
                        MOV CR3, ECX

                        ; EDX -> store data
                        ; EAX -> get data

                        TEST dword [dwaProcessorCaps1], KCPUID_FXSR
                        JZ .fpuonly

                        CPU 686
.fpuandsse:             CLTS
                        FXSAVE [EDX]
                        FXRSTOR [EAX]
                        JMP .done2

.fpuonly:               CLTS
                        FNSAVE [EDX]
                        FRSTOR [EAX]
                        JMP .done2
                        CPU 386

.done:                  CLTS
.done2:                 STI
                        POPAD
                        IRET


                        ; Function: IsrGPFHandler32
                        ; handles the GPF exception
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
IsrGPFHandler32:
                        TEST dword [ESP+3*4], EFLAGS_VM
                        JNZ .vmtrap
                        CMP word [ESP+2*4], 0x0008
                        JE .panic

.usertrap:              JMP IntHandler.errstub
                        OOPS
                        DB 1
.vmtrap:                PUSH EBX
                        PUSH EDX
                        STR BX
                        MOV DX, 0x23
                        AND EBX, 0x00000078
                        MOV DS, DX
                        MOV EBX, [lpaCurrentProcess+EBX]
                        MOV ES, DX
                        ; store GPRs
                        MOV [EBX+TDS_V8086REGS+0], EAX
                        MOV [EBX+TDS_V8086REGS+4], ECX
                        POP EDX
                        MOV [EBX+TDS_V8086REGS+8], EDX
                        MOV [EBX+TDS_V8086REGS+16], ESI
                        MOV [EBX+TDS_V8086REGS+20], EDI
                        MOV [EBX+TDS_V8086REGS+24], EBP
                        MOV ECX, [ESP+5*4]
                        MOV [EBX+TDS_V8086REGS+28], ECX
                        MOV ECX, EBX                        ; ECX -> TDS
                        POP EBX
                        MOV [ECX+TDS_V8086REGS+12], EBX

                        ; store segregs
                        MOV BX, [ESP+9*4]
                        MOV [ECX+TDS_V8086SEGS+0], BX
                        MOV BX, [ESP+8*4]
                        MOV [ECX+TDS_V8086SEGS+2], BX
                        MOV BX, [ESP+7*4]
                        MOV [ECX+TDS_V8086SEGS+4], BX
                        MOV BX, [ESP+6*4]
                        MOV [ECX+TDS_V8086SEGS+6], BX
                        MOV BX, [ESP+5*4]
                        MOV [ECX+TDS_V8086SEGS+8], BX

                        ; pack up CS:IP into ESI c.f. EnterV8086
                        MOV SI, [ESP+2*4]
                        SHL ESI, 16
                        MOV SI, [ESP+1*4]

                        ; put flags into EDI
                        MOV EDI, [ESP+3*4]

                        ; build IRET stackframe
                        PUSH 0x23                   ; SS
                        MOV EBX, [ESP+6*4]
                        SHL EBX, 4
                        ADD EBX, [ESP+5*4]
                        PUSH EBX                    ; ESP
                        PUSH EFLAGS_IF              ; flags
                        PUSH 0x1B                   ; CS
                        PUSH dword [ECX+TDS_VMEXIT] ; EIP

                        IRET

; todo: Check whether the stored state is consistent.

                        DB 2
.panic                  ;JMP IntHandler.errstub
                        OOPS
                        CLI
                        HLT



                        ; Function: IsrPFHandler32
                        ; handles the pagefault exception
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
IsrPFHandler32:         PUSHAD
                        MOV ESI, ESP
                        MOV EDI, .crashspace
                        ; Either exactly userspace, or some runover into previous kernel stackspace
                        ; both should be safe
                        MOV ECX, 14                     
                        CLD
                        REP MOVSD

                        ; flag PF on the keyboard
.w1:                        
                        IN AL, 0x64
                        TEST AL, 0x02
                        JNZ .w1
                        MOV AL, 0xED
                        OUT 0x60, AL
.w2:                    IN AL, 0x64
                        TEST AL, 0x02
                        JNZ .w2
                        MOV AL, 0x05
                        OUT 0x60, AL

                        CLI
                        MOV EAX, CR0
                        AND EAX, 0x7FFFFFFF
                        MOV CR0, EAX

                        MOV ESP, 0x80000

                        MOV EBP, .crashspace
                        CALL DumpRegState
                        
                        MOV EBP, .crashspace + 8 * 4
                        MOV EAX, 14
                        Call DumpIntErrorState
                        
                        Call DumpCRState

                        Call Debugger_Enter

                        JMP $
.hexout:                PUSH ECX
                        PUSH EAX
                        MOV ECX, EAX
.hexloop                OR ECX, ECX
                        JZ .done
                        XOR EDX, EDX
                        MOV DL, CL
                        SHR ECX, 4
                        AND DL, 0xf
                        ADD EDX, .hexdigits
                        MOV AL, [EDX]
                        MOV [EDI], AL
                        OUT 0xE9, AL
                        MOV byte [EDI+1], 0x4F
                        SUB EDI, 2
                        JMP .hexloop

.done:                  MOV AL, 13
                        OUT 0xE9, AL
                        MOV AL, 10
                        OUT 0xE9, AL
                        POP EAX
                        POP ECX
                        RET
.hexdigits:             DB "0123456789ABCDEF"

.crashspace:            TIMES 16 DD 0


                        ; Function: IrqRTCHandler
                        ; acknowledges the RTC timer interrupt and updates the time
                        ;
                        ; in:
                        ;     none
                        ;
                        ; out:
                        ;     none
                        ;
                        ; destroyed:
                        ;     none
                        ;
IrqRTCHandler:          PUSH EAX
                        INC dword [.ticks]

                        Call IntHandler.kbwait

                        ; fixme: causes panic on Bochs due to kb overflow
                        MOV AL, [.ticks]
                        AND AL, 0x7
;                        Call IntHandler.kbled

                        MOV AL, 0x0C
                        OUT 0x70, AL
                        IN AL, 0x71
                        POP EAX
                        JMP IntHandler.slavestub

.ticks:                 DD 0




; ====================================================================================================================
; Group: Data storage
; Contains all variables, pointers and preallocated datastructures
; ====================================================================================================================
ALIGN 8

                        ; lpVideoMemoryBase
                        ; contains the base address of video memory
lpVideoMemoryBase:      DQ 0


; Bootloader information

                        ; Variable: lpBootloaderData
                        ; holds the address of the structure passed by the bootloader
lpBootloaderData:       DD 0
                        ; Variable: lpRamdisk
                        ; holds the base address of the ramdisk. GRUB uses the mb structure
lpRamdisk:              DD 0

; Processor-related information

                        ; Variable: szaHostName
                        ; contains the name of the host (processor or vm) for each processor
szaHostName:            TIMES 8 * 16 DB 0
                        ; Variable: baProcessorArch
                        ; contains the family, needed for MSRs, APIC, IOAPIC and related stuff
                        ; values:
                        ;  0  - Unknown architecture
                        ;  1  - Intel 386 (external FPU)
                        ;  2  - Intel 486+ (internal FPU, intel MSRs, L.APIC)
                        ;  3  - AMD
                        ;  4  - Cyrix (Cyrix MSRs)
                        ;
baProcessorArch:        TIMES 8 DB 0
                        ; Variable: bcdaProcessorVersion
                        ; contains an unpacked BCD version number of each processor: family.major.minor.revision
bcdaProcessorVersion:   TIMES 8 DD 0
                        ; Variable: dwaProcessorCaps1
                        ; contains a bitmask of host capabilities for each processor
                        ; the bits generally correspond to cpuid flags, however, the startup code filters the
                        ; broken/unsupported/disabled features so to make these readings more accurate.
                        ; bit  0 - FPU present
dwaProcessorCaps1:      TIMES 8 DD 0
                        ; Variable: dwaProcessorCaps2
                        ; contains a bitmask of host capabilities for each processor. see <dwaProcessorCaps1>
dwaProcessorCaps2:      TIMES 8 DD 0
                        ; Variable: dwaProcessorSharedCaps
                        ; the capabilities shared by all processors
qwProcessorSharedCaps:  DD 0
                        DD 0
                        ; Variable: dwaProcessorMaxCaps
                        ; all the capabilities that are supported by at least one processor
qwProcessorMaxCaps:     DD 0
                        DD 0

                        ; Variable: dwProcessorCount
                        ; the number of logical processors in the systems
dwProcessorCount:       DD 1


; Architecture-related information

                        ; Variable: lpRSDPBase
                        ; the location of the RSDP structure
lpRSDPBase:             DD 0

                        ; Variable: lpMPTablePointer
                        ; the location of the MP pointer
lpMPTablePointer:       DD 0


; memory management info:

                        ; Variable: lpKernelBasePageTable
                        ; contains a pointer to the base kernel page table (386 paging)
lpKernelBasePageTable:  DD 0
                        ; Variable: lpKernelBasePAETable
                        ; contains a pointer to the base kernel page table (PAE/LM paging)
lpKernelBasePAETable:   DD 0
                        ; Variable: lpAddressSpaceMirror
                        ; contains a pointer to the location where the current address space
                        ; information structure is mapped
lpAddressSpaceMirror:   DD 0
                        ; Variable: lpDirectoryTableMirror
                        ; contains a pointer to the location where the current page directory is mapped
                        ; (if appropriate)
lpDirectoryTableMirror: DD 0
                        ; Variable: lp32BitGDT
                        ; the location of the 32 bit gdt
lp32BitGDT:             DD 0
                        ; Variable: lp32BitIDT
                        ; the location of the 32 bit IDT
lp32BitIDT:             DD 0
                        ; Variable: lp64BitIDT
                        ; the location of the 64 bit IDT
lp64BitIDT:             DD 0
                        ; Variable: lpL4MemoryTable
                        ; the 4th level memory reference structure.
lpL4MemoryTable:        DD 0
                        ; Variable: lpL3MemoryTable
                        ; the 3rd level memory reference structure.
lpL3MemoryTable:        DD 0
                        ; Variable: lpL2MemoryTable
                        ; the 2nd level memory reference structure. (comparable to PAE page directory)
lpL2MemoryTable:        DD 0
                        ; Variable: lpL1MemoryTable
                        ; the 1st level memory reference structure. (comparable to PAE table: covers 0-2M)
lpL1MemoryTable:        DD 0

                        ; Variable: lpHeapTop
                        ; the current top of the heap
lpHeapTop:              DD 0

                        ; Variable: lpPageTableSource
                        ; the sample table that contains all the pages required by the kernel
lpPageTableSource:      DD 0

                        ; Variable: lpPagingLocks
                        ; holds the lock bits for the paging hole
lpPagingLocks:          DD 0

                        ; Variable: lpMMTableHoles
                        ; points to four consecutive pages where parts of the paging structures can be mapped into
lpMMTableHoles:         DQ 0

                        ; Variable: lpPagingHoles
                        ; points to four consecutive pages where parts of the paging structures can be mapped into
lpPagingHoles:          DQ 0
                        ; Variable: lpPagingKernelTables
                        ; points to four consecutive pages where the paging structures for the kernel are
                        ; offset +  0k - page table
                        ; offset +  4k - page directory
                        ; offset +  8k - 3rd level
                        ; offset + 12k - top level
lpPagingKernelTables:   DQ 0
                        ; Variable: lpZeroHole
                        ; points to a page that can be mapped for clearing pages.
lpZeroHole:             DQ 0
                        ; Variable: lpFPUHole
                        ; points to a series of pages that can be mapped for
                        ; lazily storing FPU state
lpFPUHole:              DQ 0
                        ; Variable lpAddressHole
                        ; points to a pair of pages that can be mapped for creating
                        ; new address space structures
lpAddressHole:          DQ 0
                        ; Variable: lpFreePagingOffset
                        ; the first address where allocations can be done
lpFreePagingOffset:     DQ 0


; I/O address space management

; todo


; Process management info

                        ; Variable: lpaCurrentAddressSpace
                        ; contains the currently active address space for each processor
lpaCurrentAddressSpace: TIMES 8 DQ 0

                        ; Variable: lpaCurrentProcess
                        ; contains the current task's descriptor
lpaCurrentProcess:      TIMES 8 DQ 0

                        ; Variable: lpaCurrentScheduleProc
                        ; contains the location of the schedule_out function for the current thread
lpaCurrentScheduleProc: TIMES 8 DQ 0

                        ; Variable: lpaSchedulerBase
                        ; contains the base of the scheduling table
lpaSchedulerBase:       TIMES 8 DQ 0

                        ; Variable: lpaSchedulerOffset
                        ; contains the offset into the scheduler table of the currently executing task
lpaSchedulerOffset:     TIMES 8 DQ 0

                        ; Variable: dwSchedulerLocks
                        ; contains access locks to the scheduler queue
dwSchedulerLocks:       DD 0

                        ; Variable: lpaTSS
                        ; contains the location of the TSSs for each processor.
lpaTSS:                 TIMES 8 DQ 0

                        ; Variable: lpaIoBitmap
                        ; contains the location of the I/O permission bitmap
lpaIoBitmap:            DQ 0

                        ; Variable: lpaLastFPUState:
                        ; Contains the location of the FPU state to store.
lpaLastFPUState:        TIMES 8 DQ 0


; GDT and IDT allocation mask
                        ; Variable: baInterruptBitmap
                        ; contains a bitmap of the used interrupt entries
baInterruptBitmap:      TIMES (32-6) DB 0

                        ; Variable: baGDT32Bitmap
                        ; contains a bitmap of the used GDT entries
baGDT32Bitmap:          DB 0x7f
                        TIMES (0x40 - 1) DB 0



; PIC state info

                        ; Variable: bMasterMask
                        ; contains the interrupt mask for the Master PIC
bMasterMask:            DB 0
                        ; Variable: bSlaveMask
                        ; contains the interrupt mask for the Slave PIC
bSlaveMask:             DB 0


; kernel image info
                        ; Constant: kernelimageend
                        ; the first byte after the kernel
kernelimageend:
                        ; Constant: kernelimagesize
                        ; the size of the kernel image in bytes.
kernelimagesize     EQU kernelimageend - kernelimagestart
                        ; Constant: kernelbssstart
                        ; the page-aligned location where the bss should start.
kernelbssstart      EQU ((kernelimageend & 0xfffff000) + 0x2000)
