; Summary: ucode.asm
; Contains microcode uploading, proving and relocating functionality.
;
; Author:
;     Marcel Sondaar
;
; License:
;     Educational Purposes
;


                        ; Function: VerifyMicrocode
                        ; Attempts to proof that the microcode conforms
                        ; to all invariants of the system.
                        ;
                        ; To achieve this, the microcode is disassembled
                        ; and each instruction is checked against the
                        ; following security conditions
                        ; 
                        ; Isolation          - the microcode uses a fixed
                        ;                      amount of global and local
                        ;                      storage
                        ; Register invariant - Ensures that the value of
                        ;                      important registers hold
                        ;                      valid values (xSP, segments, etc)
                        ; I/O security       - Ensures that only owned
                        ;                      ports and memory areas
                        ;                      are accessed
                        ; Termination        - Verifies that the microcode
                        ;                      terminates within reasonable
                        ;                      time. Potentially valid
                        ;                      uploads can be refused based
                        ;                      on this constraint, since
                        ;                      program flow instructions
                        ;                      are required to be forward
                        ;                      (conditional) jumps
                        ; Compatibility      - Verifies that the microcode
                        ;                      does not perform privileged
                        ;                      or unsupported instructions
                        ;
                        ; The algorithm should allow for high-level
                        ; functions to pass validity to minimize the need
                        ; of assembly.
                        ;
                        ; in:
                        ;     ESI - the location of the microcode
                        ;     EDI - the virutal address where storage is
                        ;           referenced
                        ;     EDX - the size of local storage space
                        ;     EBX - the size of the microcode
                        ;
                        ; out:
                        ;     CF  - set if validity could not be established
                        ;         - clear when the program could be proven
                        ;     ESI - location of the instruction failing
                        ;           verification (valid when CF is set)
                        ;     EAX - an error code relevant to the invariant
                        ;           (valid when CF is set)
                        ;     EAX - the registers that were clobbered
                        ;           (valid when CF is clear)
                        ;
                        ; destroyed:
                        ;     - EAX, ESI (if CF clear)
                        ;     - EBX, ECX, EDX, EBP
                        ;
                        ; stackframe: 
                        ;     +0..+3f  - min/max values for GPRs
                        ;     +40..+43 - ESP difference
                        ;     +44..+47 - register usage
                        ;     +48..+4B - maximum code address
                        ;
VerifyMicrocode:        PUSH EBP
                        SUB ESP, 16 * 4 + 12
                        MOV EBP, ESP
                        MOV ECX, 8
                        MOV EAX, EBP
.minmaxinit:            MOV dword [EAX], 0x0
                        MOV dword [EAX+4], 0xffffffff
                        ADD EAX, 8
                        DEC ECX
                        JNZ .minmaxinit
                        MOV dword [EAX], 0
                        MOV dword [EAX + 4], 0
                        MOV EAX, EBX
                        ADD EAX, ESI
                        MOV [EBP+0x48], EAX

                        ; ESI = next instr, EAX, EBX, ECX free

.checkinstr:            XOR EBX, EBX
                        MOV AH, 4
.checkinstr2:           LODSB
                        MOV BL, AL
                        SHR EBX, 3
                        JMP [.optable + 4 * EBX]

.failopcode:            DEC ESI
                        ADD ESP, 16 * 4 + 12
                        POP EBP
                        STC
                        RET

.nextopcode:            CMP ESI, [EBP+0x48]
                        JBE .checkinstr
                        MOV EAX, ESI
                        ADD ESP, 16 * 4 + 12
                        POP EBP
                        STC
                        RET

.parseprefix:           CMP AL, 0x66
                        JNE .failopcode
                        CMP AH, 4
                        JNE .failopcode
                        MOV AH, 2
                        JMP .checkinstr2

.optable:
.00.optable.add:        DD VerifyAdd
.08.optable.or:         DD VerifyCompoundSix
.10.optable.adc:        DD VerifyCompoundSix
.18.optable.sbb:        DD VerifyCompoundSix
.20.optable.and:        DD VerifyCompoundSix
.28.optable.sub:        DD VerifySub
.30.optable.xor:        DD VerifyCompoundSix
.38.optable.cmp:        DD VerifyCompare
.40.optable.inc:        DD VerifyIncrement
.48.optable.dec:        DD VerifyDecrement
.50.optable.push:       DD VerifyPush
.58.optable.pop:        DD VerifyPop
.60.optable.prefix:     DD .parseprefix
.68.optable.muliox:     DD .failopcode
.70.optable.jcc1:       DD .failopcode
.78.optable.jcc2:       DD .failopcode
.80.optable.1a:         DD .failopcode
.88.optable.movlea:     DD VerifyMoveLea
.90.optable.xchg:       DD VerifyExchange
.98.optable.flags:      DD .failopcode
.A0.optable.moval:      DD VerifyMoveReg1
.A8.optable.tststr:     DD .failopcode
.B0.optable.movb:       DD VerifyMoveByte
.B8.optable.movl:       DD VerifyMoveLong
.C0.optable.retmov:     DD VerifyMoveShiftReturn
.C8.optable.intlv:      DD VerifyLeave
.D0.optable.shift:      DD VerifyShifts
.D8.optable.fpu:        DD .failopcode
.E0.optable.ioimm:      DD VerifyInOutImm
.E8.optable.jmpio:      DD VerifyInOut
.F0.optable.rephlt:     DD .failopcode
.F8.optable.tflags:     DD .failopcode

; unimplemented parts
VerifyInOutImm:
VerifyInOut:
VerifyShifts:
VerifyLeave:
VerifyMoveByte:
VerifyMoveLong:
VerifyMoveShiftReturn:
VerifyMoveReg1:
VerifyExchange:
VerifyMoveLea:
VerifyPop:
VerifyPush
VerifyIncrement:
VerifyDecrement:
VerifyCompare:
VerifySub:
VerifyAdd:
VerifyCompoundSix:
                        JMP VerifyMicrocode.failopcode




