ORG 0h
BITS 16
CPU 386

;
; Summary: stage2_ia_pc.asm
; *The stage 2 bootloader for My Operating System (MOS)*
;
; - Loads and prepares stage 3
; - Loads and constructs all other files in the root directory into a ramdisk
; - Enables Gate A20
; - Determines memory map
; - Brings the system in a state similar to that of a multiboot loader
;
;
; Author:
;     Marcel Sondaar
;
; License:
;     Educational purposes
;

; new style memory map
; 0x00010000 This file
; 0x00020000 FAT
; 0x0002F000 ramdisk header
; 0x00030000 temporary load storage
; 0x00080000 kernel

begin:
                JMP start
TIMES 4-$+begin DB 0
                DW DoneLoad.stop

start:          push cs
                pop ds
                
                XOR SP, SP
                MOV SS, SP
                
                Call InitGDT
                CLI
                MOV EAX, CR0
                OR AL, 1
                MOV CR0, EAX
                MOV CX, 0x8
                MOV DS, CX
                MOV ES, CX                
                MOV GS, CX
                MOV byte [dword 0x401000], 0x01
                AND AL, 0xFE
                MOV CR0, EAX               
                PUSH CS
                POP DS

                Call InitA20
                
                MOV AX, 0x7C0
                MOV ES, AX
                MOV FS, AX

                MOV [drv], DL                

                ; temporary to fix stage 4 crash
                PUSH ES
                MOV AX, 0x2F00
                MOV ES, AX
                MOV AX, 0xCCCC
                MOV CX, 0x8000
                REP STOSW
                POP ES

                LEA SI, [msg]

initmessage:    MOV AH, 0eh
                MOV AL, [SI]
                OR AL, AL
                JE loadfat
                INT 10h
                INC SI
                JMP initmessage

loadfat:        MOV DL, [drv]
                MOV AH, 02h
                MOV AL, [ES:bpbSectorsPerFAT]
                MOV CX, [ES:bpbReservedSectors]
                INC CL
                XOR DH, DH

                MOV BX, 0x2000
                MOV ES, BX
                XOR BX, BX

                STC
                INT 13h
                STI

                MOV CX, 0
checkdirectory: PUSH CX
                MOV DX, CX
                CALL GetDirCluster
                POP CX
                MOV BX, CX
                AND BX, 0xF
                SHL BX, 5
                ADD BX, 0x400
                MOV EAX, [ES:BX + 8]
                PUSH CX
                CMP EAX, 0x004E4942
                JE loadfile
                CMP EAX, 0x00534f4d
                JE loadfile
                JMP nextfile

loadfile:       MOV AH, 0eh
                MOV AL, '*'
                INT 10h
                
                MOV DX, [es:bx+26]  ; get cluster index

                MOV di, bx
                LEA si, [sysf1]
                MOV cx, sysfs
                CLD
                REPE CMPSB
                JNE notf1             ; ignore stage 2
                jmp nextfile

notf1:          MOV di, bx
                LEA si, [sysf2]
                MOV cx, sysfs
                CLD
                REPE CMPSB
                JNE imgfile           ; go to next case if it is not a kernel
                MOV AX, 0x8000        ; the kernel gets loaded to 0x8000:0000 onward
                CALL FileToMemory
                JMP nextfile

imgfile:        MOV EAX, [curseg]
                MOV ECX, [es:bx+28]   ; filesize
                MOV ESI, [es:bx]      ; name
                MOV EDI, [es:bx+4]    ; name

                MOV BX, 0x2F00
                MOV ES, BX
                MOV BX, [curent]
                SHL BX, 4
                MOV [ES:bx+0], ESI    ; ramdisk: name
                MOV [ES:bx+4], EDI    ; ramdisk: name
                MOV [ES:bx+8], ECX    ; ramdisk: size
                XOR ECX, ECX
                MOV ECX, EAX
                SHL ECX, 4
                SUB ECX, 0x400000     ; subtract header location
                MOV [ES:bx+12], ECX   ; ramdisk: relative offset

                MOV dword [ES:bx+16], 0
                MOV dword [ES:bx+20], 0
                MOV dword [ES:bx+24], 0
                MOV dword [ES:bx+28], 0

                MOV bx, ax
                INC word [curent]
                MOV AX, 0x3000        ; load to 0x00030000 initially
                CALL FileToMemory
                
                SUB AX, 0x3000
                MOVZX EAX, AX
                MOV ECX, EAX
                MOV EDI, [curseg]     ; save previous location
                                
                ADD [curseg], EAX                

                CMP AX, 0x8000        ; check for memory bug
                JBE movefile
                LEA SI, [sizeerror]
                JMP nokernelmsg

movefile:       ; move the file to the 4M mark
                Call InitGDT
                PUSH EDX
                PUSH DS
                PUSH ES
                MOV ESI, 0x30000
                SHL EDI, 4
                SHL ECX, 2

                CLI
                MOV EDX, CR0
                OR DL, 1
                MOV CR0, EDX
                MOV DX, gdtKDataSel
                MOV DS, DX
                MOV ES, DX
                a32 REP MOVSD
                MOV EDX, CR0
                AND DL, 0xFE
                MOV CR0, EDX
                STI                                

                POP ES
                POP DS
                POP ESI               

nextfile:       PUSHA
                MOV AH, 0eh
                MOV AL, ' '
                INT 0x10
                POPA
                
                POP CX
                INC CX
                MOV AX, 0x7C0
                MOV ES, AX
                MOV AX, [ES:bpbRootEntries]
                CMP CX, AX
                JE DoneLoad
                JMP checkdirectory

                ; Function: DoneLoad
                ; prepare to boot mos86
DoneLoad:       mov ah, 0eh
                mov al, 10
                int 0x10
                mov ah, 0eh
                mov al, 13
                int 0x10

                MOV ax, 0x8000
                MOV ES, AX
                MOV EAX, [ES:0]
                CMP EAX, 0x90909090
                JNE nokernel

                Call InitMemMap                
                Call InitGDT
                Call InitTextPos

                MOV AL, 0xff
                OUT 0x21, AL
                OUT 0xA1, AL

                CLI
                XOR EBX, EBX
                PUSH CS
                POP BX
                SHL EBX, 4
                ADD EBX, mosSeg
                MOV EDX, 0x400000
                MOV EAX, CR0
                OR AL, 1
.stop           MOV CR0, EAX
                MOV AX, gdtKDataSel
                MOV DS, AX
                MOV ES, AX
                MOV FS, AX
                MOV GS, AX
                MOV SS, AX

                MOV ESI, 0x0002F000
                MOV EDI, 0x00400000
                MOV ECX, 0x400
                a32 REP MOVSD

                JMP dword gdtKCodeSel:0x80000

nokernel:       LEA SI, [kernelerror]
nokernelmsg:    MOV AH, 0eh
                MOV AL, [SI]
                OR AL, AL
                JE nokernelend
                INT 10h
                INC SI
                JMP nokernelmsg
nokernelend:    JMP $

                ; Function: GetDirCluster
                ; DX = directory entry
                ;
GetDirCluster:  MOV AX, 0x7C0
                MOV ES, AX
                XOR AH, AH
                MOV AL, [ES:bpbNumberOfFATs]
                MOV CL, [ES:bpbSectorsPerFAT]
                MUL CL
                ADD AX, [ES:bpbReservedSectors]
                PUSH BX
                MOV BX, DX
                AND DX, 0xfff0
                CMP DX, BX
                JNE .wasloaded
                SHR DX, 4
                ADD DX, AX
                POP BX
                PUSH FS
                MOV AX, 0x7E0
                MOV FS, AX
                CALL LoadSector
                POP FS
                RET
.wasloaded:     POP BX
                RET

                ; Function FileToMemory
                ; loads a file from disk (cluster DX) to segment AX
                ;
FileToMemory:   PUSH AX
                PUSH DX

                Call LoadCluster
                POP DX
                Call GetNextCluster
                POP AX
                MOV ES, AX
                XOR DI, DI
                XOR SI, SI
                MOV BX, 0x7e0
                PUSH DS
                MOV DS, BX
                MOV CX, 512
                CLD
                REP MOVSB
                POP DS
                ADD AX, 0x20

                MOV BX, DX
                AND BX, 0x0ff0
                SUB BX, 0x0ff0
                JNZ .next
                RET

.next:          ;PUSHAD
                ;MOV AH, 0eh
                ;MOV AL, '.'
                ;INT 10h
                ;POPAD

                JMP FileToMemory

                ; Function: GetNextCluster
                ; Gets the next cluster value from the FAT table
                ;
                ; in:
                ;     - DX = current cluster number
                ;
                ; out:
                ;     - DX = new cluster number
                ;
                ; destroyed:
                ;     ES AX BX CX
                ;
GetNextCluster: MOV AX, 0x2000
                MOV ES, AX
                MOV AX, DX
                MOV BX, DX
                SHR AX, 1
                ADD BX, AX
                MOV CX, [ES:BX]
                AND DX, 1
                JNZ GetNextOdd
                AND CX, 0x0FFF
                MOV DX, CX
                RET
GetNextOdd:     SHR CX, 4
                MOV DX, CX
                RET


                ; Function: LoadCluster
                ; Loads a FAT cluster to RAM
                ;
                ; in:
                ;     - DX = clusternumber
                ;     - FS = segment to load to
                ;
                ; out:
                ;     none
                ;
                ; destroyed:
                ;     ES AX BX CX SI DI
                ;
LoadCluster:    MOV AX, 0x7C0
                MOV ES, AX
                XOR AH, AH
                MOV AL, [ES:bpbNumberOfFATs]
                MOV CL, [ES:bpbSectorsPerFAT]
                MUL CL
                ADD AX, [ES:bpbReservedSectors]
                MOV BX, AX
                XOR AH, AH
                MOV AL, [ES:bpbSectorsPerCluster]
                MOV [numsectors], AL
                SUB DX, 2
                MUL DX
                MOV DX, BX
                ADD DX, AX
                MOV CX, [ES:bpbRootEntries]
                SHR CX, 4
                ADD DX, CX


                ; Function: LoadSector
                ; Loads a sector (linear) from floppy to RAM
                ;
                ; in:
                ;     - DX = sectornumber
                ;     - FS = segment to address
                ;
                ; out:
                ;     none
                ;
                ; destroyed:
                ;     ES AX BX CX SI DI
                ;
LoadSector:     MOV AX, 0x7C0
                MOV ES, AX
                PUSH DX
                MOV CX, 3
                PUSH CX
LoadSectorRedo: MOV CL, [ES:bpbSectorsPerTrack]
                MOV AL, [ES:bpbHeadsPerCylinder]
                MUL CL                          ;AX = AL * CL
                MOV CH, 0
                MOV BX, 0
LoadSectorCyl:  CMP DX, AX
                JNGE LoadSectorHd
                INC BX
                SUB DX, AX
                JMP LoadSectorCyl
LoadSectorHd:   CMP DL, CL
                JNGE LoadSectorDo
                SUB DL, CL
                INC CH
                JMP LoadSectorHd
LoadSectorDo:   MOV DH, CH
                MOV CL, DL
                MOV CH, BL
                MOV DL, [ES:bsDriveNumber]
                MOV AH, 0x02
                MOV AL, [numsectors]
                MOV BX, 0x200
                INC CL
                PUSH ES
                PUSH FS
                POP ES
                STC
                INT 13h
                STI
                POP ES
                POP CX
                POP DX
                JC LoadSectorTry
                mov byte [numsectors], 1
LoadSectorEnd:  RET
LoadSectorTry:  DEC CX
                JZ LoadSectorEnd
                PUSH DX
                PUSH CX
                JMP LoadSectorRedo

                ; Function: InitTextPos
                ; Prepares the screen for addressing
InitTextPos:    MOV AX, 0x0e54      ; Print T
                int 0x10
                MOV AX, 0x0e58      ; Print X
                int 0x10
                MOV AX, 0x0e54      ; Print T
                int 0x10
                MOV AX, 0x0e20      ; Print space
                int 0x10

                MOV AX, 0x0500      ; select page 0
                INT 0x10
                MOV AX, 0x0103      ; change cursor
                MOV CX, 0x2000      ; invisible, scaline 0-0
                int 0x10
                MOV AX, 0x0300      ; get cursor location
                XOR DX, DX
                INT 0x10
                OR DX, DX
                MOV DWORD [mosTextBase], 0x000B8000
                JZ InitTextPosMan
                MOV [mosTextChar], DX
                RET
InitTextPosMan: MOV AX, 0xB800
                MOV ES, AX
                MOV BX, 0
                MOV CX, 40 * 25
InitTextPosLp:  MOV EAX, [ES:BX]
                CMP EAX, 0x07580754
                JE InitTextPosFnd
                ADD BX, 4
                LOOP InitTextPosLp
                MOV BX, 0
InitTextPosFnd: MOV AX, BX
                ADD AX, 8
                MOV CL, 80
                SHR AX, 1
                DIV CL
                MOV [mosTextChar], AX
                RET

                ; Function: InitA20
                ; Enables A20 gate
                ;
InitA20:        MOV AX, 0x0e41
                int 0x10
                MOV AX, 0x0e32
                int 0x10
                MOV AX, 0x0e30
                int 0x10
                Call InitA20test
                JC InitA20done
                MOV AX, 0x2403
                CLC
                INT 0x15
                JNC InitA20bx
                MOV BX, 0x0f
InitA20bx:      XOR BX, 8
                MOV [a20sel], BL
                AND BL, 8
                JZ InitA20sca
                MOV AX, 0x2401      ; A20, using bios call
                int 0x15
                Call InitA20test
                JC InitA20done
InitA20sca:     MOV BL, [a20sel]
                AND BL, 4
                JZ InitA20fg
                IN AL, 0xee         ; A20, using optional bios register
                Call InitA20test
                JC InitA20done
InitA20fg:      MOV BL, [a20sel]
                AND BL, 2
                JZ InitA20kbc
                IN AL, 0x92         ; A20, using fast A20 gate
                MOV CL, AL
                AND CL, 2
                JNZ InitA20kbc      ; if a20 bit seems set, don't touch it
                OR AL, 2
                OUT 0x92, AL
                Call InitA20test
                JC InitA20done

InitA20kbc:     CLI
                CALL InitA20waitkb
                MOV AL, 0xAD
                OUT 0x64, AL

                CALL InitA20waitkb
                MOV AL, 0xD0
                OUT 0x64, AL

                CALL InitA20waitkb2
                IN AL, 0x60
                PUSH AX

                CALL InitA20waitkb
                MOV AL, 0xD1
                OUT 0x64, AL

                CALL InitA20waitkb
                POP AX
                OR AL, 2
                OUT 0x60, AL

                CALL InitA20waitkb
                MOV AL, 0xAE
                OUT 0x64, AL

                CALL InitA20waitkb
                STI

                Call InitA20test
                JC InitA20done

                JMP InitA20error
InitA20done:    MOV AX, 0x0e20
                int 0x10
                RET
InitA20error:   MOV AH, 0x0e
                MOV AL, '!'
                int 0x10
.loop:          Call InitA20test
                JNC .loop
                RET

InitA20waitkb:  IN AL, 0x64
                TEST AL, 2
                JNZ InitA20waitkb
                RET
InitA20waitkb2: IN AL, 0x64
                TEST AL, 1
                JZ InitA20waitkb2
                RET

                ; Function: InitA20test
                ; Sets carry if A20 is enabled
                ;
InitA20test:    XOR AX, AX
                MOV FS, AX
                DEC AX
                MOV GS, AX
                MOV AL, [FS:0]
                MOV AH, [GS:16]
                CMP AL, AH
                JNE InitA20Ok
                CLI
                NOT AL
                MOV [GS:16], AL
                MOV AH, [FS:0]
                MOV CL, AL
                NOT AL
                MOV [GS:16], AL
                STI
                CMP CL, AH
                JNE InitA20Ok
InitA20Fail:    CLC
                RET
InitA20Ok:      STC
                RET

                ; Function: InitGDT
                ; Builds the bootstrap GDT
                ;
InitGDT:        XOR EAX, EAX
                PUSH CS
                POP AX
                SHL EAX, 4
                ADD EAX, gdtNULL
                MOV [gdtbase], EAX
                LGDT [gdt]
                RET

                ; Function: InitMemMap
                ; builds the memory map of this computer
                ;
InitMemMap:     MOV AH, 0eh
                MOV AL, 'M'
                INT 0x10
                MOV AH, 0eh
                MOV AL, 'E'
                INT 0x10
                MOV AH, 0eh
                MOV AL, 'M'
                INT 0x10

                XOR EBX, EBX
                PUSH CS
                POP ES
InitMemMapP1L:  XOR EAX, EAX
                MOV AX, 0xE820
                MOV EDX, 0x534D4150
                LEA EDI, [mosMemory+EBX]
                MOV ECX, 20
                CLC
                INT 15h
                JC InitMemMapP2
                INC byte [mosMemoryEntries]
                OR BX, BX
                JZ InitMemMapP2
                JMP InitMemMapP1L
InitMemMapP2:   MOV AX, [mosMemoryEntries]
                OR AX, AX
                JNZ InitMemMapDone

                MOV AH, 0eh
                MOV AL, '!'
                INT 0x10
                JMP $
InitMemMapDone: MOV AH, 0eh
                MOV AL, ' '
                INT 0x10
                RET

                
                
Printhex:       PUSHAD
                MOV BX, digits
                MOV ECX, EAX
                MOV AX, 0x0E00 + '0'
                INT 0x10
                MOV AX, 0x0E00 + 'x'
                INT 0x10

                CALL .printdigit
                
                MOV AX, 0x0E00 + ' '
                INT 0x10
                POPAD
                RET
.printdigit:    
                PUSH EAX
                MOV EAX, ECX
                SHR ECX, 4
                JZ .nomoredigits
                CALL .printdigit
.nomoredigits:
                AND AL, 0xF
                XLATB

                MOV AH, 0x0E
                INT 0x10

                POP EAX
                RET        
                
digits: DB "0123456789ABCDEF"                


spinlock: JMP $
numsectors:  DB 1
a20sel: DB 0
drv:    DB 0
msg:    DB "Loading MOS86.", 0
kernelerror: DB "No kernel was loaded", 0
sizeerror: DB "Out of memory for ramdisk", 0

curseg: DD 0x0040100
curent: DW 0

sysf1:  DB "STAGE2  BIN"
sysf2:  DB "STAGE3  BIN"
sysfs   EQU sysf2-sysf1

bpbBytesPerSector  	    EQU 11 ;DW 0
bpbSectorsPerCluster 	EQU 13 ;DB 0
bpbReservedSectors 	    EQU 14 ;DW 0
bpbNumberOfFATs 	    EQU 16 ;DB 0
bpbRootEntries 	        EQU 17 ;DW 0
bpbTotalSectors 	    EQU 19 ;DW 0
bpbMedia 	            EQU 21 ;DB 0
bpbSectorsPerFAT 	    EQU 22 ;DW 0
bpbSectorsPerTrack 	    EQU 24 ;DW 0
bpbHeadsPerCylinder 	EQU 26 ;DW 0
bpbHiddenSectors 	    EQU 28 ;DD 0
bpbTotalSectorsBig      EQU 32 ;DD 0
bsDriveNumber 	        EQU 36 ;DB 0
bsUnused 	            EQU 37 ;DB 0
bsExtBootSignature 	    EQU 38 ;DB 0
bsSerialNumber	        EQU 39 ;DD 0
bsVolumeLabel 	        EQU 43 ;DB "           "
bsFileSystem 	        EQU 51 ;DB "FAT12   "

%include "inc_info.asm"

gdt:
gdtlimit:   DW gdtend-gdtNULL
gdtbase:    DD 0

gdtNULL:
gdtNULLSel  EQU $-gdtNULL
            DD 0
            DD 0
gdtKData:
gdtKDataSel EQU $-gdtNULL
            DW 0xFFFF
            DW 0
            DB 0
            DB 0x92
            DB 0xCF
            DB 0
gdtKCode:
gdtKCodeSel EQU $-gdtNULL
            DW 0xFFFF
            DW 0
            DB 0
            DB 0x9a
            DB 0xCF
            DB 0
gdtend:
