/*
    Summary: log.c
    *Libc implementation of log*

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */


#include <math.h>

#define expansions 30

#define EULER 2.718281828459045235360287471352662497757247093699959574966967627724076630353547594571382178525166427427466391932003059L

/* Function: log
 * computes and returns the natural logarithm of x (using a taylor expansion)
 */
double log(double x)
{
    if (!(x > 0)) return -INFINITY; // handles NaN and invalid inputs
    
    // compute integer part of answer
    double base = 0;
    //normalize to 0.6-1.7
    while (x < 0.6)
    {
        x *= EULER;
        base--;
    }
    while (x > 1.7)
    {
        x /= EULER;
        base++;
    }
    
    // solve fractional part
    // for taylor series (as per wikipedia)
    //             1      1      1
    // ln(x) = 2y (-y^0 + -y^2 + -y^4 + ...)
    //             1      3      5
    // where 2y = (x-1)(x+1)

    double y = (x-1.0)/(x+1.0);
    double fraction = 0.0;
    double bottomroot = 1.0;
    double power = 1.0;
    int lpc = 0;

    while (lpc < expansions)
    {
        fraction += power / bottomroot;
        power *= y * y;
        bottomroot += 2.0;
        lpc++; 
    }
    fraction *= 2.0 * y;

    return(base + fraction);
}


#ifdef TEST
#include <_PDCLIB_test.h>

#define MARGIN 0.0000001L

static double variance (double v, double d)
{
    if (v > d) return (v - d);
    return (d - v);
}

int main(void)
{
    BEGIN_TESTS;
    TESTCASE( log(1) == 0 );
    TESTCASE( variance( log(EULER) , 1.0f ) < MARGIN );
    TESTCASE( variance( log(10.0), 2.3025850929940456840179914546844L ) < MARGIN );
    return(TEST_RESULTS);
}

#endif
