/*
    Summary: logl.c
    *Libc implementation of logl*

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */


#include <math.h>

#define expansions 35

#define EULER 2.718281828459045235360287471352662497757247093699959574966967627724076630353547594571382178525166427427466391932003059L

/* Function: logl
 * computes and returns the natural logarithm of x (using a taylor expansion)
 */
long double logl(long double x)
{
    if (!(x > 0)) return -INFINITY; // handles NaN and invalid inputs
    
    // compute integer part of answer
    long double base = 0;
    //normalize to 0.6-1.7
    while (x < 0.6)
    {
        x *= EULER;
        base--;
    }
    while (x > 1.7)
    {
        x /= EULER;
        base++;
    }
    
    // solve fractional part
    // for taylor series (as per wikipedia)
    //             1      1      1
    // ln(x) = 2y (-y^0 + -y^2 + -y^4 + ...)
    //             1      3      5
    // where 2y = (x-1)(x+1)

    long double y = (x-1.0)/(x+1.0);
    long double fraction = 0.0;
    long double bottomroot = 1.0;
    long double power = 1.0;
    int lpc = 0;

    while (lpc < expansions)
    {
        fraction += power / bottomroot;
        power *= y * y;
        bottomroot += 2.0;
        lpc++; 
    }
    fraction *= 2.0 * y;

    return(base + fraction);
}


#ifdef TEST
#include <_PDCLIB_test.h>

#define MARGIN 0.000000001L

static double variance (double v, double d)
{
    if (v > d) return (v - d);
    return (d - v);
}

int main(void)
{
    BEGIN_TESTS;
    TESTCASE( logl(1) == 0 );
    TESTCASE( variance( logl(EULER) , 1.0f ) < MARGIN );
    TESTCASE( variance( logl(10.0), 2.3025850929940456840179914546844L ) < MARGIN );
    return(TEST_RESULTS);
}

#endif
