/*
    Summary: buildtree.c
    Builds a tree from an opcode map

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <stdlib.h>
#include <stdio.h>

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>

AST_Node libgfx_nullnode = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 0, 0, 0, NULL, NULL, NULL}; // node representing the value 0

static char nodeerror[128];

/* Function: LibGFX_recursetree
 * recursive helper function for building a tree from an opcode map
 *
 * This function takes the current root, and the list of pregenerated nodes,
 * then fills in the node in question if it hasn't been visited yet, silently
 * return when the node is reachable from two sides, and complain if a cyclic
 * dependency was detected or if the opcode map contains bogus data.
 *
 * upon succesful return, the node requested is converted in a proper acyclic
 * graph, with all properties set accordingly. upon failure, the states are
 * undefined, except that node->children may have got a valid pointer assigned.
 * it is required that node->children = NULL and node->refcount = 0 upon entry
 * because cleanup becomes impossible otherwise. (this function does not do it)
 * also, node->operatorindex must be correctly set for each node in tempmap
 *
 * in:
 *     node - the current operator index
 *     tempmap - a list of pregenerated node pointers for each entry
 *     rawmap - the original raw opcode map
 *     maxnode - the amount of nodes preallocated, used in bounds checking
 *
 * out:
 *     return - 0 on success, 1 on failure.
 *     <libgfx_errormsg> - altered on failure.
 */
static int LibGFX_recursetree(int node, AST_Node ** tempmap, AST_RawOpcodeMap * rawmap, int maxnode)
{
    if (node == -1)
    {
        //special node, ignore
    }
    else if (tempmap[node]->refcount == 0)
    {
        tempmap[node]->childcount = 0;
        tempmap[node]->refcount++;
        switch(rawmap->data[4*node])
        {
            // triadic op-op-op configurations
            case UDI_GFX_OPERATOR_RGB:
            case UDI_GFX_OPERATOR_YUV:
            case UDI_GFX_OPERATOR_YIQ:
            case UDI_GFX_OPERATOR_MAD:
            case UDI_GFX_OPERATOR_FRC:
            case UDI_GFX_OPERATOR_ROR:
            case UDI_GFX_OPERATOR_ROL:
            case UDI_GFX_OPERATOR_RANGE:

                tempmap[node]->children = (AST_Node**) malloc(3 * sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode || rawmap->data[4*node + 2] >= maxnode || rawmap->data[4*node + 3] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->children[1] = tempmap[rawmap->data[4*node + 2]];
                tempmap[node]->children[2] = tempmap[rawmap->data[4*node + 3]];
                tempmap[node]->childcount = 3;
                break;
                
            // triadic op-op-arg configurations
            case UDI_GFX_OPERATOR_ADD:
            case UDI_GFX_OPERATOR_SUB:
            case UDI_GFX_OPERATOR_SHL:
            case UDI_GFX_OPERATOR_SHR:
            case UDI_GFX_OPERATOR_SAL:
            case UDI_GFX_OPERATOR_OR:
            case UDI_GFX_OPERATOR_XOR:
            case UDI_GFX_OPERATOR_BUFFER:

                tempmap[node]->children = (AST_Node**) malloc(2 * sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode || rawmap->data[4*node + 2] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->children[1] = tempmap[rawmap->data[4*node + 2]];
                tempmap[node]->const1 = rawmap->data[4*node + 3];
                tempmap[node]->childcount = 2;
                break;

            // triadic op-arg-arg configurations
            case UDI_GFX_OPERATOR_SEG:
                tempmap[node]->children = (AST_Node**) malloc(sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->const1 = rawmap->data[4*node + 2];
                tempmap[node]->const2 = rawmap->data[4*node + 3];
                tempmap[node]->childcount = 1;
                break;

            // binary op-op configurations
            case UDI_GFX_OPERATOR_ALPHA:
            case UDI_GFX_OPERATOR_MUL:
            case UDI_GFX_OPERATOR_DIV:
                tempmap[node]->children = (AST_Node**) malloc(2 * sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode || rawmap->data[4*node + 2] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->children[1] = tempmap[rawmap->data[4*node + 2]];
                tempmap[node]->childcount = 2;
                break;

            // binary op-arg configurations
            case UDI_GFX_OPERATOR_ATTR:
                tempmap[node]->children = (AST_Node**) malloc(sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->const1 = rawmap->data[4*node + 2];
                tempmap[node]->childcount = 1;
                break;

            // unary op configurations
            case UDI_GFX_OPERATOR_I:
            case UDI_GFX_OPERATOR_NOT:
            case UDI_GFX_OPERATOR_NEG:
                tempmap[node]->children = (AST_Node**) malloc(sizeof(AST_Node *));
                if (!tempmap[node]->children)
                {
                    sprintf(nodeerror, "BuildTree: Out of memory at node %i", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                if (rawmap->data[4*node + 1] >= maxnode)
                {
                    sprintf(nodeerror, "BuildTree: UDI Violation: Opcode reference on node %i out of bounds", node);
                    libgfx_errormsg = nodeerror;
                    return 1;
                }
                tempmap[node]->children[0] = tempmap[rawmap->data[4*node + 1]];
                tempmap[node]->childcount = 1;
                break;

            // unary arg configurations
            case UDI_GFX_OPERATOR_CONST:
                tempmap[node]->const1 = rawmap->data[4*node + 1];
                tempmap[node]->childcount = 0;
                break;

            // special register configurations
            case UDI_GFX_OPERATOR_X:
            case UDI_GFX_OPERATOR_Y:
            case UDI_GFX_OPERATOR_TX:
            case UDI_GFX_OPERATOR_TY:
            case UDI_GFX_OPERATOR_TXOFF:
            case UDI_GFX_OPERATOR_TYOFF:
                tempmap[node]->childcount = 0;
                break;

            default:
                sprintf(nodeerror, "BuildTree: UDI Violation: Unknown opcode at node %i", node);
                libgfx_errormsg = nodeerror;
                return 1;
        }

        tempmap[node]->visited = 1;

        for (int i = 0; i < tempmap[node]->childcount; i++)
        {
            if (tempmap[node]->children[i] == tempmap[0]) // reference to "operator" 0
                tempmap[node]->children[i] = &libgfx_nullnode;

            int retval = LibGFX_recursetree(tempmap[node]->children[i]->operatorindex, tempmap, rawmap, maxnode);
            if (retval) return retval;
        }

        tempmap[node]->visited = 0;
    }
    else
    {
        // already initialized
        if (tempmap[node]->visited)
        {
            sprintf(nodeerror, "BuildTree: UDI Violation: Cyclic structure at node %i", node);
            libgfx_errormsg = nodeerror;
            return 1;
        }
        tempmap[node]->refcount++;
    }
    return 0;
}


AST_Node * LibGFX_BuildTree (AST_RawOpcodeMap * map)
{
    if (!map) return NULL;
    if (!map->length)
    {
        libgfx_errormsg = "BuildTree: Empty opcode map";
        return NULL;
    }


    AST_Node ** inplace_map = (AST_Node **) malloc(map->length * sizeof(AST_Node*));
    if (!inplace_map)
    {
        libgfx_errormsg = "BuildTree: Out of memory";
        return NULL;
    }
    for (int i = 0; i < map->length; i++)
        inplace_map[i] = NULL;
    for (int i = 0; i < map->length; i++)
        inplace_map[i] = (AST_Node*) malloc(sizeof(AST_Node));

    int failed = 0;

    for (int i = 0; i < map->length; i++)
    {   if (inplace_map[i] == NULL)
        {
            failed = 1;
            libgfx_errormsg = "BuildTree: Out of memory";
        }
        else
        {
            inplace_map[i]->children = NULL;
            inplace_map[i]->childcount = 0;
            inplace_map[i]->opcode = map->data[4*i];
            inplace_map[i]->refcount = 0;
            inplace_map[i]->operatorindex = i;
            inplace_map[i]->data = NULL;
            inplace_map[i]->freedata = NULL;
            inplace_map[i]->clonedata = NULL;
        }
    }

    if (!failed) failed = LibGFX_recursetree(0, inplace_map, map, map->length);

    // commit point
    if (failed)
    {
        for (int i = 0; i < map->length; i++)
        {
            if (inplace_map[i])
            {
                if (inplace_map[i]->children) free(inplace_map[i]->children);
                free(inplace_map[i]);
            }
        }
        free(inplace_map);
        return NULL;
    }

    // clean up unused nodes
    for (int i = 0; i < map->length; i++)
    {
        if (inplace_map[i]->refcount == 0)
        {
            if (inplace_map[i]->children) free(inplace_map[i]->children);
            free(inplace_map[i]);
        }
    }
    // clean up construction structure
    AST_Node * root = inplace_map[0];
    free(inplace_map);
    return root;
}

#ifdef TEST
#include <libgfx/test.h>

int main(void)
{
    BEGIN_TESTS;

    TESTCASE(LibGFX_BuildTree(NULL) == NULL);

    return(TEST_RESULTS);
}

#endif
