/*
    Summary: clonetree.c
    Duplicate an entire tree

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <stdlib.h>

AST_Node * LibGFX_CloneTree (AST_Node * root)
{
    if (!root) return NULL;

    AST_Node * newnode = (AST_Node *) malloc(sizeof(AST_Node));
    if (!newnode) return NULL;

    if (root->childcount)
    {
        newnode->children = (AST_Node**) malloc(root->childcount * sizeof(AST_Node *));
        newnode->childcount = root->childcount;

        if (!newnode->children)
        {
            libgfx_errormsg = "CloneTree: Out of memory";
            free(newnode);
            return NULL;
        }

        int i = 0;
        for (i = 0; i < root->childcount; i++)
        {
            newnode->children[i] = LibGFX_CloneTree(root->children[i]);
            if (!newnode->children[i])
            {
                while (i > 0)
                {
                    i--;
                    LibGFX_FreeNode(newnode->children[i]);
                }
                free(newnode->children);
                free(newnode);
                libgfx_errormsg = "CloneTree: Out of memory";
                return NULL;
            }
        }

    }
    else
    {
        newnode->childcount = 0;
        newnode->children = NULL;
    }

    newnode->const1 = root->const1;
    newnode->const2 = root->const2;
    newnode->const3 = root->const3;
    newnode->opcode = root->opcode;
    newnode->data = NULL;
    if (root->data) newnode->data = root->clonedata(root->data);
    newnode->freedata = root->freedata;
    newnode->clonedata = root->clonedata;
    newnode->operatorindex = root->operatorindex;
    newnode->visited = 0;

    newnode->refcount = 1;

    return newnode;
}

#ifdef TEST
#include <libgfx/test.h>
#include <string.h>

int copycalls = 0;
int freecalls = 0;

static void * testcopyfunc(void * in)
{
    copycalls++;
    return (void*)((int)in+1);
}
static void testfreefunc(void * in)
{
    freecalls++;
}

int main(void)
{
    BEGIN_TESTS;

    AST_Node n1 = {NULL, 0, 1, 0, 2, 12, 0, 1, 2, NULL, &testfreefunc, &testcopyfunc};
    AST_Node n2 = {NULL, 0, 1, 2, 3, 13, 4, 5, 6, (void*)1, &testfreefunc, &testcopyfunc};

    AST_Node b1 = n1;
    AST_Node b2 = n2;

    AST_Node * ops[] = {&n1, &n2};
    AST_Node n0 = {&(ops[0]), 2, 1, 0, 3, 10, 3, 4, 5, (void*)3, &testfreefunc, &testcopyfunc};

    AST_Node b0 = n0;

    AST_Node * clone = LibGFX_CloneTree(&n0);
    TESTCASE( clone->childcount == 2 );
    TESTCASE( clone->children != n0.children );
    TESTCASE( clone->children );
    TESTCASE( clone->refcount == 1 );
    TESTCASE( clone->visited == 0 );
    TESTCASE( clone->opcode == 10 );
    TESTCASE( clone->const1 == 3 );
    TESTCASE( clone->const2 == 4 );
    TESTCASE( clone->const3 == 5 );
    TESTCASE( (int)clone->data == 4 );
    TESTCASE( clone->freedata == &testfreefunc );
    TESTCASE( clone->clonedata == &testcopyfunc );

    AST_Node * c1 = clone->children[0];
    TESTCASE( memcmp(c1, &n1, sizeof(AST_Node)) == 0 );

    AST_Node * c2 = clone->children[1];
    TESTCASE( c2->childcount == 0 );
    TESTCASE( c2->children == NULL );
    TESTCASE( c2->refcount == 1 );
    TESTCASE( c2->visited == 0 );
    TESTCASE( c2->opcode == 13 );
    TESTCASE( c2->const1 == 4 );
    TESTCASE( c2->const2 == 5 );
    TESTCASE( c2->const3 == 6 );
    TESTCASE( (int)c2->data == 2 );
    TESTCASE( c2->freedata == &testfreefunc );
    TESTCASE( c2->clonedata == &testcopyfunc );

    TESTCASE( copycalls == 2 );
    TESTCASE( freecalls == 0 );
    
    TESTCASE( memcmp(&n2, &b2, sizeof(AST_Node)) == 0 );
    TESTCASE( memcmp(&n1, &b1, sizeof(AST_Node)) == 0 );
    TESTCASE( memcmp(&n0, &b0, sizeof(AST_Node)) == 0 );


    return(TEST_RESULTS);
}

#endif
