/*
    Summary: fold_bb.c
    Fold bit listings over a tree

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>
#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <stdlib.h>

int LibGFX_BufferBitFold(AST_Node * root, AST_Bit ** bitout, int allowapprox)
{
    if (!root) return -1;

    switch(root->opcode)
    {

        // buffers:

        case UDI_GFX_OPERATOR_BUFFER:
        {
            if (root->const1 <= 0 || root->const1 > 128)
            {
                libgfx_errormsg = "BufferBitFold: broken buffer bit size";
                return -1; // sanitize
            }

            if (!bitout) return root->const1;

            AST_Bit * head = NULL;
            for (int i = root->const1 - 1; i >=0; i--)
            {
                 AST_Bit * add = (AST_Bit *) malloc(sizeof(AST_Bit));
                 if (!add)
                 {
                     LibGFX_FreeBitlist(head);
                     libgfx_errormsg = "BufferBitFold: Out of memory";
                     return -1;
                 }
                 add->next_bit = head;
                 add->source_node = root;
                 root->refcount++;
                 add->currentbit = i;
                 add->sourcebit = i;
                 head = add;
            }

            *bitout = head;
            return root->const1;

        }
        
        case AST_OPCODE_FRAMEBUFFER:
        {
            AST_FramebufferData * data = (AST_FramebufferData *)(root->data);
            if (!data) return -1;

            if (data->bufferbits <= 0 || data->bufferbits > 128)
            {
                libgfx_errormsg = "BufferBitFold: broken buffer bit size";
                return -1; // sanitize
            }

            if (!bitout) return data->bufferbits;

            AST_Bit * head = NULL;
            for (int i = data->bufferbits - 1; i >=0; i--)
            {
                 AST_Bit * add = (AST_Bit *) malloc(sizeof(AST_Bit));
                 if (!add)
                 {
                     LibGFX_FreeBitlist(head);
                     libgfx_errormsg = "BufferBitFold: Out of memory";
                     return -1;
                 }
                 add->next_bit = head;
                 add->source_node = root;
                 root->refcount++;
                 add->currentbit = i;
                 add->sourcebit = i;
                 head = add;
            }

            *bitout = head;
            return root->const1;
        }


        // operations:

        case UDI_GFX_OPERATOR_SEG:
        {
            AST_Bit * head = NULL;
            int bits = LibGFX_BufferBitFold(root->children[0], &head, allowapprox);
            if (bits == -1) return -1;

            AST_Bit * discards = NULL;
            AST_Bit ** nextpt = &head;
            // pointer abracadabra: nextpt points to the next pointer, so we can remove elements without
            // needing a "previous" pointer
            while (*nextpt)
            {
                if (((*nextpt)->currentbit >= root->const1) && ((*nextpt)->currentbit < (root->const1 + root->const2)))
                {
                    // shift bit
                    (*nextpt)->currentbit -= root->const1;
                    nextpt = &((*nextpt)->next_bit);
                }
                else
                {
                    // discard bit
                    AST_Bit * temp = *nextpt;
                    *nextpt = (*nextpt)->next_bit;
                    
                    temp->next_bit = discards;
                    discards = temp;
                    bits--; // less valid bits
                }
            }
            LibGFX_FreeBitlist(discards);
            if (!bitout)
                LibGFX_FreeBitlist(head);
            else
                *bitout = head;

            return bits;
        }

        case UDI_GFX_OPERATOR_FRC:
        {
            int nominator, denominator;
            if (LibGFX_ConstAttrFold(root->children[2], &denominator, NULL) != 1) return -1; // denominator must be constant
            int ds = 0;
            if (LibGFX_ConstAttrFold(root->children[1], &nominator, NULL) == 1)
            {
                ds = 0;
            }
            else if (LibGFX_ConstAttrFold(root->children[0], &nominator, NULL) == 1)
            {
                ds = 1;
            }
            else return -1;

            if (nominator <= 0 || denominator <= 0) return -1; // fail on identity and division by zero

            int shift = 0;
            int recurseapprox = allowapprox;


            if ((nominator & -nominator) == nominator && (denominator & -denominator) == denominator) // powers of 2
            {
                // actually a shift
            }
            else if (recurseapprox)
            {
                nominator++;
                denominator++;
                if ((nominator & -nominator) == nominator && (denominator & -denominator) == denominator) // powers of 2 -1
                {
                    // a mathematically correct rescale
                    recurseapprox = 0;
                }
                else
                {
                    return -1;
                }
            }
            else return -1;

            // determine the amount of shift
            while (nominator > denominator)
            {
               shift--;
               nominator >>= 1;
            }
            while (nominator < denominator)
            {
               shift++;
               denominator >>= 1;
            }

            AST_Bit * head = NULL;
            int bits = LibGFX_BufferBitFold(root->children[ds], &head, recurseapprox);
            if (bits == -1) return -1;

            AST_Bit * discards = NULL;
            AST_Bit ** nextpt = &head;
            // pointer abracadabra: nextpt points to the next pointer, so we can remove elements without
            // needing a "previous" pointer
            while (*nextpt)
            {
                if ((*nextpt)->currentbit >= shift)
                {
                    // shift bit
                    (*nextpt)->currentbit -= shift;
                    nextpt = &((*nextpt)->next_bit);
                }
                else
                {
                    // discard bit
                    AST_Bit * temp = *nextpt;
                    *nextpt = (*nextpt)->next_bit;

                    temp->next_bit = discards;
                    discards = temp;
                    bits--; // less valid bits
                }
            }
            LibGFX_FreeBitlist(discards);
            if (!bitout)
                LibGFX_FreeBitlist(head);
            else
                *bitout = head;

            return bits;

        }

        default:
            return -1;
    }
}

#ifdef TEST
#include <libgfx/test.h>

int main(void)
{
    BEGIN_TESTS;

    // default cases

    TESTCASE( LibGFX_BufferBitFold(NULL, NULL, 0) == -1 );

    AST_Node const_node_3 = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 3, 0, 0, NULL, NULL, NULL}; // constant value 3
    TESTCASE( LibGFX_BufferBitFold(&const_node_3, NULL, 0) == -1 );

    // test BUFFER operator

    AST_Node buf_node = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_BUFFER, 3, 0, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    AST_Bit * bit = NULL;
    TESTCASE( LibGFX_BufferBitFold(&buf_node, &bit, 0) == 3 );
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 0 );
    TESTCASE( bit->currentbit == 0 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 1 );
    TESTCASE( bit->next_bit->currentbit == 1 );
    TESTCASE( bit->next_bit->next_bit );
    TESTCASE( bit->next_bit->next_bit->next_bit == NULL );
    TESTCASE( buf_node.refcount == 4 );

    LibGFX_FreeBitlist(bit);
    buf_node.refcount = 1; // in case of screwups
    
    // test AST_OPCODE_FRAMEBUFFER
    
    AST_FramebufferData fb_data;
    fb_data.bufferbits = 3;
    AST_Node fb_node = {NULL, 0, 1, 0, -1, AST_OPCODE_FRAMEBUFFER, 0, 0, 0, &fb_data, NULL, NULL}; // buffer containing 3 bits
    bit = NULL;
    TESTCASE( LibGFX_BufferBitFold(&buf_node, &bit, 0) == 3 );
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 0 );
    TESTCASE( bit->currentbit == 0 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 1 );
    TESTCASE( bit->next_bit->currentbit == 1 );
    TESTCASE( bit->next_bit->next_bit );
    TESTCASE( bit->next_bit->next_bit->next_bit == NULL );
    TESTCASE( buf_node.refcount == 4 );

    LibGFX_FreeBitlist(bit);
    buf_node.refcount = 1; // in case of screwups

    // test SEG operator

    AST_Node * ops_0to2[] = {&buf_node};
    AST_Node seg_node = {&(ops_0to2[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_SEG, 1, 2, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    TESTCASE( LibGFX_BufferBitFold(&seg_node, &bit, 0) == 2 );
    TESTCASE( buf_node.refcount == 3 ); // test + bit1 + bit2
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 1 );
    TESTCASE( bit->currentbit == 0 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 2 );
    TESTCASE( bit->next_bit->currentbit == 1 );
    TESTCASE( bit->next_bit->next_bit == NULL );

    buf_node.refcount = 1; // in case of screwups

    // test FRC operator

    AST_Node const_node_4 = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 4, 0, 0, NULL, NULL, NULL}; // constant value 3
    AST_Node const_node_7 = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 7, 0, 0, NULL, NULL, NULL}; // constant value 3
    AST_Node const_node_8 = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 8, 0, 0, NULL, NULL, NULL}; // constant value 3

    bit = NULL;
    AST_Node * ops_bad_frac[] = {&buf_node, &const_node_8, &const_node_3};
    AST_Node frc_8_3 = {&(ops_bad_frac[0]), 3, 1, 0, -1, UDI_GFX_OPERATOR_FRC, 0, 0, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    TESTCASE( LibGFX_BufferBitFold(&frc_8_3, &bit, 0) == -1 );

    bit = NULL;
    AST_Node * ops_2m1_frac[] = {&buf_node, &const_node_7, &const_node_3};
    AST_Node frc_7_3 = {&(ops_2m1_frac[0]), 3, 1, 0, -1, UDI_GFX_OPERATOR_FRC, 0, 0, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    TESTCASE( LibGFX_BufferBitFold(&frc_7_3, &bit, 0) == -1 );
    TESTCASE( LibGFX_BufferBitFold(&frc_7_3, &bit, 1) == 3 );
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 0 );
    TESTCASE( bit->currentbit == 1 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 1 );
    TESTCASE( bit->next_bit->currentbit == 2 );
    TESTCASE( bit->next_bit->next_bit );
    TESTCASE( bit->next_bit->next_bit->next_bit == NULL );
    TESTCASE( buf_node.refcount == 4 );
    LibGFX_FreeBitlist(bit);

    AST_Node * ops_2m0_frac[] = {&buf_node, &const_node_8, &const_node_4};
    AST_Node frc_8_4 = {&(ops_2m0_frac[0]), 3, 1, 0, -1, UDI_GFX_OPERATOR_FRC, 0, 0, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    TESTCASE( LibGFX_BufferBitFold(&frc_8_4, &bit, 0) == 3 );
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 0 );
    TESTCASE( bit->currentbit == 1 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 1 );
    TESTCASE( bit->next_bit->currentbit == 2 );
    TESTCASE( bit->next_bit->next_bit );
    TESTCASE( bit->next_bit->next_bit->next_bit == NULL );
    TESTCASE( buf_node.refcount == 4 );
    LibGFX_FreeBitlist(bit);
    
    AST_Node * ops_sr1_frac[] = {&const_node_4, &buf_node, &const_node_8};
    AST_Node frc_4_8 = {&(ops_sr1_frac[0]), 3, 1, 0, -1, UDI_GFX_OPERATOR_FRC, 0, 0, 0, NULL, NULL, NULL}; // buffer containing 3 bits
    TESTCASE( LibGFX_BufferBitFold(&frc_4_8, &bit, 0) == 2 );
    TESTCASE( buf_node.refcount == 3 ); // test + bit1 + bit2
    TESTCASE( bit != NULL );
    TESTCASE( bit->source_node == &buf_node );
    TESTCASE( bit->sourcebit == 1 );
    TESTCASE( bit->currentbit == 0 );
    TESTCASE( bit->next_bit );
    TESTCASE( bit->next_bit->source_node == &buf_node );
    TESTCASE( bit->next_bit->sourcebit == 2 );
    TESTCASE( bit->next_bit->currentbit == 1 );
    TESTCASE( bit->next_bit->next_bit == NULL );


    buf_node.refcount = 1; // in case of screwups

    return(TEST_RESULTS);
}

#endif
