/*
    Summary: fold_ca.c
    Retrieves constant and attribute data from a tree

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#include <libgfx/ast.h>
#include <stdlib.h>

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>

/* Function: LibGFX_ConstAttrFold
   walks the tree and retrieves the currently applicable
   value/attribute to a simple extent

   in:
       root      - the node to start checking
       const_val - a pointer to an integer to receive the constant value
       attr_val  - a pointer to an integer to receive the attribute index

   out:
       return - 0 if there is no match, 1 for a constant match, 2 for an attribute match
       
   All arguments may be NULL without impeding functionality.
   const_val/attr_val will only be written if non-null and the corresponding match was
   found.
 */

int LibGFX_ConstAttrFold (AST_Node * root, int * const_val, int * attr_val)
{
    if (!root) return 0;

    switch (root->opcode)
    {
        case UDI_GFX_OPERATOR_CONST:
            if (const_val) *const_val = root->const1;
            return 1;

        case UDI_GFX_OPERATOR_ATTR:
        {
            int ret;
            if (LibGFX_ConstAttrFold(root->children[0], &ret, NULL) != 1) return 0;
            if (attr_val) *attr_val = ret + root->const1;
            return 2;
        }

        default:
            return 0;
    }
}

#ifdef TEST
#include <libgfx/test.h>

int main(void)
{
    BEGIN_TESTS;

    int retc = -1, reta = -1;

    // const operator
    AST_Node const_node = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 6, 0, 0, NULL, NULL, NULL};     // node representing value 6
    TESTCASE( LibGFX_ConstAttrFold(&const_node, &retc, &reta) == 1 );
    TESTCASE( retc == 6 );
    TESTCASE( reta == -1 );
    TESTCASE( LibGFX_ConstAttrFold(&const_node, NULL, NULL) == 1 );

    // attr operator
    retc = -1; reta = -1;
    AST_Node * ops_0[] = {&const_node};
    AST_Node attr1_node = {&(ops_0[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_ATTR, 0, 0, 0, NULL, NULL, NULL};     // node representing attribute 6
    TESTCASE( LibGFX_ConstAttrFold(&attr1_node, &retc, &reta) == 2 );
    TESTCASE( retc == -1 );
    TESTCASE( reta == 6 );
    retc = -1; reta = -1;
    AST_Node attr2_node = {&(ops_0[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_ATTR, 2, 0, 0, NULL, NULL, NULL};     // node representing attribute 8
    TESTCASE( LibGFX_ConstAttrFold(&attr2_node, &retc, &reta) == 2 );
    TESTCASE( retc == -1 );
    TESTCASE( reta == 8 );
    TESTCASE( LibGFX_ConstAttrFold(&attr2_node, NULL, NULL) == 2 );

    // robustness
    retc = -1; reta = -1;
    TESTCASE( LibGFX_ConstAttrFold(NULL, &retc, &reta) == 0 );
    TESTCASE( retc == -1 );
    TESTCASE( reta == -1 );

    return(TEST_RESULTS);
}

#endif
