/*
    Summary: printnode.c
    Recursively prints the a node to a C string

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#include <libgfx/ast.h>
#include <string.h>
#include <stdio.h>

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>

/* Function: strnput
 * safe write returning written length as neither strncpy nor strncat can do this efficiently
 * A null terminator will always be placed.
 *
 * in:
 *     out - pointer to the char array to write
 *     in - pointer to the null-terminated input string
 *     max - maximum number of characters to copy.
 *
 * out:
 *     return - number of characters copied
 */
static int strnput(char * out, const char * in, size_t max)
{
    char * backup = out;
    while (max && *in)
    {
        *out++ = *in++;
        max--;
    }
    *out = '\0';
    return out - backup;
}

int LibGFX_PrintNode(AST_Node * root, char * base, int maxbytes)
{
    if (!base) return 0;
    if (maxbytes <= 0) return 0;
    base[0] = '\0';
    base[maxbytes-1] = '\0';

    if (!root)
    {
        return strnput(base, "NULL", maxbytes-1);
    }

    int offset = 0;
    switch(root->opcode)
    {
        case UDI_GFX_OPERATOR_CONST:
            offset += snprintf(base + offset, maxbytes-1-offset, "%i", root->const1);
            break;

        case UDI_GFX_OPERATOR_X:
            offset += snprintf(base + offset, maxbytes-1-offset, "x");
            break;
        case UDI_GFX_OPERATOR_Y:
            offset += snprintf(base + offset, maxbytes-1-offset, "y");
            break;
        case UDI_GFX_OPERATOR_TX:
            offset += snprintf(base + offset, maxbytes-1-offset, "tile.x");
            break;
        case UDI_GFX_OPERATOR_TY:
            offset += snprintf(base + offset, maxbytes-1-offset, "tile.y");
            break;
        case UDI_GFX_OPERATOR_TXOFF:
            offset += snprintf(base + offset, maxbytes-1-offset, "tile.xoff");
            break;
        case UDI_GFX_OPERATOR_TYOFF:
            offset += snprintf(base + offset, maxbytes-1-offset, "tile.yoff");
            break;

        case UDI_GFX_OPERATOR_ATTR:
            offset += snprintf(base + offset, maxbytes-1-offset, "attr[");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            if (root->const1)
                offset += snprintf(base + offset, maxbytes-1-offset, " + %i", root->const1);
            offset += snprintf(base + offset, maxbytes-1-offset, "]");
            break;

        case UDI_GFX_OPERATOR_ADD:
            offset += snprintf(base + offset, maxbytes-1-offset, "(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, " + ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            if (root->const1)
                offset += snprintf(base + offset, maxbytes-1-offset, " + %i", root->const1);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;

        case UDI_GFX_OPERATOR_MUL:
            offset += snprintf(base + offset, maxbytes-1-offset, "(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, " * ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;
            
        case UDI_GFX_OPERATOR_MAD:
            offset += snprintf(base + offset, maxbytes-1-offset, "(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, " * ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, " + ");
            offset += LibGFX_PrintNode(root->children[2], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;
            
        case UDI_GFX_OPERATOR_FRC:
            offset += snprintf(base + offset, maxbytes-1-offset, "(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, " * ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, "/");
            offset += LibGFX_PrintNode(root->children[2], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;

        case UDI_GFX_OPERATOR_BUFFER:
            offset += snprintf(base + offset, maxbytes-1-offset, "buffer_%i[", root->const1);
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, "][");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, "]");
            break;

        case UDI_GFX_OPERATOR_RGB:
            offset += snprintf(base + offset, maxbytes-1-offset, "RGB(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ", ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ", ");
            offset += LibGFX_PrintNode(root->children[2], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;
            
        case UDI_GFX_OPERATOR_YUV:
            offset += snprintf(base + offset, maxbytes-1-offset, "YUV(");
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ", ");
            offset += LibGFX_PrintNode(root->children[1], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ", ");
            offset += LibGFX_PrintNode(root->children[2], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ")");
            break;

        case UDI_GFX_OPERATOR_SEG:
            offset += LibGFX_PrintNode(root->children[0], base + offset, maxbytes-offset);
            offset += snprintf(base + offset, maxbytes-1-offset, ":[%i..%i]", root->const1, root->const1 + root->const2 - 1);
            break;


        case AST_OPCODE_ADDRGEN:
            offset += snprintf(base + offset, maxbytes-1-offset, "xy-offset");
            break;

        default:
            offset += strnput(base+offset, "unknown", maxbytes-offset-1);
            break;
    }

    if (offset == maxbytes - 1 && maxbytes > 3)
    {
        strnput(&base[maxbytes-4], "...", 3);
    }
    return offset;
}

#ifdef TEST
#include <libgfx/test.h>
#include <string.h>


int main(void)
{
    BEGIN_TESTS;

    char buffer[256];

    strcpy(buffer, "keepme");
    TESTCASE( LibGFX_PrintNode(NULL, buffer, 0) == 0 );
    TESTCASE( strcmp(buffer, "keepme") == 0 );

    TESTCASE( LibGFX_PrintNode(NULL, buffer, 256) == 4 );
    TESTCASE( strcmp(buffer, "NULL") == 0 );

    AST_Node const_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_CONST, 5, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&const_node, buffer, 256) == 1 );
    TESTCASE( strcmp(buffer, "5") == 0 );

    AST_Node x_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_X, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&x_node, buffer, 256) == 1 );
    TESTCASE( strcmp(buffer, "x") == 0 );

    AST_Node y_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_Y, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&y_node, buffer, 256) == 1 );
    TESTCASE( strcmp(buffer, "y") == 0 );

    AST_Node tx_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_TX, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&tx_node, buffer, 256) == 6 );
    TESTCASE( strcmp(buffer, "tile.x") == 0 );

    AST_Node ty_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_TY, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&ty_node, buffer, 256) == 6 );
    TESTCASE( strcmp(buffer, "tile.y") == 0 );

    AST_Node txoff_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_TXOFF, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&txoff_node, buffer, 256) == 9 );
    TESTCASE( strcmp(buffer, "tile.xoff") == 0 );

    AST_Node tyoff_node = {NULL, 0, 1, 0, 1, UDI_GFX_OPERATOR_TYOFF, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&tyoff_node, buffer, 256) == 9 );
    TESTCASE( strcmp(buffer, "tile.yoff") == 0 );


    AST_Node * unary_ops[] = {&const_node};
    AST_Node attr_node = {&(unary_ops[0]), 1, 1, 0, 1, UDI_GFX_OPERATOR_ATTR, 3, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&attr_node, buffer, 256) == 11 );
    TESTCASE( strcmp(buffer, "attr[5 + 3]") == 0 );

    AST_Node seg_node = {&(unary_ops[0]), 1, 1, 0, 1, UDI_GFX_OPERATOR_SEG, 6, 2, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&seg_node, buffer, 256) == 8 );
    TESTCASE( strcmp(buffer, "5:[6..7]") == 0 );


    AST_Node * binary_ops[] = {&const_node, &tx_node};
    AST_Node add_node = {&(binary_ops[0]), 2, 1, 0, 1, UDI_GFX_OPERATOR_ADD, 3, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&add_node, buffer, 256) == 16 );
    TESTCASE( strcmp(buffer, "(5 + tile.x + 3)") == 0 );

    AST_Node mul_node = {&(binary_ops[0]), 2, 1, 0, 1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&mul_node, buffer, 256) == 12 );
    TESTCASE( strcmp(buffer, "(5 * tile.x)") == 0 );

    AST_Node buffer_node = {&(binary_ops[0]), 2, 1, 0, 1, UDI_GFX_OPERATOR_BUFFER, 16, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&buffer_node, buffer, 256) == 20 );
    TESTCASE( strcmp(buffer, "buffer_16[5][tile.x]") == 0 );


    AST_Node * triadic_ops[] = {&tx_node, &ty_node, &const_node};
    AST_Node rgb_node = {&(triadic_ops[0]), 3, 1, 0, 1, UDI_GFX_OPERATOR_RGB, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&rgb_node, buffer, 256) == 22 );
    TESTCASE( strcmp(buffer, "RGB(tile.x, tile.y, 5)") == 0 );

    AST_Node yuv_node = {&(triadic_ops[0]), 3, 1, 0, 1, UDI_GFX_OPERATOR_YUV, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&yuv_node, buffer, 256) == 22 );
    TESTCASE( strcmp(buffer, "YUV(tile.x, tile.y, 5)") == 0 );

    AST_Node mad_node = {&(triadic_ops[0]), 3, 1, 0, 1, UDI_GFX_OPERATOR_MAD, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&mad_node, buffer, 256) == 21 );
    TESTCASE( strcmp(buffer, "(tile.x * tile.y + 5)") == 0 );

    AST_Node frc_node = {&(triadic_ops[0]), 3, 1, 0, 1, UDI_GFX_OPERATOR_FRC, 0, 0, 0, NULL, NULL, NULL};
    TESTCASE( LibGFX_PrintNode(&frc_node, buffer, 256) == 19 );
    TESTCASE( strcmp(buffer, "(tile.x * tile.y/5)") == 0 );


    return(TEST_RESULTS);
}

#endif
