/*
    Summary: retrievemap.c
    Reads a driver's opcode map

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#include <libgfx/ast.h>
#include <libgfx/error.h>

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>
#include <GL/mgl.h>
#include <stdlib.h>

AST_RawOpcodeMap * LibGFX_RetrieveMap (uint32_t engine)
{
    int sequences = mglGetEngineRange(engine, UDI_GFX_PROP_OPERATOR_INDEX);
    if (sequences == 0)
    {
        libgfx_errormsg = "Nonconformant UDI implementation - operator index missing";
        return NULL;
    }

    int max = 0;
    for (int i = 0; i < sequences; i++)
    {
        int tmpmin, tmpmax, tmpdelta;
        mglGetLastRangeEntry(i, &tmpmin, &tmpmax, &tmpdelta);
        if (tmpdelta <= 0)
        {
            libgfx_errormsg = "RetrieveMap: Nonconformant UDI implementation - null range";
            return NULL;
        }
        if (tmpmin < 0)
        {
            libgfx_errormsg = "RetrieveMap: Nonconformant UDI implementation - negative range";
            return NULL;
        }
        if (tmpmax > 255)
        {
            libgfx_errormsg = "RetrieveMap: Too many opcodes (sanity check failure)";
            return NULL;
        }

        if (max <= tmpmax) max = tmpmax + 1;
    }

    AST_RawOpcodeMap * map = (AST_RawOpcodeMap *) malloc(sizeof(AST_RawOpcodeMap));
    if (!map)
    {
        libgfx_errormsg = "RetrieveMap: Out of memory";
        return NULL;
    }

    map->length = max;
    map->data = (uint32_t*) malloc(max * 4 * sizeof(uint32_t));
    if (!map->data)
    {
        free(map);
        libgfx_errormsg = "RetrieveMap: Out of memory";
        return NULL;
    }
    
    for (int i = 0; i < max; i++)
    {
        mglSetEngineState(engine, UDI_GFX_PROP_OPERATOR_INDEX, i);
        if (mglGetEngineState(engine, UDI_GFX_PROP_OPERATOR_INDEX) == i)
        {
            map->data[4*i + 0] = mglGetEngineState(engine, UDI_GFX_PROP_OPERATOR_OPCODE);
            map->data[4*i + 1] = mglGetEngineState(engine, UDI_GFX_PROP_OPERATOR_ARG_1);
            map->data[4*i + 2] = mglGetEngineState(engine, UDI_GFX_PROP_OPERATOR_ARG_2);
            map->data[4*i + 3] = mglGetEngineState(engine, UDI_GFX_PROP_OPERATOR_ARG_3);
        }
        else
        {
            // invalid values
            map->data[4*i + 0] = 0xffffffff;
            map->data[4*i + 1] = 0xffffffff;
            map->data[4*i + 2] = 0xffffffff;
            map->data[4*i + 3] = 0xffffffff;
        }
    }

    return map;
}

#ifdef TEST
#include <libgfx/test.h>
#include <string.h>

static const int gfx_opcode_b8g8r8x8[40] =
    {UDI_GFX_OPERATOR_RGB,    1,  2,  3,
     UDI_GFX_OPERATOR_SEG,    4, 16,  8,
     UDI_GFX_OPERATOR_SEG,    4,  8,  8,
     UDI_GFX_OPERATOR_SEG,    4,  0,  8,
     UDI_GFX_OPERATOR_BUFFER, 5,  6, 32,
     UDI_GFX_OPERATOR_ATTR,   0, UDI_GFX_PROP_CUSTOM + 0, 0,
     UDI_GFX_OPERATOR_MAD,    7,  8,  9,
     UDI_GFX_OPERATOR_ATTR,   0, UDI_GFX_PROP_SOURCE_WIDTH, 0,
     UDI_GFX_OPERATOR_Y,      0,  0,  0,
     UDI_GFX_OPERATOR_X,      0,  0,  0};

int opcodeindex = 0;
int forbiddencalls = 0;

int mglGetEngineState(int engine, unsigned int property)
{
    if (engine != 1) forbiddencalls++;

    switch(property)
    {
        case UDI_GFX_PROP_OPERATOR_INDEX:
            return opcodeindex;
        case UDI_GFX_PROP_OPERATOR_OPCODE:
            return gfx_opcode_b8g8r8x8[4*opcodeindex + 0];
        case UDI_GFX_PROP_OPERATOR_ARG_1:
            return gfx_opcode_b8g8r8x8[4*opcodeindex + 1];
        case UDI_GFX_PROP_OPERATOR_ARG_2:
            return gfx_opcode_b8g8r8x8[4*opcodeindex + 2];
        case UDI_GFX_PROP_OPERATOR_ARG_3:
            return gfx_opcode_b8g8r8x8[4*opcodeindex + 3];

        default:
            forbiddencalls++;
            return 0;
    }
}

void mglSetEngineState(int engine, unsigned int property, int value)
{
    if (engine != 1) forbiddencalls++;
    if (property != UDI_GFX_PROP_OPERATOR_INDEX || value < 0 || value > 9)
    {
        forbiddencalls++;
        return;
    }

    opcodeindex = value;
}

int mglGetEngineRange(int engine, unsigned int property)
{
    if (engine != 1 || property != UDI_GFX_PROP_OPERATOR_INDEX) forbiddencalls++;
    return 1;
}

void mglGetLastRangeEntry(int index, int * start, int * end, int * modulus)
{
    if (index != 0) forbiddencalls++;
    *start = 0;
    *end = 9;
    *modulus = 1;
}

int main(void)
{
    BEGIN_TESTS;

    AST_RawOpcodeMap * map = LibGFX_RetrieveMap(1);
    TESTCASE( forbiddencalls == 0 );
    TESTCASE( map );
    TESTCASE( map->data );
    TESTCASE( map->length == 10 );
    TESTCASE( memcmp(map->data, &(gfx_opcode_b8g8r8x8[0]), 10 * 4 * sizeof(uint32_t)) == 0 );

    return(TEST_RESULTS);
}

#endif
