/*
    Summary: substoffset.c
    Locates and substitutes framebuffer address generation patterns

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>
#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <stdlib.h>

AST_Node * LibGFX_TreeSubstOffset(AST_Node * root)
{
    int discard1, discard2, discard3, discard4, discard5, discard6, discard7, discard8;
    return LibGFX_TreeSubstOffsetPrime(root, &discard1, &discard2, &discard3, &discard4, &discard5, &discard6, &discard7, &discard8);
}

#define STATE_BASE_X 1
#define STATE_BASE_TX 2
#define STATE_BASE_Y 3
#define STATE_BASE_TY 4
#define STATE_BASE_ZERO 5

static void sortcase(int * a1, int * a2, int * a3, int * a4, int * a5, int * a6, int * a7, int * a8, int * b1, int * b2, int * b3, int * b4, int * b5, int * b6, int * b7, int * b8)
{
    if (*a1 > *b1)
    {
        int temp;
        temp = *a1; *a1 = *b1; *b1 = temp;
        temp = *a2; *a2 = *b2; *b2 = temp;
        temp = *a3; *a3 = *b3; *b3 = temp;
        temp = *a4; *a4 = *b4; *b4 = temp;
        temp = *a5; *a5 = *b5; *b5 = temp;
        temp = *a6; *a6 = *b6; *b6 = temp;
        temp = *a7; *a7 = *b7; *b7 = temp;
        temp = *a8; *a8 = *b8; *b8 = temp;
    }
}

// returns equation as (<state> * statemul) + (<state> * <attr1> * stateattrmul) + (<attr1> * attrmul) + (<attr2> * attrmul2) + add
AST_Node * LibGFX_TreeSubstOffsetPrime(AST_Node * root, int * stateindex, int * attrindex1, int * attrindex2, int * stateattrmul, int * statemul, int * attrmul1, int * attrmul2, int * add)
{
    *stateindex = 0;
    *attrindex1 = -1;
    *attrindex2 = -1;
    *attrmul1 = 0;
    *attrmul2 = 0;
    *statemul = 0;
    *stateattrmul = 0;
    *add = 0;
    
    if (!root) return root;

    int si1, ai1, eai1, sam1, sm1, am1, eam1, a1;
    int si2, ai2, eai2, sam2, sm2, am2, eam2, a2;
    AST_Node *root1, *root2;

    switch (root->opcode)
    {
        case UDI_GFX_OPERATOR_X:
            *stateindex = STATE_BASE_X;
            *statemul = 1;
            return root;
        case UDI_GFX_OPERATOR_Y:
            *stateindex = STATE_BASE_Y;
            *statemul = 1;            
            return root;
        case UDI_GFX_OPERATOR_TX:
            *stateindex = STATE_BASE_TX;
            *statemul = 1;
            return root;
        case UDI_GFX_OPERATOR_TY:
            *stateindex = STATE_BASE_TY;
            *statemul = 1;
            return root;
        case UDI_GFX_OPERATOR_ATTR:
        {
            root1 = LibGFX_TreeSubstOffsetPrime(root->children[0], &si1, &ai1, &eai1, &sam1, &sm1, &am1, &eam1, &a1);

            if (root1 != root->children[0])
            {
                //LibGFX_FreeNode(root->children[0]);
                root->children[0] = root1;
                return root;
            }

            if (si1 == 0) return root;
            if ((ai1 != -1) || (eai1 != -1)) return root;

            *stateindex = STATE_BASE_ZERO;
            *attrindex2 = root->const1 + a1;
            *attrmul2 = 1;
            return root;
        }
        case UDI_GFX_OPERATOR_CONST:
            *stateindex = STATE_BASE_ZERO;
            *add = root->const1;
            return root;

        case UDI_GFX_OPERATOR_ADD:
        {
            root1 = LibGFX_TreeSubstOffsetPrime(root->children[0], &si1, &ai1, &eai1, &sam1, &sm1, &am1, &eam1, &a1);
            root2 = LibGFX_TreeSubstOffsetPrime(root->children[1], &si2, &ai2, &eai2, &sam2, &sm2, &am2, &eam2, &a2);
            if (root1 != root->children[0] || root2 != root->children[1])
            {
                if (root1 != root->children[0]) LibGFX_FreeNode(root->children[0]);
                if (root2 != root->children[1]) LibGFX_FreeNode(root->children[1]);
                root->children[0] = root1;
                root->children[1] = root2;
                return root; // substitutions downstream, ignore
            }
            if (si1 == 0 || si2 == 0) return root;
            sortcase(&si1, &ai1, &eai1, &sam1, &sm1, &am1, &eam1, &a1, &si2, &ai2, &eai2, &sam2, &sm2, &am2, &eam2, &a2);
            if ((si1 == STATE_BASE_X || si1 == STATE_BASE_TX) && (si2 == STATE_BASE_Y || si2 == STATE_BASE_TY))
            {
                // merge cases

                if (eai1 != eai2 && eai1 != -1 && eai2 != 1) return root; // too many independent variables
                AST_Node * newnode = (AST_Node *)malloc(sizeof(AST_Node));
                if (!newnode)
                {
                    libgfx_errormsg = "TreeSubstOffset: out of memory creating node";
                    return root;
                }
                AST_LinearOffsetData * newdata = (AST_LinearOffsetData*) malloc(sizeof(AST_LinearOffsetData));
                if (!newdata)
                {
                    free(newnode);
                    libgfx_errormsg = "TreeSubstOffset: out of memory creating node";
                    return root;
                }
                newnode->childcount = 0;
                newnode->children = NULL;
                newnode->opcode = AST_OPCODE_ADDRGEN;
                newnode->refcount = 1;
                newnode->visited = 0;

                newnode->data = (void*)newdata;
                newnode->freedata = &free;
                newnode->clonedata = &LibGFX_CopyLinearOffsetData;

                // offset part
                newdata->buffersize = 0;
                newdata->offsetattr = eai1;
                if (eai2 != -1) newdata->offsetattr = eai2;
                newdata->offsetscale = eam2 + eam1;
                newdata->baseoffset = a1 + a2;

                // X and Y part
                newdata->xarg = si1 - STATE_BASE_X;
                newdata->xscaleattr = ai1;
                newdata->xmulargscale = sam1;
                newdata->xmularg = sm1;
                newdata->xmulscale = am1;

                newdata->yarg = STATE_BASE_Y - si2;
                newdata->yscaleattr = ai2;
                newdata->ymulargscale = sam2;
                newdata->ymularg = sm2;
                newdata->ymulscale = am2;

                return newnode;
            }
            else if (si2 == STATE_BASE_ZERO)
            {
                *stateindex = si1;
                *stateattrmul = sam1;
                *statemul = sm1;
                *attrmul1 = am1;
                if ((eai1 != -1) && (eai2 != -1))
                {
                    return root;
                }
                else if (eai2 != -1)
                {
                    *attrindex2 = eai2;
                    *attrmul2 = eam2;
                }
                else if (eai1 != -1)
                {
                    *attrindex2 = eai1;
                    *attrmul2 = eam1;
                }

                *add = a1 + a2 + root->const1;

                return root;
            }
            else
                return root;
        }

        case UDI_GFX_OPERATOR_MUL:
        {
            root1 = LibGFX_TreeSubstOffsetPrime(root->children[0], &si1, &ai1, &eai1, &sam1, &sm1, &am1, &eam1, &a1);
            root2 = LibGFX_TreeSubstOffsetPrime(root->children[1], &si2, &ai2, &eai2, &sam2, &sm2, &am2, &eam2, &a2);
            if (root1 != root->children[0] || root2 != root->children[1])
            {
                if (root1 != root->children[0]) LibGFX_FreeNode(root->children[0]);
                if (root2 != root->children[1]) LibGFX_FreeNode(root->children[1]);
                root->children[0] = root1;
                root->children[1] = root2;
                return root; // substitutions downstream, ignore
            }
            if (si1 == 0 || si2 == 0) return root;
            sortcase(&si1, &ai1, &eai1, &sam1, &sm1, &am1, &eam1, &a1, &si2, &ai2, &eai2, &sam2, &sm2, &am2, &eam2, &a2);
            if (si2 != STATE_BASE_ZERO)
                return root; // can't multiply coordinate with coordinate
            else if (si1 == STATE_BASE_ZERO)
            {
                if (eai1 != -1 && eai2 != -1)
                {
                    return root;
                }
                else if (eai1 != -1)
                {
                    *attrindex2 = eai1;
                    *attrmul2 = eam1 * a2;
                }
                else if (eai2 != -1)
                {
                    *attrindex2 = eai2;
                    *attrmul2 = eam2 * a1;
                }
                *stateindex = STATE_BASE_ZERO;
                *add = a1 * a2;
                return root;
            }
            else
            {
                if (eai1 != -1 && eai2 != -1)
                {
                    return root; // don't allow attr * attr
                }
                else if (ai1 != -1 && eai2 != -1)
                {
                    return root; // don't allow coordinate * attr1 * attr2
                }
                else if (eai2 != -1) // (ax + b) * (dn + e) = acvx + bcv + axd + bd
                {
                    *attrindex1 = eai2;
                    *stateattrmul = sm1 * eam2;
                    *attrmul1 = a1 * eam2;
                }
                else if (ai1 != -1 || eai1 != -1) // (avx + bx + cv + d) * e
                {
                    *attrindex1 = ai2;
                    *stateattrmul = sm1 * a2;
                    *attrmul1 = am1 * a2;
                    *attrindex2 = eai1;
                    *attrmul2 = eam1 * a2;
                }

                *stateindex = si1;
                *statemul = sm1 * a2;
                *add = a1 * a2;
                return root;
            }

        }

        default:
        {
            for (int i = 0; i < root->childcount; i++)
            {
                int discard1, discard2, discard3, discard4, discard5, discard6, discard7, discard8;
                AST_Node * newroot = LibGFX_TreeSubstOffsetPrime(root->children[i], &discard1, &discard2, &discard3, &discard4, &discard5, &discard6, &discard7, &discard8);
                if (newroot != root->children[i])
                {
                    LibGFX_FreeNode(root->children[i]);
                    root->children[i] = newroot;
                }
            }
            return root;
        }
    }
    return root;
}

#ifdef TEST
#include <libgfx/test.h>

static AST_Node * lastret;

static int Test_TreeSubstOffsetPrime(AST_Node * root, int ret, int si, int ai, int eai, int sam, int sm, int am, int eam, int add)
{
    int xsi = 666, xai = 666, xeai = 666, xsam = 666, xsm = 666, xam = 666, xeam = 666, xadd = 666;
    AST_Node * tmpret;
    lastret = LibGFX_TreeSubstOffsetPrime(root, &xsi, &xai, &xeai, &xsam, &xsm, &xam, &xeam, &xadd);
    int xret = (lastret == root) ? 0 : 1;
    if ((xret == ret) && (xsi == si) && (xai == ai) && (xeai == eai) && (xsam == sam) && (xsm == sm) && (xam == am) && (xeam == eam) && (xadd == add)) return 1;

    // useful debug information
    printf("Mismatch: {%i: %i %i %i: [%i %i %i] + %i + %i} != {%i: %i %i %i: [%i %i %i] + %i + %i}\n", ret, si, ai, eai, sam, sm, am, eam, add, xret, xsi, xai, xeai, xsam, xsm, xam, xeam, xadd);
    return 0;
}

int main(void)
{
    BEGIN_TESTS;

    int si, sa, sm, da, dm, add;
    AST_Node x_node = {NULL, 0, 9, 0, -1, UDI_GFX_OPERATOR_X, 0, 0, 0, NULL, NULL, NULL};               // node representing X
    TESTCASE(Test_TreeSubstOffsetPrime(&x_node, 0, STATE_BASE_X, -1, -1, 0, 1, 0, 0, 0));               // X*a*0 + X*1 + a*0 + 0

    AST_Node y_node = {NULL, 0, 9, 0, -1, UDI_GFX_OPERATOR_Y, 0, 0, 0, NULL, NULL, NULL};               // node representing Y
    TESTCASE(Test_TreeSubstOffsetPrime(&y_node, 0, STATE_BASE_Y, -1, -1, 0, 1, 0, 0, 0));               // Y*a*0 + Y*1 + a*0 + 0

    AST_Node const_node = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 6, 0, 0, NULL, NULL, NULL};       // node representing value 6
    TESTCASE(Test_TreeSubstOffsetPrime(&const_node, 0, STATE_BASE_ZERO, -1, -1, 0, 0, 0, 0, 6));        // 0 + 0 + 6

    AST_Node zero_node = {NULL, 0, 1, 0, -1, UDI_GFX_OPERATOR_CONST, 0, 0, 0, NULL, NULL, NULL};        // node representing value 0
    TESTCASE(Test_TreeSubstOffsetPrime(&zero_node, 0, STATE_BASE_ZERO, -1, -1, 0, 0, 0, 0, 0));         // 0 + 0 + 0

    // constant-state-propagations
    AST_Node * ops_xplus6[] = {&x_node, &const_node};
    AST_Node add_x_const = {ops_xplus6, 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};      // add ((X), (6), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x_const, 0, STATE_BASE_X, -1, -1, 0, 1, 0, 0, 6));              // X*a*0 + X*1 + a*0 + 6
    AST_Node add_x_const_v = {ops_xplus6, 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL};    // add ((X), (6), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x_const_v, 0, STATE_BASE_X, -1, -1, 0, 1, 0, 0, 12));           // X*a*0 + X*1 + a*0 + 12

    AST_Node * ops_6plus6[] = {&const_node, &const_node};
    AST_Node add_const_const = {ops_6plus6, 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};  // add ((6), (6), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_const_const, 0, STATE_BASE_ZERO, -1, -1, 0, 0, 0, 0, 12));      // 0 + 0 + 0 + 12

    AST_Node mul_x_const = {ops_xplus6, 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};      // mul ((X), (6))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_x_const, 0, STATE_BASE_X, -1, -1, 0, 6, 0, 0, 0));              // X*a*0 + X*6 + a*0 + 0
    AST_Node mul_const_const = {ops_6plus6, 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};  // mul ((6), (6))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_const_const, 0, STATE_BASE_ZERO, -1, -1, 0, 0, 0, 0, 36));      // 0 * 0 + 0 + 36

    // zero indexed attributes
    AST_Node * ops_0[] = {&zero_node};
    AST_Node attr1_node = {&(ops_0[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_ATTR, 8, 0, 0, NULL, NULL, NULL};     // node representing attribute 8
    TESTCASE(Test_TreeSubstOffsetPrime(&attr1_node, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 1, 0));             // 0 + 1e + 0
    AST_Node attr2_node = {&(ops_0[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_ATTR, 9, 0, 0, NULL, NULL, NULL};     // node representing attribute 9
    TESTCASE(Test_TreeSubstOffsetPrime(&attr2_node, 0, STATE_BASE_ZERO, -1, 9, 0, 0, 0, 1, 0));             // 0 + 1e + 0
    // constant indexed attributes
    AST_Node * ops_6[] = {&const_node};
    AST_Node attr3_node = {&(ops_6[0]), 1, 1, 0, -1, UDI_GFX_OPERATOR_ATTR, 3, 0, 0, NULL, NULL, NULL};     // node representing attribute 9
    TESTCASE(Test_TreeSubstOffsetPrime(&attr3_node, 0, STATE_BASE_ZERO, -1, 9, 0, 0, 0, 1, 0));             // 0 + 1e + 0

    // constant-attr propagations
    AST_Node * ops_6plusa[] = {&const_node, &attr1_node};
    AST_Node add_6_a8 = {&(ops_6plusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};   // add((6), (a8), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_6_a8, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 1, 6));               // 0 + 1e + 6
    AST_Node add_6_a8_6 = {&(ops_6plusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL}; // add((6), (a8), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_6_a8_6, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 1, 12));            // 0 + 1e + 12
    AST_Node mul_6_a8 = {&(ops_6plusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};   // mul((6), (a8))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_6_a8, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 6, 0));               // 0 + 6e + 0

    // coord-attr operations
    AST_Node * ops_xplusa[] = {&x_node, &attr1_node};
    AST_Node add_x_a8 = {&(ops_xplusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};   // add((x), (a8), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x_a8, 0, STATE_BASE_X, -1, 8, 0, 1, 0, 1, 0));                  // x + e + 0
    AST_Node add_x_a8_6 = {&(ops_xplusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL}; // add((x), (a8), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x_a8_6, 0, STATE_BASE_X, -1, 8, 0, 1, 0, 1, 6));                // x + e + 6
    AST_Node mul_x_a8 = {&(ops_xplusa[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};   // mul((x), (a8))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_x_a8, 0, STATE_BASE_X, 8, -1, 1, 0, 0, 0, 0));                  // x*e*1 + 0 + 0

    // complex intermediate equations
    AST_Node * ops_x6_6[] = {&mul_x_const, &const_node};
    AST_Node add_x6_6 = {&(ops_x6_6[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};     // add((6x), (6), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x6_6, 0, STATE_BASE_X, -1, -1, 0, 6, 0, 0, 6));                 // 6x + 0 + 6
    AST_Node add_x6_6_6= {&(ops_x6_6[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL};    // add((6x), (6), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_x6_6_6, 0, STATE_BASE_X, -1, -1, 0, 6, 0, 0, 12));              // 6x + 0 + 12
    AST_Node mul_x6_6 = {&(ops_x6_6[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};     // mul((6x), (6))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_x6_6, 0, STATE_BASE_X, -1, -1, 0, 36, 0, 0, 0));                // 36x + 0 + 0

    AST_Node * ops_a6_36[] = {&mul_6_a8, &mul_const_const};
    AST_Node add_a6_36 = {&(ops_a6_36[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};   // add((6x), (36), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_a6_36, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 6, 36));             // 6x + 0 + 36
    AST_Node add_a6_36_6 = {&(ops_a6_36[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL}; // add((6x), (36), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_a6_36_6, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 6, 42));           // 6x + 0 + 42
    AST_Node mul_a6_36 = {&(ops_a6_36[0]), 2, 1, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};   // mul((6x), (36))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_a6_36, 0, STATE_BASE_ZERO, -1, 8, 0, 0, 0, 216, 0));            // 36x + 0 + 0

    AST_Node * ops_x6plus6_6aplus36[] = {&add_x6_6, &add_a6_36};
    AST_Node add_6x6_6a36 = {&(ops_x6plus6_6aplus36[0]), 2, 9, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};      // add((6x+6), (6a+36), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_6x6_6a36, 0, STATE_BASE_X, -1, 8, 0, 6, 0, 6, 42));             // 6x + 6a + 42
    AST_Node add_6x6_6a36_6 = {&(ops_x6plus6_6aplus36[0]), 2, 9, 0, -1, UDI_GFX_OPERATOR_ADD, 6, 0, 0, NULL, NULL, NULL};    // add((6x+6), (6a+36), 6)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_6x6_6a36_6, 0, STATE_BASE_X, -1, 8, 0, 6, 0, 6, 48));           // 6x + 6a + 48
    AST_Node mul_6x6_6a36 = {&(ops_x6plus6_6aplus36[0]), 2, 9, 0, -1, UDI_GFX_OPERATOR_MUL, 0, 0, 0, NULL, NULL, NULL};      // mul((6x+6), (6a+36))
    TESTCASE(Test_TreeSubstOffsetPrime(&mul_6x6_6a36, 0, STATE_BASE_X, 8, -1, 36, 216, 36, 0, 216));        // 36ax+216x+36a + 0 + 216

    // Address generation
    AST_Node * ops_6x6_y[] = {&add_x6_6, &y_node};
    AST_Node add_6x6_y = {&(ops_6x6_y[0]), 2, 9, 0, -1, UDI_GFX_OPERATOR_ADD, 0, 0, 0, NULL, NULL, NULL};   // add((6x+6), (y), 0)
    TESTCASE(Test_TreeSubstOffsetPrime(&add_6x6_y, 1, 0, -1, -1, 0, 0, 0, 0, 0));                           // new output
    TESTCASE(lastret->opcode == AST_OPCODE_ADDRGEN);
    TESTCASE(lastret->children == 0);
    TESTCASE(lastret->refcount == 1);
    TESTCASE(lastret->visited == 0);
    TESTCASE(lastret->data != NULL);
    TESTCASE(lastret->freedata != NULL);
    TESTCASE(lastret->clonedata != NULL);
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->xarg == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->yarg == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->baseoffset == 6 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->offsetattr == -1 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->buffersize == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->xscaleattr == -1 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->yscaleattr == -1 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->xmularg == 6 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->xmulargscale == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->xmulscale == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->ymularg == 1 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->ymulargscale == 0 );
    TESTCASE( ((AST_LinearOffsetData*)(lastret->data))->ymulscale == 0 );
    
    // recursion (no substitute)
    AST_Node * ops_x_y[] = {&x_node, &y_node};
    AST_Node random_x_y = {&(ops_x_y[0]), 2, 9, 0, -1, 666, 0, 0, 0, NULL, NULL, NULL};   // invalid((x), (y))
    TESTCASE(Test_TreeSubstOffsetPrime(&random_x_y, 0, 0, -1, -1, 0, 0, 0, 0, 0));         // no output
    TESTCASE( random_x_y.refcount == 9 );
    TESTCASE( x_node.refcount == 9 );
    TESTCASE( y_node.refcount == 9 );
    TESTCASE( random_x_y.children == &(ops_x_y[0]) );
    TESTCASE( random_x_y.children[0] == &x_node );
    TESTCASE( random_x_y.children[1] == &y_node );

    TESTCASE( LibGFX_TreeSubstOffset(NULL) == NULL);

    return(TEST_RESULTS);
}

#endif
