/*
    Summary: unfold_mad.c
    Unfolds multiply-and-accumulate instructions into two opcodes,
    allowing simplification of other algorithms that might otherwise
    need special handling of the MAD opcode.

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>
#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <stdlib.h>

/* Function: LibGFX_UnfoldMad
 * substitutes the MAD opcode with an ADD and a MUL, replacing the
 * original node
 *
 * in:
 *     root - the root of the tree
 *
 * out:
 *     return - 0 on success, nonzero if an error occurred
 */
int LibGFX_UnfoldMad(AST_Node * root)
{
    if (!root) return 0;

    if (root->opcode == UDI_GFX_OPERATOR_MAD)
    {
        AST_Node * mulnode = (AST_Node *) malloc(sizeof(AST_Node));
        if (!mulnode)
        {
            libgfx_errormsg = "UnfoldMad: Out of memory";
            return 1;
        }
        AST_Node ** newpointer = (AST_Node **) malloc(2 * sizeof(AST_Node *));
        if (!newpointer)
        {
            free(mulnode);
            libgfx_errormsg = "UnfoldMad: Out of memory";
            return 1;
        }
        mulnode->children = root->children;
        mulnode->opcode = UDI_GFX_OPERATOR_MUL;
        mulnode->childcount = 2;
        mulnode->operatorindex = root->operatorindex;
        mulnode->refcount = 1;
        mulnode->visited = 0;
        mulnode->const1 = 0;
        mulnode->const2 = 0;
        mulnode->const3 = 0;
        mulnode->data = NULL;

        newpointer[0] = root->children[2];
        newpointer[1] = mulnode;
        root->opcode = UDI_GFX_OPERATOR_ADD;
        root->children = newpointer;
        root->childcount = 2;
        root->const1 = 0; // constant addition = 0
    }
    else
    {
        // recurse
        for (int i = 0; i < root->childcount; i++)
        {
            if (LibGFX_UnfoldMad(root->children[i])) return 1;
        }
    }
    
    return 0;
}

#ifdef TEST
#include <libgfx/test.h>
#include <string.h>

int main(void)
{
    BEGIN_TESTS;

    AST_Node x_node_1 = {NULL, 0, 1, 0, 7, UDI_GFX_OPERATOR_X, 0, 0, 0, NULL};               // node representing X
    AST_Node x_node_2 = {NULL, 0, 1, 0, 7, UDI_GFX_OPERATOR_X, 0, 0, 0, NULL};               // node representing X
    TESTCASE( LibGFX_UnfoldMad(&x_node_1) == 0);
    TESTCASE( memcmp(&x_node_1, &x_node_2, sizeof(AST_Node)) == 0);

    AST_Node const_node = {NULL, 0, 1, 0, 8, UDI_GFX_OPERATOR_CONST, 6, 0, 0, NULL};         // node representing value 6

    AST_Node * ops_x6x[] = {&x_node_1, &const_node, &x_node_2};
    AST_Node mad_x_6_x = {ops_x6x, 3, 1, 0, 9, UDI_GFX_OPERATOR_MAD, 99, 0, 0, NULL};        // x * 6 + x
    // The following sequence assumes that mad (x,y,z) unfolds to add(z, mul(x,y)), and not some other permutation
    TESTCASE( LibGFX_UnfoldMad(&mad_x_6_x) == 0)
    TESTCASE( mad_x_6_x.opcode == UDI_GFX_OPERATOR_ADD );
    TESTCASE( mad_x_6_x.childcount == 2 );
    TESTCASE( mad_x_6_x.children[0] == &x_node_2 );
    TESTCASE( mad_x_6_x.const1 == 0 );
    TESTCASE( mad_x_6_x.operatorindex == 9 );
    TESTCASE( mad_x_6_x.children[1] );
    TESTCASE( mad_x_6_x.children[1]->opcode == UDI_GFX_OPERATOR_MUL );
    TESTCASE( mad_x_6_x.children[1]->childcount == 2 );
    TESTCASE( mad_x_6_x.children[1]->children[0] == &x_node_1 );
    TESTCASE( mad_x_6_x.children[1]->children[1] == &const_node );
    TESTCASE( mad_x_6_x.children[1]->operatorindex == 9 );
    TESTCASE( memcmp(&x_node_1, &x_node_2, sizeof(AST_Node)) == 0);

    return(TEST_RESULTS);
}

#endif
