/*
    Summary: match_pfb.c
    Matches an opcode tree against the simple packed framebuffer template

    Author:
        Marcel Sondaar

    License:
        Public Domain

 */

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101
#include <udi_gfx.h>
#include <libgfx/ast.h>
#include <libgfx/error.h>
#include <libgfx/template.h>
#include <stdlib.h>
#include <string.h>

TPL_ModeTemplate * LibGFX_MatchPackedFramebuffer(AST_Node * inputtree)
{
    AST_Node * root = LibGFX_CloneTree(inputtree);
    if (!root) return NULL;
    libgfx_errormsg = "+A";
    if (root->opcode != UDI_GFX_OPERATOR_RGB) return NULL; // todo: match ARGB buffers

    LibGFX_UnfoldMad(root);
    AST_Node * saveme = LibGFX_TreeSubstOffset(root);
    if (saveme != root)
    {
        LibGFX_FreeNode(root);
        LibGFX_FreeNode(saveme);
        return NULL;
    }
    
    LibGFX_TreeSubstFramebuffer(root);

    // Now trace bits from the RGB operator to the corresponding buffers
    AST_Bit *red_series = NULL, *green_series = NULL, *blue_series = NULL;
    int red_bits = LibGFX_BufferBitFold(root->children[0], &red_series, 1);
    int green_bits = LibGFX_BufferBitFold(root->children[1], &green_series, 1);
    int blue_bits = LibGFX_BufferBitFold(root->children[2], &blue_series, 1);
    int ok = 1;

    TPL_ModeTemplate * newtpl = (TPL_ModeTemplate *) malloc(sizeof(TPL_ModeTemplate));
    TPL_PackedFramebuffer * tplout = &(newtpl->info.pfb);

    if (ok) if (!tplout)
    {
        libgfx_errormsg = "MatchPackedFramebuffer: out of memory";
        ok = 0;
    }
    if (ok) if (red_bits <= 0 || green_bits <= 0 || blue_bits <= 0) ok = 0;
    if (ok) if (red_series->source_node->opcode != AST_OPCODE_FRAMEBUFFER) ok = 0;
    if (ok) libgfx_errormsg = "+V";
    if (ok)
    {
        AST_FramebufferData * matchme = (AST_FramebufferData *) red_series->source_node->data;
        tplout->red_bits = red_bits;
        tplout->red_shift = red_series->sourcebit;
        tplout->green_bits = green_bits;
        tplout->green_shift = green_series->sourcebit;
        tplout->blue_bits = blue_bits;
        tplout->blue_shift = blue_series->sourcebit;
        tplout->alpha_bits = 0;
        tplout->alpha_shift = 0;
        int scalewidth = red_bits + red_series->currentbit;

        //todo: fix matching of seperated buffers (i.e. only offset differences)
        for (int n = 0; n < 3; n++)
        {
            AST_Bit * current = NULL;
            if (n == 0) current = red_series;
            if (n == 1) current = green_series;
            if (n == 2) current = blue_series;
            int bits[3] = {red_bits, green_bits, blue_bits};
            int shift[3] = {tplout->red_shift, tplout->green_shift, tplout->blue_shift};

            for (int i = 0; i < bits[n] && ok; i++)
            {
                if (current->sourcebit != shift[n] + i) ok = 0;
                if (ok) libgfx_errormsg = "+U";
                if (current->currentbit != scalewidth - bits[n] + i) ok = 0;
                if (ok) libgfx_errormsg = "+V";
                if (current->source_node->opcode != AST_OPCODE_FRAMEBUFFER) ok = 0;
                if (ok) libgfx_errormsg = "+X";
                if (ok) if (memcmp(current->source_node->data, matchme, sizeof(AST_FramebufferData)) != 0) ok = 0;
                current = current->next_bit;
                if (ok) libgfx_errormsg = "+Y";
                if (!current && (i + 1 < bits[n])) ok = 0;
                if (ok) libgfx_errormsg = "+Z";
            }
        }
        tplout->bitsperpixel = matchme->bufferbits;
        tplout->bitsperentry = matchme->bufferbits;
        tplout->framebuffer_const = matchme->bufferindexconst;
        tplout->framebuffer_attr = matchme->bufferindexattr;
        tplout->x_per_tile = (matchme->offset.xarg == 1) ? 0 : 1;
        tplout->y_per_tile = (matchme->offset.yarg == 3) ? 0 : 1;
    }


    LibGFX_FreeBitlist(red_series);
    LibGFX_FreeBitlist(green_series);
    LibGFX_FreeBitlist(blue_series);
    LibGFX_FreeNode(root);

    if (ok) return newtpl;
    if (tplout) free(newtpl);
    return NULL;
}

#ifdef TEST
#include <libgfx/test.h>

int main(void)
{
    BEGIN_TESTS;

    TESTCASE( NO_TESTDRIVER )

    return(TEST_RESULTS);
}

#endif
