/* Summary: dispatch.c
 * Decoding of glx opcodes and calls to the corresponding function
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <GL/gl.h>
#include <GL/glx_opcodes.h>

typedef union
{
    const void * raw;
    const GLshort * shortargs;
    const GLint * intargs;
    const GLuint * uintargs;
    const GLfloat * floatargs;
    const GLdouble * doubleargs;
} _GLX_READBUFFER;

int mglDispatchInstruction(const void * buffer, int length)
{
    if (length < 4) return 0;

    _GLX_READBUFFER data;
    data.raw = buffer;

    int oplength = data.shortargs[0];
    int opcode = data.shortargs[1];
    if (oplength > length) return 0;

    switch (opcode)
    {
        case GLOP_BEGIN:
            if (oplength != 8) return 0;
            glBegin(data.uintargs[1]);
            break;
            
        case GLOP_BINDBUFFERUDI:
            if (oplength != 12) return 0;
            glBindBufferUDI(data.intargs[1], data.intargs[2]);
            break;

        case GLOP_CLEAR:
            if (oplength != 8) return 0;
            glClear(data.uintargs[1]);
            break;

        case GLOP_CLEARCOLOR:
            if (oplength != 20) return 0;
            glClearColor(data.floatargs[1], data.floatargs[2], data.floatargs[3], data.floatargs[4]);
            break;

        case GLOP_COLOR3FV:
            if (oplength != 20) return 0;
            glColor3f(data.floatargs[1], data.floatargs[2], data.floatargs[3]);
            break;

        case GLOP_COLOR4FV:
            if (oplength != 20) return 0;
            glColor4f(data.floatargs[1], data.floatargs[2], data.floatargs[3], data.floatargs[4]);
            break;

        case GLOP_CULLFACE:
            if (oplength != 8) return 0;
            glCullFace(data.uintargs[1]);
            break;

        case GLOP_DISABLE:
            if (oplength != 8) return 0;
            glDisable(data.uintargs[1]);
            break;

        case GLOP_ENABLE:
            if (oplength != 8) return 0;
            glEnable(data.uintargs[1]);
            break;

        case GLOP_END:
            if (oplength != 4) return 0;
            glEnd();
            break;
            
        case GLOP_LOADIDENTITY:
            if (oplength != 4) return 0;
            glLoadIdentity();
            break;
            
        case GLOP_MATRIXMODE:
            if (oplength != 8) return 0;
            glMatrixMode(data.uintargs[1]);
            break;            
            
        case GLOP_SCISSOR:
            if (oplength != 20) return 0;
            glScissor(data.uintargs[1], data.uintargs[2], data.uintargs[3], data.uintargs[4]);
            break;

        case GLOP_TEXSUBIMAGE2D:
            if (oplength < 36) return 0;

            glTexSubImage2D(data.intargs[1], data.intargs[2], data.intargs[3], data.intargs[4],
                            data.intargs[5], data.intargs[6], data.intargs[7], data.intargs[8],
                            (void *)&(data.intargs[9]));

            break;

        case GLOP_VERTEX3FV:
            if (oplength != 16) return 0;
            glVertex3f(data.floatargs[1], data.floatargs[2], data.floatargs[3]);
            break;
            
        case GLOP_VIEWPORT:
            if (oplength != 20) return 0;
            glViewport(data.intargs[1], data.intargs[2], data.intargs[3], data.intargs[4]);
            break;


        default:
            return 0;
    }

    return 1;
}
