/* Summary: glxtexsubimage2d.c
 * Wraps <glTexSubImage2D> into a GLX stream
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <GL/gl.h>
#include <GL/glx_opcodes.h>
#include <mos.h>
#include <string.h>

typedef struct gltexsubimage1dstruct
{
    _glx_stream_header h;
    GLenum target;
    GLint level;
    GLint xoffset;
    GLint width;
    GLint format;
    GLenum type;
    unsigned char bytes[476];
} gltexsubimage1dstruct;

void glTexSubImage1D(GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels)
{
    gltexsubimage1dstruct op;
    op.h.opcode = GLOP_TEXSUBIMAGE1D;
    op.h.len = 8;
    op.target = target;
    op.level = level;
    op.format = format;
    op.type = type;

    int bpp = 0;
    switch(type)
    {
        case GL_UNSIGNED_BYTE:  bpp = 8; break;
        case GL_BYTE:           bpp = 8; break;
		case GL_UNSIGNED_SHORT: bpp = 16; break;
        case GL_SHORT:          bpp = 16; break;
        case GL_UNSIGNED_INT:   bpp = 32; break;
        case GL_INT:            bpp = 32; break;
        case GL_FLOAT:          bpp = 32; break;

        case GL_BITMAP:
        default:
            // Todo: error handling
            return;
    }

    int epp = 0;
    switch (format)
    {
        case GL_COLOR_INDEX:
        case GL_RED:
        case GL_GREEN:
        case GL_BLUE:
		case GL_ALPHA:
		case GL_LUMINANCE:
		        epp = 1;
		        break;

        case GL_LUMINANCE_ALPHA:
                epp = 2;
                break;

        case GL_RGB:
                epp = 3;
                break;

        case GL_RGBA:
                epp = 4;
                break;

        default:
                // Todo: error handling
                return;
    }

    int bits_per_entry = epp * bpp;
    int entries_per_packet = (476 * 8) / bits_per_entry; // round down
    int bytes_per_packet = (entries_per_packet * bits_per_entry) >> 3;
    int cx;
    unsigned char * pointer = (unsigned char *) pixels;
    int xend = xoffset + width;

    for (cx = xoffset; cx < xend; cx += entries_per_packet)
    {
        int bytes = bytes_per_packet;
        op.width = entries_per_packet;

        if (cx + entries_per_packet > xend)
        {
            // the leftover value of bytes can be precomputed if necessary.
            bytes = ((xend - cx) * bits_per_entry + 7) >> 3; // round up
            op.width = xend - cx;
        }
        memcpy(op.bytes, pointer, bytes);
        pointer += bytes;
        op.xoffset = cx;
        op.h.len = bytes + 36;
        _glx_schedule(&op, op.h.len);
        yield();
    }
}
