/* Summary: writebuffer_bgra32.c
 * Decodes pixel data to a B8G8R8A8 buffer
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <GL/gl.h>
#include <stdlib.h>
#include <swgl/gltexture.h>
#include <swgl/glbuffer.h>

static GLubyte popdata(GLubyte ** readptr, GLenum type)
{
    GLubyte * ptr = *readptr;
    switch(type)
    {
        case GL_BYTE:
            readptr++;
            if (*((GLbyte *) ptr) < 0) return 0;
            return *((GLbyte *) ptr);

        case GL_UNSIGNED_BYTE:
            readptr++;
            return *ptr;

        case GL_UNSIGNED_SHORT:
            readptr += 2;
            if (*((GLushort *) ptr) > 255) return 255;
            return *((GLushort *) ptr);

        case GL_SHORT:
            readptr += 2;
            if (*((GLshort *) ptr) < 0)   return 0;
            if (*((GLshort *) ptr) > 255) return 255;
            return *((GLshort *) ptr);

        case GL_UNSIGNED_INT:
            readptr += 4;
            if (*((GLuint *) ptr) > 255) return 255;
            return *((GLuint *) ptr);

        case GL_INT:
            readptr += 4;
            if (*((GLint *) ptr) < 0)   return 0;
            if (*((GLint *) ptr) > 255) return 255;
            return *((GLint *) ptr);

        case GL_FLOAT:
            readptr += 4;
            return 0; // TODO: implement GL_FLOAT 
            
        default:
            return 0;            

    }
    return 0;
}

void _gl_writebuffer_b8g8r8x8(GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels, GLbuffer * dest)
{

    int x, y;

    GLubyte * basept = dest->base;;
    GLint outwidth = dest->width;
    GLint outheight = dest->height;
    GLint outpitch = dest->pitch;

    if (xoffset + width > outwidth) return;
    if (yoffset + height > outheight) return;

    GLubyte * inptr = (GLubyte *) pixels;

    // sort out common cases before doing the slow brute-force thing
    if (format == GL_RGB && type == GL_UNSIGNED_BYTE)
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 4;
            for (x = 0; x < width; x++)
            {
                outptr[2] = *inptr++;
                outptr[1] = *inptr++;
                outptr[0] = *inptr++;
                outptr[3] = 0;
                outptr += 4;
            }
        }
    }
    else if (format == GL_RGBA && type == GL_UNSIGNED_BYTE)
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 4;
            for (x = 0; x < width; x++)
            {
                outptr[2] = *inptr++;
                outptr[1] = *inptr++;
                outptr[0] = *inptr++;
                outptr[3] = *inptr++;
                outptr += 4;
            }
        }
    }
    else if (format == GL_COLOR_INDEX && (type == GL_UNSIGNED_BYTE || type == GL_BYTE))
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 4;
            for (x = 0; x < width; x++)
            {
                GLubyte v = *inptr++;
                *outptr++ = v;
                *outptr++ = v;
                *outptr++ = v;
                *outptr++ = v;
            }
        }
    }
    else
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 4;
            for (x = 0; x < width; x++)
            {
                switch(format)
                {
                    case GL_RED:
                        outptr[0] = 0;
                        outptr[1] = 0;
                        outptr[2] = popdata(&inptr, type);
                        outptr[3] = 0;
                        break;
                    case GL_GREEN:
                        outptr[0] = 0;
                        outptr[1] = popdata(&inptr, type);
                        outptr[2] = 0;
                        outptr[3] = 0;
                        break;
                    case GL_BLUE:
                        outptr[0] = popdata(&inptr, type);;
                        outptr[1] = 0;
                        outptr[2] = 0;
                        outptr[3] = 0;
                        break;
                    case GL_ALPHA:
                        outptr[0] = 0;
                        outptr[1] = 0;
                        outptr[2] = 0;
                        outptr[3] = popdata(&inptr, type);
                        break;

                    case GL_RGB:
                        outptr[2] = popdata(&inptr, type);
                        outptr[1] = popdata(&inptr, type);
                        outptr[0] = popdata(&inptr, type);
                        outptr[3] = 0;
                        break;

                    case GL_RGBA:
                        outptr[2] = popdata(&inptr, type);
                        outptr[1] = popdata(&inptr, type);
                        outptr[0] = popdata(&inptr, type);
                        outptr[3] = popdata(&inptr, type);
                        break;

                    case GL_LUMINANCE:
                        {
                            GLubyte data = popdata(&inptr, type);
                            outptr[0] = data;
                            outptr[1] = data;
                            outptr[2] = data;
                            outptr[3] = 0;
                        }
                        break;

                    case GL_LUMINANCE_ALPHA:
                        {
                            GLubyte data = popdata(&inptr, type);
                            outptr[0] = data;
                            outptr[1] = data;
                            outptr[2] = data;
                            outptr[3] = popdata(&inptr, type);
                        }
                        break;

                    case GL_COLOR_INDEX:
                        break;
                        
                    default:
                        // TODO: error handling
                        break;
                }
                outptr += 4;
            }
        }
    }
}
