/* Summary: writebuffer_i8.c
 * Decodes pixel data to a 8-bits indexed buffer
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <GL/gl.h>
#include <stdlib.h>
#include <swgl/gltexture.h>
#include <swgl/glbuffer.h>

static GLubyte popdata(GLubyte ** readptr, GLenum type)
{
    GLubyte * ptr = *readptr;
    switch(type)
    {
        case GL_BYTE:
            readptr++;
            if (*((GLbyte *) ptr) < 0) return 0;
            return *((GLbyte *) ptr);

        case GL_UNSIGNED_BYTE:
            readptr++;
            return *ptr;

        case GL_UNSIGNED_SHORT:
            readptr += 2;
            if (*((GLushort *) ptr) > 255) return 255;
            return *((GLushort *) ptr);

        case GL_SHORT:
            readptr += 2;
            if (*((GLshort *) ptr) < 0)   return 0;
            if (*((GLshort *) ptr) > 255) return 255;
            return *((GLshort *) ptr);

        case GL_UNSIGNED_INT:
            readptr += 4;
            if (*((GLuint *) ptr) > 255) return 255;
            return *((GLuint *) ptr);

        case GL_INT:
            readptr += 4;
            if (*((GLint *) ptr) < 0)   return 0;
            if (*((GLint *) ptr) > 255) return 255;
            return *((GLint *) ptr);

        case GL_FLOAT:
            readptr += 4;
            return 0; // TODO: implement GL_FLOAT 
            
        default:
            return 0;            

    }
    return 0;
}

void _gl_writebuffer_i8(GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels, GLbuffer * dest)
{

    int x, y;

    GLubyte * basept = dest->base;;
    GLint outwidth = dest->width;
    GLint outheight = dest->height;
    GLint outpitch = dest->pitch;

    if (xoffset + width > outwidth) return;
    if (yoffset + height > outheight) return;

    GLubyte * inptr = (GLubyte *) pixels;

    // sort out common cases before doing the slow brute-force thing
    if (format == GL_COLOR_INDEX && ((type == GL_UNSIGNED_BYTE) || (type == GL_BYTE)))
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset;
            for (x = 0; x < width; x++)
            {
                *outptr++ = *inptr++;
            }
        }
    }
    else
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 3;
            for (x = 0; x < width; x++)
            {
                switch(format)
                {
                    case GL_RED:
                        *outptr++ = popdata(&inptr, type) / 3;
                        break;

                    case GL_GREEN:
                        *outptr++ = popdata(&inptr, type) / 3;
                        break;

                    case GL_BLUE:
                        *outptr++ = popdata(&inptr, type) / 3;
                        break;

                    case GL_ALPHA:
                        *outptr++ = 0;
                        break;

                    case GL_RGB:
                        {
                            int v = 0;
                            v += (int)popdata(&inptr, type);
                            v += (int)popdata(&inptr, type);
                            v += (int)popdata(&inptr, type);
                            v /= 3;
                            *outptr++ = (unsigned char)v;
                        }
                        break;

                    case GL_RGBA:
                        {
                            int v = 0;
                            v += (int)popdata(&inptr, type);
                            v += (int)popdata(&inptr, type);
                            v += (int)popdata(&inptr, type);
                            v /= 3;
                            *outptr++ = (unsigned char)v;
                            popdata(&inptr, type);
                        }
                        break;

                    case GL_LUMINANCE:
                        *outptr++ = popdata(&inptr, type);
                        break;

                    case GL_LUMINANCE_ALPHA:
                        *outptr++ = popdata(&inptr, type);
                        popdata(&inptr, type);
                        break;

                    case GL_COLOR_INDEX:
                        *outptr++ = popdata(&inptr, type);
                        break;
                        
                    default:
                        // TODO: error handling
                        break;
                }
            }
        }
    }
}
