/* Summary: writebuffer_rgb16.c
 * Decodes pixel data to a R5G6B5 buffer
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <GL/gl.h>
#include <stdlib.h>
#include <swgl/gltexture.h>
#include <swgl/glbuffer.h>

static GLubyte popdata(GLubyte ** readptr, GLenum type)
{
    GLubyte * ptr = *readptr;
    switch(type)
    {
        case GL_BYTE:
            readptr++;
            if (*((GLbyte *) ptr) < 0) return 0;
            return *((GLbyte *) ptr);

        case GL_UNSIGNED_BYTE:
            readptr++;
            return *ptr;

        case GL_UNSIGNED_SHORT:
            readptr += 2;
            if (*((GLushort *) ptr) > 255) return 255;
            return *((GLushort *) ptr);

        case GL_SHORT:
            readptr += 2;
            if (*((GLshort *) ptr) < 0)   return 0;
            if (*((GLshort *) ptr) > 255) return 255;
            return *((GLshort *) ptr);

        case GL_UNSIGNED_INT:
            readptr += 4;
            if (*((GLuint *) ptr) > 255) return 255;
            return *((GLuint *) ptr);

        case GL_INT:
            readptr += 4;
            if (*((GLint *) ptr) < 0)   return 0;
            if (*((GLint *) ptr) > 255) return 255;
            return *((GLint *) ptr);

        case GL_FLOAT:
            readptr += 4;
            return 0; // TODO: implement GL_FLOAT 
            
        default:
            return 0;            

    }
    return 0;
}

void _gl_writebuffer_r5g6b5(GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const GLvoid *pixels, GLbuffer * dest)
{

    int x, y;

    GLubyte * basept = dest->base;
    GLint outwidth = dest->width;
    GLint outheight = dest->height;
    GLint outpitch = dest->pitch;

    if (xoffset + width > outwidth) return;
    if (yoffset + height > outheight) return;

    GLubyte * inptr = (GLubyte *) pixels;

    // sort out common cases before doing the slow brute-force thing
    if (format == GL_RGB && type == GL_UNSIGNED_BYTE)
    {
        for (y = 0; y < height; y++)
        {
            GLushort * outptr = (GLushort *) (basept + (y + yoffset) * outpitch + xoffset * 2);
            for (x = 0; x < width; x++)
            {
                int cr = *inptr++;
                int cg = *inptr++;
                int cb = *inptr++;
                *outptr++ = (cr >> 3) | ((cg & 0xfc) << 3) | ((cb & 0xf8) << 8);
            }
        }
    }
    else if (format == GL_RGBA && type == GL_UNSIGNED_BYTE)
    {
        for (y = 0; y < height; y++)
        {
            GLushort * outptr = (GLushort *) (basept + (y + yoffset) * outpitch + xoffset * 2);
            for (x = 0; x < width; x++)
            {
                int cr = *inptr++;
                int cg = *inptr++;
                int cb = *inptr++;
                inptr++; // discard alpha
                *outptr++ = (cr >> 3) | ((cg & 0xfc) << 3) | ((cb & 0xf8) << 8);
            }
        }
    }
    else if (format == GL_COLOR_INDEX && (type == GL_UNSIGNED_BYTE || type == GL_BYTE))
    {
        for (y = 0; y < height; y++)
        {
            GLushort * outptr = (GLushort *) (basept + (y + yoffset) * outpitch + xoffset * 2);
            for (x = 0; x < width; x++)
            {
                GLubyte v = *inptr++;
                *outptr++ = (v >> 3) | ((v & 0xfc) << 3) | ((v & 0xf8) << 8);
            }
        }
    }
    else
    {
        for (y = 0; y < height; y++)
        {
            GLubyte * outptr = basept + (y + yoffset) * outpitch + xoffset * 2;
            for (x = 0; x < width; x++)
            {
                int cg = 0;
                int cr = 0;
                int cb = 0;
                switch(format)
                {
                    case GL_RED:
                        cr = popdata(&inptr, type);
                        break;
                    case GL_GREEN:
                        cg = popdata(&inptr, type);
                        break;
                    case GL_BLUE:
                        cb = popdata(&inptr, type);;
                        break;
                    case GL_ALPHA:
                        popdata(&inptr, type);
                        break;

                    case GL_RGB:
                        cr = popdata(&inptr, type);
                        cg = popdata(&inptr, type);
                        cb = popdata(&inptr, type);
                        break;

                    case GL_RGBA:
                        cr = popdata(&inptr, type);
                        cg = popdata(&inptr, type);
                        cb = popdata(&inptr, type);
                        break;

                    case GL_LUMINANCE:
                        {
                            GLubyte data = popdata(&inptr, type);
                            cr = data;
                            cg = data;
                            cb = data;
                        }
                        break;

                    case GL_LUMINANCE_ALPHA:
                        {
                            GLubyte data = popdata(&inptr, type);
                            cr = data;
                            cg = data;
                            cb = data;
                            popdata(&inptr, type);
                        }
                        break;

                    case GL_COLOR_INDEX:
                        break;
                        
                    default:
                        // TODO: error handling
                        break;
                }
                *outptr++ = (cr >> 3) | ((cg & 0xfc) << 3) | ((cb & 0xf8) << 8);
            }
        }
    }
}
