/**
 * Summary: buf_inject.c
 * Insert parts into a buffer object
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <udi_env.h>
#include <stdlib.h>

void _udi_buf_inject(mos_buf_t * buf, mos_buf_part_t * part, udi_size_t off, udi_size_t len)
{
    mos_buf_part_t ** nextptr = &(buf->first);
    mos_buf_part_t * prevptr = NULL;
    
    while (*nextptr)
    {
        mos_buf_part_t * thisptr = *nextptr;
        if (thisptr->size >= off)
        {
            if (len >= thisptr->size - off)
            {
                mos_buf_part_t * oldnext = thisptr->next;
                thisptr->next = part;
                part->prev = thisptr;
                
                // cut the trailing part from this node and possibly others too
                len -= (thisptr->size - off);
                thisptr->size = off;    
                if (thisptr->size == 0)
                {
                    // cut entirely
                    *nextptr = part;
                    part->prev = prevptr;
                    thisptr->data->refcount--;
                    if (thisptr->data->refcount == 0) thisptr->data->free_op(thisptr->data);
                    free(thisptr);
                }
                
                // cutloop
                while (oldnext != NULL && len > 0)
                {
                    if (oldnext->size <= len)
                    {
                        // full cut
                        thisptr = oldnext;
                        oldnext = oldnext->next;
                        thisptr->data->refcount--;
                        if (thisptr->data->refcount == 0) thisptr->data->free_op(thisptr->data);
                        len -= thisptr->size;
                        free(thisptr);    
                    }
                    else
                    {
                        // shift
                        oldnext->offset += len;
                        oldnext->size -= len;
                        len -= len;
                    }
                }
                if (oldnext)
                {
                    oldnext->prev = part;
                    part->next = oldnext;
                }
                else
                {
                    part->next = NULL;
                    buf->last = part;
                }
                buf->allocated += part->size - len;
                return;
                
            }
            else
            {                
                if (off == 0)
                {
                    // add at beginning
                    *nextptr = part;
                    part->next = thisptr;
                    thisptr->prev = part;
                    part->prev = prevptr;                    
                    thisptr->size = thisptr->size - off - len;
                    thisptr->offset = thisptr->offset + off + len;
                } else {
                    // split part, add in the middle
                    mos_buf_part_t * splitpart = (mos_buf_part_t *) malloc(sizeof(mos_buf_part_t));
                    splitpart->next = thisptr->next;
                    splitpart->prev = part;
                    thisptr->next = part;
                    part->prev = thisptr;
                    part->next = splitpart;
                    splitpart->data = thisptr->data;
                    splitpart->data->refcount++;
                    splitpart->size = thisptr->size - off - len;
                    splitpart->offset = thisptr->offset + off + len;
                    thisptr->size = off;
                    
                    // fixed tail pointer if necessary
                    if (buf->last == thisptr) 
                        buf->last = splitpart;
                    else 
                        splitpart->next->prev = splitpart;
                }
                buf->allocated += part->size - len;
                return;
            }
            
        }
        else
        {
            off -= thisptr->size;
            prevptr = thisptr;
            nextptr = &(prevptr->next);            
        }        
    }
    
    //if (*nextptr == NULL)
    
    if (off > 0)
    {
        mos_buf_part_t * nullpart = (mos_buf_part_t *) malloc(sizeof(mos_buf_part_t));
        mos_buf_impl_t * nullimpl = (mos_buf_impl_t *) malloc(sizeof(mos_buf_impl_t));
        udi_assert(nullpart != NULL);
        udi_assert(nullimpl != NULL);
        nullimpl->refcount = 1;
        nullimpl->free_op = &_udi_generic_buf_free_op;
        nullimpl->copy_op = NULL;
        nullimpl->lock_op = NULL;
        nullpart->data = nullimpl;
        nullpart->next = part;
        nullpart->prev = prevptr;
        nullpart->size = off;
        nullpart->offset = 0;
        part->prev = nullpart;
        part->next = NULL;
        *nextptr = nullpart;
        buf->last = part;
        buf->allocated += part->size + off;
    }
    else
    {
        *nextptr = part;
        buf->last = part;
        part->prev = prevptr;
        part->next = NULL;
        buf->allocated += part->size;
    }
}



#ifdef TEST
#include <test.h>

typedef struct test_bufpair
{
    mos_buf_t hidden;
    udi_buf_t public;
} test_bufpair_t;

int main(void)
{
    BEGIN_TESTS;

    // cases:                           yields:
    // without overwrite
    // 1 [new]                            [new]
    // 2 ...[new]                         [...][new]
    // 3 [aaa][bbb][new]                  [aaa][bbb][new]
    // 4 [aaa][bbb]..[new]                [aaa][bbb][...][new]
    // 5 [aaa][new][bbb]                  [aaa][new][bbb]
    // 6 [a[new]aa]                       [a][new][aa]            (duplicate a)
    // 7 [a[new]aa][bbb]                  [a][new][aa][bbb]       (duplicate a)
    // 8 [[new]aaa]                       [new][aaa]
    
    // with overwrite        
    // 9 [new]a] cut left                 [new][a]
    //10 [a[new]a] cut center             [a][new][a]             (duplicate a)
    //11 [a[new]a][bbb] cut center        [a][new][a][bbb]        (duplicate a)
    //12 [a[new] cut right                [a][new]                
    //13 [a[new][bbb] cut right           [a][new][bbb]
    //14 [a[new]b]                        [a][new][b]
    //15 [[new]]                          [new]                   (delete a)
    //16 [[new]][bbb]                     [new][bbb]              (delete a)
    //17 [a[new]c]                        [a][new][c]             (delete b)
    
    test_bufpair_t buf;    
    mos_buf_impl_t impl_a, impl_b, impl_c, impl_new;
    impl_a.free_op = &_udi_null_buf_free_op;
    impl_b.free_op = &_udi_null_buf_free_op;
    impl_c.free_op = &_udi_null_buf_free_op;
    impl_new.free_op = &_udi_null_buf_free_op;
    impl_a.refcount = 1;
    impl_b.refcount = 1;
    impl_c.refcount = 1;
    impl_new.refcount = 1;
    
    mos_buf_part_t part_new, part_a, part_b, part_c, *part_doomed;
    
    // case 1
    buf.hidden.allocated = 0;
    buf.hidden.first = NULL;
    buf.hidden.last = NULL;    
    part_new.next = NULL;
    part_new.prev = NULL;
    part_new.size = 10;
    part_new.offset = 5;
    part_new.data = &impl_new;
    _udi_buf_inject(&(buf.hidden), &part_new, 0, 0);
    TESTCASE (buf.hidden.first == &part_new);
    TESTCASE (buf.hidden.last == &part_new);
    TESTCASE (buf.hidden.allocated == 10);
    TESTCASE (part_new.prev == NULL);
    TESTCASE (part_new.next == NULL);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    
    // case 2
    buf.hidden.allocated = 0;
    buf.hidden.first = NULL;
    buf.hidden.last = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 8, 0);
    part_doomed = buf.hidden.first;
    TESTCASE (buf.hidden.last == &part_new);
    TESTCASE (buf.hidden.allocated == 18);
    TESTCASE (part_new.prev == part_doomed);
    TESTCASE (part_new.next == NULL);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_doomed->prev == NULL);
    TESTCASE (part_doomed->next == &part_new);
    TESTCASE (part_doomed->size == 8);
    TESTCASE (part_doomed->offset == 0);
    free(part_doomed);
    
    // case 3
    buf.hidden.allocated = 10;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_b;
    part_a.prev = NULL;
    part_a.next = &part_b;
    part_b.prev = &part_a;
    part_b.next = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_b.data = &impl_b;
    part_b.size = 4;
    part_b.offset = 7;
    _udi_buf_inject(&(buf.hidden), &part_new, 10, 0);
    TESTCASE (buf.hidden.last == &part_new);
    TESTCASE (buf.hidden.first == &part_a);
    TESTCASE (buf.hidden.allocated == 20);
    TESTCASE (part_new.prev == &part_b);
    TESTCASE (part_new.next == NULL);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_b);
    TESTCASE (part_a.size == 6);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 1);
    TESTCASE (part_b.prev == &part_a);
    TESTCASE (part_b.next == &part_new);
    TESTCASE (part_b.size == 4);
    TESTCASE (part_b.offset == 7);
    TESTCASE (part_b.data->refcount == 1);
    
    // case 4
    buf.hidden.allocated = 10;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_b;
    part_b.next = NULL;
    part_new.prev = NULL;
    part_new.next = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 13, 0);
    TESTCASE (buf.hidden.last == &part_new);
    TESTCASE (buf.hidden.first == &part_a);
    TESTCASE (buf.hidden.allocated == 23);
    part_doomed = part_new.prev;
    TESTCASE (part_doomed != &part_a);
    TESTCASE (part_doomed != &part_b);
    TESTCASE (part_doomed != &part_new);
    /* printf("data: -> 0x%lx | 0x%lx <- 0x%lx -> 0x%lx | 0x%lx <- 0x%lx -> 0x%lx | 0x%lx <- 0x%lx -> 0x%lx | 0x%lx <- 0x%lx -> 0x%lx | 0x%lx <-\n",
           (long) buf.hidden.first,
           (long) part_a.prev, (long) (&part_a), (long) part_a.next,
           (long) part_b.prev, (long) (&part_b), (long) part_b.next,
           (long) part_doomed->prev, (long) part_doomed, (long) part_doomed->next,
           (long) part_new.prev, (long) (&part_new), (long) part_new.next,
           (long) buf.hidden.last);*/    
    TESTCASE (part_new.next == NULL);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_b);
    TESTCASE (part_a.size == 6);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 1);
    TESTCASE (part_b.prev == &part_a);
    TESTCASE (part_b.next == part_doomed);
    TESTCASE (part_b.size == 4);
    TESTCASE (part_b.offset == 7);
    TESTCASE (part_b.data->refcount == 1);
    TESTCASE (part_doomed->prev == &part_b);
    TESTCASE (part_doomed->next == &part_new);
    TESTCASE (part_doomed->size == 3);
    TESTCASE (part_doomed->offset == 0);
    free(part_doomed);
    
    // case 5
    buf.hidden.allocated = 10;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_b;
    part_b.next = NULL;
    part_b.prev = &part_a;
    part_a.next = &part_b;
    part_a.prev = NULL;
    part_new.prev = NULL;
    part_new.next = NULL;
    part_b.prev = &part_a;
    _udi_buf_inject(&(buf.hidden), &part_new, 6, 0);
    TESTCASE (buf.hidden.last == &part_b);
    TESTCASE (buf.hidden.first == &part_a);
    TESTCASE (buf.hidden.allocated == 20);
    TESTCASE (part_new.prev == &part_a);
    TESTCASE (part_new.next == &part_b);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_new);
    TESTCASE (part_a.size == 6);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 1);
    TESTCASE (part_b.prev == &part_new);
    TESTCASE (part_b.next == NULL);
    TESTCASE (part_b.size == 4);
    TESTCASE (part_b.offset == 7);
    TESTCASE (part_b.data->refcount == 1);    
    
    // case 6
    buf.hidden.allocated = 6;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_a;
    part_a.next = NULL;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 2, 0);
    TESTCASE (buf.hidden.first == &part_a);
    part_doomed = buf.hidden.last;
    TESTCASE (part_doomed != &part_a);
    TESTCASE (part_doomed != &part_new);
    TESTCASE (buf.hidden.allocated == 16);
    TESTCASE (part_new.prev == &part_a);
    TESTCASE (part_new.next == part_doomed);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_new);
    TESTCASE (part_a.size == 2);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 2);
    TESTCASE (part_doomed->prev == &part_new);
    TESTCASE (part_doomed->next == NULL);
    TESTCASE (part_doomed->size == 4);
    TESTCASE (part_doomed->offset == 11);
    TESTCASE (part_doomed->data == part_a.data);
    free(part_doomed);
    
    // case 7
    buf.hidden.allocated = 10;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_b;
    part_a.next = &part_b;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_a.data->refcount = 1;
    part_b.prev = &part_a;
    part_b.next = NULL;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 2, 0);
    TESTCASE (buf.hidden.first == &part_a);
    TESTCASE (buf.hidden.last == &part_b);
    part_doomed = part_new.next;
    TESTCASE (part_doomed != &part_a);
    TESTCASE (part_doomed != &part_b);
    TESTCASE (part_doomed != &part_new);
    TESTCASE (buf.hidden.allocated == 20);
    TESTCASE (part_new.prev == &part_a);
    TESTCASE (part_new.next == part_doomed);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_new);
    TESTCASE (part_a.size == 2);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 2);
    TESTCASE (part_b.prev == part_doomed);
    TESTCASE (part_b.next == NULL);
    TESTCASE (part_b.size == 4);
    TESTCASE (part_b.offset == 7);
    TESTCASE (part_b.data->refcount == 1);
    TESTCASE (part_doomed->prev == &part_new);
    TESTCASE (part_doomed->next == &part_b);
    TESTCASE (part_doomed->size == 4);
    TESTCASE (part_doomed->offset == 11);
    TESTCASE (part_doomed->data == part_a.data);
    free(part_doomed);
    
    // case 8
    buf.hidden.allocated = 6;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_a;
    part_a.next = NULL;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_a.data->refcount = 1;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 0, 0);
    TESTCASE (buf.hidden.last == &part_a);
    TESTCASE (buf.hidden.first == &part_new);
    TESTCASE (buf.hidden.allocated == 16);
    TESTCASE (part_new.prev == NULL);
    TESTCASE (part_new.next == &part_a);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == &part_new);
    TESTCASE (part_a.next == NULL);
    TESTCASE (part_a.size == 6);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 1);
    
    // case 9
    buf.hidden.allocated = 6;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_a;
    part_a.next = NULL;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_a.data->refcount = 1;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 0, 4);
    TESTCASE (buf.hidden.last == &part_a);
    TESTCASE (buf.hidden.first == &part_new);
    TESTCASE (buf.hidden.allocated == 12);
    TESTCASE (part_new.prev == NULL);
    TESTCASE (part_new.next == &part_a);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == &part_new);
    TESTCASE (part_a.next == NULL);
    TESTCASE (part_a.size == 2);
    TESTCASE (part_a.offset == 13);
    TESTCASE (part_a.data->refcount == 1);

    // case 10
    buf.hidden.allocated = 6;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_a;
    part_a.next = NULL;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 1, 3);
    TESTCASE (buf.hidden.first == &part_a);
    part_doomed = buf.hidden.last;
    TESTCASE (part_doomed != &part_a);
    TESTCASE (part_doomed != &part_new);
    TESTCASE (buf.hidden.allocated == 13);
    TESTCASE (part_new.prev == &part_a);
    TESTCASE (part_new.next == part_doomed);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_new);
    TESTCASE (part_a.size == 1);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 2);
    TESTCASE (part_doomed->prev == &part_new);
    TESTCASE (part_doomed->next == NULL);
    TESTCASE (part_doomed->size == 2);
    TESTCASE (part_doomed->offset == 13);
    TESTCASE (part_doomed->data == part_a.data);
    free(part_doomed);
    
    // case 11
    buf.hidden.allocated = 10;
    buf.hidden.first = &part_a;
    buf.hidden.last = &part_b;
    part_a.next = &part_b;
    part_a.prev = NULL;
    part_a.data = &impl_a;
    part_a.size = 6;
    part_a.offset = 9;
    part_a.data->refcount = 1;
    part_b.next = NULL;
    part_b.prev = &part_a;
    part_b.data = &impl_b;
    part_b.size = 4;
    part_b.offset = 7;
    part_new.next = NULL;
    part_new.prev = NULL;
    _udi_buf_inject(&(buf.hidden), &part_new, 1, 3);
    TESTCASE (buf.hidden.first == &part_a);
    TESTCASE (buf.hidden.last == &part_b);
    part_doomed = part_b.prev;
    TESTCASE (part_doomed != &part_a);
    TESTCASE (part_doomed != &part_new);
    TESTCASE (buf.hidden.allocated == 17);
    TESTCASE (part_new.prev == &part_a);
    TESTCASE (part_new.next == part_doomed);
    TESTCASE (part_new.size == 10);
    TESTCASE (part_new.offset == 5);
    TESTCASE (part_a.prev == NULL);
    TESTCASE (part_a.next == &part_new);
    TESTCASE (part_a.size == 1);
    TESTCASE (part_a.offset == 9);
    TESTCASE (part_a.data->refcount == 2);
    TESTCASE (part_b.prev == part_doomed);
    TESTCASE (part_b.next == NULL);
    TESTCASE (part_b.size == 4);
    TESTCASE (part_b.offset == 7);
    TESTCASE (part_b.data->refcount == 1);
    TESTCASE (part_doomed->prev == &part_new);
    TESTCASE (part_doomed->next == &part_b);
    TESTCASE (part_doomed->size == 2);
    TESTCASE (part_doomed->offset == 13);
    TESTCASE (part_doomed->data == part_a.data);
    free(part_doomed);
    
    return(TEST_RESULTS);
}

#endif
