/**
 * Summary: buf_read.c
 * Read from a logical buffer
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#include <udi_env.h>
#include <string.h>

static void _udi_buf_read_prime(mos_buf_part_t * part, udi_size_t src_off, udi_size_t src_len, void * dst_mem)
{
    if (part == NULL)
    {
        memset(dst_mem, src_len, 0);
    }
    if (part->size <= src_off)
    {        
        _udi_buf_read_prime(part->next, src_off - part->size, src_len, dst_mem);
    }
    else if (part->size >= src_off+src_len)
    {
        mos_buf_impl_t * data = part->data;
        udi_assert(data != NULL);
        data->copy_op(data, dst_mem, src_off + part->offset, src_len);
    }
    else // part->size < src_off && part->size > src_off+src_len
    {        
        // split transaction
        udi_size_t size_r = src_off+src_len - part->size;
        udi_size_t size_l = src_len - size_r;
        
        // head operation
        mos_buf_impl_t * data = part->data;
        data->copy_op(data, dst_mem, src_off + part->offset, size_l);
        
        // tail recursion        
        char * nextpos = ((char *)dst_mem) + size_l;
        _udi_buf_read_prime(part->next, 0, size_r, nextpos);        
    }
}


void udi_buf_read (udi_buf_t *src_buf, udi_size_t src_off, udi_size_t src_len, void *dst_mem )
{
    mos_buf_t * actual_buf = ((mos_buf_t *)src_buf) - 1;
    udi_size_t end = src_off + src_len;
    udi_assert(src_off <= actual_buf->allocated);
    udi_assert(end <= actual_buf->allocated);

    // TODO: optimize for back-to-front copying
    _udi_buf_read_prime(actual_buf->first, src_off, src_len, dst_mem);
}

#ifdef TEST
#include <test.h>

typedef struct test_bufpair
{
    mos_buf_t hidden;
    udi_buf_t public;
} test_bufpair_t;

int main(void)
{
    BEGIN_TESTS;
    
    // data
    mos_malloc_buf_impl_t storage1 = {
        {1, NULL, &_udi_malloc_buf_copy_op, NULL, NULL},
        "Testcase"};
    mos_malloc_buf_impl_t storage2 = {
        {1, NULL, &_udi_malloc_buf_copy_op, NULL, NULL},
        "1234567890"};
           
    // buffer parts    
    mos_buf_part_t buf1_part1 = {&storage1, NULL, NULL, 8, 0}; // Stores "Testcase"
    
    mos_buf_part_t buf2_part1 = {&storage1, NULL, NULL, 8, 0}; // Stores "Testcase"
    mos_buf_part_t buf2_part2 = {&storage2, NULL, &buf2_part1, 10, 0}; // Stores "1234567890"
    buf2_part1.next = &buf2_part2;
    
    mos_buf_part_t buf3_part1 = {&storage1, NULL, NULL, 4, 4}; // Stores "case"
    mos_buf_part_t buf3_part2 = {&storage2, NULL, &buf3_part1, 4, 4}; // Stores "5678"
    buf3_part1.next = &buf3_part2;
    
    // buffers    
    test_bufpair_t buf1 = {{&buf1_part1, &buf1_part1, NULL, 8}, 8}; // Stores "Testcase"        
    test_bufpair_t buf2 = {{&buf2_part1, &buf2_part2, NULL, 18}, 18}; // Stores "Testcase1234567890"
    test_bufpair_t buf3 = {{&buf3_part1, &buf3_part2, NULL, 8}, 8}; // Stores "case5678"
    
    // Actual tests
    char output[64];
    
    // simple ops (also check not too much was written)
    memset(output, 0, 64);
    udi_buf_read(&(buf1.public), 0, 8, output);
    TESTCASE ( memcmp(output, "Testcase", 9) == 0 );
    
    memset(output, 0, 64);
    udi_buf_read(&(buf1.public), 2, 4, output);
    TESTCASE ( memcmp(output, "stca", 5) == 0 );
    
    memset(output, 'a', 64);
    udi_buf_read(&(buf1.public), 2, 4, output);
    TESTCASE ( memcmp(output, "stcaaaaa", 8) == 0 );

    // linked buffers
    memset(output, 0, 64);
    udi_buf_read(&(buf2.public), 0, 18, output);
    TESTCASE ( memcmp(output, "Testcase1234567890", 19) == 0 );
    
    memset(output, 0, 64);
    udi_buf_read(&(buf2.public), 6, 4, output);
    TESTCASE ( memcmp(output, "se12", 5) == 0 );
    
    memset(output, 0, 64);
    udi_buf_read(&(buf2.public), 10, 4, output);
    TESTCASE ( memcmp(output, "3456", 5) == 0 );
    
    // offsets
    memset(output, 0, 64);
    udi_buf_read(&(buf3.public), 0, 8, output);
    TESTCASE ( memcmp(output, "case5678", 9) == 0 );
    
    memset(output, 0, 64);
    udi_buf_read(&(buf3.public), 2, 4, output);
    TESTCASE ( memcmp(output, "se56", 5) == 0 );
    
    memset(output, 0, 64);
    udi_buf_read(&(buf3.public), 5, 2, output);
    TESTCASE ( memcmp(output, "67", 3) == 0 );
    
    return(TEST_RESULTS);
}

#endif
