/**
 * Summary: cb_park.c
 * Manage control blocks during a request
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#define UDI_VERSION 0x101

#include <udi_env.h>
#include <udi.h>
#include <stdlib.h>

typedef struct MAPLEVEL
{
    void * pointers[16];
    void * parent;
    int available;
} MAPLEVEL;

static MAPLEVEL * prefixroot = NULL;
static int msgindex = 1;

udi_ubit32_t _udi_cb_park(udi_cb_t * cb)
{
    MAPLEVEL ** mappointer = &prefixroot;
    void * createparent = NULL;
    
    // generate sequential request numbers by default.
    // these are unique enough that any bad cb/protocol use can be detected.
    //  (mostly for detecting doubled replies as numbers are not reused)
    int preferredindex = msgindex++;     
    
    // at each loop:
    //   mappointer points to the location holding the address
    //   createparent points to the MAPLEVEL struct in which the address lies
    //     (or NULL if it is the root)
    //   the entered node is known to have available space.    
    for (int i = 0; i < 8; i++) 
    {
        // the node is empty, create it
        if (*mappointer == NULL)
        {
            *mappointer = (MAPLEVEL *) malloc(sizeof(MAPLEVEL));
            udi_assert(*mappointer != NULL);    
            for (int n = 0; n < 16; n++) (*mappointer)->pointers[n] = NULL;
            (*mappointer)->available = 16;
            (*mappointer)->parent = createparent;
        }
        
        // we can now iterate the map
        MAPLEVEL * level = *mappointer;
        
        int trying = 1;
        int levelindex = (preferredindex >> (4*i)) & 0xf;
        while (trying)
        {
            if (level->pointers[levelindex] == NULL)
            {                
                trying = 0; // the level does not exist yet, select it.                        
            }
            else if ((i != 7) && ((MAPLEVEL *) level->pointers[levelindex])->available > 0)                
                trying = 0; // we have free space in the next level (take care not to ask a cb_t for ->available)
            else
            {
                // find new slot
                int mask = 0xf << (4 * i);
                int incr = 0x1 << (4 * i);
                preferredindex = (preferredindex & ~mask) | ((preferredindex + incr) & mask);
                levelindex = (levelindex+1) & 0xf;
            }            
        }        
        createparent = level;                
        mappointer = (MAPLEVEL **) &(level->pointers[levelindex]);
    }
    
    // update free list
    ((MAPLEVEL *)createparent)->available--;    
    while (createparent)
    {
        MAPLEVEL * current = (MAPLEVEL *)createparent;
        MAPLEVEL * parent = (MAPLEVEL *)(current->parent);
        if (current->available == 0)
        {
            if (parent)
            {
                parent->available--;
            }
            createparent = current->parent;
        }
        else createparent = NULL;
    }
    
    *mappointer = (MAPLEVEL *) cb;
    return preferredindex;
}

udi_cb_t * _udi_cb_unpark(udi_ubit32_t ticket)
{
    void * node = prefixroot;
    for (int i = 0; i < 8; i++)
    {
        udi_assert(node != NULL);
        MAPLEVEL * level = (MAPLEVEL *) node;
        int index = (ticket >> (4*i)) & 0xf;
        if (i < 7)
            node = level->pointers[index];
        else
        {
            // leaf level
            MAPLEVEL * backtrack = level;
            node = level->pointers[index];
            level->pointers[index] = NULL;
            level->available++;
            int done = 0;
            int reason = 0;
            if (level->available == 1) reason = 1;
            
            while (!done)
            {                           
                for (int n = 0; n < 16; n++)
                {
                    if (backtrack->pointers[n] != NULL) done = 1;
                }
                if (reason == 1 && backtrack->available == 1)
                {
                    if (backtrack->parent)
                    {
                        backtrack = (MAPLEVEL *) backtrack->parent;
                        backtrack->available++;
                        done = 0;
                    }
                }
                else if (!done)
                {
                    if (backtrack->parent)
                    {
                        MAPLEVEL * parent = (MAPLEVEL *)backtrack->parent;
                        int deleteindex = -1;
                        for (int n = 0; n < 16; n++)
                        {
                            if (parent->pointers[n] == (void*)backtrack)
                                deleteindex = n;                            
                        }
                        udi_assert(deleteindex != -1);
                        parent->pointers[deleteindex] = NULL;
                        free(backtrack);
                        backtrack = parent;
                    }
                    else
                    {
                        done = 1;
                    }
                }
            }
        }            
    }
    return (udi_cb_t *) node;
}
