/* Summary: decode_gfx.c
 * Decoder for GFX calls
 *
 *  Author:
 *      Marcel Sondaar
 *
 *  License:
 *      Public Domain
 */

#define UDI_VERSION 0x101
#define UDI_GFX_VERSION 0x101
#include <udi.h>
#include <udi_gfx.h>
#include <udi_env.h>
#include <mos/gfx.h>
#include <mos/driver.h>
#include <mos/drivercom.h>
#include <mos/event.h>
#include <stdlib.h>
#include <string.h>

udi_meta_initproc_t udi_decode_udi_gfx_provider;
static const udi_gfx_provider_ops_t * metalanguage_pointers;
static int metalanguage_sockets, metalanguage_engines;

//GFXCOMMAND_GETCOUNTS = GFXCOMMANDBASE + 4
//' returns nothing
//GFXCOMMAND_SETATTRIB = GFXCOMMANDBASE + 5
//GFXCOMMAND_GETATTRIB = GFXCOMMANDBASE + 6
//GFXCOMMAND_GETRANGES = GFXCOMMANDBASE + 7

//GFXCOMMAND_GLMESSAGE = GFXCOMMANDBASE + 10
//GFXCOMMAND_GLMESSAGE_L = GFXCOMMANDBASE + 11

int handle_gfx_getcounts(int, int, unsigned char *);
int handle_gfx_setattrib(int, int, unsigned char *);
int handle_gfx_getattrib(int, int, unsigned char *);
int handle_gfx_getranges(int, int, unsigned char *);
int handle_gfx_glmessage(int, int, unsigned char *);
int handle_gfx_invalid(int sender, int bytes, unsigned char * data);
int handle_sys_queryinterface(int, int, unsigned char *);

MOS_EVENTHANDLER gfx_decoder_handles[11] = {
    {&handle_gfx_invalid},
    {&handle_gfx_invalid},
    {&handle_gfx_invalid},
    {&handle_gfx_invalid},
    {&handle_gfx_getcounts},
    {&handle_gfx_setattrib},
    {&handle_gfx_getattrib},
    {&handle_gfx_getranges},
    {&handle_gfx_invalid},
    {&handle_gfx_invalid},
    {&handle_gfx_glmessage} };
    
MOS_EVENTHANDLER gfx_query_handles[10] = {
    {&handle_sys_queryinterface} };

void udi_decode_udi_gfx_provider(int region, int meta)
{
    udi_assert(region == 0); // currently hackily hardcoded    
    //udi_assert(meta == 0);
    udi_ops_vector_t * entrypoints = udi_init_info.ops_init_list[meta].ops_vector;    
    metalanguage_pointers = (const udi_gfx_provider_ops_t *) entrypoints;
    
    // inject into UDI environment    
    udi_gfx_bind_cb_t * cb = (udi_gfx_bind_cb_t *) _udi_cb_enter(region, sizeof(udi_gfx_bind_cb_t));
    metalanguage_pointers->gfx_bind_req_op(cb);        
}

void udi_gfx_bind_ack(udi_gfx_bind_cb_t * cb, udi_index_t sockets, udi_index_t engines, udi_status_t status)
{
    metalanguage_sockets = sockets;
    metalanguage_engines = engines;
    
    udi_assert(status == 0);
    
    MOS_EVENTHANDLER_TREE * msg_tree = mos_eventhandles->sys_trees;
    // register GFX events
    msg_tree[3].handles = gfx_decoder_handles;
    msg_tree[3].max_message_sub = 11;
    
    // register Query events
    msg_tree[1].handles = gfx_query_handles;
    msg_tree[1].max_message_sub = 1;
    
    int devid = DriverInit();
    udi_ubit32_t msg[2] = {7, devid};
    drv_sendmessage((1 << 16) + 0, 8, (char *) msg); // TODO: fix driver management protocol
        
    _udi_cb_exit(UDI_GCB(cb));
}

void udi_gfx_unbind_ack(udi_gfx_bind_cb_t * cb)
{
    _udi_cb_exit(UDI_GCB(cb));
}  

void udi_gfx_set_connector_ack(udi_gfx_state_cb_t * cb)
{
    _udi_cb_exit(UDI_GCB(cb));
}  

void udi_gfx_set_engine_ack(udi_gfx_state_cb_t * cb)
{
    _udi_cb_exit(UDI_GCB(cb));
}

void udi_gfx_get_connector_ack(udi_gfx_state_cb_t * cb, udi_ubit32_t value)
{
    mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
    udi_ubit32_t msg[5] = {GFXCOMMAND_GETATTRIB, 0, cb->subsystem, cb->attribute, value};
    drv_sendmessage(sys_cb->origin, 20, (char*) &(msg[0]));
    _udi_cb_exit(UDI_GCB(cb));
}  

void udi_gfx_get_engine_ack(udi_gfx_state_cb_t * cb, udi_ubit32_t value)
{
    mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
    udi_ubit32_t msg[5] = {GFXCOMMAND_GETATTRIB, 1, cb->subsystem, cb->attribute, value};
    drv_sendmessage(sys_cb->origin, 20, (char*) &(msg[0]));
    _udi_cb_exit(UDI_GCB(cb));
}  

int handle_sys_queryinterface(int sender, int bytes, unsigned char * data)
{
    udi_assert(bytes == 4);
    if (data) {}
    
    udi_ubit32_t msg[2] = {INTERFACE_SYSCOMMAND, INTERFACE_GRAPHICS2};
    drv_sendmessage(sender, 8, (char *) msg);    
    return 0;
}

int handle_gfx_invalid(int sender, int bytes, unsigned char * data)
{
    if (sender) if (bytes) if (data) {}
    //__asm volatile(".byte 0xeb, 0xfe");
    //udi_assert(1 == 0);
    return 1;
}

int handle_gfx_getranges(int sender, int bytes, unsigned char * data)
{
    const int region = 0;
    
    udi_assert(bytes == 16);
    //if (bytes != 16) return 0;
    
    udi_ubit32_t * values = (udi_ubit32_t *) data;
    if (values[1] == 0) // connector range
    {
        udi_gfx_range_cb_t * cb = (udi_gfx_range_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_range_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        cb->rangedata = NULL;
        cb->sender_hack = sender;
        mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
        sys_cb->origin = sender;
        metalanguage_pointers->gfx_range_connector_req_op(cb);
    }
    else if (values[1] == 1) // engine range
    {
        udi_gfx_range_cb_t * cb = (udi_gfx_range_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_range_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        cb->rangedata = NULL;
        cb->sender_hack = sender;
        mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
        sys_cb->origin = sender;
        metalanguage_pointers->gfx_range_engine_req_op(cb);
    }
    
    return 0;
}

int handle_gfx_getcounts(int sender, int bytes, unsigned char * data)
{
    if (bytes != 4) return 0;
    if (data) {} // not needed anymore

    udi_ubit32_t reply[3] = {*(udi_ubit32_t *) data, metalanguage_sockets, metalanguage_engines};
    drv_sendmessage(sender, 12, (char *) &(reply[0]));
    return 0;
}

int handle_gfx_setattrib(int sender, int bytes, unsigned char * data)
{
    if (sender) {}    
    
    const int region = 0;
    
    udi_assert(bytes == 20);
    udi_ubit32_t * values = (udi_ubit32_t *) data;    
    if (values[1] == 0) // connector write
    {
        udi_gfx_state_cb_t * cb = (udi_gfx_state_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_state_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        metalanguage_pointers->gfx_set_connector_req_op(cb, values[4]);
    }
    else if (values[1] == 1) // engine write
    {
        udi_gfx_state_cb_t * cb = (udi_gfx_state_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_state_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        metalanguage_pointers->gfx_set_engine_req_op(cb, values[4]);
    }
    
    return 0;
}

int handle_gfx_getattrib(int sender, int bytes, unsigned char * data)
{
    
    const int region = 0;

    //udi_assert(0 == 1);
    
    if (bytes != 16) return 0;
    udi_ubit32_t * values = (udi_ubit32_t *) data;
    if (values[1] == 0) // connector read
    {
        udi_gfx_state_cb_t * cb = (udi_gfx_state_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_state_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
        sys_cb->origin = sender;
        metalanguage_pointers->gfx_get_connector_req_op(cb);
        
    }
    else if (values[1] == 1) // engine read
    {
        udi_gfx_state_cb_t * cb = (udi_gfx_state_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_state_cb_t));
        cb->subsystem = values[2];
        cb->attribute = values[3];
        mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
        sys_cb->origin = sender;
        metalanguage_pointers->gfx_get_engine_req_op(cb);
    }             
    
    return 0;
}

int handle_gfx_glmessage(int sender, int bytes, unsigned char * data)
{
    if (sender) if (bytes) if (data) {}
    
    mos_malloc_buf_impl_t * lazy_copy = (mos_malloc_buf_impl_t *) malloc(sizeof(mos_malloc_buf_impl_t));
    lazy_copy->data = data + 4;
    lazy_copy->buf.refcount = 2; // one for this function, one for the metalanguage entry point called
    lazy_copy->buf.free_op = &_udi_generic_buf_free_op;
    lazy_copy->buf.copy_op = &_udi_malloc_buf_copy_op;
    lazy_copy->buf.lock_op = &_udi_malloc_buf_lock_op;
    lazy_copy->buf.unlock_op = &_udi_malloc_buf_unlock_op;
    mos_buf_part_t * bufpart = (mos_buf_part_t *) malloc(sizeof(mos_buf_part_t));
    bufpart->offset = 0;
    bufpart->size = bytes - 4;
    bufpart->next = NULL;
    bufpart->prev = NULL;
    bufpart->data = &(lazy_copy->buf);
    
    mos_buf_t * buf = (mos_buf_t *) malloc(sizeof(mos_buf_t) + sizeof(udi_buf_t));
    buf->first = bufpart;
    buf->last = bufpart;
    buf->allocated = bytes - 4;
    udi_buf_t * publicbuf = (udi_buf_t *)(buf + 1);
    publicbuf->buf_size = bytes - 4;
    
    const int region = 0;
    
    udi_gfx_command_cb_t * cb = (udi_gfx_command_cb_t*) _udi_cb_enter(region, sizeof(udi_gfx_command_cb_t));
    mos_system_cb_t * sys_cb = _udi_cb_system(UDI_GCB(cb));
    sys_cb->origin = sender;
    cb->commanddata = publicbuf;
    
    metalanguage_pointers->gfx_command_op(cb);
    
    _udi_cb_exit(UDI_GCB(cb));
    
    
    if (lazy_copy->buf.refcount > 1)
    {
        udi_ubit8_t * copy = (udi_ubit8_t *) malloc(bytes - 8);
        memcpy(copy, lazy_copy->data, bytes - 8);
        lazy_copy->data = copy;
        lazy_copy->buf.refcount--;
    }
    else free(lazy_copy);
    
    return 0;
}
