/**
* Summary: mei_call.c
* convert UDI calls to native calls
*
* Author:
*     Marcel Sondaar
*
* License:
*     <Public Domain>
*  
*/

#define UDI_VERSION 0x101
#include <udi_env.h>
#include <udi.h>

#include <mos/driver.h>
#include <mos/drivercom.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

void bochs_puts(const char * s);
void bochs_putu(uint32_t i);

udi_ubit32_t udi_get_marshal_size(udi_layout_t * layout);

udi_ubit32_t udi_get_marshal_size(udi_layout_t * layout)
{
    udi_ubit32_t count = 0;
    while (*layout != UDI_DL_END)
    {
        switch(*layout)
        {
            case UDI_DL_UBIT32_T:
            case UDI_DL_STATUS_T:
            case UDI_DL_INDEX_T:
                count += (4 - (count) % 4) % 4; // aligned 32-bit
                count += 4;
                layout++;
                break;
                
            case UDI_DL_BUF:
                count += 4;              
                layout += 4;
                break;
            
            default:
                bochs_puts("PANIC: unknown layout type 0x");
                bochs_putu(*layout);
                return 0;
        }        
    }
    return count;
}

void udi_mei_call (udi_cb_t *gcb, udi_mei_init_t *meta_info, udi_index_t meta_ops_num, udi_index_t vec_idx, ... )
{
    va_list args;
    va_start(args, vec_idx);
    
    bochs_puts("][message out: 0x");
    
    mos_channel_t * channel = (mos_channel_t*) gcb->channel;
    udi_mei_ops_vec_template_t * templates = meta_info->ops_vec_template_list;
    
    int message_id_base = channel->system.fn_offset[meta_ops_num];
    
     bochs_putu(message_id_base);

    vec_idx--; // discard channel_event_ind from pointer lists
    //bochs_puts("+0x"); bochs_putu(vec_idx);
    
    while (templates->meta_ops_num != meta_ops_num) templates++;
    bochs_puts(":"); bochs_puts(templates->op_template_list[vec_idx].op_name);
    //udi_assert(channel->channel_funcs == templates);
    
    udi_mei_op_template_t * function = templates->op_template_list + vec_idx;
    
    udi_ubit32_t cb_count = udi_get_marshal_size(function->visible_layout);
    udi_ubit32_t call_count = udi_get_marshal_size(function->marshal_layout);
    //bochs_puts(" size:8+0x"); bochs_putu(cb_count);
    //bochs_puts("+0x"); bochs_putu(call_count);
    
    // get buffers+sizes
    udi_layout_t * cbargs = function->visible_layout;
    unsigned char * cbbuffer = ((unsigned char *)gcb) + sizeof(udi_cb_t);
    int buftransfersize = 0;
    while (*cbargs != UDI_DL_END)
    {
        switch (*cbargs)
        {
            case UDI_DL_INDEX_T:
            case UDI_DL_UBIT32_T:
            case UDI_DL_STATUS_T:
                cbargs++;
                cbbuffer += 4;
                break;
            case UDI_DL_BUF:
            {
                udi_buf_t * read_buf = *((udi_buf_t **)cbbuffer);
                if (read_buf) buftransfersize += read_buf->buf_size;
                bochs_puts("out buffer total: 0x"); bochs_putu(buftransfersize);
                cbargs += 4;
                cbbuffer += 4;
                break;
            }
            
            default:
                bochs_puts("PANIC: can't export layout type 0x");
                bochs_putu(*cbargs);
                return;
        }
    }
    
    char * messagebuffer = (char *) malloc(cb_count+call_count+buftransfersize+8);
    bochs_puts("malloc(0x"); bochs_putu((uint32_t)messagebuffer); bochs_puts(");");
    udi_ubit32_t * intmsg = (udi_ubit32_t *)messagebuffer;
    intmsg[0] = message_id_base + vec_idx;
    intmsg[1] = (udi_ubit32_t) gcb->initiator_context;
    
    // marshal CB data
    cbargs = function->visible_layout;
    cbbuffer = ((unsigned char *)gcb) + sizeof(udi_cb_t);
    char * cbdestbuffer = messagebuffer + 8;
    char * buftransfer = messagebuffer + 8 + cb_count + call_count;
    while (*cbargs != UDI_DL_END)
    {
        switch (*cbargs)
        {
            case UDI_DL_INDEX_T:
            case UDI_DL_UBIT32_T:
            case UDI_DL_STATUS_T:
                cbargs++;
                memcpy(cbdestbuffer, cbbuffer, 4);
                cbbuffer += 4;
                cbdestbuffer += 4;                
                break;
            case UDI_DL_BUF:
            {
                udi_buf_t * read_buf = *((udi_buf_t **)cbbuffer);
                int addsize = 0;
                if (read_buf) 
                {
                    addsize = read_buf->buf_size;
                    udi_buf_read(read_buf, 0, addsize, buftransfer);
                }
                cbargs += 4;
                cbbuffer += 4;
                cbdestbuffer += 4;
                buftransfer += addsize;
                break;
            }
            
            default:
                bochs_puts("PANIC: can't export layout type 0x");
                bochs_putu(*cbargs);
                return;
        }
    }
    memcpy(messagebuffer+8, ((unsigned char *)gcb) + sizeof(udi_cb_t), cb_count);
    
    // marshal arguments
    char * marshalbuffer = messagebuffer + 8 + cb_count;
    udi_layout_t * marshalargs = function->marshal_layout;
    while (*marshalargs != UDI_DL_END)
    {
        switch (*marshalargs)
        {
            case UDI_DL_INDEX_T:
            case UDI_DL_UBIT32_T:
            case UDI_DL_STATUS_T:
                {
                    *((uint32_t *)marshalbuffer) = va_arg(args, uint32_t);
                    //bochs_puts(", 0x"); bochs_putu(*((uint32_t *)marshalbuffer));
                    marshalbuffer += 4;
                    marshalargs++;
                }
                break;
                
            default:
                bochs_puts("PANIC: can't marshal layout type 0x");
                bochs_putu(*marshalargs);
                return;
        }
    }
    
    //bochs_puts(" to 0x"); bochs_putu(channel->system.destination_address);
    drv_sendmessage(channel->system.destination_address, cb_count+call_count+buftransfersize+8, messagebuffer);
    free(messagebuffer);
    udi_cb_free(gcb);
    va_end(args);
}

