/* Summary: udimain.c
 * Contains the new entry point for UDI drivers
 *
 *  Author:
 *      Marcel Sondaar
 * 
 *  License:
 *      Public Domain
 */

#include <mos.h>
#include <mos/drivercom.h>
#include <mos/event.h>

#define UDI_VERSION 0x101
#include <udi.h>
#include <x86.h>
#include <udi_env.h>

#include <mos/driver.h>

#include <string.h>

void bochs_puts(const char * s);
void bochs_putu(uint32_t i);
udi_ubit32_t udi_get_marshal_size(udi_layout_t * layout);

typedef struct udi_entry_tuple
{
    udi_mei_op_template_t * optemplate;
    mos_channel_t * channel;
    udi_op_t * entry_point;
} udi_entry_tuple_t;

void bochs_puts(const char * s)
{
    while (*s)
    {
        outportb(0xe9, *s);
        s++;
    }
}

void bochs_putu(uint32_t i)
{
    if (i == 0)
    {
        outportb(0xe9, '0');
    }
    else
    {
        int j = i & 0xf;
        int k = i >> 4;
        if (k > 0) bochs_putu(k);
        const char * digits = "0123456789ABCDEF";
        outportb(0xe9, digits[j]);
    }
}

static mos_region_t * region;

/*
extern struct _PDCLIB_headnode_t _PDCLIB_memlist;
static void dump_heap(void)
{
    struct _PDCLIB_memnode_t * current = _PDCLIB_memlist.first;
    struct _PDCLIB_memnode_t * last = current;
    bochs_puts("dump:");
    while (current)
    {
        bochs_puts("[");
        bochs_putu(current->size);
        bochs_puts("@");
        bochs_putu((uint32_t)current + 8);
        bochs_puts("->");
        bochs_putu((uint32_t)current->next);
        bochs_puts("]");
        last = current;
        current = current->next;
    }
    if (last != _PDCLIB_memlist.last) bochs_puts("[last check failed]");
    bochs_puts("\r\n");
}
*/

static int udi_fail_message(void * ign1, unsigned int ign2, unsigned int ign3, unsigned char * ign4)
{
    bochs_puts("[UDI ABORT: unknown message 0x");
    bochs_putu(*((uint32_t*)ign3));
    bochs_puts("]\r\n");
    
    udi_assert((intptr_t)ign1);
    udi_assert(ign2);
    udi_assert(ign3);
    udi_assert((intptr_t)ign4); // always fails
    
    return 1;
}

//static int crash_counter = 10;

static int udi_inject_message(void * custom, unsigned int sender, unsigned int bytes, unsigned char * data)
{
    bochs_puts("[incoming ");
    
    udi_entry_tuple_t * customdata = (udi_entry_tuple_t *) custom;
    bochs_puts(customdata->optemplate->op_name);
    
    udi_ubit32_t cb_size = udi_get_marshal_size(customdata->optemplate->visible_layout);
    udi_ubit32_t vec_size = udi_get_marshal_size(customdata->optemplate->marshal_layout);
    //bochs_putu(bytes); bochs_puts("/"); bochs_putu(cb_size + vec_size + 8);
        
    //udi_ubit32_t * intdata = (udi_ubit32_t *)(data + 8 + cb_size);
    //for (int i = 0; i < vec_size >> 2; i++)
    //{
    //    if (i == 0) bochs_puts("(0x"); else bochs_puts(", 0x");
    //    bochs_putu(intdata[i]);
    //}
    //if (vec_size < 4) bochs_puts(" (");
    //bochs_puts(")");
    
    udi_cb_t * cb = _udi_cb_enter(region, cb_size + sizeof(udi_cb_t));
    
    // demarshal controlblock data
    udi_layout_t * layout = customdata->optemplate->visible_layout;
    char * output = ((char*)cb) + sizeof(udi_cb_t);
    unsigned char * input = data + 8;
    unsigned char * bufstart = data + 8 + cb_size + vec_size;
    int bufmax = bytes - (8 + cb_size + vec_size);
    for (int cb_remaining = cb_size; cb_remaining > 0;  )
    {
        //if (cb_remaining == cb_size) bochs_puts("{0x"); else bochs_puts(", 0x");
        switch(*layout)
        {
            case UDI_DL_STATUS_T:
            case UDI_DL_INDEX_T:
            case UDI_DL_UBIT32_T:
                {
                    *((udi_ubit32_t *)output) = *((udi_ubit32_t *)input);
                    //bochs_putu(*((udi_ubit32_t *)output));
                    output += 4;
                    input += 4;
                    cb_remaining -= 4;
                    layout++;
                }
                break;
            case UDI_DL_BUF:
                {
                    int buflen = *((udi_ubit32_t *)input);
                    bochs_putu(buflen);
                    //bochs_putu(buflen);
                    if (buflen > bufmax)
                    {
                        buflen = bufmax;
                        bochs_puts("<missing>");
                    }                     
                    
                    // create hidden buffer structures
                    mos_malloc_buf_impl_t * buf_copy = (mos_malloc_buf_impl_t *) malloc(sizeof(mos_malloc_buf_impl_t));
                    bochs_puts("malloc(0x"); bochs_putu((uint32_t)buf_copy); bochs_puts(");");
                    buf_copy->buf.refcount = 1; // one for this function, one for the metalanguage entry point called
                    buf_copy->buf.free_op = &_udi_generic_buf_free_op;
                    buf_copy->buf.copy_op = &_udi_malloc_buf_copy_op;
                    buf_copy->buf.lock_op = &_udi_malloc_buf_lock_op;
                    buf_copy->buf.unlock_op = &_udi_malloc_buf_unlock_op;
                    mos_buf_part_t * bufpart = (mos_buf_part_t *) malloc(sizeof(mos_buf_part_t));
                    bochs_puts("malloc(0x"); bochs_putu((uint32_t)bufpart); bochs_puts(");");
                    bufpart->offset = 0;
                    bufpart->size = buflen;
                    bufpart->next = NULL;
                    bufpart->prev = NULL;
                    bufpart->data = &(buf_copy->buf);
                    
                    // copy data into buffer
                    unsigned char * datacopy = (unsigned char *) malloc(buflen);
                    bochs_puts("malloc(0x"); bochs_putu((uint32_t)datacopy); bochs_puts(");");
                    memcpy(datacopy, bufstart, buflen);                    
                    buf_copy->data = datacopy;
                    
                    // create the top-level buffer structure
                    mos_buf_t * buf = (mos_buf_t *) malloc(sizeof(mos_buf_t) + sizeof(udi_buf_t));
                    bochs_puts("malloc(0x"); bochs_putu((uint32_t)buf); bochs_puts(");");
                    buf->first = bufpart;
                    buf->last = bufpart;
                    buf->allocated = buflen;
                    udi_buf_t * publicbuf = (udi_buf_t *)(buf + 1);
                    publicbuf->buf_size = buflen;
                    
                    // stick the buffer into the control block
                    *((udi_buf_t **)output) = publicbuf;
                    //bochs_puts("->0x");
                    //bochs_putu((uint32_t)buf);
                    
                    // pointermagic
                    input += 4;
                    output += 4;
                    cb_remaining -= 4;
                    layout += 4;
                    bufmax -= buflen;
                    bufstart += buflen;
                }
                break;
            
            default:
                bytes -= 4;
            
        }
    }
    //if (cb_size < 4) bochs_puts(" {");
    //bochs_puts("}");
    
    //memcpy(((char*)cb) + sizeof(udi_cb_t), data + 8, cb_size);
    
    //bochs_puts("cb 0x"); bochs_putu((uint32_t)cb);
    
    mos_system_cb_t * syscb = _udi_cb_system(cb);
    
    syscb->origin = sender;
    cb->channel = (udi_channel_t) customdata->channel;
    customdata->channel->system.destination_address = sender;
    
/*    bochs_puts(" be 0x"); bochs_putu((uint32_t)(customdata->template->backend_stub));
    bochs_puts(" ep 0x"); bochs_putu((uint32_t)(customdata->entry_point));*/
    
    bochs_puts("]\n[call ");
    customdata->optemplate->backend_stub(customdata->entry_point, cb, data + 8 + cb_size);
    bochs_puts(" ok]\n");
    
    return 0;
}

static int udi_reply_supports(void * custom, unsigned int sender, unsigned int bytes, unsigned char * data)
{
    bochs_puts("[incoming queryinterfaces]");
    if (data) { }
    if (bytes) { }
    
    int sendsize = 0;
    uint32_t * senddata = (uint32_t *) custom;
    uint32_t * counter = senddata;
    while (*counter)
    {
        counter++;
        sendsize++;
    }
    drv_sendmessage(sender, sendsize * 4, (char *) senddata);
    
    return 0;
}

int main(void)
{
    allocateiobitmap(0x0, 0x100, (int *) -1);
    portalloc(0xE8, 4);
    
    int devno = DriverInit();
    drv_setname(0,1);
    
    bochs_putu(0xabcd1234);
    bochs_puts(" - udimain.c\r\n");

    region = _mos_region_enter();
    
    //for (int i = 0; i < 16; i++) {malloc(16); bochs_puts("*");}
    
    // initialize receivers
    int extents[16];
    for (int i = 0; i < 16; i++) extents[i] = 0;
    extents[1] = 1; // reserve for queryinterfaces
    
    // get maximum function numbers
    for(udi_hidden_meta_init_t * initptrs = udi_props_meta_list; initptrs->metalanguage; initptrs++)
    {
        bochs_puts("[meta ");
        udi_ops_init_t * opslist = udi_init_info.ops_init_list;
        while (opslist->ops_idx != initptrs->bound_index) opslist++;
        
        int meta_index = opslist->meta_ops_num;
        bochs_putu(meta_index);
        uint32_t end = initptrs->native_function_base[meta_index];
        bochs_puts(" "); bochs_putu(end);
        end += 16; // FIXME: make the right amount of room
        bochs_puts("-"); bochs_putu(end);
        uint32_t group = end >> 16;
        end &= 0xffff;
        if (extents[group] < end) extents[group] = end;
        bochs_puts("]");
    }
    bochs_puts("[ok]\r\n[");
    //for (int i = 0; i < 16; i++) {malloc(16); bochs_puts("*");}
    
    for (int i = 0; i < 16; i++)
    {
        bochs_putu(i); bochs_puts(":"); bochs_putu(extents[i]); bochs_puts(", ");
        if (extents[i] > 0)
        {
            MOS_EVENTHANDLER_TREE tree;
            MOS_EVENTHANDLER * handlers = (MOS_EVENTHANDLER *) malloc(sizeof(MOS_EVENTHANDLER) * extents[i]);
            ;
            //bochs_puts("allocated branch at 0x"); bochs_putu((uint32_t) handlers);
            //bochs_puts(", branch "); bochs_putu(i); 
            //bochs_puts(", size "); bochs_putu(extents[i]);            
            //bochs_puts(", next "); bochs_putu((uint32_t)malloc(sizeof(MOS_EVENTHANDLER) * extents[i]));
            //bochs_puts("\r\n"); 
            tree.handles = handlers;
            tree.max_message_sub = extents[i];
            for (int j = 0; j < extents[i]; j++)
            {
                handlers[j].callback = &udi_fail_message;
                handlers[j].data = NULL;
            }
            mos_eventhandles->sys_trees[i] = tree;
            //memcpy(&(mos_eventhandles->sys_trees[i]), &tree, sizeof(MOS_EVENTHANDLER_TREE));
        }
    }
    bochs_puts("]\r\n");
    
    for(udi_hidden_meta_init_t * initptrs = udi_props_meta_list; initptrs->metalanguage; initptrs++)
    {
        udi_ops_init_t * opslist = udi_init_info.ops_init_list;
        while (opslist->ops_idx != initptrs->bound_index) opslist++;
        
        int meta_index = opslist->meta_ops_num;
        uint32_t decode_start = initptrs->native_function_base[meta_index];
        uint32_t group = decode_start >> 16;
        
        bochs_puts("decode meta "); bochs_putu(meta_index); bochs_puts(" @ 0x"); bochs_putu(decode_start);
        
        decode_start &= 0xffff;
        
        udi_mei_ops_vec_template_t * templates = initptrs->metalanguage->ops_vec_template_list;
        while (templates->meta_ops_num != meta_index) 
        {
            templates++;
            udi_assert(templates->op_template_list != NULL);
        }
         
        //bochs_puts(" = "); bochs_putu(templates->meta_ops_num); bochs_puts("\r\n");
        
        udi_mei_op_template_t * functions = templates->op_template_list;
        //bochs_puts("root at: 0x"); bochs_putu((uint32_t)mos_eventhandles);
        //bochs_puts(" tree at: 0x"); bochs_putu((uint32_t)(mos_eventhandles->sys_trees));
        //bochs_puts(" branch at: 0x"); bochs_putu((uint32_t)(mos_eventhandles->sys_trees[group].handles));
        MOS_EVENTHANDLER * handlers = mos_eventhandles->sys_trees[group].handles + decode_start;
        //bochs_puts(" starting entry at: 0x"); bochs_putu((uint32_t)handlers);
        udi_ops_vector_t * entrypoints = opslist->ops_vector;
        //bochs_puts("\r\n");
        
        // initialize per-channel data
        mos_channel_t * channeldata = (mos_channel_t *) malloc(sizeof(mos_channel_t));
        channeldata->channel_funcs = templates;
        channeldata->system.fn_offset = initptrs->native_function_base;
        
        // FIXME: deal with *_channel_event_ind
        entrypoints++; // skip channel_event_ind op in driver's pointer table
        
        bochs_puts(", binding: \r\n");
        while (functions->op_name)
        {
            bochs_puts(functions->op_name);  bochs_puts(", ");
            
            udi_entry_tuple_t * data = (udi_entry_tuple_t *) malloc(sizeof(udi_entry_tuple_t));
            //bochs_puts(" stored at: 0x"); bochs_putu((uint32_t)data);
            //bochs_puts(" meta template at: 0x"); bochs_putu((uint32_t)functions);
            data->optemplate = functions;            
            //bochs_puts("\r\nentry point at: 0x"); bochs_putu((uint32_t)*entrypoints);
            data->entry_point = *entrypoints;
            //bochs_puts(" handlers at: 0x"); bochs_putu((uint32_t)handlers);
            data->channel = channeldata;
            
            handlers->callback = &udi_inject_message;
            handlers->data = data;            
            
            handlers++;
            functions++;
            entrypoints++;            
        }        
        bochs_puts("\r\n");
        
    }
    bochs_puts("[meta pointers ok][driver supports: ");
    
    int meta_count = 0;
    for(udi_hidden_meta_init_t * initptrs = udi_props_meta_list; initptrs->metalanguage; initptrs++)
        meta_count++;
    uint32_t * proto_ids = (uint32_t*) malloc((meta_count + 2) * sizeof(uint32_t));
    uint32_t * current_proto = proto_ids;
    *current_proto++ = 0x00010000;
    for(udi_hidden_meta_init_t * initptrs = udi_props_meta_list; initptrs->metalanguage; initptrs++)
    {
        udi_ops_init_t * opslist = udi_init_info.ops_init_list;
        while (opslist->ops_idx != initptrs->bound_index) opslist++;
        
        int meta_index = opslist->meta_ops_num;
        *current_proto = initptrs->native_proto_indices[meta_index];
        bochs_putu(*current_proto);
        bochs_puts(", ");
        current_proto++;
    }
    bochs_puts("]");
    *current_proto = 0;
    MOS_EVENTHANDLER * queryhandler = mos_eventhandles->sys_trees[1].handles;
    queryhandler->data = proto_ids;
    queryhandler->callback = &udi_reply_supports;
    
    // initialisation complete
    int initok_msg[2] = {7, devno};
    drv_sendmessage(0x00010000 + 0, 8, (char *) &(initok_msg[0]) );
    bochs_puts("[UDI live]\r\n");
    
    mos_eventloop();
    
    udi_assert(1 == 0);
}
