/**
 * Summary: pio_memio.c
 * Direct raw MMIO accesses stubs
 * 
 * Implements a template PIO handler that accesses memory in processor order.
 * This might not match the actual platform requirements for MMIO, but it
 * will suffice when the area is uncacheable and strictly ordered by architecture.
 * 
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#define UDI_VERSION 0x101

#include <udi_env.h>
#include <udi.h>
#include <stdlib.h>

static void _udi_mmio_write_8  (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit8_t data)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    *pointer = data;
}

static void _udi_mmio_write_16 (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit16_t data)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    *((udi_ubit16_t*)pointer) = data;
}

static void _udi_mmio_write_32 (struct mos_pio_impl * impl, udi_size_t offset, udi_ubit32_t data)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    *((udi_ubit32_t*)pointer) = data;
}

static udi_ubit8_t  _udi_mmio_read_8   (struct mos_pio_impl * impl, udi_size_t offset)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    return *pointer;
}

static udi_ubit16_t _udi_mmio_read_16  (struct mos_pio_impl * impl, udi_size_t offset)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    return *((udi_ubit16_t *)pointer);
}

static udi_ubit32_t _udi_mmio_read_32  (struct mos_pio_impl * impl, udi_size_t offset)
{
    mos_mmio_pio_impl_t * mmio = (mos_mmio_pio_impl_t *) impl;
    udi_ubit8_t * pointer = (udi_ubit8_t *)mmio->base + offset;
    return *((udi_ubit32_t *)pointer);
}

mos_pio_impl_t * _udi_create_pio_mmio(void * base, udi_size_t length)
{
    mos_mmio_pio_impl_t * impl = (mos_mmio_pio_impl_t *) malloc(sizeof(mos_mmio_pio_impl_t)); 
    udi_assert(impl != NULL);
    
    impl->pio_impl.read_8   = &_udi_mmio_read_8;
    impl->pio_impl.read_16  = &_udi_mmio_read_16;
    impl->pio_impl.read_32  = &_udi_mmio_read_32;
    impl->pio_impl.write_8  = &_udi_mmio_write_8;
    impl->pio_impl.write_16 = &_udi_mmio_write_16;
    impl->pio_impl.write_32 = &_udi_mmio_write_32;
    impl->pio_impl.limit = length;
    impl->base = base;
    return &(impl->pio_impl);
}

