/**
 * Summary: pio_trans.c
 * Physical IO transfers
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Public Domain>
 */

#define UDI_VERSION 0x101
#define UDI_PHYSIO_VERSION 0x101

#include <udi.h>
#include <udi_physio.h>
#include <udi_env.h>

void bochs_puts(const char *);
void bochs_putu(udi_ubit32_t);

void udi_pio_trans (udi_pio_trans_call_t *callback, udi_cb_t *gcb, udi_pio_handle_t pio_handle, udi_index_t start_label, udi_buf_t *buf, void *mem_ptr)
{
    udi_pio_trans_internal_t * trans = (udi_pio_trans_internal_t *) pio_handle;
    bochs_puts("[assert 0]");
    udi_assert(trans != 0);
    __asm volatile("xchgw %bx, %bx\nxchgw %di, %di");
    udi_ubit32_t registers[8] = {0xc0c0c0c0, 0xc1c1c1c1, 0xc2c2c2c2, 0xc3c3c3c3, 0xc4c4c4c4, 0xc5c5c5c5, 0xc6c6c6c6, 0xc7c7c7c7};
    
    udi_size_t instructionpointer = 0;
    if (start_label != 0)
    {
        for (; instructionpointer < trans->opcodecount; instructionpointer++)
        {
            if (trans->opcodes[instructionpointer].pio_op == UDI_PIO_LABEL && trans->opcodes[instructionpointer].operand == start_label)
            {
                break;
            }
        }
        bochs_puts("[assert 1]");
        udi_assert(instructionpointer < trans->opcodecount);
    }
    
    for (; instructionpointer < trans->opcodecount; instructionpointer++)
    {
        udi_pio_local_trans_t instruction = trans->opcodes[instructionpointer];
        //bochs_puts("[");
        //bochs_putu(instruction.pio_op);
        //bochs_puts("]");

        if (instruction.pio_op < 0x80)
        {
            // Class A
            if ((instruction.pio_op & 0xE0) == UDI_PIO_IN || (instruction.pio_op & 0xE0) == UDI_PIO_STORE)
            {
                // read pio/reg, write memory                
                udi_ubit32_t value = 0;
                if ((instruction.pio_op & 0xE0) == UDI_PIO_IN)
                {
                    udi_size_t write_offset = instruction.operand;
                    bochs_puts("[assert 6]");
                    udi_assert(write_offset + (1 << instruction.tran_size) <= trans->regsize);
                    write_offset += trans->regoffset;
                    
                    switch (instruction.tran_size)
                    {
                        case 0: value = trans->ioaccess->read_8(trans->ioaccess, write_offset); break; 
                        case 1: value = trans->ioaccess->read_16(trans->ioaccess, write_offset); break;
                        case 2: value = trans->ioaccess->read_32(trans->ioaccess, write_offset); break; 
                        
                        default: 
                            bochs_puts("[assert 7]");
                            udi_assert(0);
                    }
                }
                else // UDI_PIO_STORE
                {
                    // regular load
                    udi_assert(instruction.operand < 8);
                    value = registers[instruction.operand & 0x07];
                }
                
                switch (instruction.pio_op & 0x18)
                {
                    case UDI_PIO_DIRECT:
                    {
                        bochs_puts("[assert 3]");
                        udi_ubit8_t regindex = instruction.pio_op & 0x7;                        
                        switch(instruction.tran_size)
                        {
                            case 0: registers[regindex] = value & 0xff; break;
                            case 1: registers[regindex] = value & 0xffff; break;
                            case 2: registers[regindex] = value & 0xffffffff; break;
                            default: 
                                bochs_puts("[assert G]");
                                udi_assert(0);
                        }
                    }
                    break;

                    case UDI_PIO_SCRATCH:
                    case UDI_PIO_MEM:
                    {
                        //bochs_puts("[");
                        void * ptr = ((instruction.pio_op & 0x18) == UDI_PIO_SCRATCH) ? gcb->scratch : mem_ptr;
                        //bochs_putu((udi_ubit32_t)ptr);
                        udi_ubit8_t * byteptr = (udi_ubit8_t *) ptr;
                        udi_size_t offset = registers[instruction.pio_op & 0x07];
                        //bochs_puts("+");
                        //bochs_putu(offset);

                        switch(instruction.tran_size)
                        {
                            case 0: *(udi_ubit8_t  *)(byteptr + offset) = (udi_ubit8_t)value; break;
                            case 1: *(udi_ubit16_t  *)(byteptr + offset) = (udi_ubit16_t)value; break;
                            case 2: *(udi_ubit32_t  *)(byteptr + offset) = (udi_ubit32_t)value; break;
                            bochs_puts("[assert 4]");
                            default: udi_assert(0);
                        }
                        //bochs_puts("=");
                        //bochs_putu(value);
                        //bochs_puts("]");
                    }
                    break;

                    case UDI_PIO_BUF:
                    default:
                        bochs_puts("[assert 5]");
                        udi_assert(0);
                }                                
                
            }
            else // UDI_PIO_OUT || UDI_PIO_LOAD 
            {
                udi_ubit32_t value = 0;
                // read memory, write pio/reg
                switch (instruction.pio_op & 0x18)
                {
                    case UDI_PIO_DIRECT:
                        bochs_puts("[assert 3]");                        
                        value = registers[instruction.pio_op & 0x7];
                        switch(instruction.tran_size)
                        {
                            case 0: value = value & 0xff; break;
                            case 1: value = value & 0xffff; break;
                            case 2: value = value & 0xffffffff; break;
                            default: 
                                bochs_puts("[assert G]");
                                udi_assert(0);
                        }
                        break;

                    case UDI_PIO_SCRATCH:
                    case UDI_PIO_MEM:
                    {
                        //bochs_puts("[");
                        void * ptr = ((instruction.pio_op & 0x18) == UDI_PIO_SCRATCH) ? gcb->scratch : mem_ptr;
                        //bochs_putu((udi_ubit32_t)ptr);
                        udi_ubit8_t * byteptr = (udi_ubit8_t *) ptr;
                        udi_size_t offset = registers[instruction.pio_op & 0x07];
                        //bochs_puts("+");
                        //bochs_putu(offset);

                        switch(instruction.tran_size)
                        {
                            case 0: value = *(udi_ubit8_t  *)(byteptr + offset); break;
                            case 1: value = *(udi_ubit16_t *)(byteptr + offset); break;
                            case 2: value = *(udi_ubit32_t *)(byteptr + offset); break;
                            bochs_puts("[assert 4]");
                            default: udi_assert(0);
                        }
                        //bochs_puts("=");
                        //bochs_putu(value);
                        //bochs_puts("]");
                    }
                    break;

                    case UDI_PIO_BUF:
                    default:
                        bochs_puts("[assert 5]");
                        udi_assert(0);
                }
                
                if ((instruction.pio_op & 0xE0) == UDI_PIO_OUT)
                {
                    udi_size_t write_offset = instruction.operand;
                    bochs_puts("[assert 6]");
                    udi_assert(write_offset + (1 << instruction.tran_size) <= trans->regsize);
                    write_offset += trans->regoffset;
                    
                    switch (instruction.tran_size)
                    {
                        case 0: trans->ioaccess->write_8(trans->ioaccess, write_offset, (udi_ubit8_t)(value & 0xff)); break; 
                        case 1: trans->ioaccess->write_16(trans->ioaccess, write_offset, (udi_ubit16_t)(value & 0xffff)); break;
                        case 2: trans->ioaccess->write_32(trans->ioaccess, write_offset, (udi_ubit32_t)(value & 0xffffffff)); break; 
                        
                        default: 
                            bochs_puts("[assert 7]");
                            udi_assert(0);
                    }
                }
                else // UDI_PIO_LOAD
                {
                    // regular load
                    udi_assert(instruction.operand < 8);
                    registers[instruction.operand & 0x07] = value;
                }
            }
        }
        else if (instruction.pio_op < 0xF0)
        {
            // Class B
            udi_ubit8_t regindex = instruction.pio_op & 0x07;
            switch (instruction.pio_op & 0xF8)
            {
                case UDI_PIO_LOAD_IMM: // 80
                    switch (instruction.tran_size)
                    {
                        case 0: registers[regindex] = instruction.operand & 0xff; break;
                        case 1: registers[regindex] = instruction.operand & 0xffff; break;
                        case 2: 
                            registers[regindex] = ((udi_ubit32_t)instruction.operand) | ((udi_ubit32_t)(trans->opcodes[instructionpointer + 1].operand)) << 16;
                            instructionpointer++;
                            break;
                        default:
                            bochs_puts("[assert 8]");
                            udi_assert(0);
                    }
                    break;
                    
                case UDI_PIO_CSKIP: // 88
                    {
                        udi_ubit32_t regvalue = registers[regindex];
                        udi_ubit32_t sign = 0;
                        switch (instruction.tran_size)
                        {
                            case 0: 
                                sign = regvalue & 0x80; 
                                regvalue = regvalue & 0xff; 
                                break;
                            case 1: 
                                sign = regvalue & 0x8000;
                                regvalue = regvalue & 0xffff;
                                break;
                            case 2: 
                                sign = regvalue & 0x80000000; 
                                regvalue = regvalue & 0xffffffff;
                                break;

                            default:
                                bochs_puts("[assert C]");
                                udi_assert(0);
                        }
                        
                        //bochs_puts("[");
                        //bochs_putu(regvalue);
                        //bochs_puts("~");
                        //bochs_putu(sign);

                        udi_boolean_t skip = 0;
                        switch (instruction.operand)
                        {
                            case UDI_PIO_Z:    skip = (regvalue == 0); break;
                            case UDI_PIO_NZ:   skip = (regvalue != 0); break;
                            case UDI_PIO_NEG:  skip = (sign != 0); break;
                            case UDI_PIO_NNEG: skip = (sign == 0); break;
                            default:
                                bochs_puts("[assert D]");
                                udi_assert(0);
                        }
                        //bochs_puts("~");
                        //bochs_putu(skip);
                        //bochs_puts("]");

                        if (skip) 
                        {
                            instructionpointer++;
                        }
                    }
                    break;
                
                    
                    case UDI_PIO_OUT_IND: // 98
                    {
                        udi_ubit32_t value = registers[regindex];
                        udi_assert(instruction.operand < 8);
                        udi_ubit32_t write_offset = registers[instruction.operand & 0x7];
                        udi_assert(write_offset + (1 << instruction.tran_size) <= trans->regsize);
                        write_offset += trans->regoffset;
                        
                        //bochs_puts("[");
                        //bochs_putu(write_offset);
                        //bochs_puts("=");
                        //bochs_putu(value);
                        //bochs_puts("]");
                        
                        switch (instruction.tran_size)
                        {
                            case 0: trans->ioaccess->write_8(trans->ioaccess, write_offset, (udi_ubit8_t)(value & 0xff)); break; 
                            case 1: trans->ioaccess->write_16(trans->ioaccess, write_offset, (udi_ubit16_t)(value & 0xffff)); break;
                            case 2: trans->ioaccess->write_32(trans->ioaccess, write_offset, (udi_ubit32_t)(value & 0xffffffff)); break; 
                            
                            default: 
                                bochs_puts("[assert 7b]");
                                udi_assert(0);
                        }
                    }
                    break;
                    
                    case UDI_PIO_OR: // C0
                    {
                        udi_ubit32_t left = registers[regindex];
                        udi_ubit32_t right = registers[instruction.operand & 0x7];
                        switch (instruction.tran_size)
                        {
                            case 0: registers[regindex] = (udi_ubit8_t)left  | (udi_ubit8_t)right; break;
                            case 1: registers[regindex] = (udi_ubit16_t)left | (udi_ubit16_t)right; break;
                            case 2: registers[regindex] = (udi_ubit32_t)left | (udi_ubit32_t)right; break;
                            default:
                                bochs_puts("[assert I]");
                                udi_assert(0);
                        }
                    }
                    break;
                                
                    
                    case UDI_PIO_ADD: // D8
                    {
                        udi_ubit32_t left = registers[regindex];
                        udi_ubit32_t right = registers[instruction.operand & 0x7];
                        switch (instruction.tran_size)
                        {
                            case 0: registers[regindex] = (udi_ubit8_t)left  + (udi_ubit8_t)right; break;
                            case 1: registers[regindex] = (udi_ubit16_t)left + (udi_ubit16_t)right; break;
                            case 2: registers[regindex] = (udi_ubit32_t)left + (udi_ubit32_t)right; break;
                            default:
                                bochs_puts("[assert E]");
                                udi_assert(0);
                        }
                    }
                    break;
                    
                    case UDI_PIO_ADD_IMM: // E0
                    {
                        udi_sbit32_t value = (udi_sbit16_t)instruction.operand;
                        registers[regindex] = registers[regindex] + value;
                    }
                    break;
                        
                    case UDI_PIO_SUB: // E8
                    {
                        udi_ubit32_t left = registers[regindex];
                        udi_ubit32_t right = registers[instruction.operand & 0x7];
                        switch (instruction.tran_size)
                        {
                            case 0: registers[regindex] = (udi_ubit8_t)left  - (udi_ubit8_t)right; break;
                            case 1: registers[regindex] = (udi_ubit16_t)left - (udi_ubit16_t)right; break;
                            case 2: registers[regindex] = (udi_ubit32_t)left - (udi_ubit32_t)right; break;
                            default:
                                bochs_puts("[assert F]");
                                udi_assert(0);
                        }
                    }
                    break;
                    
                default:
                    bochs_puts("[assert 9]");
                    udi_assert(0);
            }
        }
        else
        {
            // Class C
            switch (instruction.pio_op)
            {
                case UDI_PIO_BRANCH: // F0
                    for (instructionpointer = 0; instructionpointer < trans->opcodecount; instructionpointer++)
                    {
                        if (trans->opcodes[instructionpointer].pio_op == UDI_PIO_LABEL && trans->opcodes[instructionpointer].operand == instruction.operand)
                        {
                            break;
                        }
                    }
                    break;
                
                case UDI_PIO_LABEL: // F1
                    // nothing to do here
                    break;
                
                case UDI_PIO_END: // FE
                    udi_assert(instruction.tran_size == 1);
                    udi_assert(instruction.operand < 8);
                    callback(gcb, buf, UDI_OK, registers[instruction.operand & 0x7]);
                    return;
                    
                case UDI_PIO_END_IMM: // FF
                    callback(gcb, buf, UDI_OK, instruction.operand);
                    return;
                    
                default:
                    bochs_puts("[assert A]");
                    udi_assert(0);
            }
        }
    }

    bochs_puts("[assert B]");
    udi_assert(0);
}
