/**
* Summary: genemit.c
* contains the assembly emitter for compiled udiprops
*
* Author:
*     Marcel Sondaar
*
* License:
*     <Public Domain>
*  
*/


#include "udigen.h"

int emit_asm(FILE * f, udi_udiprops_parse_t * parse, int arch)
{
    int i;
    
    if (arch != ARCH_I386)
    {
        fprintf(stderr, "Architecture not implemented\n");
        return 2;
    }

    // header
    switch(arch)
    {
        case ARCH_I386:
            fprintf(f, "; Generated by udigen\n"
                       "; Driver name: %s\n\n"
                       "SECTION .rodata\n"
                       "global udi_props_meta_list\n"
                       "global udi_props_device_list\n"
                       "udi_props_meta_list: DD .list\n\n"
                       , parse->shortname);
            break;
        
        default:
            ;
    }
    
    // imports
    udiprops_module_tuple ** modules = parse->modules;
    while (modules != NULL && *modules != NULL)
    {
        udiprops_region_tuple ** regions = (*modules)->regions;
        while (regions != NULL && *regions != NULL)
        {
            udiprops_metalanguage_tuple ** metas = (*regions)->metalanguages;
            while (metas != NULL && *metas != NULL)
            {                
                switch(arch)
                {
                    case ARCH_I386:
                        fprintf(f, "; metalanguage %s\n"
                                   "extern %s_meta_info\n"
                                   "extern %s_meta_pointers\n"
                                   "extern %s_meta_protos\n",
                                    (*metas)->meta_name,
                                    (*metas)->meta_name,
                                    (*metas)->meta_name,
                                    (*metas)->meta_name );
                        break;
                        
                    default:
                        ;
                }
                metas++;
            }
            regions++;
        }        
        modules++;
    }
    
    
    // prefix binding data
    switch (arch)
    {
        case ARCH_I386:
            fprintf(f, "\n\n; bindings follow:\n.list:\n");
        
        default:
            ;
    }
    
    // emit binding data
    udiprops_bind_tuple ** bindings = parse->parent_bindings;
    int parent = 1;
    if (!bindings)
    {
        bindings = parse->child_bindings;
        parent--;
    }
    while (bindings != NULL && *bindings != NULL)
    {
        // locate corresponding binding
        udiprops_bind_tuple * bind = *bindings;
        if (!parse->modules) 
        {
            fprintf(stderr, "Error: modules missing\n");
            return 1;
        }
        udiprops_module_tuple * module = parse->modules[0];
        udiprops_region_tuple ** regions = module->regions;
        while (1)
        {
            if (regions == NULL || *regions == NULL)
            {
                fprintf(stderr, "Error: undefined region number %i\n", bind->region_index);
                return 1;
            }
            if ((*regions)->index == bind->region_index) break;
            regions++;
        }
        udiprops_metalanguage_tuple ** metas = (*regions)->metalanguages;
        while (1)
        {
            if (metas == NULL || *metas == NULL)
            {
                fprintf(stderr, "Error: undefined metalanguage %i\n", bind->meta_index);
                return 1;
            }
            if ((*metas)->binding == bind->meta_index) break;
            metas++;
        }
        
        // emit data
        switch(arch)
        {
            case ARCH_I386:
                fprintf(f, "    DD %s_meta_info\n    DD %s_meta_pointers\n    DD %s_meta_protos\n    DD %i\n\n", 
                        (*metas)->meta_name,
                        (*metas)->meta_name,
                        (*metas)->meta_name,
                        bind->meta_index);
                break;
            default:
                ;
        }        
        
        bindings++;
        if (parent && *bindings == NULL)
        {
            bindings = parse->child_bindings;
            parent--;
        }
    }
    // suffix bind data
    switch (arch)
    {
        case ARCH_I386:
            fprintf(f, "    DD 0, 0, 0, 0\n\n");
        
        default:
            ;
    }        
    
    // emit device data
    udiprops_device_tuple ** devices = parse->devices;
    int device_count = 0;    
    
    while (devices != NULL && *devices != NULL)
    {
        udiprops_device_tuple * device = *devices;
    
        int constraint_count = 0;
        udiprops_keyconstraint_tuple ** constraints = device->constraints;
        while (constraints != NULL && *constraints != NULL)
        {   
            udiprops_keyconstraint_tuple * constraint = *constraints;
            
            switch(arch)
            {
                case ARCH_I386:
                    fprintf(f, "dc_%i_%i_key: DB \"%s\", 0\n", device_count, constraint_count, constraint->key);
                    fprintf(f, "dc_%i_%i_val: DB \"%s\", 0\n", device_count, constraint_count, constraint->value);
                    break;
                    
                default:
                    ;
            }
            
            constraint_count++;
            constraints++;
        }
        
        switch (arch)
        {
            case ARCH_I386:
                fprintf(f, "device_%i:   DD %i, %i \n", device_count, device->devicename, device->metaname);
                for (i = 0; i < constraint_count; i++)
                {
                    fprintf(f, "            DD dc_%i_%i_key, dc_%i_%i_val\n", device_count, i, device_count, i);
                }
                fprintf(f, "            DD 0, 0\n");
                break;
                
            default:
                ;
        }
        
        device_count++;
        devices++;        
    }
    
    switch (arch)
    {
        case ARCH_I386:
            fprintf(f, "udi_props_device_list: \n");
            for (i = 0; i < device_count; i++)
            {
                fprintf(f, "    DD device_%i \n", i );
            }
            fprintf(f, "    DD 0\n");
            break;
            
        default:
            ;
    }
    
    
    
    fflush(f);
    return 0;
}
