/**
* Summary: genfree.c
* contains the cleanup code
*
* Author:
*     Marcel Sondaar
*
* License:
*     <Public Domain>
*  
*/


#include "udigen.h"
#include <stdlib.h>

static void poplist(void ** pointer, void(*freefunc)(void*))
{
    if (pointer == NULL) return;
    void ** oldpointer = pointer;

    while (*pointer)
    {
        freefunc(*pointer);
        pointer++;
    }
    free(oldpointer);
}

void udi_free_udiprops_parse(udi_udiprops_parse_t * parse)
{
    if (!parse) return;
    
    if(parse->contact) free(parse->contact);
    if(parse->shortname) free(parse->shortname);
    if(parse->release_str) free(parse->release_str);
  
    poplist((void**)parse->messages, &free_message_tuple);
    poplist((void**)parse->devices, &free_device_tuple);
    poplist((void**)parse->configurations, &free_configchoice_tuple);
    poplist((void**)parse->modules, &free_module_tuple);
    poplist((void**)parse->global_dependencies, &free_dependency_tuple);
    poplist((void**)parse->parent_bindings, &free_bind_tuple);
    poplist((void**)parse->child_bindings, &free_bind_tuple);
    poplist((void**)parse->source_sets, &free_sourceset_tuple);
    poplist((void**)parse->symbols, &free_symbol_tuple);
    free(parse);
}

void free_module_tuple(void *ptr)
{
    udiprops_module_tuple * t = (udiprops_module_tuple *)ptr;
    if (!t) return;
    
    if (t->module_name) free(t->module_name);
    poplist((void**)t->regions, &free_region_tuple);
    poplist((void**)t->dependencies, &free_dependency_tuple);
    free(t);
}

void free_region_tuple(void * ptr)
{
    udiprops_region_tuple * t = (udiprops_region_tuple *) ptr;
    if (!t) return;
    poplist((void**) t->metalanguages, &free_metalanguage_tuple);
    free(t);
}

void free_sourceset_tuple(void * ptr)
{
    udiprops_sourceset_tuple * t = (udiprops_sourceset_tuple *)ptr;
    if (!t) return;
    if (t->compile_options) free(t->compile_options);
    poplist((void**) t->source_files, &free);
    free(t);
}

void free_bind_tuple(void * ptr)
{
    if (ptr) free(ptr);
}

void free_metalanguage_tuple(void * ptr)
{
    udiprops_metalanguage_tuple * t = (udiprops_metalanguage_tuple *)ptr;
    if (!t) return;
    if (t->meta_name) free(t->meta_name);
    free(t);
}

void free_dependency_tuple(void * ptr)
{
    udiprops_dependency_tuple * t = (udiprops_dependency_tuple*) ptr;
    if (!t) return;
    if (t->dependency_name) free(t->dependency_name);
    free(t);
}

void free_device_tuple(void * ptr)
{
    udiprops_device_tuple * t = (udiprops_device_tuple*) ptr;
    if (!t) return;
    poplist((void**) t->constraints, &free_keyconstraint_tuple);
    free(t);
}

void free_configchoice_tuple(void * ptr)
{
    udiprops_configchoice_tuple * t = (udiprops_configchoice_tuple*) ptr;
    if (!t) return;
    poplist((void**)t->config_items, &free_configitem_tuple);
    free(t);
}

void free_configitem_tuple(void * ptr)
{
    udiprops_configitem_tuple * t = (udiprops_configitem_tuple *) ptr;
    if (!t) return;
    free_keyconstraint_tuple(t->key_and_default);
    poplist((void**)t->value_alternatives, &free);
    free(t);
}

void free_keyconstraint_tuple(void * ptr)
{
    udiprops_keyconstraint_tuple * t = (udiprops_keyconstraint_tuple *) ptr;
    if (!t) return;
    if (t->key) free(t->key);
    if (t->value) free(t->value);
    free(t);
}

void free_message_tuple(void * ptr)
{
    udiprops_message_tuple * t = (udiprops_message_tuple *) ptr;
    if (!t) return;
    if (t->content) free(t->content);
    if (t->locale) free(t->locale);
    free(t);
}

void free_symbol_tuple(void * ptr)
{
    udiprops_symbol_tuple * t = (udiprops_symbol_tuple *) ptr;
    if (!t) return;
    if (t->export_name) free(t->export_name);
    if (t->real_name) free(t->real_name);
    free(t);
}
