/* Summary: genparse.c
 * Parses an udiprops file to a memory structure.
 *
 * Author:
 *     Marcel "(Com)Buster" Sondaar
 *
 * License:
 *     <Public Domain>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "udigen.h"

udi_udiprops_parse_t * udi_parse_propsfile(FILE * f)
{
    if (!f) return NULL;
    
    udi_udiprops_parse_t * parse = calloc(sizeof(udi_udiprops_parse_t), 1);
    if (!parse) return NULL;
    
    int rb = 0;
    unsigned char readbuf[512];
    unsigned char parsebuf[512];
    int linecounter = 0;
    
    int errors = 0;
    
    while ((!feof(f) || rb > 0) && errors < 10)
    {
        // get up to 512 bytes in the buffer
        if (!feof(f))
        {
            int bytes_read = fread(readbuf + rb, 1, 512-rb, f);
            rb += bytes_read;
        }
        
        int bytes_parsed = 0;
        int bytes_out = 0;
        int state = 0;
        int done = 0;
        while (bytes_parsed < rb && done == 0)
        {            
            unsigned char b = readbuf[bytes_parsed];
            bytes_parsed++;
            
            //fprintf(stderr, "state: %i, rb: %i, parsed: %i, out: %i, byte: %i\n", state, rb, bytes_parsed, bytes_out, (int)b);
            
            if (b == '\\' && state == 0)
            {
                state = 1; // escape next
            }
            else if (b == '\\' && state == 1)
            {
                state = 0; // default
                parsebuf[bytes_out++] = b;
            }
            else if (b == 0x0D && (state == 0 || state == 4))
            {
                state = 2;
            }
            else if (b == 0x0D && state == 1)
            {
                state = 3;
            }
            else if (b == 0x0A && (state == 0 || state == 2 || state == 4))
            {                
                parsebuf[bytes_out++] = 0;
                linecounter++;
                done = 1;
            }
            else if (b == 0x0A && (state == 1 || state == 3))
            {
                parsebuf[bytes_out++] = ' ';
                linecounter++;
                state = 0; // escaped newline
            }            
            else if (b == '#' && (state == 0 || state == 2))
            {
                state = 4; // comment
            }
            else if (state == 2 || state == 3)
            {
                fprintf(stderr, "Parse error: stray 0x0D on line %i\n", linecounter);
                udi_free_udiprops_parse(parse);
                return NULL;
            }
            else if (b == 0x09 && state == 0)
            {
                parsebuf[bytes_out++] = ' '; // convert TABs to whitespace
            }
            else if (b == 0x09 && (state == 1 || state == 4))
            {
                // ignored tab
            }
            else if ((b < 0x20) || (b == 0x7F))
            {
                fprintf(stderr, "Parse error: stray 0x%2x on line %i\n", (unsigned int) b, linecounter);
                udi_free_udiprops_parse(parse);
                return NULL;
            }
            else if (state == 1)
            {
                fprintf(stderr, "Parse error: unrecognised escape \\0x%2x on line %i\n", (unsigned int) b, linecounter);
                udi_free_udiprops_parse(parse);
                return NULL;
            }
            else if (state == 0)
            {
                parsebuf[bytes_out++] = b;
            }
            else if (state == 4)
            {             
                // comment
            }
            else
            {
                fprintf(stderr, "Internal parser error: state %i, input 0x%2x, line %i, \n", state, (unsigned int) b, linecounter);
                udi_free_udiprops_parse(parse);
                return NULL;
            }
        }
        
        if (done == 0)
        {
            fprintf(stderr, "Parse error: line %i: length exceeds limits\n", linecounter);
            udi_free_udiprops_parse(parse);
                return NULL;
        }
        else
        {
            errors += udi_parse_line(parse, parsebuf, linecounter);
        }
    
        if (bytes_parsed < 512)
        {
            memmove(readbuf, readbuf + bytes_parsed, 512 - bytes_parsed);            
        }
        rb -= bytes_parsed;
    }

    if (errors)
    {
        if (errors == 10)
        {
            fprintf(stderr, "Error limit reached, parsing stopped\n");
        }
        udi_free_udiprops_parse(parse);
        return NULL;
    }
    return parse;
}


//
// textual parsing tools
//
typedef struct bstring
{
    const unsigned char * off;
    unsigned int len;
} bstring;

// bstring tools
udiprops_keyconstraint_tuple * udi_parse_keyconstraint (bstring * base, bstring * after, int * errors, int linecounter);
static void trim (bstring * s);
static void split(bstring * input, bstring * left, bstring * right);
static int bstrint(bstring * input, int * error, int linecounter);
static void pushobj(void *** base, void * added, int * error, void(*freefunc)(void*));

static void trim (bstring * s)
{
    if (!s->len) return;
    while (*(s->off) == ' ')
    {
        s->off++;
        s->len--;
        if (!s->len) return;
    }
    while (s->off[s->len - 1] == ' ')
    {
        s->len--;
        if (!s->len) return;
    }
}

static void split(bstring * input, bstring * left, bstring * right)
{
    left->off = input->off;
    left->len = input->len;
    right->len = 0;
    right->off = NULL;
    
    void * divptr = memchr(input->off, ' ', input->len);
    if (divptr)
    {
        int off = (const unsigned char *)divptr - input->off;
        right->off = input->off + off;
        right->len = input->len - off;
        left->len = off;
        trim(left);
        trim(right);
    }
}

static inline int bstrequal(bstring * input, const char * reference)
{
    if (input->len != strlen(reference))
        return 0;
    if (memcmp(reference, input->off, input->len) != 0)
        return 0;
    return 1;
}

static int bstrint(bstring * input, int * error, int linecounter)
{
    if (input->len == 0)
    {        
        (*error)++;
        fprintf(stderr, "Line %i: missing numerical argument", linecounter);
        return 0;
    }
    
    int n = 0;
    int ord = 10;
    unsigned int i;
    for (i = 0; i < input->len; i++)
    {
        if( input->off[i] >= '0' && input->off[i] <= '9' && (input->off[i] - '0' < ord))
        {
            n = n * ord + (input->off[i] - '0');
        }
        else if ( (input->off[i] == 'b' || input->off[i] == 'B') && i == 1 && n == 0)
        {
            ord = 2;
        }
        else if ( input->off[i] >= 'a' && input->off[i] <= 'f' && ord == 16)
        {
            n = n * ord + (input->off[i] - 'a' + 10);
        }
        else if ( input->off[i] >= 'A' && input->off[i] <= 'O' && ord == 16)
        {
            n = n * ord + (input->off[i] - 'A' + 10);
        }
        else if ( (input->off[i] == 'o' || input->off[i] == 'O') && i == 1 && n == 0)
        {
            ord = 8;
        }
        else if ( (input->off[i] == 'x' || input->off[i] == 'X') && i == 1 && n == 0)
        {
            ord = 16;
        }
        else
        {
            (*error)++;
            fprintf(stderr, "Line %i: invalid format for numerical argument", linecounter);
        }
    }
    return n;
}
 
// add an object to a list
static void pushobj(void *** base, void * added, int * error, void(*freefunc)(void*))
{
    if (!*base)
    {
        *base = (void **) malloc(2 * sizeof(void *));
        if (!*base)
        {
            (*error)++;
            fprintf(stderr, "Out of memory");
            if (freefunc) freefunc(added);
            return;
        }
        (*base)[0] = added;
        (*base)[1] = NULL;
        
        return;
    }
    
    int count = 0;
    while ((*base)[count] != NULL) count++;
    
    void ** newblock = (void**)realloc(*base, (count + 2) * sizeof(void *));
    if (!newblock)
    {
        (*error)++;
        fprintf(stderr, "Out of memory");
        if (freefunc) freefunc(added);
    }
    else
    {
        newblock[count] = added;
        newblock[count + 1] = NULL;
        *base = newblock;
    }
}


udiprops_keyconstraint_tuple * udi_parse_keyconstraint(bstring * base, bstring * after, int * errors, int linecounter)
{
    bstring devkey, devtype, devval;
    split(base, &devkey, &devtype);
    split(&devtype, &devtype, &devval);
    split(&devval, &devval, after);
    int type = 0;
    if (bstrequal(&devtype, "string"))
    {
        type = TYPE_STRING;
    } 
    else if (bstrequal(&devtype, "ubit32"))
    {
        type = TYPE_UBIT32;
    }
    else if (bstrequal(&devtype, "boolean"))
    {
        type = TYPE_BOOL;
    }
    else if (bstrequal(&devtype, "array"))
    {
        type = TYPE_BYTEARRAY;
    }
    
    if (type == 0)
    {
        fprintf(stderr, "Line %i: Unknown type: ", linecounter);
        fwrite(devtype.off, devtype.len, 1, stderr);
        fputs("\n", stderr);
    }
    else if (devkey.len == 0 || devtype.len == 0 || devval.len == 0)
    {
        fprintf(stderr, "Line %i: Missing argument", linecounter);
    }
    else
    {
        char * copy1 = malloc(devkey.len + 1);
        char * copy2 = malloc(devval.len + 1);
        udiprops_keyconstraint_tuple * constraint = malloc(sizeof(udiprops_keyconstraint_tuple));
        if (copy1 && copy2 && constraint)
        {
            memcpy(copy1, devkey.off, devkey.len);
            memcpy(copy2, devval.off, devval.len);
            copy1[devkey.len] = 0;
            copy2[devval.len] = 0;
            constraint->type = type;
            constraint->key = copy1;
            constraint->value = copy2;
            return constraint;
        }
        else
        {
            fprintf(stderr, "Out of memory\n");            
        }
    }
    
    (*errors)++;
    
    return NULL;
}





int udi_parse_line(udi_udiprops_parse_t * parse, const unsigned char * line, int linecounter)
{
    bstring baseline = {line, strlen((const char*)line)};
    trim(&baseline);
    
    bstring field1, field2;
    split(&baseline, &field1, &field2);
    
    int errors = 0;
    
    //fwrite(field1.off, field1.len, 1, stderr);
    
    if (bstrequal(&field1, ""))
    {
        // whitespace
    }
    else if (bstrequal(&field1, "properties_version"))
    {
        int version = bstrint(&field2, &errors, linecounter);
        parse->ver_major = version >> 8;
        parse->ver_minor = version & 0xff;
    }
    else if (parse->ver_major == 0 && parse->ver_minor == 0)
    {
        fprintf(stderr, "Line %i: first command is not properties_version\n", linecounter);
        errors++;
    }
    else if (bstrequal(&field1, "message") || bstrequal(&field1, "disaster_message"))
    {
        int disaster = bstrequal(&field1, "disaster_message");
        
        bstring field3;
        split(&field2, &field2, &field3);        
        int val = bstrint(&field2, &errors, linecounter);        
        if (val > 0 && field2.len > 0 && !errors)
        {
            udiprops_message_tuple * info = malloc(sizeof(udiprops_message_tuple));
            unsigned char * clone = malloc(field2.len + 1);
            if (!clone || !info)
            {
                fprintf(stderr, "Out of memory");
                errors++;                
            }
            else
            {
                memcpy(clone, field2.off, field2.len);
                clone[field2.len] = 0;
                info->number = val;
                info->content = clone;
                info->disaster = disaster;
                info->locale = NULL;
                pushobj((void***)&(parse->messages), (void*)info, &errors, &free_message_tuple);
            }
        }
        else if (!errors && val == 0)
        {
            fprintf(stderr, "Line %i: Invalid message number\n", linecounter);
            errors++;
        }
        else if (!errors)
        {
            fprintf(stderr, "Line %i: Message missing argument\n", linecounter);
            errors++;
        }
    }
    else if (bstrequal(&field1, "supplier"))
    {
        int val = bstrint(&field2, &errors, linecounter);
        if (val == 0 || parse->supplier)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else parse->supplier = val;
    }
    else if (bstrequal(&field1, "contact"))
    {
        int val = bstrint(&field2, &errors, linecounter);
        if (val == 0)
        {
            fprintf(stderr, "Line %i: Invalid message number\n", linecounter);
            errors++;
        }
        else pushobj((void ***) &(parse->contact), (void *)((intptr_t)val), &errors, NULL);
    }
    else if (bstrequal(&field1, "name"))
    {
        int val = bstrint(&field2, &errors, linecounter);
        if (val == 0 || parse->name)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else parse->name = val;
    }
    else if (bstrequal(&field1, "shortname"))
    {
        unsigned char * copy = malloc(field2.len + 1);
        if (field2.len == 0 || parse->shortname != NULL)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
            if (copy) free(copy);
        }
        else if (copy == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;            
        }
        else 
        {
            memcpy(copy, field2.off, field2.len);
            copy[field2.len] = 0;
            parse->shortname = copy;
        }
    }
    else if (bstrequal(&field1, "category"))
    {
        int val = bstrint(&field2, &errors, linecounter);
        if (val == 0 || parse->category)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else parse->category = val;
    }
    else if (bstrequal(&field1, "release"))
    {
        bstring field3;
        split(&field2, &field2, &field3);
        unsigned char * copy = malloc(field3.len + 1);
        int val = bstrint(&field2, &errors, linecounter);
        if (field3.len == 0 || parse->release_str != NULL || val == 0)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
            if (copy) free(copy);
        }
        else if (copy == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;            
        }
        else 
        {
            memcpy(copy, field3.off, field3.len);
            copy[field3.len] = 0;
            parse->release_str = copy;
        }
    }
    else if (bstrequal(&field1, "device"))
    {
        bstring field3, remainder;
        split(&field2, &field2, &field3);
        split(&field3, &field3, &remainder);
        int devname = bstrint(&field2, &errors, linecounter);
        int metaname = bstrint(&field3, &errors, linecounter);
        udiprops_device_tuple * device = malloc(sizeof(udiprops_device_tuple));
        
        if (devname == 0 || metaname == 0 || remainder.len == 0)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
            if (device) free(device);
        }
        else if (device == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;            
        }
        else 
        {
            device->devicename = devname;
            device->metaname = metaname;
            device->constraints = NULL;
            while (remainder.len)
            {
                udiprops_keyconstraint_tuple * constraint = udi_parse_keyconstraint(&remainder, &remainder, &errors, linecounter);
                if (constraint)
                    pushobj((void***) &(device->constraints), constraint, &errors, &free_keyconstraint_tuple);
            }
            
            pushobj( (void***) &(parse->devices), device, &errors, &free_device_tuple);
        }
        
    }
    else if (bstrequal(&field1, "config_choices"))
    {
        bstring remainder, range;
        split(&field2, &field2, &remainder);
        int devname = bstrint(&field2, &errors, linecounter);
        udiprops_configchoice_tuple * config;
        if (devname == 0 || remainder.len == 0)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else if ((config = (udiprops_configchoice_tuple *) malloc(sizeof(udiprops_configchoice_tuple))) == NULL)
        {
            fprintf(stderr, "Out of memory");
            errors++;            
        }
        else
        {
            config->devicename = devname;
            config->config_items = NULL;
            while (remainder.len)
            {
                udiprops_keyconstraint_tuple * constraint = udi_parse_keyconstraint(&remainder, &range, &errors, linecounter);
                udiprops_configitem_tuple * option = malloc(sizeof(udiprops_configitem_tuple));
                
                if (!option)
                {
                    fprintf(stderr, "Out of memory");
                    errors++;            
                    if (constraint) free(constraint);
                }
                else if (!constraint)
                {
                    if (option) free(option);                    
                }
                else if (range.len == 0)
                {
                    fprintf(stderr, "Line %i: Missing constraint keyword\n", linecounter);
                    errors++;
                    if (option) free(option);
                    if (constraint) free(constraint);
                }
                else
                {
                    option->key_and_default = constraint;
                    option->min = 0;
                    option->max = 0;
                    option->stride = 0;
                    option->value_alternatives = NULL;
                    option->optionmode = 0;                    
                    split(&range, &range, &remainder);
                    
                    if (bstrequal(&range, "any"))
                    {
                        option->optionmode = OPTION_ANY;
                    }
                    else if (bstrequal(&range, "only"))
                    {
                        option->optionmode = OPTION_ONLY;
                    } 
                    else if (bstrequal(&range, "range"))
                    {
                        bstring r1, r2, rs;
                        split(&remainder, &r1, &r2);
                        split(&r2, &r2, &rs);
                        split(&rs, &rs, &remainder);
                        option->min = bstrint(&r1, &errors, linecounter);
                        option->max = bstrint(&r2, &errors, linecounter);
                        option->stride = bstrint(&rs, &errors, linecounter);
                        option->optionmode = OPTION_RANGE;
                    }
                    else
                    {
                        fprintf(stderr, "Line %i: Unknown constraint keyword: ", linecounter);
                        fwrite(range.off, range.len, 1, stderr);
                        fputs("\n", stderr);
                    }
                    
                    pushobj( (void***) &(config->config_items), option, &errors, &free_configitem_tuple);
                }
            }
            pushobj( (void***) &(parse->configurations), config, &errors, &free_configchoice_tuple);
        }
        
    }
    else if (bstrequal(&field1, "module"))
    {
        unsigned char * copy = malloc(field2.len + 1);
        udiprops_module_tuple * module = (udiprops_module_tuple*) malloc(sizeof(udiprops_module_tuple));
        if (field2.len == 0)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
            if (copy) free(copy);
            if (module) free(module);
        }
        else if (copy == NULL || module == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;
            if (copy) free(copy);
            if (module) free(module);
        }
        else 
        {
            memcpy(copy, field2.off, field2.len);
            copy[field2.len] = 0;
            module->module_name = copy;
            pushobj((void***) &(parse->modules), module, &errors, &free_module_tuple);
            if (!errors) 
            {
                parse->active_module = module;
                parse->active_region = NULL;
            }
        }
    }
    else if (bstrequal(&field1, "region"))
    {
        int val = bstrint(&field2, &errors, linecounter);
        if (errors != 0 || field2.len == 0 || parse->active_module == NULL )
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else 
        {
            udiprops_region_tuple * region = (udiprops_region_tuple *) calloc(sizeof(udiprops_region_tuple), 1);
            if (region)
            {
                region->index = val;
                pushobj((void***) &(parse->active_module->regions), region, &errors, &free_region_tuple);
                if (!errors)
                {
                    parse->active_region = region;
                }
            }
            else
            {
                fprintf(stderr, "Out of memory\n");
                errors++;
            }            
        }
    }
    else if (bstrequal(&field1, "requires"))
    {
        bstring field3;
        split(&field2, &field2, &field3);
        int val = bstrint(&field3, &errors, linecounter);
        udiprops_dependency_tuple * dependency = (udiprops_dependency_tuple *) calloc(sizeof(udiprops_dependency_tuple), 1);
        unsigned char * copy = malloc(field2.len + 1);
        if (errors != 0 || field2.len == 0 || val == 0)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else if (copy == NULL || dependency == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;
            if (copy) free(copy);
            if (dependency) free(dependency);
        }
        else
        {
            memcpy(copy, field2.off, field2.len);
            copy[field2.len] = 0;
            dependency->version = val;
            dependency->dependency_name = copy;
            
            if (parse->active_module)
            {
                pushobj((void***) &(parse->active_module->dependencies), dependency, &errors, &free_dependency_tuple);
            }
            else
            {
                pushobj((void***) &(parse->global_dependencies), dependency, &errors, &free_dependency_tuple);
            }
        }
    }
    else if (bstrequal(&field1, "provides"))
    {
        bstring field3;
        split(&field2, &field2, &field3);
        int val = bstrint(&field3, &errors, linecounter);
        udiprops_dependency_tuple * dependency = (udiprops_dependency_tuple *) calloc(sizeof(udiprops_dependency_tuple), 1);
        unsigned char * copy = malloc(field2.len + 1);
        if (errors != 0 || field2.len == 0 || val == 0 || parse->active_module == NULL)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            if (copy) free(copy);
            if (dependency) free(dependency);
            errors++;
        }
        else if (copy == NULL || dependency == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;
            if (copy) free(copy);
            if (dependency) free(dependency);
        }
        else
        {
            memcpy(copy, field2.off, field2.len);
            copy[field2.len] = 0;
            dependency->version = val;
            dependency->dependency_name = copy;
            
            pushobj((void***) &(parse->active_module->exports), dependency, &errors, &free_dependency_tuple);
        }
    }
    else if (bstrequal(&field1, "meta"))
    {
        bstring field3;
        split(&field2, &field2, &field3);
        int val = bstrint(&field2, &errors, linecounter);
        udiprops_metalanguage_tuple * metalanguage = (udiprops_metalanguage_tuple *) calloc(sizeof(udiprops_metalanguage_tuple), 1);
        unsigned char * copy = malloc(field3.len + 1);
        if (errors != 0 || field3.len == 0 || val == 0 || parse->active_region == NULL)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else if (copy == NULL || metalanguage == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            errors++;
            if (copy) free(copy);
            if (metalanguage) free(metalanguage);
        }
        else
        {
            memcpy(copy, field3.off, field3.len);
            copy[field3.len] = 0;
            metalanguage->binding = val;
            metalanguage->meta_name = copy;
            
            pushobj((void***) &(parse->active_region->metalanguages), metalanguage, &errors, &free_metalanguage_tuple);
        }
    }
    else if (bstrequal(&field1, "child_bind_ops") || bstrequal(&field1, "parent_bind_ops"))
    {
        bstring field3, field4, field5;
        split(&field2, &field2, &field3);
        split(&field3, &field3, &field4);
        split(&field4, &field4, &field5);
        int meta = bstrint(&field2, &errors, linecounter);
        int region = bstrint(&field3, &errors, linecounter);
        int ops = bstrint(&field4, &errors, linecounter);
        int cb = 0;
        udiprops_bind_tuple * binding = malloc(sizeof(udiprops_bind_tuple));
        
        if (parse->ver_major >= 1 && bstrequal(&field1, "parent_bind_ops"))
        {
            // required for parents in latest UDI
            cb = bstrint(&field5, &errors, linecounter);
        }
        
        if (errors)
        {
            // message already printed
            if (binding) free(binding);
        }
        else if (
                 meta == 0 
              || ops == 0 
              || field3.len == 0 
              || (field5.len == 0 && parse->ver_major >= 1 && bstrequal(&field1, "parent_bind_ops")) 
              || (field5.len > 0 && parse->ver_major == 0)
              || (field5.len > 0 && !bstrequal(&field1, "parent_bind_ops"))
                )
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            if (binding) free(binding);
            errors++;
        }        
        else if (binding == NULL)
        {            
            fprintf(stderr, "Out of memory\n");
            errors++;            
        }
        else
        {
            binding->meta_index = meta;
            binding->region_index = region;
            binding->ops_index = ops;
            binding->cb_index = cb;
            if (bstrequal(&field1, "parent_bind_ops"))
            {
                pushobj((void***)&(parse->parent_bindings), binding, &errors, &free_bind_tuple);
            } 
            else
            {
                pushobj((void***)&(parse->child_bindings), binding, &errors, &free_bind_tuple);
            }
        }
    }  
    else if (bstrequal(&field1, "symbols"))
    {
        bstring field3, field4, field5;
        field4.len = 0; field5.len = 0;
        split(&field2, &field2, &field3);
        if (field3.len > 0)
        {
            split(&field3, &field3, &field4);
            split(&field4, &field4, &field5);
        }
        if (field5.len != 0 || (field3.len > 0 && field4.len == 0) || (field3.len > 0 && !bstrequal(&field3, "as")))
        {
            fprintf(stderr, "%i %i %i %i", field2.len, field3.len, field4.len, field5.len);
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        }
        else
        {
            unsigned char * part1 = malloc(field2.len + 1);
            unsigned char * part2 = NULL;
            udiprops_symbol_tuple * symbol = (udiprops_symbol_tuple *) malloc(sizeof(udiprops_symbol_tuple));
            if (field3.len > 0)
            {
                part2 = part1;
                part1 = malloc(field4.len + 1);
                if (!part1 || !part2)
                {
                    fprintf(stderr, "Out of memory\n");
                    errors++;
                    if (part1) free(part1);
                    if (part2) free(part2);
                }
                else
                {
                    memcpy(part2, field2.off, field2.len);
                    part2[field2.len] = 0;
                    memcpy(part1, field4.off, field4.len);
                    part1[field4.len] = 0;
                }
            } 
            else if (!part1)
            {
                fprintf(stderr, "Out of memory\n");
                errors++;                
            }
            else
            {
                memcpy(part1, field2.off, field2.len);
                part1[field2.len] = 0;
            }
            
            if (errors)
            {
                if (symbol) free(symbol);
            }
            else if (!symbol)
            {
                fprintf(stderr, "Out of memory\n");
                errors++;
                if (part1) free(part1);
                if (part2) free(part2);
            }
            else
            {
                symbol->export_name = part1;
                symbol->real_name = part2;
                pushobj((void***)&(parse->symbols), symbol, &errors, &free_symbol_tuple);
            }            
        }
    }
    else if (bstrequal(&field1, "compile_option") /*0.95*/ || bstrequal(&field1, "compile_options") /*1.01*/ )
    {
        udiprops_sourceset_tuple * sourceset = malloc(sizeof(udiprops_sourceset_tuple));
        unsigned char * copy = malloc(field2.len + 1);
        if (sourceset == NULL || copy == NULL)
        {
            fprintf(stderr, "Out of memory\n");
            if (copy) free(copy);
            if (sourceset) free(sourceset);
            errors++;            
        }
        else
        {
            memcpy(copy, field2.off, field2.len);
            copy[field2.len] = 0;
            sourceset->compile_options = copy;
            sourceset->source_files = NULL;
            pushobj((void***)&(parse->source_sets), sourceset, &errors, &free_sourceset_tuple);
            if (!errors) parse->active_sourceset = sourceset;
        }
    }
    else if (bstrequal(&field1, "source_files"))
    {
        bstring remainder = field2;
        if (parse->active_sourceset == NULL)
        {
            udiprops_sourceset_tuple * sourceset = malloc(sizeof(udiprops_sourceset_tuple));
            unsigned char * empty = malloc(1);
            if (!sourceset || !empty)
            {
                fprintf(stderr, "Out of memory\n");
                if (empty) free(empty);
                if (sourceset) free(sourceset);
                errors++;            
            }
            else
            {
                empty[0] = 0;
                sourceset->compile_options = empty;
                sourceset->source_files = NULL;
                pushobj((void***)&(parse->source_sets), sourceset, &errors, &free_sourceset_tuple);
                if (!errors) parse->active_sourceset = sourceset;
            }
        }
        
        if (remainder.len == 0 || parse->active_sourceset == NULL)
        {
            fprintf(stderr, "Line %i: Invalid directive use\n", linecounter);
            errors++;
        } 
        else 
        {
            while (remainder.len)
            {
                split(&remainder, &field2, &remainder);
                unsigned char * copy = malloc(field2.len + 1);
                if (!copy)
                {
                    fprintf(stderr, "Out of memory\n");
                }
                else
                {
                    memcpy(copy, field2.off, field2.len);
                    copy[field2.len] = 0;
                    pushobj((void***)&(parse->active_sourceset->source_files), copy, &errors, &free);
                }
            }
        }
    }
    else if (bstrequal(&field1, "rank"))    
    {
        fprintf(stderr, "Warning: Line %i: specification violation found and ignored\n", linecounter);
    }
    else
    {
        fprintf(stderr, "Line %i: Unknown directive: ", linecounter);
        fwrite(field1.off, field1.len, 1, stderr);
        fputs("\n", stderr);
        return 1;
    }
    
    return (errors ? 1 : 0);
    
}
