/* Summary: udigen.c
 * Generates the bootcode portion of an udi driver
 *
 * Author:
 *     Marcel "(Com)Buster" Sondaar
 *
 * License:
 *     <Public Domain>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "udigen.h"

int main(int argc, const char ** argv)
{
    const char * fn_in;
    const char * fn_out;

    // basic usage check
    if (argc != 4 && argc != 5)
    {
        fprintf(stderr, "Error: invalid arguments\nUsage: %s operation architecture input [output]\n", argv[0]);
        return(1);
    }

    int in_arch = 0;
    if (strcmp(argv[2], "i386") == 0)
        in_arch = ARCH_I386;
    else if (strcmp(argv[2], "ia") == 0)
        in_arch = ARCH_I386;
    else if (strcmp(argv[2], "ia32") == 0)
        in_arch = ARCH_I386;
    else
    {
        fprintf(stderr, "Error: unknown architecture %s\nKnown architectures are: i386\n", argv[2]);
        return(1);
    }

    // open input
    fn_in = argv[3];
    FILE * fin = fopen(fn_in, "r");
    if (!fin)
    {
        fprintf(stderr, "Error opening %s: %s\n", fn_in, strerror(errno));
        return(1);
    }

    // parse
    udi_udiprops_parse_t * parse = udi_parse_propsfile(fin);
    
    fclose(fin);
    
    if (!parse)
        return 1;

    // open output if specified, stdout otherwise
    FILE * fout = NULL;
    if (argc == 5)
    {
        fn_out = argv[4];
        fout = fopen(fn_out, "w");
    }
    else
    {
        fn_out = "stdout";
        fout = stdout;
    }
    if (!fout)
    {
        fprintf(stderr, "Error opening %s: %s\n", fn_in, strerror(errno));
        udi_free_udiprops_parse(parse);
        return(1);
    }
    
    // render
    int result = 0;
    if (strcmp(argv[1], "udiasm") == 0)
    {
        result = emit_asm(fout, parse, in_arch);
    }
    else
    {
        fprintf(stderr, "Error: Unknown operation %s\nKnown operations are: udiasm\n", argv[1]);
        result = 1;
    }
    
        
    if (argc == 5)
        fclose(fout);
    udi_free_udiprops_parse(parse);
    return result;
}
