' Summary: isaprobe.bas
' Detection of ISA devices that are part of the PC standard.
'
' Author:
'     Marcel Sondaar
'
' License:
'     <Public Domain>
'

#include "mos.bi"
#include "x86.bi"
#include "mos/drivercom.bi"
#include "mos/devmgr.bi"

' Devices that never have an PNP variant
Declare Function ProbeFDC(portbase As Integer) As Integer
Declare Function ProbeRTC(portbase As Integer) As Integer
Declare Function ProbePS2(portbase As Integer) As Integer
Declare Function ProbeISADMA(portbase As Integer) As Integer
Declare Function ProbeUART(portbase As Integer) As Integer
Declare Function ProbeCmos(portbase As Integer) As Integer
Declare Function ProbeParport(portbase As Integer) As Integer

' Devices that may be plug-and-play compatible
Declare Function ProbeVGA(portbase As Integer) As Integer ' can be a SVGA
Declare Function ProbePIT(portbase As Integer) As Integer ' can be an HPET
Declare Function ProbeIDE(portbase As Integer) As Integer ' can be PCI-IDE or SATA

Type DetectScript
    ProbeFunction As Function (portbase As Integer) As Integer
    ExcludeFunction As Function (portbase As Integer) As Integer
    Status as Integer
    Portbase As Integer
    Registername as String * 16
End Type


Dim Shared probelist() As DetectScript
Dim Shared probecount As Integer

Declare Sub ScheduleProbe(Devname As String, Port As Integer, Probe as Function(portbase As Integer) As Integer, Exclude as Function(portbase As Integer) As Integer)

Declare Function DetectionIdle(portbase As Integer) As Integer
Declare Function ExcludeNever(portbase As Integer) As Integer
Declare Function ExcludePciVGA(portbase As Integer) As Integer
Declare Function ExcludePciIDE(portbase As Integer) As Integer

Sub modmain CDecl Alias "main" ()

    ' Allocate port ranges
    allocateiobitmap(0, &HD00, CPtr(Byte Ptr, &HFFFFFFFF))

    ' Init communication port
    drv_setname(DRIVER_BUS, 4)
    Yield

    ScheduleProbe "ISA KBC", 0, @ExcludeNever, @ExcludeNever
    ScheduleProbe "ISA RTC", 0, @ExcludeNever, @ExcludeNever
    ScheduleProbe "ISA PIT", 0, @ExcludeNever, @ExcludeNever
    ScheduleProbe "ISA DMA", 0, @ExcludeNever, @ExcludeNever
    ScheduleProbe "ISA VGA", 0, @ProbeVGA, @ExcludePciVGA
    ScheduleProbe "ISA IDE 1F0", &H1F0, @ExcludeNever, @ExcludePciIDE
    ScheduleProbe "ISA IDE 170", &H170, @ExcludeNever, @ExcludePciIDE
    ScheduleProbe "CMOS RAM", 0, @ExcludeNever, @ExcludeNever

    Dim done As Integer
    done = 0
    While Not done
        done = 1
        Dim lp As Integer
        For lp = 1 to probecount

            If probelist(lp).status = 0 Then

                Dim status As Integer
                status = probelist(lp).ExcludeFunction(probelist(lp).Portbase)
                If status = 1 Then
                    status = probelist(lp).ProbeFunction(probelist(lp).Portbase)
                End If
                probelist(lp).status = status

                If status = 1 Then
                    Dim message(0 to 1) As Integer
                    message(0) = DEVMGRCOMMANDS_ADD
                    message(1) = 0
                    drv_sendmessage(DRIVER_MGR * &H10000, 8, CPtr(Byte Ptr, @(message(0)) )  )

                    While drv_peekmessage() = 0
                        Yield
                    Wend

                    drv_readmessage(CPtr(Byte Ptr, @(message(0)) ) )

                    Dim ws As String
                    ws = "        " & probelist(lp).RegisterName
                    Dim bp As Byte Ptr
                    bp = *Cptr(Byte Ptr Ptr, @ws)
                    Cptr(Integer Ptr, bp)[0] = DEVMGRCOMMANDS_SETNAME
                    Cptr(Integer Ptr, bp)[1] = message(0)
                    drv_sendmessage(DRIVER_MGR * &H10000, len(ws), bp)
                ElseIf status = 0 Then
                    done = 0
                End If
            End if
        Next lp
        Yield
    Wend


End Sub

Sub ScheduleProbe(Devname As String, Port As Integer, Probe as Function(portbase As Integer) As Integer, Exclude as Function(portbase As Integer) As Integer)
    probecount = probecount + 1
    Redim Preserve probelist(probecount)

    probelist(probecount).portbase = port
    probelist(probecount).status = 0
    probelist(probecount).registername = devname
    probelist(probecount).probefunction = probe
    probelist(probecount).excludefunction = exclude

End Sub

Function DetectionIdle(portbase As Integer) As Integer
    Static lastcheck As Integer = 0
    Static lastcount As Integer = 1
    Yield
    Dim i(0 to 1) as Integer
    i(0) = DEVMGRCOMMANDS_GETNAME
    i(1) = lastcount
    drv_sendmessage(DRIVER_MGR * &H10000, 8, CPtr(Byte Ptr, @(i(0))) )
    Yield
    While drv_peekmessage() = 0
        Yield
    Wend
    Dim rs as string
    Dim rv As Integer
    rv = drv_peekmessage()
    rs = space$(rv)
    drv_readmessage(*CPtr(Byte Ptr Ptr, @rs))
    function = 0
    If rv = 1 Then
        lastcheck = 0
        lastcount = lastcount + 1
    Else
        lastcheck = lastcheck + 1
        If lastcheck > 4 Then Function = 1
    End If
End Function

Function ExcludePciVGA(portbase As Integer) As Integer

    Static lastdevice As Integer = 1

    If DetectionIdle(portbase) = 0 then
        Function = 0
        Exit Function
    End If

    Dim i(0 to 1) as Integer
    i(0) = DEVMGRCOMMANDS_GETNAME
    i(1) = lastdevice
    drv_sendmessage(DRIVER_MGR * &H10000, 8, CPtr(Byte Ptr, @(i(0))) )
    Yield
    While drv_peekmessage() = 0
        Yield
    Wend

    Dim rs as string
    Dim rv As Integer
    rv = drv_peekmessage()
    rs = space$(rv)
    drv_readmessage(*CPtr(Byte Ptr Ptr, @rs))

    If rv = 22 Then
        If Right$(rs, 8) = "03:00:00" Then
            Function = -1
        ElseIf Right$(rs, 8) = "03:00:80" Then
            Function = -1
        Else
            Function = 0
            lastdevice = lastdevice + 1
        End If
        'PCI 1234:5678 ab:cd:ef
    ElseIf rv <= 1 Then
        Function = 1
    Else
        lastdevice = lastdevice + 1
        Function = 0
    End If
End Function

Function ExcludePciIDE(portbase As Integer) As Integer

    Static lastdevice As Integer = 1

    If DetectionIdle(portbase) = 0 then
        Function = 0
        Exit Function
    End If

    Dim i(0 to 1) as Integer
    i(0) = DEVMGRCOMMANDS_GETNAME
    i(1) = lastdevice
    drv_sendmessage(DRIVER_MGR * &H10000, 8, CPtr(Byte Ptr, @(i(0))) )
    Yield
    While drv_peekmessage() = 0
        Yield
    Wend

    Dim rs as string
    Dim rv As Integer
    rv = drv_peekmessage()
    rs = space$(rv)
    drv_readmessage(*CPtr(Byte Ptr Ptr, @rs))

    If rv = 22 Then
        If Left$(Right$(rs, 8), 5) = "01:01" Then 'A PCI IDE
            'Todo: check if some idiot put an PCI IDE card in a machine with an onboard ISA IDE controller
            Function = -1
        Else
            Function = 0
            lastdevice = lastdevice + 1
        End If
    ElseIf rv <= 1 Then
        Function = 1
    Else
        lastdevice = lastdevice + 1
        Function = 0
    End If
End Function

Function ExcludeNever(portbase As Integer) As Integer
    Function = 1
End Function

Function ProbeVGA(portbase As Integer) As Integer
    Portalloc(&H3B0, &H30)

    Function = 1

    Dim b1 as Byte
    Dim b2 as Byte
    Dim b3 as Byte
    Dim b4 as Byte

    ' Expects a bios-configured video card
    b4 = inportb(&H3CE)                 ' back up byte in register

    outportb(&H3CE, 6)                  ' select GC.6
    b1 = inportb(&H3CE)                 ' check if writable
    If b1 <> 6 Then Function = -1
    b2 = inportb(&H3CF)                 ' read GC.6
    outportb(&H3CE, 5)                  ' select GC.5
    b1 = inportb(&H3CE)                 ' check if writable
    If b1 <> 5 Then Function = -1
    b3 = inportb(&H3CF)                 ' read GC.5
    if b2 = b3 Then Function = -1       ' GC.5 != GC.6

    ' re-read registers, to check if they are constant
    outportb(&H3CE, 6)                  ' select GC.6
    b1 = inportb(&H3CE)                 ' check writability
    If b1 <> 6 Then Function = -1
    b1 = inportb(&H3CF)                 ' check GC.6 for constantness
    if b1 <> b2 Then Function = -1

    outportb(&H3CE, 5)                  ' select GC.5
    b1 = inportb(&H3CE)                 ' check writability
    If b1 <> 5 Then Function = -1
    b1 = inportb(&H3CF)                 ' check GC.5 for constantness
    if b1 <> b3 Then Function = -1

    outportb(&H3CE, b4)                 ' restore register

    ' dealloc ports
End Function