' Summary: r600test.bas
' Testing module for ATI R6xx graphics stuff
'
' Author:
'     Marcel Sondaar
'
' License:
'     Educational Purposes
'

#include "../gfx/vga_io.bi"
#include "mos.bi"
#include "x86.bi"
#include "mos/pci.bi"

Declare Sub ModMain Cdecl Alias "main" ()
Declare Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
Declare Function GetYesNo () As Integer
Declare Function ReadKB () as Integer
Declare Sub ClearCon (vram As Byte Ptr)
Declare Function R600_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer, ByRef mmiobase As Long) As Long

Declare Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
Declare Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
Declare Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
Declare Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)

Declare Sub TestR600(board_model As Long, board_version As Long, lfb As Long, portbase As Long, mmiobase As Long)


Public Sub ModMain Cdecl Alias "main" ()
    Dim vram As Byte Ptr
    Dim cram As Byte Ptr
    Dim bios As Byte Ptr
    Dim vaddress as Byte Ptr

    vaddress = CPtr(Byte Ptr, &HA0000)
    vram = CPtr(Byte Ptr, &HA0000)
    cram = CPtr(Byte Ptr, &HB8000)
    bios = CPtr(Byte Ptr, &HC0000)

    blockallocphys(64, vram, vram)
    allocateiobitmap(0, &HE000, CPtr(Byte Ptr, &HFFFFFFFF))
    PortAlloc(&HCF8, 8)
    PortAlloc(&H3C0, 32)
    PortAlloc(&H60, 2)

    Dim lfb as Integer
    Dim portbase as Integer
    Dim mmiobase as Integer

    Dim chiptype as Integer
    Dim chipsubtype as Integer
    Dim bioshash as Integer
    Dim lp as Long
    Dim devname As String

    ClearCon cram
    PrintString "BASIC For teh win (>O.O)>", cram, 25 * 80 - 27

    PrintString "Ati R6xx test program", cram, 0
    PrintString "Searching PCI device list for a compatible ATI board...", cram, 80

    chiptype = R600_PCIProbe(lfb, portbase, mmiobase)
    PrintString "Hashing BIOS...", cram, 160
    bioshash = 0
    for lp = 0 to 32767
        bioshash = bioshash + ((16381 * lp + 1) * CLng(bios[lp]))
    next lp
    PrintString "Probe result: " + Hex$(chiptype), cram, 240
    If chiptype > 0 Then

        Select case chiptype

            Case &H94C1
                devname = "[RV610] Radeon HD 2400 XT"
                chipsubtype = 610
            Case &H94C3
                devname = "[RV610] Radeon HD 2400 PRO"
                chipsubtype = 610
            Case &H94C4
                devname = "[RV610] Radeon HD 2400 PRO AGP"
                chipsubtype = 610

            Case Else
                devname = "Unknown (0x" + hex$(chiptype) + ")"
                chipsubtype = 600

        End Select

        PrintString "Model: " + devname, cram, 400

        PrintString "GPU: " + str$(chipsubtype), cram, 320


        PrintString "LFB Address: 0x" + hex$(lfb), cram, 480
        PrintString "Port Base:   0x" + hex$(portbase), cram, 560
        PrintString "MMIO Base:   0x" + hex$(mmiobase), cram, 640

        PrintString "Press Y to start the test at your hardware's peril", cram, 720
        PrintString "Press N to skip this test", cram, 800

        If GetYesNo() = 1 Then
            TestR600(chiptype, chipsubtype, lfb, portbase, mmiobase)
        End If

        ClearCon cram
        PrintString "Test Ended. Please hit the power or the reset button", cram, 0

    Else
        PrintString "No R6xx board found", cram, 320
        PrintString "Test Ended. Please hit the power or the reset button", cram, 400
    End if
    While 1 = 1

    Wend

End Sub

Public Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        vram[lp * 2 - 2 + 2 * offset] = ch
        vram[lp * 2 - 1 + 2 * offset] = 7
    Next lp

End Sub

Public Function GetYesNo () As Integer
    ' Y = &H15
    ' N = &H31

    Dim status As Integer = -1
    Dim key As Integer
    While status = -1
        key = ReadKB
        if key = &H15 then status = 1
        if key = &H31 then status = 0
    Wend
    GetYesNo = status
End Function

Public Function ReadKB () as Integer
    Dim temp as Unsigned Byte
    Dim key as Unsigned Byte
    Dim breakexit as Unsigned Byte = 0
    Dim done as Unsigned Byte = 0
    Dim isescaped as Unsigned Byte = 0
    While done = 0
        key = inportb(&H60)
        temp = inportb(&H61)
        outportb(&H61,temp Or  &H80)    'disable
        outportb(&H61,temp And &H7F)    'and reenable
        If key = &HE0 Then
            isescaped = &H80
        ElseIf key > &H80 Then
            If key = breakexit Then
                done = 1
            Else
                breakexit = 0
                isescaped = 0
            End If
        Else
            breakexit = key or &H80
            ReadKB = key or isescaped
        End If
    Wend
End Function

Public Sub ClearCon (vram As Byte Ptr)
    Dim lp As Long
    For lp = 1 to 80 * 25 * 2
        vram[2 * lp - 2] = 0
        vram[2 * lp - 1] = 7
    Next lp
End Sub

Public Function R600_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer, ByRef mmiobase As Long) As Long

    base_lfb = 0
    base_pio = 0

    Dim lp as long, bus as long, dev as long, fn as long
    Dim lastdevice as Unsigned Short
    Dim mybus as long, mydev as long, myfn as long
    Dim vendor as Unsigned Short, myvendor as unsigned short
    Dim device as Unsigned Short, mydevice as unsigned short
    Dim atidev as Unsigned Short

    mybus = -1
    atidev = &HFFFF
    lastdevice = &HFFFF
    lp = 0
    For bus = 0 to 2
        For dev = 0 to 31
            For fn = 0 to 7
                vendor = PCI_type1_readword(bus, dev, fn, 0)
                device = PCI_type1_readword(bus, dev, fn, 2)
                If vendor = &H1002 And (   _
                        device = &H94C1 Or _
                        device = &H94C3 Or _
                        device = &H94C4 )  Then
                    mybus = bus
                    mydev = dev
                    myfn = fn
                    myvendor = vendor
                    mydevice = device
                    exit for
                ElseIf vendor = &H1002 Then
                    atidev = device
                End If
            Next fn
        Next dev
    Next bus

    if mybus = -1 Then
        function = -1
        exit function
    end if

    Dim bar as Unsigned Long, barmask As Unsigned Long

    For lp = 5 to 0 step -1
        barmask = PCI_bar_readmask(mybus,mydev,myfn, lp)
        bar = PCI_bar_readaddress(mybus,mydev,myfn, lp)

        If bar = 0 then

        Elseif (barmask and &H1) = 1 Then
            base_pio = bar and &HFFF8&
        Elseif ((barmask and &HFFFF0) = 0) And (base_lfb = 0) then
            base_lfb = bar
        Else
            mmiobase = bar
        End If
    Next lp

    function = mydevice

End Function

Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
    Dim offset As Long
    Select case bpp
        Case 4
            Dim temp as Byte
            offset = x \ 2 + y * pitch
            If (x And &H1) = 1 Then
                temp = vram[offset] and &HF
                vram[offset] = temp + 16 * (col and &HF)
            Else
                temp = vram[offset] and &HF0
                vram[offset] = temp + (col and &HF)
            End If

        Case 8
            offset = x + y * pitch
            vram[offset] = CByte(col)

        Case 16
            offset = 2 * x + pitch
            Dim mempos as Short Ptr
            mempos = CPtr(Short Ptr, @vram[offset])
            mempos[0] = CShort(col)

        Case 32
            offset = 4 * x + pitch
            Dim mempos as Long Ptr
            mempos = CPtr(Long Ptr, @vram[offset])
            mempos[0] = col

    End Select
End Sub

Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        lfb[lp * 8 - 8 + 8 * offset] = ch
        lfb[lp * 8 - 7 + 8 * offset] = &H1F
    Next lp
End Sub

Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
    WriteGfxString vram, s, @font(0), column * 8, row * 16, col, bpp, vw
End Sub

Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    Dim lp as long
    For lp = 1 to len(s)
        WriteGfxChar vram, asc(mid$(s,lp,1)), font, x + 8 * (lp - 1), y, col, bpp, vw
    next lp
End Sub
Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    dim lp as long
    Dim fontbyte as byte
    For lp = 0 to 15
        fontbyte = font[32 * ch + lp]
        if (fontbyte and &H01) = &H01 then PutPixel vram, x + 7, y + lp, col, bpp, vw
        if (fontbyte and &H02) = &H02 then PutPixel vram, x + 6, y + lp, col, bpp, vw
        if (fontbyte and &H04) = &H04 then PutPixel vram, x + 5, y + lp, col, bpp, vw
        if (fontbyte and &H08) = &H08 then PutPixel vram, x + 4, y + lp, col, bpp, vw
        if (fontbyte and &H10) = &H10 then PutPixel vram, x + 3, y + lp, col, bpp, vw
        if (fontbyte and &H20) = &H20 then PutPixel vram, x + 2, y + lp, col, bpp, vw
        if (fontbyte and &H40) = &H40 then PutPixel vram, x + 1, y + lp, col, bpp, vw
        if (fontbyte and &H80) = &H80 then PutPixel vram, x + 0, y + lp, col, bpp, vw
    Next lp
End Sub

Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)
    Dim lpx as long, lpy as long
    For lpx = 1 To w - 1 Step 2
        PutPixel vram, lpx, 0, 9, bpp, vw
        PutPixel vram, lpx, 1, 9, bpp, vw
        If lpx mod 10 = 9 then
            PutPixel vram, lpx, 2, 9, bpp, vw
        End If
        If lpx mod 40 = 39 then
            PutPixel vram, lpx, 3, 9, bpp, vw
        End If
        PutPixel vram, lpx, h-1, 9, bpp, vw
        PutPixel vram, lpx, h-2, 9, bpp, vw
    Next lpx
    For lpy = 1 To h - 1 Step 2
        PutPixel vram, 0, lpy, 9, bpp, vw
        PutPixel vram, 1, lpy, 9, bpp, vw

        PutPixel vram, w-1, lpy, 9, bpp, vw
        PutPixel vram, w-2, lpy, 9, bpp, vw

        If lpy mod 10 = 9 then
            PutPixel vram, w-3, lpy, 9, bpp, vw
        End If
        If lpy mod 40 = 39 then
            PutPixel vram, w-4, lpy, 9, bpp, vw
        End If
    Next lpy
End Sub

' Function: TestR600
' runs the test suite for the R6xx series graphics card
'
' Parameters:
'     board_model   - the board type. 1 for a V1k, 2 for a V2k
'     board_version - the board rating.
'                     1 for a V2100, 2 for a V2200, 0 if unknown
'     lfb           - address of the framebuffer in memory
'     portbase      - the base I/O port for the Verite registers
'
Sub TestR600(board_model As Long, board_version As Long, lfb As Long, portbase As Long, mmiobase As Long)
    Dim vaddress as Byte Ptr
    Dim vram as Unsigned Byte Ptr
    Dim mmio As Unsigned Integer Ptr

    Dim lp as long
    Dim ws as String, ws2 as String

    Dim cram As Byte Ptr
    cram = CPtr(Byte Ptr, &HB8000)

    ' claim board-specific addresses

    vram = CPtr(Unsigned Byte Ptr, &HB0000000)
    mmio = CPtr(Unsigned Integer Ptr, &HB8000000)

    PrintString "1", cram, 60

    vaddress = CPtr(Byte Ptr, mmiobase)
    ManageMemoryL2(CPtr(Byte Ptr, &HFFFFFFFF), vaddress)
    PrintString "4", cram, 66
    BlockAllocPhysL(1, CPtr(Byte Ptr, mmio), vaddress)
    PrintString "5", cram, 68

    vaddress = CPtr(Byte Ptr, lfb)
    'ManageMemoryL2(CPtr(Byte Ptr, &HFFFFFFFF), vaddress)
    PrintString "2", cram, 62
    BlockAllocPhysL(1, CPtr(Byte Ptr, vram), vaddress)
    PrintString "3", cram, 64

    portalloc portbase, &H100
    PrintString "6", cram, 70

    mmio = CPtr(Unsigned Integer Ptr, &HB8000000 + (mmiobase and &H003FF000))

    ' ATI has the LFB enabled by default

    PrintString "7", cram, 72

    ReadKB

    ' enter native mode.
    ' 60f0 (latch from vga)
    ' 0330 (use vga for display 1)
    ' 0338 (use vga for display 2)

    ' disable VGA CRTC
    mmio[&H300 \ 4] = mmio[&H300 \ 4] and &HFFFCFFFF
    ' disable VGA Palette
    mmio[&H308 \ 4] = mmio[&H308 \ 4] and &HFFFFFFCF
    ' disable Axxxx-Cxxxx aperture and suspend the VGA
    mmio[&H308 \ 4] = mmio[&H308 \ 4] or &H00010010
    ' detach VGA from display pipelines 1 & 2
    mmio[&H330 \ 4] = mmio[&H330 \ 4] and &HFFFFFFFE
    mmio[&H338 \ 4] = mmio[&H338 \ 4] and &HFFFFFFFE
    ReadKB
 
    for lp = 0 to &H10000
        vram[lp] = lp and &HFF
    next lp
    ReadKB

    ' primary graphics enable
    mmio[&H6100 \ 4] = mmio[&H6100 \ 4] Or 1
    ReadKB
    ' 32 bit B8G8R8A8
    'mmio[&H6104 \ 4] = (mmio[&H6104 \ 4] And &HFFFFF8FC) Or &H00000002
    mmio[&H6104 \ 4] = &H00000002
    ReadKB
    ' enable video memory access
    mmio[&H6080 \ 4] = mmio[&H6080 \ 4] And &HFEFFFFFF
    ReadKB

    for lp = 0 to &H10000
        vram[lp] = lp and &HFF
    next lp
    ReadKB
    
    ' virtual width
    mmio[&H6110 \ 4] = 0 + mmio[&H542C]   ' offset + framebuffer start in GPU address space
    mmio[&H6120 \ 4] = 1024 ' virtual width
    mmio[&H6124 \ 4] = 0    ' target x offset
    mmio[&H6128 \ 4] = 0    ' target y offset
    mmio[&H612C \ 4] = 0    ' viewport x start
    mmio[&H6130 \ 4] = 0    ' viewport y start
    mmio[&H6134 \ 4] = 640  ' viewport x
    mmio[&H6138 \ 4] = 480  ' viewport y
    ReadKB

    mmio[&H652C] = 480 ' undocumented reg
    mmio[&H6528] = 0   ' undocumented reg
    ReadKB

    for lp = 0 to &H10000
        vram[lp] = (lp and &HFF) xor &HAA
    next lp
    ReadKB

    for lp = 0 to &H10000
        vram[lp] = lp and &HFF
    next lp
    ReadKB

    ' timings. For some wierd reason, pixel 0 must align with the sync pulse.
    ' h.total
    'mmio[&H6000 \ 4] = 800 - 1
    'ReadKB
    ' h.blank
    'mmio[&H6004 \ 4] = &H10000 * 96
    'ReadKB
    ' h.sync
    'mmio[&H600C \ 4] = &H10000 * 96
    'ReadKB
    ' v.total
    'mmio[&H6020 \ 4] = 524 - 1
    'ReadKB
    ' v.blank
    'mmio[&H6024 \ 4] = &H10000 * 2
    'ReadKB
    ' v.sync
    'mmio[&H6028 \ 4] = &H10000 * 2
    'ReadKB


    PrintString "8", cram, 74

    ReadKB

End Sub

