' Summary: tweak.bas
' Universal graphics poke tool (based on TWEAK for DOS)
'
' Author:
'     Marcel Sondaar
'
' License:
'     Public Domain
'

#include "mos.bi"
#include "x86.bi"
#include "../gfx/vga_io.bi"
#include "mos/pci.bi"
#include "./tweak.bi"

Declare Sub ModMain Cdecl Alias "main" ()
Declare Function ReadKey() As Byte
Declare Sub InitKeyboard()

Declare Sub ReadRegisters
Declare Sub WriteRegisters

' Graphics Routines
Enum VideoModes
    ' standard addressing modes
    linear = 1      ' linear mode
    unchained = 2   ' planar 256 color
    planar = 3      ' planar 16 color

    ' tweaked addressing modes
    halfchained = 4 ' planar 256 color in word mode
    chained = 5     ' emulating linear mode with chain-4 bit clear (planar 256 in dword mode)
    planar2 = 6     ' 16 color in word mode
    planar4 = 7     ' 16 color in dword mode
    concatl = 8     ' concatenated 16 color mode on a left-handed system
    concatr = 9     ' concatenated 16 color mode on a right-handed system
    
    ' lfb modes
    r5g6b5  = 10
    b5g6r5  = 11
    r8g8b8x8 = 12
    b8g8r8x8 = 13
    
    
End Enum
Declare Sub PutPixel(ByVal vram as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode as byte, ByVal vw as long)
Declare Sub RenderTest(ByVal vram as Byte Ptr, ByVal w as long, ByVal h as long, ByVal mode as byte, ByVal vw as long)
Declare Sub ClearLinear(ByVal vram as Byte Ptr)
Declare Sub ClearUnchained(ByVal vram as Byte Ptr, ByVal pixels as long)
Declare Function PlaneSelect(ByVal planeno as long) As Unsigned Byte
Declare Sub WriteGfxString(ByVal vram as Byte Ptr, ByRef s as String, ByVal font as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode as Byte, ByVal vw as Long)
Declare Sub WriteGfxChar(ByVal vram as Byte Ptr, ByVal ch as Integer, ByVal font as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode as Byte, ByVal vw as Long)
Declare Sub ReadFont(ByVal vram as byte ptr, ByVal dumper as byte ptr)
Declare Sub WriteFont(ByVal vram as byte ptr, ByVal dumper as byte ptr)


Common Shared device_lfb As Byte Ptr
Common Shared device_lfb_offset As Long
Common Shared device_lfb_size As Long
Common Shared device_mmio As Byte Ptr
Common Shared device_mmio_offset As Long
Common Shared device_mmio_size As Long
Common Shared device_port As Short
Common Shared device_port_size As Long
Common Shared device_vendor As Short
Common Shared device_devid As Short

Dim Shared reg_listing() As RegisterType
Dim Shared reg_count As Integer

Public Sub AssignIO
    Dim lp as long, bus as long, dev as long, fn as long
    Dim lastdevice as Unsigned Short
    Dim mybus as long, mydev as long, myfn as long
    Dim vendor as Unsigned Short, myvendor as unsigned short
    Dim device as Unsigned Short, mydevice as unsigned short
    Dim class1 As Unsigned Byte

    mybus = -1
    lastdevice = &HFFFF
    lp = 0
    For bus = 0 to 15
        For dev = 0 to 31
            For fn = 0 to 7
                vendor = PCI_type1_readword(bus, dev, fn, 0)
                device = PCI_type1_readword(bus, dev, fn, 2)
                If (vendor <> 0) And (vendor <> &HFFFF) And (device <> 0) And (device <> &HFFFF) Then
                    class1 = PCI_type1_readbyte(bus, dev, fn, &H0B)
                    if (class1 = &H03) Then                
                        mybus = bus
                        mydev = dev
                        myfn = fn
                        myvendor = vendor
                        mydevice = device
                        Exit For                
                    End If
                End If
            Next fn
        Next dev
    Next bus

    if mybus = -1 Then        
        Exit Sub
    end if

    Dim bar as Unsigned Long, barmask As Unsigned Long

    For lp = 5 to 0 step -1
        barmask = PCI_bar_readmask(mybus,mydev,myfn, lp)
        bar = PCI_bar_readaddress(mybus,mydev,myfn, lp)

        If bar = 0 then
            ' do nothing
        Elseif (barmask and &H1) = 1 Then
            device_port = bar and &HFFF8&            
            device_port_size = barmask
        Elseif ((barmask and &HFFFF0) = 0) And (device_lfb_offset = 0) then
            device_lfb_offset = bar
            device_lfb_size = barmask
        Else
            device_mmio_offset = bar
            device_mmio_size = barmask
        End If
    Next lp

    device_vendor   = myvendor
    device_devid    = mydevice
    
End Sub

Sub debug(ByRef s As String)
    Dim lp As Integer
    For lp = 1 To len(s)
        outportb &HE9, asc(mid$(s,lp,1))
    Next lp
    outportb &HE9, 10
    outportb &HE9, 13
End Sub

Public Sub ModMain Cdecl Alias "main" ()
    Dim vram As Byte Ptr
    Dim cram As Byte Ptr
    Dim vaddress as Byte Ptr

    vram = CPtr(Byte Ptr, &HA0000)
    cram = CPtr(Byte Ptr, &HB8000)

    ' claim video memory
    blockallocphys(64, vram, vram)
    ' allocate io bitmap
    allocateiobitmap(0, &HE000, CPtr(Byte Ptr, &HFFFFFFFF))
    ' claim KBC
    InitKeyboard
    ' claim VGA
    portalloc &H3B0, 48
    ' claim PCI
    portalloc &HCF8, 8
    ' claim bochs debugger
    portalloc &HE9, 1        
    
    ' Claim ranges for this device
    AssignIO        
    If (device_lfb_size <> 0) Then
        device_lfb = CPtr(Byte Ptr, &H80000000)
        ManageMemoryL2(CPtr(Byte Ptr, &HFFFFFFFF), CPtr(Byte Ptr, device_lfb_offset))
        BlockAllocPhysL(1, device_lfb, CPtr(Byte Ptr, device_lfb_offset))
    End If
    If (device_mmio_size <> 0) Then
        device_mmio = CPtr(Byte Ptr, &H81000000)
        ManageMemoryL2(CPtr(Byte Ptr, &HFFFFFFFF), CPtr(Byte Ptr, device_mmio_offset))
        BlockAllocPhysL(1, device_mmio, CPtr(Byte Ptr, device_mmio_offset))
        
        device_mmio = device_mmio + (device_mmio_offset And &H3FFFFF)
    End If
    If (device_port_size > 0) Then
        portalloc device_port, &H400
    End If
    
    ReadKey
    
    ClearCon cram
    PrintString "BASIC For teh win (>O.O)>", cram, 25 * 80 - 27
    
    Dim vstr As String, dstr As String
    DeviceNames vstr, dstr, device_vendor, device_devid
    
    PrintStringC "Device vendor: ", cram, 15, 10, &H0E
    PrintStringC "Device name: ", cram, 15, 11, &H0E
    PrintStringC "Detected LFB: ", cram, 15, 12, &H0E
    PrintStringC "Detected MMIO: ", cram, 15, 13, &H0E
    PrintStringC "Detected PIO: ", cram, 15, 14, &H0E
    PrintStringC "0x" + hex$(device_vendor) + ": " + vstr, cram, 30, 10, &H09
    PrintStringC "0x" + hex$(device_devid) + ": " + dstr, cram, 30, 11, &H09
    PrintStringC "0x" + hex$(CInt(device_lfb_offset)), cram, 30, 12, &H09
    PrintStringC "0x" + hex$(CInt(device_mmio_offset)) + " at 0x" + hex$(Cint(device_mmio)), cram, 30, 13, &H09
    PrintStringC "0x" + hex$(device_port), cram, 30, 14, &H09    
    
    ReadKey
    
    Reglist_VGA    
    
    ReadRegisters
    
    ClearCon cram
    UIEventLoop cram, TweakUI
    
    While 1 = 1
        Yield
    Wend

End Sub

Sub TestSettings
    Dim backupregs() As RegisterType
    Redim backupregs(reg_count)
    
    'Dim oldcharset as Byte Ptr
    'oldcharset = Allocate(8192)
    
    'VGAUnlockCRTC
    'ReadFont CPtr(Byte Ptr, &HA0000), oldcharset 
    
    Dim lp As Integer
    For lp = 0 To reg_count - 1
        'debug "backup copy: " + str$(lp)
        backupregs(lp) = reg_listing(lp)
        'debug "backup get: " + str$(lp)
        backupregs(lp).getter(backupregs(lp))
    Next lp
    
    For lp = 0 To reg_count - 1
        'debug "live set: " + str$(lp)
        reg_listing(lp).setter(reg_listing(lp))
    Next lp
    
    ReadKey
    
    
    For lp = 0 To reg_count - 1
        'debug "debug set: " + str$(lp)
        backupregs(lp).setter(backupregs(lp))
    Next lp
    
    'VGAUnlockCRTC
    'WriteFont CPtr(Byte Ptr, &HA0000), oldcharset
    'VGAUnlockCRTC
    'VGASetTextMode
    'VGAEnableDisplay
    
    'Deallocate oldcharset
    
End Sub

Sub CloneSettings(ByVal modeno As Integer)
    Yank modeno
    Dim lp As Integer
    For lp = 0 To reg_count - 1        
        reg_listing(lp).getter(reg_listing(lp))
    Next lp
    Yank 3
End Sub

Sub ReadFont(ByVal vram as byte ptr, ByVal dumper as byte ptr)
    dim lp as long
    VGASet320x200mode
    VGASetModeX
    Write3CE &H4, &H2
    for lp = 0 to 8191
        dumper[lp] = vram[lp]
    next lp
    'WriteRegs regs()
End Sub
Sub WriteFont(ByVal vram as byte ptr, ByVal dumper as byte ptr)
    Dim lp As Long
    VGASet320x200mode
    VGASetModeX
    Write3C4 &H2, &H4
    For lp = 0 to 8191
        vram[lp] = dumper[lp]
    Next lp
    'WriteRegs regs()
End Sub

Sub ClearUnchained(ByVal vram as Byte Ptr, ByVal pixels as long)
    Dim lp as long
    Write3C4 &H2, &HF
    for lp = 0 to (pixels \ 4) - 1
        vram[lp] = 0
    next lp
End Sub

Sub RenderTest(ByVal vram as Byte Ptr, ByVal w as long, ByVal h as long, ByVal mode as byte, ByVal vw as long)
    Dim lpx as long, lpy as long
    For lpx = 1 To w - 1 Step 2
        PutPixel vram, lpx, 0, 9, mode, vw
        PutPixel vram, lpx, 1, 9, mode, vw

        PutPixel vram, lpx, h-1, 9, mode, vw
        PutPixel vram, lpx, h-2, 9, mode, vw
    Next lpx
    For lpy = 1 To h - 1 Step 2
        PutPixel vram, 0, lpy, 9, mode, vw
        PutPixel vram, 1, lpy, 9, mode, vw

        PutPixel vram, w-1, lpy, 9, mode, vw
        PutPixel vram, w-2, lpy, 9, mode, vw
    Next lpy
End Sub

Sub PutPixel(ByVal vram as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode As Byte, ByVal vw As Long)
    Select Case mode
        Case VideoModes.linear
            vram[x + vw * y] = col And &HFF
        Case VideoModes.unchained
            Write3C4 &H2, PlaneSelect(x Mod 4)
            vram[(x + vw * y) \ 4] = col And &HFF
        Case VideoModes.halfchained
            Write3C4 &H2, PlaneSelect(x Mod 4)
            vram[((x + vw * y) \ 2) And &HFFFFFFFE] = col And &HFF
        Case VideoModes.chained
            Write3C4 &H2, PlaneSelect(x Mod 4)
            vram[(x + vw * y) And &HFFFFFFFC] = col And &HFF

        Case VideoModes.planar
            Dim setbyte as Byte
            Dim clearbyte as Byte
            Dim address as Long
            setbyte = PlaneSelect(7-(x Mod 8))
            clearbyte = 255 - setbyte
            address = (x + vw * y) \ 8
            Write3C4 &H2, 1
            Write3CE &H4, 0
            if (col and 1) = 1 then
                vram[address] = vram[address] or setbyte
            else
                vram[address] = vram[address] and clearbyte
            end if
            Write3C4 &H2, 2
            Write3CE &H4, 1
            if (col and 2) = 2 then
                vram[address] = vram[address] or setbyte
            else
                vram[address] = vram[address] and clearbyte
            end if
            Write3C4 &H2, 4
            Write3CE &H4, 2
            if (col and 4) = 4 then
                vram[address] = vram[address] or setbyte
            else
                vram[address] = vram[address] and clearbyte
            end if
            Write3C4 &H2, 8
            Write3CE &H4, 3
            if (col and 8) = 8 then
                vram[address] = vram[address] or setbyte
            else
                vram[address] = vram[address] and clearbyte
            end if
        case VideoModes.planar2
            PutPixel vram, (x and &H7) + 2 * (x And &HFFF8), y, col, VideoModes.planar, vw * 2
        case VideoModes.planar4
            PutPixel vram, (x and &H7) + 4 * (x And &HFFF8), y, col, VideoModes.planar, vw * 4

        case VideoModes.concatl
            Dim readbyte as byte
            Write3CE &H4, (x \ 2) Mod 4
            Write3C4 &H2, PlaneSelect((x \ 2) Mod 4)
            readbyte = vram[(x + vw * y) \ 8]
            if (x and 1) = 1 then
                readbyte = (readbyte and &HF0) or (col and &H0F)
            else
                readbyte = (readbyte and &H0F) or ((col * 16) and &HF0)
            end if
            vram[(x + vw * y) \ 8] = readbyte

        case VideoModes.concatr
            PutPixel vram, x xor 1, y, col, Videomodes.concatl, vw

    End Select
End Sub

Function PlaneSelect(ByVal planeno As Long) As Unsigned Byte
    Select Case planeno
        Case 0
            PlaneSelect = 1
        Case 1
            PlaneSelect = 2
        Case 2
            PlaneSelect = 4
        Case 3
            PlaneSelect = 8
        Case 4
            PlaneSelect = 16
        Case 5
            PlaneSelect = 32
        Case 6
            PlaneSelect = 64
        Case 7
            PlaneSelect = 128
    End Select
End Function

Sub WriteGfxString(ByVal vram as Byte Ptr, ByRef s as String, ByVal font as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode as Byte, ByVal vw as Long)
    Dim lp as long
    For lp = 1 to len(s)
        WriteGfxChar vram, asc(mid$(s,lp,1)), font, x + 8 * (lp - 1), y, col, mode, vw
    next lp
End Sub
Sub WriteGfxChar(ByVal vram as Byte Ptr, ByVal ch as Integer, ByVal font as Byte Ptr, ByVal x as long, ByVal y as long, ByVal col as long, ByVal mode as Byte, ByVal vw as Long)
    dim lp as long
    Dim fontbyte as byte
    For lp = 0 to 15
        fontbyte = font[32 * ch + lp]
        if (fontbyte and &H01) = &H01 then PutPixel vram, x + 7, y + lp, col, mode, vw
        if (fontbyte and &H02) = &H02 then PutPixel vram, x + 6, y + lp, col, mode, vw
        if (fontbyte and &H04) = &H04 then PutPixel vram, x + 5, y + lp, col, mode, vw
        if (fontbyte and &H08) = &H08 then PutPixel vram, x + 4, y + lp, col, mode, vw
        if (fontbyte and &H10) = &H10 then PutPixel vram, x + 3, y + lp, col, mode, vw
        if (fontbyte and &H20) = &H20 then PutPixel vram, x + 2, y + lp, col, mode, vw
        if (fontbyte and &H40) = &H40 then PutPixel vram, x + 1, y + lp, col, mode, vw
        if (fontbyte and &H80) = &H80 then PutPixel vram, x + 0, y + lp, col, mode, vw
    Next lp
End Sub

' Reglist
Sub AddRegister(ByVal Address As Integer, ByVal Index As Short, ByVal Getter As Sub(ByRef reg As RegisterType), ByVal Setter As Sub(ByRef reg As RegisterType), ByVal model As Byte, ByRef Description As String)
    Dim reg As RegisterType
    reg.address = address
    reg.index = index
    reg.setter = Setter
    reg.getter = Getter
    reg.uitype = model
    reg.description = left$(Description, 15)
    reg.enabled = 1
    
    reg_count = reg_count + 1
    redim preserve reg_listing(reg_count)
    reg_listing(reg_count - 1) = reg    
End Sub

Sub ReadRegisters
    Dim lp As Integer
    For lp = 0 To reg_count - 1
        Dim reg As RegisterType        
        reg = reg_listing(lp)
        reg_listing(lp).Getter(reg)
        reg_listing(lp) = reg
    Next lp
End Sub

Sub WriteRegisters
    Dim lp As Integer
    For lp = 0 To reg_count - 1
        Dim reg As RegisterType        
        reg = reg_listing(lp)
        If (reg.enabled = 1) Then
            reg.Setter(reg)
            reg_listing(lp) = reg
        End If
    Next lp
End Sub
