' Summary: tweak_ui.bas
' User interface for the graphics poke tool (based on TWEAK for DOS)
'
' Author:
'     Marcel Sondaar
'
' License:
'     Public Domain
'

#include "./tweak.bi"
#include "mos.bi"

Declare Function ReadKey() As Byte ' character
Declare Function ReadKeyboard() As Unsigned Byte ' scancode

Dim Shared TweakUI As UIState

'
' Console functions
'

Public Sub ClearCon (ByVal vram As Byte Ptr)
    Dim lp As Long
    For lp = 1 to 80 * 25 * 2
        vram[2 * lp - 2] = 0
        vram[2 * lp - 1] = 7
    Next lp
End Sub

Public Sub PrintStringC (ByRef s As String, ByVal vram As Byte Ptr, ByVal x As Integer, ByVal y As Integer, ByVal colour As Byte)
    Dim lp As Long
    Dim ch As Byte
    Dim offset As Long
    offset = 2 * x + 160 * y

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        vram[lp * 2 - 2 + offset] = ch
        vram[lp * 2 - 1 + offset] = colour
    Next lp
End Sub

Sub DrawWindow(Byval cram As Byte Ptr, ByVal sx as integer, ByVal sy as integer, ByVal w as integer, ByVal h as integer, ByVal colour As Byte)
    Dim lpx as long
    Dim lpy as long

    for lpy = sy - 1 to sy + h
    for lpx = sx - 1 to sx + w
        cram[2*lpx + 160*lpy] = 0
        cram[2*lpx + 160*lpy + 1] = colour
    next lpx
    next lpy

    for lpx = sx to sx + w
        cram[2*lpx + 160 * (sy - 1)] = CByte(&HCD)
        cram[2*lpx + 160 * (sy+h)] = CByte(&HCD)
    next lpx
    for lpy = sy to sy + h
        cram[2*sx-2   + 160 * lpy] = CByte(&HBA)
        cram[2*sx+2*w + 160 * lpy] = CByte(&HBA)
    next lpy

    cram[2 * sx + 160 * sy - 162] = CByte(&HC9)
    cram[2 * (sx+w) + 160 * sy - 160] = CByte(&HBB)
    cram[2 * sx + 160 * (sy+h) - 2] = CByte(&HC8)
    cram[2 * (sx+w) + 160 * (sy+h)] = CByte(&HBC)

End Sub


Public Sub PrintString (ByRef s As String, ByVal vram As Byte Ptr, ByVal offset As Integer)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        vram[lp * 2 - 2 + 2 * offset] = ch
        vram[lp * 2 - 1 + 2 * offset] = 7
    Next lp
End Sub

Public Function GetYesNo () As Integer
    Dim status As Integer = -1
    Dim key As Integer
    While status = -1
        key = ReadKey
        if key = &H15 then status = 1
        if key = &H31 then status = 0
    Wend
    GetYesNo = status
End Function

'
' Render functions
'
Sub RenderRow(ByRef state As UIState, ByVal cram As Byte Ptr, ByVal position As Integer)
    Dim screenpos As Integer = position - state.row_offset
    
    Dim reg As RegisterType
    reg = reg_listing(position)
    
    Dim row As Integer
    Dim col As Integer
    row = screenpos mod 20
    col = screenpos \ 20
    
    Dim desc As String
    desc = reg.description
    PrintStringC "               =        ", cram, 25 * col, row, 7
    PrintStringC desc, cram, 25 * col + 15 - len(desc), row, iif(reg.enabled, &H08 + (reg.UIType SHR 4), &H08)
    'PrintStringC hex$(CInt(reg.value)), cram, 25*col + 16, row, iif(reg.enabled, &H0E, &H08)
    Rendernum CInt(reg.value), (reg.UIType And &H0F) SHL 1, iif(position = state.Selected_index, 1, 0), 25*col + 16, row, cram
    
End Sub

Sub RenderList(ByRef state As UIState, ByVal cram As Byte Ptr)
    Dim count as Integer
    count = reg_count - state.row_offset
    If count > 60 Then count = 60
    
    Dim lp As Integer
    For lp = 0 to count - 1
        RenderRow state, cram, lp + state.row_offset        
    Next lp    
End Sub

Sub RenderNum(ByVal number As Integer, ByVal digits As Byte, ByVal selected As Byte, ByVal x As Integer, ByVal y As Integer, ByVal cram As Byte Ptr)
    Dim text As String
    text = Hex$(number)
    While Len(text) < digits
        text = "0" + text
    Wend    
    Dim colour As byte
    colour = &H7
    If (selected <> 0) Then colour = &HF
    PrintStringC text, cram, x, y, colour
End Sub


' 
' UI root
'
Sub UIEventLoop(ByVal cram As Byte Ptr, ByRef state As UIState)
    Dim done As Integer = 0
    RenderList TweakUI, cram
    
    While done = 0
        Dim key As Integer
        key = 0
        While key = 0
            Yield
            key = ReadKeyboard      
        Wend
        
        Dim lastIndex As Integer
        Dim typedchar As Integer        
        lastIndex = state.Selected_index
        
        Select Case key
            Case &H48, &HC8 ' up
                
                If (state.Selected_index mod 20) = 0 Then
                    state.Selected_index = state.Selected_index + 19
                    If (state.Selected_index >= reg_count) Then state.Selected_index = reg_count - 1
                Else
                    state.Selected_index = state.Selected_index - 1
                End If
                RenderRow state, cram, lastIndex
                RenderRow state, cram, state.Selected_index
                
            Case &H50, &HD0 ' down
                If (state.Selected_index mod 20) = 19 Then
                    state.Selected_index = state.Selected_index - 19                
                Else
                    state.Selected_index = state.Selected_index + 1
                    If (state.Selected_index >= reg_count) Then state.Selected_index = state.Selected_index - (state.Selected_index mod 20)
                End If
                RenderRow state, cram, lastIndex
                RenderRow state, cram, state.Selected_index
                
            Case &H4B, &HCB ' left
                If state.Selected_index >= 20 Then
                    state.Selected_index = state.Selected_index - 20
                    If state.Selected_index < state.row_offset Then
                        state.row_offset = state.row_offset - 20
                    End If
                End If
                RenderRow state, cram, lastIndex
                RenderRow state, cram, state.Selected_index
                
            Case &H4D, &HCD ' right
                If state.Selected_index + 20 < reg_count Then
                    state.Selected_index = state.Selected_index + 20
                    If state.Selected_index >= state.row_offset + 60 Then
                        state.row_offset = state.row_offset + 20
                    End If
                End If
                RenderRow state, cram, lastIndex
                RenderRow state, cram, state.Selected_index
                
            Case &H39 ' space = toggle write state
                reg_listing(lastIndex).enabled = 1 - reg_listing(lastIndex).enabled
                RenderRow state, cram, lastIndex
                
            Case &H3B To &H42 'F1..F8 = toggle bit
                Dim bitindex As Integer = key - &H3B
                reg_listing(lastIndex).value = reg_listing(lastIndex).value Xor (1 SHL bitindex)
                RenderRow state, cram, lastIndex
            
            Case &H02 to &H0A ' 1..9
                PushDigit key - 1, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            
            Case &H0B ' 0
                PushDigit 0, state
                RenderRow state, cram, lastIndex
                typedchar = 1       
            Case &H1E ' A
                PushDigit &HA, state
                RenderRow state, cram, lastIndex
                typedchar = 1           
            Case &H30 ' B
                PushDigit &HB, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            Case &H2E ' C
                PushDigit &HC, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            Case &H20 ' D
                PushDigit &HD, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            Case &H12 ' E
                PushDigit &HE, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            Case &H21 ' F
                PushDigit &HF, state
                RenderRow state, cram, lastIndex
                typedchar = 1
            
            Case &H1C ' enter = test
                TestSettings
                RenderList TweakUI, cram
            
            Case &H32 ' M = get mode
                Dim mode As Integer
                mode = UIModeLoop(cram)
                If mode > 0 Then
                    CloneSettings mode
                End If
                RenderList TweakUI, cram
                
            Case &H13 ' R = reset video mode
                Yank &H3
                RenderList TweakUI, cram
                
            Case &H1A ' Test left
            
            Case &H1B ' Test right
            
            Case Else
                PrintStringC "key: " + hex$(key) + " " , cram, 70, 24, &H01
            
        End Select      
        
        If typedchar = 0 Then
            state.Charnum = 0
        End If
    Wend
End Sub

Function UIModeLoop(ByVal cram As Byte Ptr) As Integer
    PrintStringC "mode: 0x     " , cram, 60, 24, &H07
    
    Dim done As Integer = 0
    Dim mode As Integer
    
    While done = 0    
        mode = mode And &HFFFF
        PrintStringC "     ", cram, 68, 24, &H0F
        PrintStringC hex$(mode) , cram, 68, 24, &H0F
    
        Dim key As Integer
        key = 0
        While key = 0
            Yield
            key = ReadKeyboard      
        Wend                
        
        Select Case key
            Case &H02 to &H0A ' 1..9
                mode = mode * 16 + (key - 1)
            Case &H0B ' 0
                mode = mode * 16
            Case &H1E ' A
                mode = mode * 16 + &HA
            Case &H30 ' B
                mode = mode * 16 + &HB
            Case &H2E ' C
                mode = mode * 16 + &HC
            Case &H20 ' D
                mode = mode * 16 + &HD
            Case &H12 ' E
                mode = mode * 16 + &HE
            Case &H21 ' F
                mode = mode * 16 + &HF            
            Case &H1C 
                done = 1
            Case Else
                mode = 0
                done = 1
        End Select        
    Wend

    PrintStringC "             " , cram, 60, 24, &H07
    Function = mode
End Function

Sub PushDigit(ByVal digit As Integer, ByRef state As UIState)
    If state.Charnum = 0 Then
        reg_listing(state.Selected_index).Value = digit
    Else
        reg_listing(state.Selected_index).Value = (reg_listing(state.Selected_index).Value SHL 4) Or digit
    End If
    
    state.Charnum = state.Charnum + 1
    
    If (state.Charnum SHR 1) = (reg_listing(state.Selected_index).UIType And &H0F) Then
        state.Charnum = 0
    End If
End Sub
