' Summary: verite2k.bas
' Testing module for Verite V2x00 graphics stuff
'
' Author:
'     Marcel Sondaar
'
' License:
'     Educational Purposes
'

#include "../gfx/vga_io.bi"
#include "../gfx/vgasignal.bi"
#include "../gfx/verite/verite.bi"
#include "mos.bi"
#include "x86.bi"
#include "mos/pci.bi"

Declare Sub ModMain Cdecl Alias "main" ()
Declare Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
Declare Function GetYesNo () As Integer
Declare Function ReadKB () as Integer
Declare Sub ClearCon (vram As Byte Ptr)
Declare Function Verite_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer) As Long

Declare Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
Declare Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
Declare Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
Declare Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
Declare Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)

Declare Sub TestVerite(gpu_model As Long, gpu_version As Long, lfb As Long, portbase As Long)
Declare Sub Profile_insn(insn As Integer, portbase As Integer, vram As Byte Ptr, font As Byte Ptr, ypos As Integer)
Declare Sub Profile_condjump(insn As Integer, portbase As Integer, vram As Byte Ptr, font As Byte Ptr, ypos As Integer)

Declare Function malloc Cdecl Alias "malloc" (byval bytes as long) as Byte Ptr
Declare Sub free Cdecl Alias "free" (byval pointer as Byte Ptr)

Declare Sub syncb(port As Integer, value as Unsigned Byte)
Declare Sub syncd(port As Integer, value as Integer)

Declare Function Verite_CalcClock(WantedMHz As Double) As Integer
Declare Function NPQ_to_freq(NPQ As Integer) As Double
Declare Function NPQ_to_V2x00(NPQ As Integer) As Integer
Declare Function V2x00_to_NPQ(NPQ As Integer) As Integer

Declare Sub Verite_ExecuteInsn(portbase As Integer, insn As Integer)
Declare Function Verite_ReadGPUReg(portbase As Integer, register As Byte) As Integer
Declare Function Verite_GetPC(portbase As Integer) As Integer
Declare Function Verite_GetIR(portbase As Integer) As Integer

Extern microcode Alias "verite_microcode" As Byte

Public Sub ModMain Cdecl Alias "main" ()
    Dim vram As Byte Ptr
    Dim cram As Byte Ptr
    Dim bios As Byte Ptr
    Dim vaddress as Byte Ptr

    vaddress = CPtr(Byte Ptr, &HA0000)
    vram = CPtr(Byte Ptr, &HA0000)
    cram = CPtr(Byte Ptr, &HB8000)
    bios = CPtr(Byte Ptr, &HC0000)

    blockallocphys(64, vram, vram)
    allocateiobitmap(0, &HE000, CPtr(Byte Ptr, &HFFFFFFFF))
    PortAlloc(&HCF8, 8)
    PortAlloc(&H3C0, 32)
    PortAlloc(&H60, 2)

    Dim lfb as Long
    Dim portbase as Long
    Dim chiptype as Long
    Dim chipsubtype as Long
    Dim bioshash as Long
    Dim lp as Long
    Dim devname As String

    ClearCon cram
    PrintString "BASIC For teh win (>O.O)>", cram, 25 * 80 - 27

    PrintString "Verite test program", cram, 0
    PrintString "Searching PCI device list for a Verite board...", cram, 80

    chiptype = Verite_PCIProbe(lfb, portbase)
    PrintString "Hashing BIOS...", cram, 160
    bioshash = 0
    for lp = 0 to 32767
        bioshash = bioshash + ((16381 * lp + 1) * CLng(bios[lp]))
    next lp
    PrintString "Probe result: " + hex$(chiptype), cram, 240
    If chiptype = 1 Or chiptype = 2 Then

        Select case bioshash
            Case &H4696828B
                devname = "Stealth II S220"
                chipsubtype = 1
            Case Else
                devname = "Unknown (0x" + hex$(bioshash) + ")"
                chipsubtype = 0
        End Select
        PrintString "Model: " + devname, cram, 400

        If chiptype = 1 Then
            devname = "Verite V1000"
        Else
            If chipsubtype = 2 Then
                devname = "Verite V2200"
            ElseIf chipsubtype = 1 Then
                devname = "Verite V2100"
            Else
                devname = "Verite V2x00"
            End If
        End If
        PrintString "GPU: " + devname, cram, 320


        PrintString "LFB Address: 0x" + hex$(lfb), cram, 480
        PrintString "Port Base: 0x" + hex$(portbase), cram, 560

        PrintString "Press Y to start the test at your hardware's peril", cram, 720
        PrintString "Press N to skip this test", cram, 800

        If GetYesNo() = 1 Then
            TestVerite(chiptype, chipsubtype, lfb, portbase)
        End If

        ClearCon cram
        PrintString "Test Ended. Please hit the power or the reset button", cram, 0

    Else
        PrintString "No Verite board found", cram, 320
        PrintString "Test Ended. Please hit the power or the reset button", cram, 400
    End if
    While 1 = 1

    Wend

End Sub

Public Sub PrintString (s As String, vram As Byte Ptr, offset As Integer)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        vram[lp * 2 - 2 + 2 * offset] = ch
        vram[lp * 2 - 1 + 2 * offset] = 7
    Next lp

End Sub

Public Function GetYesNo () As Integer
    ' Y = &H15
    ' N = &H31

    Dim status As Integer = -1
    Dim key As Integer
    While status = -1
        key = ReadKB
        if key = &H15 then status = 1
        if key = &H31 then status = 0
    Wend
    GetYesNo = status
End Function

Public Function ReadKB () as Integer
    Dim temp as Unsigned Byte
    Dim key as Unsigned Byte
    Dim breakexit as Unsigned Byte = 0
    Dim done as Unsigned Byte = 0
    Dim isescaped as Unsigned Byte = 0
    While done = 0
        key = inportb(&H60)
        temp = inportb(&H61)
        outportb(&H61,temp Or  &H80)    'disable
        outportb(&H61,temp And &H7F)    'and reenable
        If key = &HE0 Then
            isescaped = &H80
        ElseIf key > &H80 Then
            If key = breakexit Then
                done = 1
            Else
                breakexit = 0
                isescaped = 0
            End If
        Else
            breakexit = key or &H80
            ReadKB = key or isescaped
        End If
    Wend
End Function

Public Sub ClearCon (vram As Byte Ptr)
    Dim lp As Long
    For lp = 1 to 80 * 25 * 2
        vram[2 * lp - 2] = 0
        vram[2 * lp - 1] = 7
    Next lp
End Sub

Public Function Verite_PCIProbe(ByRef base_lfb As Long, ByRef base_pio As Integer) As Long

    base_lfb = 0
    base_pio = 0

    Dim lp as long, bus as long, dev as long, fn as long
    Dim lastdevice as Unsigned Short
    Dim mybus as long, mydev as long, myfn as long
    Dim vendor as Unsigned Short, myvendor as unsigned short
    Dim device as Unsigned Short, mydevice as unsigned short
    Dim s3dev as Unsigned Short

    mybus = -1
    s3dev = &HFFFF
    lastdevice = &HFFFF
    lp = 0
    For bus = 0 to 2
        For dev = 0 to 31
            For fn = 0 to 7
                vendor = PCI_type1_readword(bus, dev, fn, 0)
                device = PCI_type1_readword(bus, dev, fn, 2)
                If vendor = &H1163 And device = &H2000 Then
                    mybus = bus
                    mydev = dev
                    myfn = fn
                    myvendor = vendor
                    mydevice = device
                    exit for
                ElseIf vendor = &H1163 Then
                    s3dev = device
                End If
            Next fn
        Next dev
    Next bus

    if mybus = -1 Then
        function = -1
        exit function
    end if

    Dim bar as Unsigned Long, barmask As Unsigned Long

    For lp = 5 to 0 step -1
        barmask = PCI_bar_readmask(mybus,mydev,myfn, lp)
        bar = PCI_bar_readaddress(mybus,mydev,myfn, lp)

        If bar = 0 then

        Elseif (barmask and &H1) = 1 Then
            base_pio = bar and &HFFF8&
        Elseif ((barmask and &HFFFF0) = 0) And (base_lfb = 0) then
            base_lfb = bar
        Else
            'mmiobase = bar
        End If
    Next lp

    function = 2

End Function

Sub PutPixel(vram As Byte Ptr, x As Long, y As Long, col As Long, bpp As Byte, pitch As Long)
    Dim offset As Long
    Select case bpp
        Case 4
            Dim temp as Byte
            offset = x \ 2 + y * pitch
            If (x And &H1) = 1 Then
                temp = vram[offset] and &HF
                vram[offset] = temp + 16 * (col and &HF)
            Else
                temp = vram[offset] and &HF0
                vram[offset] = temp + (col and &HF)
            End If

        Case 8
            offset = x + y * pitch
            vram[offset] = CByte(col)

        Case 16
            offset = 2 * (x + pitch * y)
            Dim mempos as Short Ptr
            mempos = CPtr(Short Ptr, @vram[offset])
            mempos[0] = CShort(col)

        Case 32
            offset = 4 * (x + pitch * y)
            Dim mempos as Long Ptr
            mempos = CPtr(Long Ptr, @vram[offset])
            mempos[0] = col

    End Select
End Sub

Sub WriteLfbString(lfb As Byte Ptr, s as String, offset as Long)
    Dim lp As Long
    Dim ch As Byte

    For lp = 1 To len(s)
        ch = asc(mid$(s,lp,1))
        lfb[lp * 8 - 8 + 8 * offset] = ch
        lfb[lp * 8 - 7 + 8 * offset] = &H1F
    Next lp
End Sub

Sub WriteConioString(vram As Byte Ptr, s as String, font() As Byte, row As Long, column As Long, col As Long, bpp as Byte, vw As Long)
    WriteGfxString vram, s, @font(0), column * 8, row * 16, col, bpp, vw
End Sub

Sub WriteGfxString(vram as Byte Ptr, s as String, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    Dim lp as long
    For lp = 1 to len(s)
        WriteGfxChar vram, asc(mid$(s,lp,1)), font, x + 8 * (lp - 1), y, col, bpp, vw
    next lp
End Sub
Sub WriteGfxChar(vram as Byte Ptr, ch as Integer, font as Byte Ptr, x as long, y as long, col as long, bpp as Byte, vw as Long)
    dim lp as long
    Dim fontbyte as byte
    For lp = 0 to 15
        fontbyte = font[32 * ch + lp]
        if (fontbyte and &H01) = &H01 then PutPixel vram, x + 7, y + lp, col, bpp, vw
        if (fontbyte and &H02) = &H02 then PutPixel vram, x + 6, y + lp, col, bpp, vw
        if (fontbyte and &H04) = &H04 then PutPixel vram, x + 5, y + lp, col, bpp, vw
        if (fontbyte and &H08) = &H08 then PutPixel vram, x + 4, y + lp, col, bpp, vw
        if (fontbyte and &H10) = &H10 then PutPixel vram, x + 3, y + lp, col, bpp, vw
        if (fontbyte and &H20) = &H20 then PutPixel vram, x + 2, y + lp, col, bpp, vw
        if (fontbyte and &H40) = &H40 then PutPixel vram, x + 1, y + lp, col, bpp, vw
        if (fontbyte and &H80) = &H80 then PutPixel vram, x + 0, y + lp, col, bpp, vw
    Next lp
End Sub

Sub RenderTest(vram as Byte Ptr, w as long, h as long, bpp as byte, vw as long)
    Dim lpx as long, lpy as long
    For lpx = 1 To w - 1 Step 2
        PutPixel vram, lpx, 0, 9, bpp, vw
        PutPixel vram, lpx, 1, 9, bpp, vw
        If lpx mod 10 = 9 then
            PutPixel vram, lpx, 2, 9, bpp, vw
        End If
        If lpx mod 40 = 39 then
            PutPixel vram, lpx, 3, 9, bpp, vw
        End If
        PutPixel vram, lpx, h-1, 9, bpp, vw
        PutPixel vram, lpx, h-2, 9, bpp, vw
    Next lpx
    For lpy = 1 To h - 1 Step 2
        PutPixel vram, 0, lpy, 9, bpp, vw
        PutPixel vram, 1, lpy, 9, bpp, vw

        PutPixel vram, w-1, lpy, 9, bpp, vw
        PutPixel vram, w-2, lpy, 9, bpp, vw

        If lpy mod 10 = 9 then
            PutPixel vram, w-3, lpy, 9, bpp, vw
        End If
        If lpy mod 40 = 39 then
            PutPixel vram, w-4, lpy, 9, bpp, vw
        End If
    Next lpy
End Sub

' Function: TestVerite
' runs the test suite for the Verite graphics card
'
' Parameters:
'     board_model   - the board type. 1 for a V1k, 2 for a V2k
'     board_version - the board rating.
'                     1 for a V2100, 2 for a V2200, 0 if unknown
'     lfb           - address of the framebuffer in memory
'     portbase      - the base I/O port for the Verite registers
'
Sub TestVerite(board_model As Long, board_version As Long, lfb As Long, portbase As Long)
    Dim vaddress as Byte Ptr
    Dim vram as Byte Ptr

    Dim lp as long, lp2 As Long
    Dim ws as String, ws2 as String

    Dim fontdata() As Byte
    Dim font As Byte Ptr

    ' claim board-specific addresses
    vaddress = CPtr(Byte Ptr, lfb)
    vram = CPtr(Byte Ptr, &HB0000000)

    ManageMemoryL2(CPtr(Byte Ptr, &HFFFFFFFF), vaddress)
    BlockAllocPhysL(1, vram, vaddress)

    portalloc portbase, &H100

    Do
        outportb(portbase + VREGS.VERITE_MEMENDIAN, 1)
    Loop until inportb(portbase + VREGS.VERITE_MEMENDIAN) = 1

    For lp = 0 to 80 * 25 * 8 step 8
        vram[lp+0] = asc("B")
    Next lp

    Do
        outportb(portbase + VREGS.VERITE_MEMENDIAN, 0)
    Loop until inportb(portbase + VREGS.VERITE_MEMENDIAN) = 0

    For lp = 0 to 80 * 25 * 8 step 8
        vram[lp+0] = &H20
        vram[lp+1] = &H1F
    Next lp

    inportb(portbase + VREGS.VERITE_GPUSTAT)
    font = malloc(8192)
    For lp = 0 to 8191
        font[lp] = vram[lp*4+2]
    Next lp

    WriteLfbString(vram, "Set System Byte ordering", 0)

    lp = inportb(portbase + VREGS.VERITE_GPUCTL)
    ws = ""
    if (lp and &H1) = &H1 then ws2 = ws + " Reset": ws = ws2
    if (lp and &H2) = &H2 then ws2 = ws + " Hold" : ws = ws2
    if (lp and &H4) = &H4 then ws2 = ws + " Debug": ws = ws2
    ws2 = "GPU Control: [" + ws + " ]"
    WriteLfbString vram, ws2, 80

    inportb(portbase + VREGS.VERITE_GPUSTAT)

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCCTL)
    WriteLfbString vram, "CRTC Control: 0x" + hex$(lp), 160

    inportb(portbase + VREGS.VERITE_GPUSTAT)

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCH)
    WriteLfbString vram, "CRTC Horizontal: 0x" + hex$(lp), 240

    inportb(portbase + VREGS.VERITE_GPUSTAT)

    ReadKB

    lp = inportd(portbase + VREGS.VERITE_CRTCV)
    WriteLfbString vram, "CRTC Vertical: 0x" + hex$(lp), 320

    inportb(portbase + VREGS.VERITE_GPUSTAT)

    ReadKB

    Dim vdp As vesa_display_properties
    Dim vmp As vesa_mode_properties
    Dim vt As vesa_timings

    LoadDefaultDisplayValues(vdp)
    vmp.X_resolution = 640
    vmp.Y_resolution = 480
    vmp.interlace = 0
    vmp.margins = 0
    lp = GTF_ComputeFromRefresh(60, vmp, vdp, vt)

    WriteLfbString vram, "GTF calc: 0x" + hex$(lp), 400
    WriteLfbString vram, "GTF V: " + str$(vt.V_total) + ", " + str$(vt.V_active) + ", " + str$(vt.V_margin) + ", " + str$(vt.V_frontporch)+ ", " + str$(vt.V_sync)+ ", " + str$(vt.V_backporch), 480
    WriteLfbString vram, "GTF H: " + str$(vt.H_total) + ", " + str$(vt.H_active) + ", " + str$(vt.H_margin) + ", " + str$(vt.H_frontporch)+ ", " + str$(vt.H_sync)+ ", " + str$(vt.H_backporch), 560
    lp = vt.dotclock
    WriteLfbString vram, "GTF dotclock: " + str$(lp), 640

    ReadKB

    ' native mode
    outportb portbase + VREGS.VERITE_MODE, 1

    ReadKB

    ' H sync: (from vga 640x480)
    ' Active = 80 character cells -> 0x4F
    ' Left = 3 character cells -> 0x2
    ' Right = 5 character cells -> 0x4
    ' Sync = 12 character cells ->  0xA
    'outportd portbase + VREGS.VERITE_CRTCH, &H004A084F
    outportd portbase + VREGS.VERITE_CRTCV, &H002B0A4F

    ReadKB

    ' V sync:
    ' Active = 480 -> 0x1DF
    ' Top = 25 -> 0x18
    ' Bottom = 3 -> 0x2
    ' Sync = 2 -> 0x1
    'outportd portbase + VREGS.VERITE_CRTCV, &H018211DF
    outportd portbase + VREGS.VERITE_CRTCV, &H009301DF

    ReadKB

    ' crtc black magic
    outportd portbase + VREGS.VERITE_WIDTH, 128
    outportd portbase + &H94, &H0


    ' dramctl black magic
    lp = inportd(portbase + &HA0)
    lp = (lp and &HC7FF&) Or &H330000
    outportd(portbase + &HA0, lp)

    ReadKB

    ' 32 bpp, enable output
    outportd portbase + VREGS.VERITE_CRTCCTL, &H1C1C

    ReadKB

    ' 32 bit DAC
    outportb portbase + VREGS.VERITE_DAC_CR1, &H82 ' access enable, 8-bit DAC
    ReadKB
    outportb portbase + VREGS.VERITE_DAC_CR1, &H18 ' 24 bpp, no palette
    ReadKB
    outportb portbase + VREGS.VERITE_DAC_CR2, &H20 ' black magic
    ReadKB
    outportb portbase + VREGS.VERITE_DAC_PIXELMASK, &HFF ' enable all pixels

    ReadKB

    for lp = 0 to 640*480*4 step 4
        vram[lp] = lp
        vram[lp+1] = lp \ 2
        vram[lp+2] = lp \ 3
        vram[lp+3] = lp \ 4
    next lp

    ReadKB

    For lp = 0 to 255
        For lp2 = 0 to 8
            Dim fd as byte
            fd = font[32*lp+lp2]

            vram[lp*4 + 640*4*lp2 + 0] = fd
            vram[lp*4 + 640*4*lp2 + 1] = fd SHL 2
            vram[lp*4 + 640*4*lp2 + 2] = fd SHL 4
            vram[lp*4 + 640*4*lp2 + 3] = fd SHL 6
        Next lp2
    Next lp

    ReadKB

    for lp = 0 to 640*480*4 step 4
        vram[lp] = 24
        vram[lp+1] = 24
        vram[lp+2] = 32
        vram[lp+3] = 24
    next lp

    ReadKB

    WriteGfxString(vram, "Welcome to V2x00", font, 20, 16, &HFFFFFFFF, 32, 640)
    WriteGfxString(vram, "Resolution: 640x480x32", font, 20, 32, &HFFFFFFFF, 32, 640)

    ReadKB

    for lp = 0 to 640*4 step 4
        If (lp mod 8) = 0 Then
            vram[lp+0] = -1
            vram[lp+1] = -1
            vram[lp+2] = -1
            vram[lp+3] = -1

            vram[lp+0 + 479*640*4] = -1
            vram[lp+1 + 479*640*4] = -1
            vram[lp+2 + 479*640*4] = -1
            vram[lp+3 + 479*640*4] = -1
        Else
            vram[lp+0] = &H00
            vram[lp+1] = &H00
            vram[lp+2] = &H00
            vram[lp+3] = &H00

            vram[lp+0 + 479*640*4] = &H00
            vram[lp+1 + 479*640*4] = &H00
            vram[lp+2 + 479*640*4] = &H00
            vram[lp+3 + 479*640*4] = &H00
        End if
    next lp

    ReadKB

    Dim Clock60 As Integer
    Dim Clock85 As Integer
    Dim ClockCur As Integer

    Clock60 = Verite_CalcClock(CDbl(800*524*60) / CDbl(1000000))
    Clock85 = Verite_CalcClock(CDbl(800*524*85) / CDbl(1000000))

    Dim myfreq As Double
    myfreq = NPQ_to_freq(Clock60) * 1000
    ClockCur = myfreq


    WriteGfxString(vram, "Clock settings: 60: 0x" + hex$(Clock60) + " 85: 0x" + hex$(Clock85) + " cur: 0x" + hex$(inportd(portbase + VREGS.VERITE_DOTCLOCK)), font, 20, 48, &HFFFFFFFF, 32, 640)
    WriteGfxString(vram, str$(ClockCur) + "MHz, " + str$(NPQ_to_freq(Clock85) * 1000.0) + "MHz", font, 20, 64, &HFFFFFFFF, 32, 640)
    WriteGfxString(vram, "PLL Values: 0x" + hex$(NPQ_to_V2x00(Clock60)) + ", 0x" + hex$(NPQ_to_V2x00(Clock85)), font, 20, 80, &HFFFFFFFF, 32, 640)

    ReadKB

    Dim oldclock As Integer
    oldclock = inportd(portbase+VREGS.VERITE_DOTCLOCK)
    outportd portbase + VREGS.VERITE_CRTCCTL, &H0C1C
    outportd(portbase+VREGS.VERITE_DOTCLOCK, NPQ_to_V2x00(Clock60))
    ReadKB
    outportd portbase + VREGS.VERITE_CRTCCTL, &H1C1C
    ReadKB

    'outportd portbase + VREGS.VERITE_CRTCCTL, &H0C1C
    'outportd(portbase+VREGS.VERITE_DOTCLOCK, NPQ_to_V2x00(Clock85))
    'ReadKB
    'outportd portbase + VREGS.VERITE_CRTCCTL, &H1C1C
    'ReadKB

    'outportd portbase + VREGS.VERITE_CRTCCTL, &H0C1C
    'outportd(portbase+VREGS.VERITE_DOTCLOCK, oldclock)
    'ReadKB
    'outportd portbase + VREGS.VERITE_CRTCCTL, &H1C1C
    'ReadKB

    ' dump GPU regs
    ws = ""
    For lp = 64 to 72
        ws = ws + " 0x" + hex$(Verite_ReadGPUReg(portbase, lp))

        if (lp mod 4) = 3 then
            WriteGfxString(vram, "Regs:" + ws , font, 20, 32 + 4*(lp-63), &H80FFFFFF, 32, 640)
            ws = ""
        End if
    Next lp

    ReadKB
    for lp = 0 to 1023
        vram[lp] = lp
    next lp

    ReadKB

    ' opcodes with priority:
    ' 44
    ' 78
    ' 64
    ' 62
    ' 47
    ' 68
    ' 4f

    ' jz test


    Profile_condjump(&H60, portbase, vram, font, 140)
    Profile_condjump(&H61, portbase, vram, font, 155)
    Profile_condjump(&H62, portbase, vram, font, 170)
    Profile_condjump(&H63, portbase, vram, font, 185)
    Profile_condjump(&H64, portbase, vram, font, 200)
    Profile_condjump(&H65, portbase, vram, font, 215)
    Profile_condjump(&H66, portbase, vram, font, 230)
    Profile_condjump(&H67, portbase, vram, font, 245)


    'Profile_insn(&H70, portbase, vram, font, 140)
    'ReadKB
    'Profile_insn(&H71, portbase, vram, font, 155)
    'ReadKB
    'Profile_insn(&H72, portbase, vram, font, 170)
    'ReadKB
    'Profile_insn(&H78, portbase, vram, font, 185)
    'ReadKB
    'Profile_insn(&H79, portbase, vram, font, 200)
    'ReadKB
    'Profile_insn(&H7A, portbase, vram, font, 215)
    'ReadKB

    for lp = 0 to 1023
        vram[lp] = (lp SHR 4)
    next lp

    ReadKB

    'Profile_insn(&H70, portbase, vram, font, 230)
    'ReadKB
    'Profile_insn(&H71, portbase, vram, font, 245)
    'ReadKB
    'Profile_insn(&H72, portbase, vram, font, 260)
    'ReadKB
    'Profile_insn(&H78, portbase, vram, font, 275)
    'ReadKB
    'Profile_insn(&H79, portbase, vram, font, 290)
    'ReadKB
    'Profile_insn(&H7A, portbase, vram, font, 305)
    'ReadKB

    'Profile_insn(&H73, portbase, vram, font, 125)
    'ReadKB
    'Profile_insn(&H74, portbase, vram, font, 140)
    'ReadKB
    'Profile_insn(&H75, portbase, vram, font, 155)
    'ReadKB
    'Profile_insn(&H76, portbase, vram, font, 170)
    'ReadKB
    'Profile_insn(&H77, portbase, vram, font, 185)
    'ReadKB
    'Profile_insn(&H78, portbase, vram, font, 200)
    'ReadKB
    'Profile_insn(&H79, portbase, vram, font, 215)
    'ReadKB
    'Profile_insn(&H7A, portbase, vram, font, 230)
    'ReadKB
    'Profile_insn(&H7B, portbase, vram, font, 245)
    'ReadKB
    'Profile_insn(&H7C, portbase, vram, font, 260)
    'ReadKB
    'Profile_insn(&H7D, portbase, vram, font, 275)
    'ReadKB
    'Profile_insn(&H7E, portbase, vram, font, 290)
    'ReadKB
    'Profile_insn(&H7F, portbase, vram, font, 305)
    'ReadKB
    'Profile_insn(&H6C, portbase, vram, font, 320)
    'ReadKB
    'Profile_insn(&H61, portbase, vram, font, 335)
    'ReadKB
    'Profile_insn(&H62, portbase, vram, font, 350)
    'ReadKB
    'Profile_insn(&H63, portbase, vram, font, 365)
    'ReadKB


    Dim ucstart As Byte Ptr
    ucstart = @microcode
    For lp = 0 to 1 * 4096 - 1 Step 4
        vram[lp+3] = ucstart[lp]
    Next lp
    For lp = 1 to 1 * 4096 - 1 Step 4
        vram[lp+1] = ucstart[lp]
    Next lp
    For lp = 2 to 1 * 4096 - 1 Step 4
        vram[lp-1] = ucstart[lp]
    Next lp
    For lp = 3 to 1 * 4096 - 1 Step 4
        vram[lp-3] = ucstart[lp]
    Next lp
    ReadKB

    'For lp = 0 to 16
    '    outportd(portbase + VREGS.VERITE_FRAMESTART, lp * 640 * 4)
    '    for lp2 = 0 to 500000
    '    next lp2
    'Next lp
    'ReadKB



    lp = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H60000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    lp2 = Verite_GetPC(portbase)
    WriteGfxString(vram, "insn 60 00 00 00: 0x" + hex$(lp) + " -> 0x" + hex$(lp2), font, 20, 080, &HFF80FFFF, 32, 640)

    lp = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H61000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    lp2 = Verite_GetPC(portbase)
    WriteGfxString(vram, "insn 61 00 00 00: 0x" + hex$(lp) + " -> 0x" + hex$(lp2), font, 20, 095, &HFF80FFFF, 32, 640)

    lp = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H62000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    lp2 = Verite_GetPC(portbase)
    WriteGfxString(vram, "insn 62 00 00 00: 0x" + hex$(lp) + " -> 0x" + hex$(lp2), font, 20, 110, &HFF80FFFF, 32, 640)

    lp = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H6C000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    lp2 = Verite_GetPC(portbase)
    WriteGfxString(vram, "insn 6C 00 00 00: 0x" + hex$(lp) + " -> 0x" + hex$(lp2), font, 20, 125, &HFF80FFFF, 32, 640)


    ReadKB



    'Verite_ExecuteInsn(portbase, &H6C000000 + &H1800 \ 4)
    Verite_ExecuteInsn(portbase, &H6C000000 + &H0000 \ 4)
    Verite_ExecuteInsn(portbase, &H00000000)
    WriteGfxString(vram, "GPU ready to go at 0x" + hex$(Verite_GetPC(portbase)), font, 20, 140, &HFF80FFFF, 32, 640)
    ReadKB

    lp = inportb(portbase+VREGS.VERITE_GPUCTL)
    'outportb(portbase+VREGS.VERITE_GPUCTL, lp and &HFD)
    WriteGfxString(vram, "GPU activated", font, 20, 155, &HFF80FFFF, 32, 640)

    outportd(portbase, &HABCD)
    outportd(portbase, &H00001ABC)
    outportd(portbase, 640)
    outportd(portbase, 2)
    outportd(portbase, &HFFFFFFFF)

    ws = ""
    for lp = 1 to 4
        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " "
    next lp
    ws = ws + "0x" + hex$(Verite_GetIR(portbase))
    WriteGfxString(vram, ws, font, 20, 215, &HFF80FFFF, 32, 640)
    ReadKB

    ws = ""
    Dim gpuctl As Byte
    gpuctl = inportb(portbase+VREGS.VERITE_GPUCTL)
    outportb(portbase+VREGS.VERITE_GPUCTL, gpuctl and &H02)
    For lp = 0 to 15
        outportb(portbase + VERITE_GPUCTL, gpuctl Or &H06)
        inportd(portbase + VERITE_GPUDATA)
        inportd(portbase + VERITE_GPUDATA)

        lp2 = 0
        While (( inportb(portbase + VERITE_GPUCTL) And 6 ) <> 2) And (lp2 < 100)
            lp2 = lp2 + 1
        Wend

        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " 0x" + hex$(Verite_GetIR(portbase)) + ", "
        if (lp2 = 100) Then ws = ws + "!, "
        if (lp mod 2) = 1 then
            WriteGfxString(vram, ws, font, 20, 170 + 7 * (lp-1), &HFF8080FF, 32, 640)
            ws = ""
        end if
    Next lp

    ReadKB

    For lp = &H40 to &H4F
        ws = ws + "0x" + hex$(Verite_ReadGPUReg(portbase, lp)) + " "
        if lp = &H47 then
            WriteGfxString(vram, ws, font, 20, 315, &HFF8080FF, 32, 640)
            ws = ""
        elseif lp = &H4F then
            WriteGfxString(vram, ws, font, 20, 330, &HFF8080FF, 32, 640)
            ws = ""
        end if
    Next lp

    ReadKB

    outportb(portbase+VREGS.VERITE_GPUCTL, lp and &HFD)
    ReadKB


    'outportd(portbase + 0, 256)
    For lp = 0 to 11
        outportb(portbase + VERITE_GPUCTL, gpuctl Or &H06)
        inportd(portbase + VERITE_GPUDATA)
        inportd(portbase + VERITE_GPUDATA)

        lp2 = 0
        While (( inportb(portbase + VERITE_GPUCTL) And 6 ) <> 2) And (lp2 < 100)
            lp2 = lp2 + 1
        Wend

        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " 0x" + hex$(Verite_GetIR(portbase)) + ", "
        if (lp mod 2) = 1 then
            WriteGfxString(vram, ws, font, 20, 345 + 7 * (lp-1), &HFF8080FF, 32, 640)
            ws = ""
        end if
    Next lp

    'outportd(portbase + 0, 50)
    ws = ""
    for lp = 1 to 4
        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " "
    next lp
    ws = ws + "0x" + hex$(Verite_GetIR(portbase))
    WriteGfxString(vram, ws, font, 20, 420, &HFF80FFFF, 32, 640)

    'outportd(portbase + 0, 3)
    ws = ""
    for lp = 1 to 4
        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " "
    next lp
    ws = ws + "0x" + hex$(Verite_GetIR(portbase))
    WriteGfxString(vram, ws, font, 20, 435, &HFF80FFFF, 32, 640)

    'outportd(portbase + 0, &HFF00FF00)
    ws = ""
    for lp = 1 to 4
        ws = ws + "0x" + hex$(Verite_GetPC(portbase)) + " "
    next lp
    ws = ws + "0x" + hex$(Verite_GetIR(portbase))
    WriteGfxString(vram, ws, font, 20, 450, &HFF80FFFF, 32, 640)
    ReadKB


    ' VGA mode (= bork me)
    outportb portbase + VREGS.VERITE_MODE, 2

    free(font)
End Sub

Sub Profile_insn(insn As Integer, portbase As Integer, vram As Byte Ptr, font As Byte Ptr, ypos As Integer)

    Dim old37 As Integer, oldpc As Integer, oldindex as byte, newpc As Integer

    old37 = Verite_ReadGPUReg(portbase, 37)



    'Verite_ExecuteInsn(portbase, &H00410055) ' addi R41, zero, 0x55
    'Verite_ExecuteInsn(portbase, &H004200AA) ' addi R42, zero, 0xAA
    'Verite_ExecuteInsn(portbase, &H0043005A) ' addi R43, zero, 0x5A
    'Verite_ExecuteInsn(portbase, &H004400AA) ' to test for immediates
    Verite_ExecuteInsn(portbase, &H00440004) ' addi R41, zero, 0x55
    Verite_ExecuteInsn(portbase, &H00480008) ' addi R42, zero, 0xAA
    Verite_ExecuteInsn(portbase, &H004C000C) ' addi R43, zero, 0x5A
    Verite_ExecuteInsn(portbase, &H0050000C) ' to test for immediates

    Dim r1 As Integer, r2 as Integer, r3 as Integer, r4 as Integer

    oldindex = inportb(portbase + VERITE_GPUINDEX)
    outportb(portbase + VERITE_GPUINDEX, 129)
    syncb(portbase + VERITE_GPUINDEX, 129)
    oldpc = inportd(portbase + VERITE_GPUDATA)
    outportb(portbase + VERITE_GPUINDEX, oldindex)

    Verite_ExecuteInsn(portbase, &H00400000) ' addi R41, zero, 0x55
    Verite_ExecuteInsn(portbase, (insn * &H1000000) Or &H400044 ) ' op R40, R41, R42
    Verite_ExecuteInsn(portbase, &H00000000)
    'Verite_ExecuteInsn(portbase, &H00000000)
    r1 = Verite_ReadGPUReg(portbase, &H40)

    Verite_ExecuteInsn(portbase, &H00400000) ' addi R41, zero, 0x55
    Verite_ExecuteInsn(portbase,  (insn * &H1000000) Or &H400448 ) ' op R40, R41, R43
    Verite_ExecuteInsn(portbase, &H00000000)
    'Verite_ExecuteInsn(portbase, &H00000000)
    r2 = Verite_ReadGPUReg(portbase, &H40)

    Verite_ExecuteInsn(portbase, &H00400000) ' addi R41, zero, 0x55
    Verite_ExecuteInsn(portbase,  (insn * &H1000000) Or &H400850 ) ' op R40, R41, R44
    Verite_ExecuteInsn(portbase, &H00000000)
    'Verite_ExecuteInsn(portbase, &H00000000)
    r3 = Verite_ReadGPUReg(portbase, &H40)

    Verite_ExecuteInsn(portbase, &H00400000) ' addi R41, zero, 0x55
    Verite_ExecuteInsn(portbase,  (insn * &H1000000) Or &H40FC4C ) ' op R40, R41, zero
    Verite_ExecuteInsn(portbase, &H00000000)
    'Verite_ExecuteInsn(portbase, &H00000000)
    r4 = Verite_ReadGPUReg(portbase, &H40)

    oldindex = inportb(portbase + VERITE_GPUINDEX)
    outportb(portbase + VERITE_GPUINDEX, 129)
    syncb(portbase + VERITE_GPUINDEX, 129)
    newpc = inportd(portbase + VERITE_GPUDATA)
    outportb(portbase + VERITE_GPUINDEX, oldindex)

    Dim ws as String
    ws = "opcode " + hex$(insn) + ": 0x" + hex$(r1) + " 0x" + hex$(r2) + " 0x" + hex$(r3) + " 0x" + hex$(r4)
    if (Verite_ReadGPUReg(portbase, &H44) <> &H04) Then ws = ws & " R44 changed"
    if (Verite_ReadGPUReg(portbase, &H48) <> &H08) Then ws = ws & " R48 changed"
    if (Verite_ReadGPUReg(portbase, &H4C) <> &H0C) Then ws = ws & " R4C changed"
    if (Verite_ReadGPUReg(portbase, &H50) <> &H0C) Then ws = ws & " R50 changed"

    if (Verite_ReadGPUReg(portbase, 37) <> old37) Then ws = ws & " R37 changed"


    if (newpc - 64) <> oldpc Then ws = ws + " PC 0x" + hex$(newpc) + " d:0x" + hex$(newpc-oldpc)

    WriteGfxString(vram, ws, font, 20, ypos, &HFF80FFFF, 32, 640)

End Sub

Sub syncd(port As Integer, value as Integer)
    Dim rv as Integer
    do
        rv = inportd(port)
    loop while rv <> value
End Sub

Sub syncb(port As Integer, value as Unsigned Byte)
    Dim rv as Unsigned Byte
    do
        rv = inportb(port)
    loop while rv <> value
End Sub

Function Verite_ReadGPUReg(portbase As Integer, register As Byte) As Integer

    Dim oldgpuctl As Byte
    Dim insn As Integer
    Dim oldindex As Byte

    oldgpuctl = inportb(portbase + VERITE_GPUCTL)
    oldindex = inportb(portbase + VERITE_GPUINDEX)

    ' 10 00 00 nn = add zero, zero, nn
    insn = VINSN_ADD * &H1000000 + register

    outportb(portbase + VERITE_GPUCTL, oldgpuctl Or &H02)

    ' send instruction feed without setting index?
    outportd(portbase + VERITE_GPUDATA, insn)

    outportb(portbase + VERITE_GPUINDEX, 128) ' Instruction Register
    syncd(portbase + VERITE_GPUDATA, insn)

    outportb(portbase + VERITE_GPUINDEX, 130) ' Status Register 1
    inportb(portbase + VERITE_GPUINDEX)
    inportb(portbase + VERITE_GPUINDEX)
    Function = inportd (portbase + VERITE_GPUDATA)

    outportb(portbase + VERITE_GPUINDEX, oldindex)
    outportb(portbase + VERITE_GPUCTL, oldgpuctl)

End Function

Sub Verite_ExecuteInsn(portbase As Integer, insn As Integer)
    Dim oldgpuctl As Byte
    Dim oldindex As Byte

    oldgpuctl = inportb(portbase + VERITE_GPUCTL)
    oldindex = inportb(portbase + VERITE_GPUINDEX)

    outportb(portbase + VERITE_GPUINDEX, 128)
    syncb(portbase + VERITE_GPUINDEX, 128)

    outportd(portbase + VERITE_GPUDATA, insn)
    syncd(portbase + VERITE_GPUDATA, insn)

    outportb(portbase + VERITE_GPUCTL, oldgpuctl Or &H06)
    inportd(portbase + VERITE_GPUDATA)
    inportd(portbase + VERITE_GPUDATA)

    While ( inportb(portbase + VERITE_GPUCTL) And 6 ) <> 2
    Wend

    outportb(portbase + VERITE_GPUINDEX, oldindex)
End Sub

Function Verite_GetPC(portbase As Integer) As Integer
    Dim oldindex As Byte
    Dim oldctl As Byte

    oldctl = inportb(portbase + VERITE_GPUCTL)
    outportb(portbase + VERITE_GPUCTL, oldctl or &H02)

    oldindex = inportb(portbase + VERITE_GPUINDEX)
    outportb(portbase + VERITE_GPUINDEX, 129)
    syncb(portbase + VERITE_GPUINDEX, 129)
    Function = inportd(portbase + VERITE_GPUDATA)
    outportb(portbase + VERITE_GPUINDEX, oldindex)

    outportb(portbase + VERITE_GPUCTL, oldctl)
End Function

Function Verite_GetIR(portbase As Integer) As Integer
    Dim oldindex As Byte
    Dim oldctl As Byte

    oldctl = inportb(portbase + VERITE_GPUCTL)
    outportb(portbase + VERITE_GPUCTL, oldctl or &H02)

    oldindex = inportb(portbase + VERITE_GPUINDEX)
    outportb(portbase + VERITE_GPUINDEX, 128)
    syncb(portbase + VERITE_GPUINDEX, 128)
    Function = inportd(portbase + VERITE_GPUDATA)
    outportb(portbase + VERITE_GPUINDEX, oldindex)

    outportb(portbase + VERITE_GPUCTL, oldctl)
End Function

Public Function Verite_CalcClock(WantedMHz As Double) As Integer
    Dim N As integer, P as integer, Q as integer
    Dim Nl As integer, Pl as integer, Ql as integer
    Dim Nh as integer, Ph as integer, Qh as integer
    Dim ldiff As Double, hdiff as double

    ldiff = -1000
    hdiff =  1000

    Dim divfreq As Double
    Dim outfreq As Double
    Dim clock As Double

    For N = 1 to &H3f                               ' denominator
        divfreq = 14.31818 / CDbl(N)
        If (divfreq > 1) and (divfreq < 3) Then
            For Q = 1 to 255                        ' nominator

                outfreq = divfreq * CDbl(Q)
                If (outfreq > 125) And (outfreq < 250) Then
                    For P = 1 to 15                 ' postdivider
                        clock =  outfreq / CDbl(P)
                        If (WantedMHZ - clock) >= 0 Then
                            If (WantedMHZ - clock) < hdiff Then
                                hdiff = WantedMHZ - clock
                                Nh = N
                                Ph = P
                                Qh = Q
                            End if
                        Else
                            If (WantedMHZ - clock) > ldiff Then
                                ldiff = WantedMHZ - clock
                                Nl = N
                                Pl = P
                                Ql = Q
                            End if
                        End if
                    Next P
                End If
            Next Q
        End if
    next N
'7 100 3
'14 197 3

    If (ldiff = -1000) or (hdiff = 1000) Then ' out of range
        Function = 0
        Exit Function
    End if

    If (0 - ldiff) > hdiff Then
        N = Nh
        P = Ph
        Q = Qh
    Else
        N = Nl
        P = Pl
        Q = Ql
    End If

    Function = (N * &H10000) + Q * (&H100) + P

End Function

Function NPQ_to_V2x00(NPQ As Integer) As Integer
    Dim N as integer
    Dim P as integer
    Dim Q as integer
    N = NPQ SHR 16
    Q = (NPQ SHR 8) And &HFF&
    P = NPQ And &HFF&

    Function = (N Shl 13) or (P SHL 9) or Q
End Function

Function V2x00_to_NPQ(pll As Integer) As Integer
    Dim N as integer
    Dim P as integer
    Dim Q as integer
    N = pll SHR 13
    P = (pll SHR 9) And &HF&
    Q = pll And &HFF&

    Function = (N Shl 16) or (Q SHL 8) or P
End Function

Function NPQ_to_freq(NPQ As Integer) As Double
    Dim N as integer
    Dim P as integer
    Dim Q as integer
    N = NPQ SHR 16
    Q = (NPQ SHR 8) And &HFF&
    P = NPQ And &HFF&

    Function = (14.31818 * CDbl(Q)) / (CDbl(N) * CDbl(P))

End Function

Sub Profile_condjump(insn As Integer, portbase As Integer, vram As Byte Ptr, font As Byte Ptr, ypos As Integer)
    Dim cj0 As Integer, cj1 As Integer, cj2 As Integer, cjm1 As Integer

    Verite_ExecuteInsn(portbase, &H6C000400)
    Verite_ExecuteInsn(portbase, &H76400000)
    Verite_ExecuteInsn(portbase,   &HFFFB40 + insn * &H1000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    cj0 = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H6C000400)
    Verite_ExecuteInsn(portbase, &H76400001)
    Verite_ExecuteInsn(portbase,   &HFFFB40 + insn * &H1000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    cj1 = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H6C000400)
    Verite_ExecuteInsn(portbase, &H76400002)
    Verite_ExecuteInsn(portbase,   &HFFFB40 + insn * &H1000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    cj2 = Verite_GetPC(portbase)
    Verite_ExecuteInsn(portbase, &H6C000400)
    Verite_ExecuteInsn(portbase, &H01400001)
    Verite_ExecuteInsn(portbase,   &HFFFB40 + insn * &H1000000)
    Verite_ExecuteInsn(portbase, &H00000000)
    cjm1 = Verite_GetPC(portbase)

    Dim ws As String
    ws = "insn " + hex$(insn) + ": (0) 0x" + hex$(cj0) + " (1) 0x" + hex$(cj1) + " (2) 0x" + hex$(cj2) + " (-1) 0x" + hex$(cjm1)
    WriteGfxString(vram, ws, font, 20, ypos, &HFF80FFFF, 32, 640)

End Sub

