' Summary: gtf.bas
' Implementation of the General Timing Formulae
'
' Author:
'     Marcel Sondaar
'
' License:
'     <Public Domain>
'

#include "vgasignal.bi"

' Function: LoadDefaultDisplayValues
' Gets a standard value for the monitor settings
'
' in:
'     vdp - A pointer to a <VESA_dislpay_properties> structure that needs to be filled out
'
' out:
'     A <MonitorSettings> Structure containing default values
'
Sub LoadDefaultDisplayValues(ByRef vdp As VESA_display_properties)

    vdp.C_prime = 40
    vdp.M_prime = 600
    vdp.min_retrace_time = 550.0 / 1000000.0 ' 550 us
    vdp.min_porch = 1
    vdp.Cellwidth = 8
    vdp.vsync = 3
    vdp.hsyncpercent = 8
    vdp.reducedblank = 0


    '    m.M = 600
    '    m.C = 40
    '    m.K = 128
    '    m.J = 20

End Sub

' Function: GTF_ComputeFromRefresh
' Computes the CRTC settings given a resolution and frequency
'
' in:
'     vfreq - The refresh rate to use in Hz
'     vmp   - The mode's properties
'     vdp   - The display's properties
'     vt    - A pointer to a structure of vesa timings to be computed.
'
' out:
'     Returns zero on success, nonzero if the timing could not be established
'
Function GTF_ComputeFromRefresh(ByRef Refresh As Single, ByRef vmp As VESA_mode_properties, ByRef vdp As VESA_display_properties, ByRef vt As VESA_timings) As Integer
    Dim retval As Integer

    retval = GTF_Common_Precalc(vmp, vdp, vt)
    If retval <> 0 Then
        Function = retval
        Exit Function
    End if

    Function = 2

    Dim v_field_rate As Single
    v_field_rate = refresh
    If vt.interlace = 1 Then
        v_field_rate = refresh / 2
    End If

    Dim v_signal_lines As Single
    Dim h_period_estimate as Double

    v_signal_lines = vt.V_active + 2 * vt.V_margin + vdp.min_porch
    if vt.interlace = 1 Then v_signal_lines = v_signal_lines + 0.5
    h_period_estimate = ( 1 / v_field_rate - vdp.min_retrace_time) / v_signal_lines
    'h period in s
    If h_period_estimate <= 0 then Exit Function ' way too many fps

    Dim v_retrace_lines As Integer
    v_retrace_lines = vdp.min_retrace_time / h_period_estimate
    vt.V_sync = vdp.Vsync
    vt.V_backporch = v_retrace_lines - vdp.Vsync
    if vt.V_backporch < vdp.min_porch then Exit Function ' way too little data

    vt.V_frontporch = vdp.min_porch
    vt.V_total = vt.V_sync + vt.V_active + 2 * vt.V_margin + vt.V_backporch + vt.V_frontporch


    Dim v_field_estimate As Double, v_total As Double
    v_total = vt.V_total
    If vt.interlace = 1 Then v_total = v_total + 0.5
    v_field_estimate = 1 / (h_period_estimate * v_total)

    Dim hperiod As Double, dutycycle As Double, h_pixels As Integer
    hperiod = h_period_estimate * v_field_rate / v_field_estimate

    h_pixels = vt.H_active + 2 * vt.H_margin
    dutycycle = vdp.C_prime - vdp.M_prime * hperiod * 1000
    If (dutycycle <= 0) or (dutycycle >= 100) Then
        Function = 3
        Exit Function
    End If

    Dim h_blank as Integer, h_blank_dif As Integer
    h_blank = (h_pixels * dutycycle) / (100 - dutycycle)
    If h_blank < 0 Then Exit Function
    h_blank_dif = h_blank Mod (vdp.cellwidth * 2)
    If h_blank_dif >= vdp.cellwidth Then
        h_blank = h_blank + vdp.cellwidth * 2 - h_blank_dif
    Else
        h_blank = h_blank - h_blank_dif
    End if

    vt.H_total = h_pixels + h_blank

    vt.dotclock = vt.H_total / hperiod

    retval = GTF_Common_Postcalc(vmp, vdp, vt)
    If retval <> 0 Then
        Function = retval
        Exit Function
    End if

    Function = 0
End Function


' Function: GTF_ComputeFromHFreq
' Computes the CRTC settings given a resolution and horizontal frequency
'
' in:
'     h_freq - The horizontal frequency in Hz
'     vmp    - The mode's properties
'     vdp    - The display's properties
'     vt     - A pointer to a structure of vesa timings to be computed.
'
' out:
'     Returns zero on success, nonzero if the timing could not be established
'
Function GTF_ComputeFromHfreq(ByVal H_Freq As Single, ByRef vmp As VESA_mode_properties, ByRef vdp As VESA_display_properties, ByRef vt As VESA_timings) As Integer
    ' TODO: write me

    Function = 5
End Function


' Function: GTF_ComputeFromDotclock
' Computes the CRTC settings given a resolution and dot clock
'
' in:
'     dotclock - The dot clock to use in MHz
'     vmp      - The mode's properties
'     vdp      - The display's properties
'     vt       - A pointer to a structure of vesa timings to be computed.
'
' out:
'     Returns zero on success, nonzero if the timing could not be established
'
Function GTF_ComputeFromDotclock(ByVal Dotclock As Single, ByRef vmp As VESA_mode_properties, ByRef vdp As VESA_display_properties, ByRef vt As VESA_timings) As Integer
    ' TODO: write me

    Function = 5
End Function



Function GTF_Common_Precalc(ByRef vmp As VESA_mode_properties, ByRef vdp As VESA_display_properties, ByRef vt As VESA_timings) As Integer
    Function = 1

    vt.H_active = vmp.X_resolution
    If (vmp.interlace = 1) Then
        vt.V_active = vmp.Y_resolution / 2
        vt.interlace = 1
    ElseIf vmp.interlace = 0 Then
        vt.V_active = vmp.Y_resolution
        vt.interlace = 0
    Else
        Exit Function
    End If

    If vt.H_active <= 0 Then Exit Function
    If vt.V_active <= 0 Then Exit Function
    If (vt.H_active mod vdp.cellwidth <> 0) Then Exit Function

    Dim x_margin As Integer
    Dim y_margin As Integer

    x_margin = csng(vt.H_active) * vmp.Margins
    y_margin = csng(vt.V_active) * vmp.Margins
    x_margin = x_margin / vdp.cellwidth
    vt.H_margin = x_margin * vdp.cellwidth
    vt.V_margin = y_margin

    Function = 0
End Function

Function GTF_Common_Postcalc(ByRef vmp As VESA_mode_properties, ByRef vdp As VESA_display_properties, ByRef vt As VESA_timings) As Integer
    Function = 4

    Dim hblank as Integer
    hblank = vt.H_total - (vt.H_active + 2 * vt.H_margin)
    If hblank < 4 * vdp.cellwidth Then Exit Function

    ' backporch
    vt.H_backporch = hblank / 2

    ' sync
    Dim hsync As Integer
    hsync = ( CSng(vdp.hsyncpercent) / 100) * csng(vt.H_total) / vdp.cellwidth
    hsync = hsync * vdp.cellwidth
    if hsync < vdp.cellwidth Then Exit Function
    if hsync >= hblank / 2 Then Exit Function
    vt.H_Sync = hsync
    vt.H_frontporch = (hblank / 2) - hsync

    if vt.H_frontporch < vdp.min_porch Then Exit Function
    if vt.H_backporch < vdp.min_porch Then Exit Function

    Function = 0
End Function
