/* Summary: vdisasm.c
 * A utility to disassemble Verite microcode
 *
 * Author:
 *     Marcel Sondaar
 *
 * License
 *     <Public Domain>
 */

#include <stdio.h>
#include <stdint.h>

const char * opcodes[256] = {"addi"  , "subi"  , "andni" , "rsubi" , "andi"  , "ori"   , "nori"  , "xori"  , // 00
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 08
                             "add"   , "sub"   , "andn"  , "rsub"  , "and"   , "or"    , "nor"   , "xor"   , // 10
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 18
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 20
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 28
                             NULL    , NULL    , NULL    , NULL    , "rfifo" , NULL    , NULL    , NULL    , // 30
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 38
                             "add16i", NULL    , NULL    , NULL    , "rori"  , "shli"  , "sari"  , "shri"  , // 40
                             NULL    , NULL    , NULL    , "add8i" , NULL    , NULL    , NULL    , NULL    , // 48
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 50
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 58
                             "jz"    , "jnz"   , "js"    , "jns"   , "ja"    , "jna"   , NULL    , NULL    , // 60
                             NULL    , NULL    , NULL    , NULL    , "jmp"   , NULL    , NULL    , "jmpr"  , // 68
                             "ldb"   , "ldh"   , "ldw"   , NULL    , NULL    , NULL    , "ldi"   , "ldhi"  , // 70
                             "stb"   , "sth"   , "stw"   , NULL    , NULL    , NULL    , NULL    , NULL    , // 78
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 80
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 88
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 90
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // 98
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // a0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // a8
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // b0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // b8
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // c0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // c8
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // d0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // d8
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // e0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // e8
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , // f0
                             NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    , NULL    };// f8

const char insn_format[32] = {1, 9, 0, 9, 9, 9, 1, 9,
                              1, 1, 9, 9, 2, 9, 7, 8,
                              9, 9, 9, 9, 9, 9, 9, 9,
                              9, 9, 9, 9, 9, 9, 9, 9};


int main(int argc, const char * argv[])
{
	if (argc != 2)
	{
		fprintf(stderr, "Verite disassembler\nusage: %s <filename>\n", argv[0]);
		return 2;
	}
	FILE * fin = fopen(argv[1], "rb");
	if (fin == NULL)
	{
		fprintf(stderr, "Error opening %s\n", argv[1]);
		return 1;
	}

	uint8_t bytes[4];
	size_t rb;

	int count = 0x1000;

	rb = fread(&(bytes[0]), 1, 4, fin);
	while (rb == 4)
	{
	    printf("%8x: ", count);
	    count += 4;

	    if (bytes[0] == 0 && bytes[1] == 0 && bytes[2] == 0 && bytes[3] == 0)
            printf("nop\n");
	    else if(opcodes[bytes[0]] != NULL)
	    {
	        switch (insn_format[bytes[0] >> 3])
	        {
	            case 0:
                    printf("%s R%.2x, R%.2x, R%.2x \n", opcodes[bytes[0]], (int)bytes[1], (int)bytes[2], (int)bytes[3]);
                    break;

	            case 1:
                    printf("%s R%.2x, R%.2x, 0x%.2x \n", opcodes[bytes[0]], (int)bytes[1], (int)bytes[2], (int)bytes[3]);
                    break;

                case 2:
                    printf("%s .%+i, R%.2x \n", opcodes[bytes[0]], (((int)(((int8_t*)bytes)[1]) << 8) + (int)bytes[2] - 4) << 2, (int)bytes[3]);
                    break;

                case 7:
                    if ((bytes[0] & 0x7) < 4)
                        printf("%s R%.2x, [R%.2x + %u] \n", opcodes[bytes[0]], (int)bytes[1], (int)bytes[3], ((int)bytes[2]) << (bytes[0] & 7));
                    else
                        printf("%s R%.2x, 0x%.4x \n", opcodes[bytes[0]], (int)bytes[1], (((int)bytes[2]) << 8) + ((int)bytes[3]));
                    break;

                case 8:
                    printf("%s [R%.2x + %u], R%.2x \n", opcodes[bytes[0]], (int)bytes[3], ((int)bytes[1]) << (bytes[0] & 7), (int)bytes[2]);
                    break;

	            default:
                    printf("%s 0x%.2x, 0x%.2x, 0x%.2x \n", opcodes[bytes[0]], (int)bytes[1], (int)bytes[2], (int)bytes[3]);
	        }

	    }
	    else
            printf("db %.2x, %.2x, %.2x, %.2x \n", (int)bytes[0], (int)bytes[1], (int)bytes[2], (int)bytes[3]);
		rb = fread(&(bytes[0]), 1, 4, fin);
	}


	fclose(fin);

	return 0;
}

