' Summary: verite.bi
' Contains the register definitions for the Rendition Verite video cards
' (V1000 and V2x00)
'
' Author:
'     Marcel Sondaar
'
' License
'     <Public Domain>
'


' Enumeration: VREGS
' Registers common to all Verite-based boards
Enum VREGS

    V1K_FIFO = &H00

    ' Constant: VERITE_MEMENDIAN
    ' This register controls the byteswapping performed when
    ' writing to the linear framebuffer.
    '
    ' Size:
    '  8 bits
    '
    ' valid register values:
    ' 0 - no byteswapping
    ' 1 - 32-bit byteswapping
    ' 2 - 16-bit byteswapping
    ' 3 - swap halfwords
    VERITE_MEMENDIAN = &H43

    ' Constant: VERITE_GPUCTL
    ' This register controls the state of the RISC processor
    ' It holds several bitflags:
    '
    ' Size:
    '  _unknown_
    '
    ' Bitflags:
    '     01h - reset chip
    '     02h - hold risc. Setting this bit will stop the main GPU from executing more instructions
    '     04h - single step. Setting this bit will cause a single step in the pipeline to be executed. The bit will be cleared by the GPU when the instruction is executed.
    '     08h - divide by 2 disable
    '     10h - VGA reset
    '     20h - assert XReset output to ext devices
    VERITE_GPUCTL = &H48

    ' Constant: VERITE_GPUSTAT
    ' This register contains the current status of the device
    ' Only present on V2x00
    '
    ' Size:
    '     _unknown_
    '
    ' 02h - risc held
    VERITE_GPUSTAT = &H4C

    ' Constant: VERITE_GPUINDEX
    ' index register to the GPU external registers
    '
    ' Size:
    '     _unknown_
    '
    ' Values:
    '     80 - Intruction Register
    '     82 - Status Register 1
    VERITE_GPUINDEX = &H60

    ' Constant: VERITE_GPUDATA
    ' This register contains the value of the GPU register
    ' selected by <VERITE_GPUINDEX>. A write will load the
    ' corresponding GPU register
    '
    VERITE_GPUDATA = &H64

    ' Constant: VERITE_MODE
    ' This register allows one to select between CRTC modes.
    ' One can choose between VGA, VESA and Native mode by
    ' encoding the proper bits
    '
    ' Size:
    '  8 bits
    '
    ' Bitflags:
    '  01h - Decode 0xA0000.
    '  02h - Operate in VGA Mode. This will enable A0000-BFFFF
    '        indepent of bit 1
    '  04h - 32-bit VGA
    VERITE_MODE = &H72

    ' Constant: VERITE_CRTCCTL
    ' CRTC Control register. 32-bits in size, contains several flags
    ' and pixel formats
    '
    ' Size:
    '  32 bits
    '
    ' Bitflags:
    '  0000000F - color format
    '  00000010 - CRTCCTL_VIDEOFIFOSIZE128
    '  00000020 - CRTCCTL_ENABLEDDC
    '  00000040 - CRTCCTL_DDCOUTPUT
    '  00000080 - CRTCCTL_DDCDATA
    '  00000100 - CRTCCTL_VSYNCHI
    '  00000200 - CRTCCTL_HSYNCHI
    '  00000400 - CRTCCTL_VSYNCENABLE
    '  00000800 - CRTCCTL_HSYNCENABLE
    '  00001000 - CRTCCTL_VIDEOENABLE
    '  00002000 - CRTCCTL_STEREOSCOPIC
    '  00004000 - CRTCCTL_FRAMEDISPLAYED
    '  00008000 - CRTCCTL_FRAMEBUFFERBGR
    '  00010000 - CRTCCTL_EVENFRAME
    '  00020000 - CRTCCTL_LINEDOUBLE
    '  00040000 - CRTCCTL_FRAMESWITCHED
    '
    ' Color formats:
    '  4  - 16 bpp (R5-G6-B5)
    '  5  - 16 bpp (R4-G4-B4-X4)
    '  12 - 32 bpp (R8-G8-B8-X8)
    VERITE_CRTCCTL = &H84

    ' Constant: VERITE_CRTCH
    ' Horizontal timing register
    '
    ' Size:
    '  32 bits
    VERITE_CRTCH = &H88

    ' Constant: VERITE_CRTCV
    ' Vertical timing register
    '
    ' Size:
    '  32 bits
    VERITE_CRTCV = &H8C

    ' Constant: VERITE_FRAMESTART
    ' Frame offset
    '
    ' Size:
    '  unknown
    VERITE_FRAMESTART = &H94

    ' Constant: VERITE_WIDTH
    ' Virtual width?
    '
    ' Size:
    '  32 bits
    VERITE_WIDTH = &H98

    ' Constant: VERITE_DAC_PIXELMASK
    ' RAMDAC configuration
    '
    ' Size:
    '  8 bits
    '
    VERITE_DAC_PIXELMASK = &HB2

    ' Constant: VERITE_DAC_CR0
    ' RAMDAC configuration
    '
    ' Size:
    '  8 bits
    '
    VERITE_DAC_CR0 = &HB6

    ' Constant: VERITE_DAC_CR1
    ' RAMDAC configuration
    '
    ' Size:
    '  8 bits
    '
    ' Bitfields:
    '  08 - Use 565 RGB
    '  10 - Bypass color lookup (disable palette)
    '  E0 - Bits per pixel
    '       00 = 24 bits
    '       20 = 16 bits
    '       40 = 8 bits
    '       60 = 4 bits
    VERITE_DAC_CR1 = &HB8

    ' Constant: VERITE_DAC_CR2
    ' RAMDAC configuration
    '
    ' Size:
    '  8 bits
    '
    VERITE_DAC_CR2 = &HB9

    ' Constant: VERITE_DOTCLOCK
    ' Access to the V2x00 dot clock PLL
    '
    ' Size:
    '  unknown
    '
    ' Bitfields:
    '
    VERITE_DOTCLOCK = &HC0

End Enum

' Enumeration: VINSNS
' List of verite GPU opcodes
' Each instruction is exactly 32-bits, of the form ooxxxxxx, where oo is the opcode and xx are the arguments
' Most opcodes follow a triadic instruction: ooddxxyy, where dd is destination register, xx and yy are source registers
' Jumps are pipelined and therefore have a delay of a cycle.
'
Enum VINSNS

    ' Constant: VINSN_ADDI
    ' Add a register and a byte immediate
    '
    ' Format:
    '     00 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss + xx
    VINSN_ADDI = &H00

    ' Constant: VINSN_SUBI
    ' Subtract a byte immediate from a register
    '
    ' Format:
    '     01 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss - xx
    VINSN_SUBI = &H01

    ' Constant: VINSN_ANDNI
    ' Computes the binary and-not of a register and a byte immediate
    '
    ' Format:
    '     02 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss & ~xx
    VINSN_ANDNI = &H02

    ' Constant: VINSN_RSUBI
    ' Subtract a register from a byte immediate
    '
    ' Format:
    '     03 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = xx - ss

    ' Constant: VINSN_ANDI
    ' Computes the inclusive or of a register and a byte immediate
    '
    ' Format:
    '     04 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss & xx
    VINSN_ANDI = &H04

    ' Constant: VINSN_ORI
    ' Computes the inclusive or of a register and a byte immediate
    '
    ' Format:
    '     05 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss | xx
    VINSN_ORI = &H05

    ' Constant: VINSN_NORI
    ' Computes the binary nor of a register and a byte immediate
    '
    ' Format:
    '     06 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ~ (ss | xx)
    VINSN_NORI = &H06

    ' Constant: VINSN_XORI
    ' Computes the exclusive or of a register and a byte immediate
    '
    ' Format:
    '     07 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte value
    '
    ' operation:
    '     dd = ss ^ xx
    VINSN_XORI = &H07

        ' 08 -

        ' 09 -

        ' 0A -

        ' 0B -

        ' 0C -

        ' 0D -

        ' 0E - yields 1

        ' 0F - yields 0

    ' constant: VINSN_ADD
    ' add two registers and put the result in the target register
    '
    ' format:
    '     10 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx + yy
    VINSN_ADD = &H10

    ' constant: VINSN_SUB
    ' subtract the second source from the first source register and put the result in the target register
    '
    ' format:
    '     11 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx - yy
    VINSN_SUB = &H11

    ' constant: VINSN_ANDN
    ' compute the and-not of two registers and put the result in the target register
    '
    ' format:
    '     12 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx & ~yy
    VINSN_ANDN = &H12

    ' constant: VINSN_RSUB
    ' subtract the first source register from the second source register and put the result in the target register
    '
    ' format:
    '     13 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = yy - xx
    VINSN_RSUB = &H13


    ' constant: VINSN_AND
    ' compute the binary and of two registers and put the result in the target register
    '
    ' format:
    '     14 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx & yy
    VINSN_AND = &H14

    ' constant: VINSN_OR
    ' compute the binary inclusive or of two registers and put the result in the target register
    '
    ' format:
    '     15 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx | yy
    VINSN_OR = &H15

    ' constant: VINSN_NOR
    ' compute the binary nor two registers and put the result in the target register
    '
    ' format:
    '     16 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = ~(xx | yy)
    VINSN_NOR = &H16

    ' constant: VINSN_XOR
    ' compute the binary exclusive or of two registers and put the result in the target register
    '
    ' format:
    '     17 dd xx yy
    '
    '     dd - destination register
    '     xx - source register 1
    '     yy - source register
    '
    ' operation:
    '     dd = xx ^ yy
    VINSN_XOR = &H17

        ' 18-19 undefined?

        ' 1A-3F todo

    ' Constant: VINSN_RFIFO
    ' Pops a slot from the FIFO, stall if the fifo is empty.
    '
    ' format:
    '     43 dd xx yy
    '
    '     dd - destination register
    '     xx - mask register (between 0 and F)
    '     yy - control byte
    '
    ' operation:
    '     dd = fifo.pop() & xx
    '
    '     The mask does not seem to work properly with general purpose registers. The following can be used consistently:
    '
    '     08 - and the fifo value with 0xffff0000
    '     0D - mask with 0xffffffff (i.e. read all bytes)
    '
    '     The control byte specifies one of the fifo formats
    '
    '     20 - read fifo in big-endian format
    '     21 - read fifo in little-endian format

    ' 34 - something that stalls until a FIFO write 34 dd xx yy - yy endianness?, xx defines read format
    ' looks like a half-broken AND that may stall in some cases r20/r21



    ' constant: fixme-VINSN_PLD16I
    ' load the high 16 bits of a register with an immediate value
    '
    ' format:
    '     40 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = (ss & 0xffff) | (xx << 16)
    VINSN_PLD16I = &H40

        ' 41 (~xx) << 16 + reg (maybe -xx)
        ' 42 reg+imm?
        ' 43 xx << 16 - reg

    ' constant: VINSN_RORI
    ' rotate register right by a constant number of positions.
    '
    ' format:
    '     44 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = ss ROR xx
    VINSN_RORI = &H44

    ' constant: VINSN_SHLI
    ' shift register right by a constant number of positions, sign extending in the process.
    '
    ' format:
    '     45 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = ss << xx
    VINSN_SHLI = &H45

    ' constant: VINSN_SARI
    ' shift register right by a constant number of positions, sign extending in the process.
    '
    ' format:
    '     46 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = ss >>> xx
    VINSN_SARI = &H46

    ' constant: VINSN_SHRI
    ' shift register right by a constant number of positions, adding zeros on the far end
    '
    ' format:
    '     47 dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = ss >> xx
    VINSN_SHRI = &H47


        ' 48 zero (reg shift reg?)
        ' 49 zero (reg shift reg?)

        ' 4A ?

    ' constant: fixme-VINSN_PLD8I
    ' load the high 24 bits of a register with an immediate value
    '
    ' format:
    '     4b dd ss xx
    '
    '     dd - destination register
    '     ss - source register
    '     xx - byte immediate
    '
    ' operation:
    '     dd = (ss & 0xff) | (xx << 8)
    VINSN_PLD8I = &H4B

        ' 4C - 4D do nothing?


        ' 4F reg + imm?

    ' Constant: VINSN_JZ
    ' Jump conditional if register is zero
    '
    ' format:
    '     60 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss = 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JZ = &H60

    ' Constant: VINSN_JNZ
    ' Jump conditional if register is non-zero
    '
    ' format:
    '     61 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss != 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JNZ = &H61

    ' Constant: VINSN_JS
    ' Jump conditional if register is negative
    '
    ' format:
    '     62 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss < 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JS = &H62

    ' Constant: VINSN_JNS
    ' Jump conditional if register is not negative
    '
    ' format:
    '     63 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss >= 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JNS = &H63

    ' Constant: VINSN_JA
    ' Jump conditional if register is above zero
    '
    ' format:
    '     64 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss > 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JA = &H64

    ' Constant: VINSN_JNA
    ' Jump conditional if register is not above zero
    '
    ' format:
    '     64 nn nn ss
    '
    '     nnnn - signed 16-bit displacement. Contains the number of instructions to jump if the condition is met. A displacement of -1 would be the jump instruction.
    '     ss   - the source register whose contents to check
    '
    ' operation:
    '     if (ss > 0) schedule_jump(pc+4 - 4*nnnn)
    VINSN_JNA = &H65


    ' Constant: VINSN_JMP
    ' Jump to an absolute address
    '
    ' format:
    '     6c nn nn nn
    '
    '     nnnnnn - 24-bit absolute jump target. This value is shifted left by two to create the actual address (and keep the program counter dword aligned)
    VINSN_JMP = &H6C

    ' Constant: VINSN_JMPR
    ' Jump to address specified in register
    '
    ' format:
    '     6f xx xx nn
    '
    '     nn - The register that holds the jump target.
    '     xxxx - unused?
    '
    VINSN_JUMPR = &H6F

    ' Constant: VINSN_LDB. The result is not available until after the next cycle.
    ' Load a byte from RAM
    '
    ' format:
    '     70 dd xx ss
    '
    '     dd - destination register
    '     xx - unsigned immediate displacement
    '     ss - memory offset register
    '
    ' operation:
    '     dd = mem[ss + xx]
    VINSN_LDB = &H70

    ' Constant: VINSN_LDH
    ' Load a 16-bit halfword from RAM. The result is not available until after the next cycle.
    '
    ' format:
    '     71 dd xx ss
    '
    '     dd - destination register
    '     xx - unsigned immediate displacement
    '     ss - memory offset register
    '
    ' operation:
    '     dd = mem[ss + 2*xx]
    VINSN_LDH = &H71

    ' Constant: VINSN_LDW
    ' Load a 32-bit word from RAM. The result is not available until after the next cycle.
    '
    ' format:
    '     72 dd xx ss
    '
    '     dd - destination register
    '     xx - unsigned immediate displacement
    '     ss - memory offset register
    '
    ' operation:
    '     dd = mem[ss + 4*xx]
    VINSN_LDW = &H72

    ' Constant: VINSN_LDI
    ' Load an 16-bit immediate into the selected register
    '
    ' format:
    '     76 dd nn nn
    '
    '     dd - destination register
    '     nnnn - 16 bit constant
    '
    ' operation:
    '     dd = nnnn

    VINSN_LDI = &H76

    ' Constant: VINSN_LDHI
    ' Load a shifted 16-bit immediate into the selected register.
    '
    ' format:
    '     76 dd nn nn
    '
    '     dd - destination register
    '     nnnn - 16 bit constant
    '
    ' operation:
    '     dd = nnnn << 16

    VINSN_LDHI = &H77

    ' 78 - write to RAM


End Enum


' Verite GPU registers
' seems to be 256 x 32b (including control in the first 64)
'   R00 - 0x00000000 - Zero
'   R01 -
'   R02 - 0x00008000
'   R03 - 0x80000000
'   R04 - 0xFF000000
'   R05 - 0x00FF0000
'   R06 - 0x0000FF00
'   R07 - 0x000000FF
'   R08 - 0xFFFF0000
'   R09 - 0x0000FFFF
'   R0A - 0xFF00FF00
'   R0B - 0x00FF00FF
'   R0C -
'   R0D - 0xFFFFFFFF
'   R0E -
'   R0F - 0x00010000
'   R10 - 0xFFFFFFFF
'   R11 - 0x00010000
'   R12 - 0x007FFFFF
'   R13 - 0x02000000
'   R14 - 0x00800000
'   R15 - 0x00020000
'   R16 - 0x0000B000
'   R17 - 0x0000B800
'   R18 - 0x0000C000
'   R19 - 0x0000C800
'   R1A - 0x0000D000
'   R1B - 0x0000D800
'   R1C - 0x0000E000
'   R1D - 0x0000E800
'   R1E - 0x0000F000
'   R1F - 0x0000F800
'   R20 -
'   R21 -
'   R22 - 0x46AA0000
'   R23 - 0xEF7088AC
'   R24 - 0x0000003F
'   R25 - 0x000307FC
'   R26 - 0x0000009C
'   R27 - random?
'   R28 -
'   R29 -
'   R2A - 0x46AA0000 = 22
'   R2B - 0xEF7088AC = 23
'   R2C - 0x0000003F = 24
'   R2D - 0x000307FC = 25
'   R2E - 0x0000001C
'   R2F - 0x00000001
'   R30 -
'   R31 -
'   R32 -
'   R33 -
'   R34 -
'   R35 -
'   R36 -
'   R37 -
'   R38 - 0xFFFFF000
'   R39 - 0xFFFFF800
'   R3A -
'   R3B -
'   R3C -
'   R3D -
'   R3E -
'   R3F -
'   R40 -
'   R41 -
'   R42 -
'   R43 - 0x01000004
