/* Summary: vgasocket.c
 * Handles VGA Connector states
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     <Educational Purposes>
 */

#define UDI_GFX_VERSION 0x101
#define UDI_VERSION 0x101

#include "vgaif.h"
#include "../vga_io.h"
#include <udi_gfx.h>

VGACONNECTORSTATE socket =
    {0, 0,              // disabled, alpha w/o cursor
    720, 400,           // 80x25 tiles
    9, 9, 8, 8,         // 1 char overscan
    2*9, 4*9, 24, 2,    // blanking
    12*9, 2,            // sync
    9,                  // 9dot mode
    28000};             // 28MHz dot clock

int VGA_ConnectorSetState (void * info, int index, int prop, int value)
{
    if (index != 0) return 0;
    if (info) {} // TODO: separate state
    
    switch(prop)
    {
        case UDI_GFX_PROP_ENABLE:
            if (value != 0) value = 1; // sanitize
            if (socket.enabled == value) return 0;

            if (value)
            {
                // from the CRTC test sample
                // compute parameter values for horizontal timings
                int hclocks = (socket.width + socket.os_l + socket.os_r + socket.blank_l + socket.blank_r + socket.h_sync) / socket.hmod;
                int htotal = hclocks - 5;
                int hdispend = socket.width / socket.hmod - 1;
                int hblankstart = (socket.width + socket.os_r) / socket.hmod - 1;
                int hsyncstart = (socket.width + socket.os_r + socket.blank_r) / socket.hmod - 1;
                int hsyncend = (socket.width + socket.os_r + socket.blank_r + socket.h_sync) / socket.hmod - 1;
                int hblankend = (socket.width + socket.os_r + socket.blank_r + socket.h_sync + socket.blank_l) / socket.hmod - 1;
                hsyncend = (hsyncend % (hclocks - 1)) & 0x1F;
                hblankend = (hblankend % (hclocks - 1)) & 0x3F;

                // compute parameter values for vertical timings
                int vtotal = socket.height + socket.os_t + socket.os_b + socket.blank_t + socket.blank_b + socket.v_sync;
                int vdispend = socket.height - 1;
                int vblankstart = socket.height + socket.os_b - 1;
                int vsyncstart = socket.height + socket.os_b + socket.blank_b;
                int vsyncend = socket.height + socket.os_b + socket.blank_b + socket.v_sync;
                int vblankend = socket.height + socket.os_b + socket.blank_b + socket.v_sync + socket.blank_t;

                // prepare registers, mangle and pack to VGA register format
                //   hsyncend register: 7 = HBE5 6..5 = reserved 4..0 = hsyncend
                if (hblankend >= 0x20) hsyncend |= 0x80;
                //   hblankend register: 7 = reserved(1) 6..5 = skew(0) 4..0 = hblankend
                hblankend = (hblankend & 0x1F) | 0x80; // drop bits, add reserved bit
                //   overflow register: contains bits 8/9 of several other registers
                unsigned char overflowregister = 0x10; // LC8 is set by default, the rest is set as follows:
                if (vtotal & 0x100)      overflowregister = overflowregister + 0x01;
                if (vdispend & 0x100)    overflowregister = overflowregister + 0x02;
                if (vsyncstart & 0x100)  overflowregister = overflowregister + 0x04;
                if (vblankstart & 0x100) overflowregister = overflowregister + 0x08;
                if (vtotal & 0x200)      overflowregister = overflowregister + 0x20;
                if (vdispend & 0x200)    overflowregister = overflowregister + 0x40;
                if (vsyncstart & 0x200)  overflowregister = overflowregister + 0x80;
                //   max scan line register: contains bit 9 of vertical blank start register, and the doublescan bit
                //   as well as the character height
                unsigned char scanlineregister = 0x40; //Read3D4(9);
                if (vblankstart & 0x200) scanlineregister = scanlineregister + 0x20;
                if (socket.input == 2 || socket.input == 1)
                {
                    //if (engines[socket.input].tile_h == 2)
                    //    scanlineregister = scanlineregister + 0x80;
                    scanlineregister |= (engines[socket.input].tile_h - 1);
                }
                else
                {
                    scanlineregister |= (engines[0].tile_h - 1);
                }
                //   vertical retrace end: contains memory control and crtc protect
                vsyncend = (Read3D4(0x11) & 0xF0) | (vsyncend & 0x0F);

                Write3D4(0x0, htotal);
                Write3D4(0x1, hdispend);
                Write3D4(0x2, hblankstart);
                Write3D4(0x3, hblankend);
                Write3D4(0x4, hsyncstart);
                Write3D4(0x5, hsyncend);
                Write3D4(0x6, vtotal);
                Write3D4(0x7, overflowregister);
                Write3D4(0x9, scanlineregister);
                // Register gap here, in case you wonder.
                Write3D4(0x10, vsyncstart);
                Write3D4(0x11, vsyncend);
                Write3D4(0x12, vdispend);
                Write3D4(0x15, vblankstart);
                Write3D4(0x16, vblankend);

                // Display timing generation
                unsigned char clockmode = Read3C4(0x01) & 0xC2; // mask off reserved bits, disables CD and shifts
                if (socket.hmod == 8) clockmode |= 0x01; // select character width
                unsigned char miscreg = Read3C2() & 0x33; // mask clock and polarity
                if (socket.clock == 28000) miscreg |= 4;
                miscreg |= 0x40; // Todo: Sync Polarity

                miscreg = 0xe3;

                // Operation mode
                unsigned char memctl = (Read3C4(0x04) & 0xF1) | 0x02 ; // mask off Chain4 and Odd/even, enable >64k mem
                unsigned char gfxmode = Read3CE(0x05) & 0x8F;
                unsigned char gfxmisc = Read3CE(0x06) & 0xF0;
                unsigned char attribctl = Read3C0(0x10) & 0x10;
                unsigned char attribpas = 0x00;
                unsigned char vwidth = 0x00;
                unsigned char underline = 0x00;
                unsigned char pixelpan = 0x00; // no panning by default

                if (socket.input == 1)
                {
                    // 4-bit modes
                    gfxmisc |= 0x05; // Axxxx, graphics
                    attribctl |= 1;
                    memctl |= 0x04; // planar access
                    if (engines[1].enabled) attribpas = 0x20;
                    vwidth = engines[1].width / 16;
                }
                else if (socket.input == 2)
                {
                    gfxmisc |= 0x05;   // Axxxx, graphics
                    gfxmode |= 0x40;   // 256-linear shift
                    attribctl |= 0x41; // 8-bpp pixels
                    memctl |= 0x04;    // planar access                     
                    if (engines[2].enabled) attribpas = 0x20;
                    vwidth = engines[2].width / 16;
                }
                else
                {
                    gfxmisc |= 0x0E;   // B8xxx, alphanumeric
                    gfxmode |= 0x10;   // odd,even
                    attribctl |= 0x0C; // text properties
                    attribpas = 0x20;
                    underline = 0x1F;
                    pixelpan = 0x08;
                    vwidth = engines[0].width / 2;
                }

                Write3C4(0x01, clockmode);
                Write3C4(0x04, memctl);
                Write3C2(miscreg);
                Write3CE(0x05, gfxmode);
                Write3CE(0x06, gfxmisc);
                Write3D4(0x13, vwidth);
                Write3D4(0x14, underline);

                Write3C0(0x11, 0x0E);
                Write3C0(0x13, pixelpan);
                Write3C0(0x10 | attribpas, attribctl); // register 0x10 and the P bit

                    //register file for 320x240x8 mode-x, useful for regression testing
                    //Write3C2(0xe3);
                    //Write3D4(0, 0x5f);
                    //Write3D4(1, 0x4f);
                    //Write3D4(2, 0x50);
                    //Write3D4(3, 0x82);
                    //Write3D4(4, 0x54);
                    //Write3D4(5, 0x80);
                    //Write3D4(6, 0x0d);
                    //Write3D4(7, 0x3e);
                    //Write3D4(8, 0x00);
                    //Write3D4(9, 0x41);

                    //Write3D4(0x10, 0xea);
                    //Write3D4(0x11, 0x2c);
                    //Write3D4(0x12, 0xdf);
                    //Write3D4(0x13, 0x28);
                    //Write3D4(0x14, 0x00);
                    //Write3D4(0x15, 0xe7);
                    //Write3D4(0x16, 0x03);
                    //Write3C4(1, 0x01);
                    //Write3C4(4, 0x06);
                    //Write3CE(5, 0x40);
                    //Write3CE(6, 0x05);
                    //Write3C0(0x10, 0x41);
                    //Write3C0(0x13, 0);
                    //Write3C0(0x33, 0);

                if (socket.input == 0 || socket.input == 3)
                    Write3D4(0x17, 0xA3); // sync enable, word mode
                else
                    Write3D4(0x17, 0xE3); // sync enable, byte mode
            }
            else
            {
                Write3D4(0x17, Read3D4(0x17) & 0x7f); // sync disable
            }

            return 0;

        case UDI_GFX_PROP_INPUT:
            if (value < 0 || value > 3) return 0; // sanitize input
            if ((socket.input == 0 || socket.input == 3) && (value == 0 || value == 3))
            {
                // toggle cursor
                unsigned char cd = Read3D4(0x0A) & 0xDF; // mask off the CD bit
                if (value == 0 || (engines[3].enabled == 0)) cd |= 0x20;
                Write3D4(0x0A, cd);
                socket.input = value;
            }
            else if (socket.enabled)
            {
                return 0; // won't change input when active
            }
            else if (socket.input == value)
            {
                return 0;
            }
            else
            {
                int newgran = 8;
                if (value == 0 || value == 3)
                    if (engines[0].tile_w == 9)
                        newgran = 9;

                if (newgran != socket.hmod)
                {
                    socket.width = socket.width + 4 - ((socket.width + 4) % newgran);
                    socket.blank_r = socket.blank_r + 4 - ((socket.blank_r + 4) % newgran);
                    socket.blank_l = socket.blank_l + 4 - ((socket.blank_l + 4) % newgran);
                    socket.os_r = socket.os_r + 4 - ((socket.os_r + 4) % newgran);
                    socket.os_l = socket.os_l + 4 - ((socket.os_l + 4) % newgran);
                    socket.h_sync = socket.h_sync + 4 - ((socket.h_sync + 4) % newgran);

                    socket.hmod = newgran;
                }
                socket.input = value;
            }
            break;

        case UDI_GFX_PROP_WIDTH:
            if (socket.enabled) return 0;
            if (value < socket.hmod || value / socket.hmod >= 128) return 0;
            socket.width = value - (value % socket.hmod);
            break;

        case UDI_GFX_PROP_HEIGHT:
            if (socket.enabled) return 0;
            if (value < 1 || value > 1000) return 0;
            if (value + socket.os_t + socket.os_b + socket.v_sync + socket.blank_t + socket.blank_b >= 1023) return 0;
            socket.height = value;
            break;

        case UDI_GFX_PROP_VGA_H_FRONT_PORCH:
            if (socket.enabled) return 0;
            if (value - socket.os_r < socket.hmod || value / socket.hmod >= 128) return 0;
            // todo: check against limits
            socket.blank_r = value - (value % socket.hmod) - socket.os_r;
            break;

        case UDI_GFX_PROP_VGA_H_BACK_PORCH:
            if (socket.enabled) return 0;
            if (value - socket.os_l < socket.hmod || value / socket.hmod >= 128) return 0;
            // todo: check against limits
            socket.blank_l = value - (value % socket.hmod) - socket.os_l;
            break;

        case UDI_GFX_PROP_VGA_H_SYNC:
            if (value < socket.hmod || value / socket.hmod >= 128) return 0;
            socket.h_sync = value - (value % socket.hmod);
            break;

        case UDI_GFX_PROP_VGA_V_FRONT_PORCH:
            if (socket.enabled) return 0;
            if (value - socket.os_b < 1 || value > 1000) return 0;
            if (socket.height + socket.os_t + socket.v_sync + socket.blank_t + value >= 1023) return 0;
            socket.blank_b = value - socket.os_b;
            break;

        case UDI_GFX_PROP_VGA_V_BACK_PORCH:
            if (socket.enabled) return 0;
            if (value - socket.os_t < 1 || value > 1000) return 0;
            if (socket.height + socket.os_b + socket.v_sync + socket.blank_b + value >= 1023) return 0;
            socket.blank_t = value - socket.os_t;
            break;

        case UDI_GFX_PROP_VGA_V_SYNC:
            if (socket.enabled) return 0;
            if (value < 1 || value >= 64) return 0;
            if (value + socket.os_t + socket.os_b + socket.height + socket.blank_t + socket.blank_b >= 1023) return 0;
            socket.v_sync = value;
            break;

        case UDI_GFX_PROP_DOT_CLOCK:
            if (socket.enabled) return 0;
            if (value < 0) return 0;
            if (value <= (25000 + 28000) / 2)
                value = 25000;
            else
                value = 28000;
                socket.clock = value;
            break;

        default:
            return 0;
    }

    return 0;
}

int VGA_ConnectorGetState (void * info, int index, int prop)
{
    if (index != 0) return 0;
    if (info) {} // TODO: separate state

    switch(prop)
    {
        case UDI_GFX_PROP_ENABLE:           return socket.enabled;
        case UDI_GFX_PROP_INPUT:            return socket.input;
        case UDI_GFX_PROP_WIDTH:            return socket.width;
        case UDI_GFX_PROP_HEIGHT:           return socket.height;

        case UDI_GFX_PROP_SIGNAL:           return UDI_GFX_SIGNAL_RGBHV;
        case UDI_GFX_PROP_CONNECTOR_TYPE:   return UDI_GFX_CONNECTOR_VGA;
        case UDI_GFX_PROP_VGA_H_FRONT_PORCH:return (socket.blank_r + socket.os_r);
        case UDI_GFX_PROP_VGA_H_BACK_PORCH: return (socket.blank_l + socket.os_l);
        case UDI_GFX_PROP_VGA_H_SYNC:       return (socket.h_sync);
        case UDI_GFX_PROP_VGA_V_FRONT_PORCH:return (socket.blank_b + socket.os_b);
        case UDI_GFX_PROP_VGA_V_BACK_PORCH: return (socket.blank_t + socket.os_t);
        case UDI_GFX_PROP_VGA_V_SYNC:       return (socket.v_sync);
        case UDI_GFX_PROP_DOT_CLOCK:        return socket.clock;

        default:
            return 0;
    }
}

int VGA_ConnectorGetRange (void * info, int index, int prop, int * startval, int * endval, int * modval)
{
    if (index != 0) return 0;
    if (info) {} // TODO: separate state
    *modval = 1;
    switch(prop)
    {
        case UDI_GFX_PROP_ENABLE:
            *startval = 0;
            *endval = 1;
            break;
        case UDI_GFX_PROP_INPUT:
            *startval = 0;
            *endval = 3;
            break;
        
        //case UDI_GFX_PROP_WIDTH:
        //case UDI_GFX_PROP_HEIGHT:

        case UDI_GFX_PROP_SIGNAL:
            *startval = UDI_GFX_SIGNAL_RGBHV;
            *endval = *startval;
            break;
        case UDI_GFX_PROP_CONNECTOR_TYPE:
            *startval = UDI_GFX_CONNECTOR_VGA;
            *endval = *startval;
            break;
        case UDI_GFX_PROP_VGA_H_FRONT_PORCH:
            *startval = 0;
            *endval = 256;
            *modval = 8;
            break;
        case UDI_GFX_PROP_VGA_H_BACK_PORCH:
            *startval = 0;
            *endval = 256;
            *modval = 8;
            break;
        case UDI_GFX_PROP_VGA_H_SYNC:
            *startval = 0;
            *endval = 128;
            *modval = 8;
            break;
        case UDI_GFX_PROP_VGA_V_FRONT_PORCH:
            *startval = 0;
            *endval = 128;
            *modval = 1;
            break;        
        case UDI_GFX_PROP_VGA_V_BACK_PORCH:
            *startval = 0;
            *endval = 128;
            break;
        case UDI_GFX_PROP_VGA_V_SYNC:
            *startval = 0;
            *endval = 64;
            break;
        case UDI_GFX_PROP_DOT_CLOCK:
            *startval = 25000;
            *endval = 28000;
            *modval = 28000-25000;
            break;

        default:
            return 0;
    }
    return 1;
}
