' Summary: vga_io.bas
' Contains IO primitives for VGA based displays,
' as well as some mode setters.
'
' Author:
'     Marcel Sondaar
'
' License:
'     Public Domain
'

#include "x86.bi"
#include "vga_io.bi"


' Function: Write3C0
' Writes an index/value pair to VGA register 3C0 (Attribute Control Register)
'
' in:
'     index - the AC register to access
'     value - the value to write
'
Sub Write3C0 Cdecl Alias "Write3C0" (ByVal index as byte, ByVal value as Unsigned byte)
    inportb &H3DA
	outportb &H3C0, index
	outportb &H3C0, value
End Sub


' Function: Write3C2
' Writes to VGA register 3C2 (Miscellaneous Output Register)
'
' in:
'     value - the value to write
'
Sub Write3C2 Cdecl Alias "Write3C2" (ByVal value as Unsigned byte)
    outportb &H3C2, value
End Sub


' Function: Write3C4
' Writes an index/value pair to VGA register 3C4 (Sequencer Register)
'
' in:
'     index - the Sequencer register to access
'     value - the value to write
'
Sub Write3C4 Cdecl Alias "Write3C4" (ByVal index as byte, ByVal value as Unsigned byte)
	If index = 1 Then
	    outportw &H3C4, &H0100
		outportb &H3C4 ,    index
        outportb &H3C4 + 1, value
		outportw &H3C4, &H0300
	Else
		outportb &H3C4 ,    index
        outportb &H3C4 + 1, value
	End If
End Sub


' Function: Write3C6
' Writes to VGA register 3C6 (DAC Mask Register)
'
' in:
'     value - the value to write
'
Sub Write3C6 Cdecl Alias "Write3C6" (ByVal value as Unsigned byte)
    outportb &H3C6, value
End Sub


' Function: Write3CE
' Writes an index/value pair to VGA register 3CE (Graphics Controller Register)
'
' in:
'     index - the Graphics Register to access
'     value - the value to write
'
Sub Write3CE Cdecl Alias "Write3CE" (ByVal index as byte, ByVal value as Unsigned byte)
	outportb &H3CE ,    index   
    outportb &H3CE + 1, value
End Sub


' Function: Write3D4
' Writes an index/value pair to VGA register 3D4 (CRTC Controller)
'
' in:
'     index - the CRTC register to access
'     value - the value to write
'
Sub Write3D4 Cdecl Alias "Write3D4"(ByVal index as byte, ByVal value as Unsigned byte)
	outportb &H3D4 ,    index
    outportb &H3D4 + 1, value
End Sub


' Accesses to io:0x03C0 cause the display enable to be turned off

' Function: Read3C0
' Reads an value from the indexed VGA register 3C0 (Attribute Control Register)
'
' in:
'     index - the AC register to access
'
' out:
'     return - the value of the register
'
Function Read3C0 Cdecl Alias "Read3C0" (ByVal index As Byte) As Unsigned Byte
    inportb &H3DA
	outportb &H3C0, index
	Read3C0 = inportb(&H3C0 + 1)
End Function


' Function: Read3C2
' Reads the value from VGA register 3C2 (Miscellaneous Output Register)
'
' out:
'     return - the value of the Miscellaneous Output Register
'
Function Read3C2 Cdecl Alias "Read3C2" () As Unsigned Byte
    Read3C2 = inportb(&H3CC)
End Function


' Function: Read3C4
' Reads an value from the indexed VGA register 3C4 (Sequencer Register)
'
' in:
'     index - the Sequencer register to access
'
' out:
'     return - the value of the register
'
Function Read3C4 Cdecl Alias "Read3C4" (ByVal index As Byte) As Unsigned Byte
    outportb &H3C4, index
    Read3C4 = inportb(&H3C4+1)
End Function


' Function: Read3C6
' Reads the value from VGA register 3C6 (DAC Mask)
'
' out:
'     return - the value of the DAC Mask Register
'
Function Read3C6 Cdecl Alias "Read3C6" As Unsigned Byte
    Read3C6 = inportb(&H3C6)
End Function


' Function: Read3CE
' Reads an value from the indexed VGA register 3CE (Graphics Controller Register)
'
' in:
'     index - the GC register to access
'
' out:
'     return - the value of the register
'
Function Read3CE Cdecl Alias "Read3CE" (ByVal index As Byte) As Unsigned Byte
    outportb &H3CE, index
    Read3CE = inportb(&H3CE+1)
End Function


' Function: Read3D4
' Reads an value from the indexed VGA register 3D4 (CRTC Register)
'
' in:
'     index - the CRTC register to access
'
' out:
'     return - the value of the register
'
Function Read3D4 Cdecl Alias "Read3D4" (ByVal index As Byte) As Unsigned Byte
    outportb &H3D4, index
    Read3D4 = inportb(&H3D4+1)
End Function


' Function: VGAUnlockCRTC
' unlocks the CRTC registers
'
Sub VGAUnlockCRTC Cdecl Alias "VGAUnlockCRTC" ()
    Dim temp as Byte
    temp = Read3D4(&H11) And &H7F
    Write3D4 &H11, temp
End Sub


' Function: VGAEnableDisplay
' Enables video output on the VGA
'
Sub VGAEnableDisplay Cdecl Alias "VGAEnableDisplay" ()
    inportb &H3DA
    outportb &H3C0, &H20
End Sub


' Function: VGASet320x200Mode
' Set the VGA registers to a 320x200 Mode
'
Sub VGASet320x200Mode Cdecl Alias "VGASet320x200Mode" ()
    VGASet320x240Mode()

    Write3C2 &H63

    ' CRTC
    Write3D4 0, &H5f
    Write3D4 1, &H4f
    Write3D4 2, &H50
    Write3D4 3, &H82
    Write3D4 4, &H54
    Write3D4 5, &H80
    Write3D4 6, &Hbf
    Write3D4 7, &H1f
    Write3D4 8, &H00
    Write3D4 9, &H41
    Write3D4 &H10, &H9c
    Write3D4 &H11, &H0e
    Write3D4 &H12, &H8f
    Write3D4 &H13, &H28
    Write3D4 &H14, &H40
    Write3D4 &H15, &H96
    Write3D4 &H16, &Hb9
    Write3D4 &H17, &Ha3

    ' Seq
    Write3C4 1, &H01
    Write3C4 2, &H0f
    Write3C4 4, &H0e

    ' GC
    Write3CE 0, &H00
    Write3CE 1, &H00
    Write3CE 2, &H00
    Write3CE 3, &H00
    Write3CE 4, &H00
    Write3CE 5, &H40
    Write3CE 6, &H05
    Write3CE 7, &H0F
    Write3CE 8, &HFF

    Write3C0 &H10, &H41
    Write3C0 &H11, 0
    Write3C0 &H12, &H0f
    Write3C0 &H13, 0
    Write3C0 &H14, 0
End Sub


' Function: VGASet320x240Mode
' Set the VGA registers to a 320x240 Mode
'
Sub VGASet320x240Mode Cdecl Alias "VGASet320x240Mode"()
    Write3C2 &He3
    Write3D4 0, &H5f
    Write3D4 1, &H4f
    Write3D4 2, &H50
    Write3D4 3, &H82
    Write3D4 4, &H54
    Write3D4 5, &H80
    Write3D4 6, &H0d
    Write3D4 7, &H3e
    Write3D4 8, &H00
    Write3D4 9, &H41
    Write3D4 &H10, &Hea
    Write3D4 &H11, &H2c
    Write3D4 &H12, &Hdf
    Write3D4 &H13, &H28
    Write3D4 &H14, &H40
    Write3D4 &H15, &He7
    Write3D4 &H16, &H03
    Write3D4 &H17, &Ha3
    Write3C4 1, &H01
    Write3C4 4, &H0e
    Write3CE 5, &H40
    Write3CE 6, &H05
    Write3C0 &H10, &H41
    Write3C0 &H13, 0
End Sub


' Function: VGASet400x300Mode
' Set the VGA registers to a 400x300 Mode
'
Sub VGASet400x300Mode Cdecl Alias "VGASet400x300Mode"()
    Write3C2 &Heb
    Write3D4 0, &H89
    Write3D4 1, &H63
    Write3D4 2, &H64
    Write3D4 3, &H8c
    Write3D4 4, &H6f
    Write3D4 5, &H09
    Write3D4 6, &H70
    Write3D4 7, &Hf0
    Write3D4 8, &H00
    Write3D4 9, &He0
    Write3D4 &H10, &H58
    Write3D4 &H11, &H0c
    Write3D4 &H12, &H57
    Write3D4 &H13, &H32
    Write3D4 &H14, &H00
    Write3D4 &H15, &H58
    Write3D4 &H16, &H70
    Write3D4 &H17, &He3
    Write3C4 1, &H01
    Write3C4 4, &H06
    Write3CE 5, &H40
    Write3CE 6, &H05
    Write3C0 &H10, &H41
    Write3C0 &H13, 0
End Sub

' Function: VGASet640x480x16Mode
' Set the VGA registers to 640x480 16 colors
'

Sub VGASet640x480x16Mode Cdecl Alias "VGASet640x480x16Mode"()

    Write3C2 &HE3

    Write3C4 1, &H01
    Write3C4 2, &H08
    Write3C4 4, &H06

    Write3D4 0, &H5f
    Write3D4 1, &H4f
    Write3D4 2, &H50
    Write3D4 3, &H82
    Write3D4 4, &H54
    Write3D4 5, &H80
    Write3D4 6, &H0b
    Write3D4 7, &H3e
    Write3D4 8, &H00
    Write3D4 9, &H40
    Write3D4 &H10, &Hea
    Write3D4 &H11, &H0c
    Write3D4 &H12, &Hdf
    Write3D4 &H13, &H28
    Write3D4 &H14, &H00
    Write3D4 &H15, &He7
    Write3D4 &H16, &H04
    Write3D4 &H17, &He3

    Write3CE 0, &H00
    Write3CE 1, &H00
    Write3CE 2, &H00
    Write3CE 3, &H00
    Write3CE 4, &H03
    Write3CE 5, &H00
    Write3CE 6, &H05
    Write3CE 7, &H0F
    Write3CE 8, &HFF

    Write3C0 &H10, &H01
    Write3C0 &H11, 0
    Write3C0 &H12, &H0f
    Write3C0 &H13, 0
    Write3C0 &H14, 0

End Sub


' Function: VGASetTextMode
' Set the VGA registers to standard text mode
'
Sub VGASetTextMode Cdecl Alias "VGASetTextMode" ()

    Write3C2 &H67
    Write3D4 0, &H5f
    Write3D4 1, &H4f
    Write3D4 2, &H50
    Write3D4 3, &H82
    Write3D4 4, &H55
    Write3D4 5, &H81
    Write3D4 6, &Hbf
    Write3D4 7, &H1f
    Write3D4 8, &H00
    Write3D4 9, &H4f

    Write3D4 &H10, &H9c
    Write3D4 &H11, &H0e
    Write3D4 &H12, &H8f
    Write3D4 &H13, &H28
    Write3D4 &H14, &H1f
    Write3D4 &H15, &H96
    Write3D4 &H16, &Hb9
    Write3D4 &H17, &Ha3

    Write3C4 1, &H00
    Write3C4 3, &H00
    Write3C4 4, &H00
    Write3CE 5, &H10
    Write3CE 6, &H0e

    Write3C0 &H10, &H0c
    Write3C0 &H13, &H08

End Sub


' Function: VGASetModeX
' Modifies the CRTC and Sequencer to enter planar addressing mode from a linear mode
'
' To avoid confusion, some only consider "Mode X" to be unchained 320x240x8,
' but in general it can apply to any 256-color mode
Sub VGASetModeX Cdecl Alias "VGASetModeX"()
    Write3D4 &H14, Read3D4(&H14) And &HBF
    Write3D4 &H17, Read3D4(&H17) Or &H40
    Write3C4 4, Read3C4(4) And &HF7
End Sub
