/*
 * Summary: comproxy.c
 * routing and proxy code for communication ports
 *
 * Author:
 *     Marcel Sondaar
 *
 * License:
 *     Public Domain
 */

#include <mos/drivercom.h>
#include <string.h>

/* Function: drv_peekmessageany
 * Checks if any message is waiting, including messages to others
 *
 * in:
 *     target - pointer to a variable to hold the destination address
 *
 * out:
 *     return - size of the waiting message, 0 if none present
 *     target - if not null, set to the destination address
 */
int drv_peekmessageany(int * target)
{
    int nextmsg = ((int)_MOS_bootinfo->readpt) & 0x7ff;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    int *dst = (int*) &(bufferbase[nextmsg]);
    if (*dst == 0) return 0;
    if(target) *target = *dst;
    nextmsg = (nextmsg + 8) & 0x7ff;
    int *size = (int*) &(bufferbase[nextmsg]);
    return *size;
}

/* Function: drv_peeksender
 * Checks the sender of the message currently at the front of the queue
 *
 * in:
 *     none
 *
 * out:
 *     return - sender of the waiting message, 0 if none present
 */
int drv_peeksender()
{
    int nextmsg = ((int)_MOS_bootinfo->readpt) & 0x7ff;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    int * dst = (int*) &(bufferbase[nextmsg]);
    if (*dst == 0) return 0;
    nextmsg = (nextmsg + 4) & 0x7ff;
    int * src = (int*) &(bufferbase[nextmsg]);
    return *src;
}

/* Function: drv_readmessageany
 * Reads a message from the queue, independent of target
 *
 * in:
 *     dest - pointer to buffer to hold a copy
 *
 * out:
 *     return - address of the sending object
 */
int drv_readmessageany(char * dest)
{
    int nextmsg = ((int)_MOS_bootinfo->readpt) & 0x7ff;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    int *dst = (int*) &(bufferbase[nextmsg]);
    if (*dst == 0) return 0;

    *dst = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;
    int src = *((int*) &(bufferbase[nextmsg]));
    *((int*) &(bufferbase[nextmsg])) = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;
    int size = *((int*) &(bufferbase[nextmsg]));
    *((int*) &(bufferbase[nextmsg])) = 0;
    nextmsg = (nextmsg + 4) & 0x7ff;

    if (size > 0x210) *(char *)0xE1710F10 = 0;

    if (nextmsg + size > 0x800)
    {
        int sizea = 0x800 - nextmsg;
        memcpy(dest, bufferbase + nextmsg, sizea);
        memcpy(&dest[sizea], bufferbase, size-sizea);
        memset(bufferbase+nextmsg, 0, sizea);
        memset(bufferbase, 0, size-sizea);
    }
    else
    {
        memcpy(dest, bufferbase + nextmsg, size);
        memset(bufferbase+nextmsg, 0, size);
    }

    unsigned int realsize = size + 12 + (4-(size%4))%4;
    __asm__ __volatile__("lock addl %%eax, (%%ecx)" : : "a"(realsize), "c"(&(_MOS_bootinfo->readpt)) : "memory");

    return src;
}


/* Function: drv_sendmessageany
 * sends a message over communication port
 *
 * in:
 *     dest   - address of the receiving object
 *     src    - sender address to attach to the message
 *     size   - size of the data to send
 *     buffer - pointer to the data to send
 *
 */
void drv_sendmessageany(unsigned int dest, unsigned int src, unsigned int size, char * buffer)
{
    int outptr;
    char * bufferbase = &(((char *) _MOS_bootinfo)[0x800]);
    unsigned int realsize = size + 12 + (4-(size%4))%4;
    __asm__ __volatile__("lock xaddl %%eax, (%%ecx)" : "=a"(outptr) : "a"(realsize), "c"(&(_MOS_bootinfo->writept)) : "memory");
    outptr = outptr & 0x7ff;
    int outptr2 = (outptr + 4) & 0x7ff;
    *((int *)(bufferbase+outptr2)) = src;
    outptr2 = (outptr2 + 4) & 0x7ff;
    *((int *)(bufferbase+outptr2)) = size;
    outptr2 = (outptr2 + 4) & 0x7ff;

    if (outptr2 + size > 0x800)
    {
       int sizea = 0x800 - outptr2;
       memcpy(bufferbase+outptr2, buffer, sizea);
       memcpy(bufferbase, &(buffer[sizea]), size - sizea);
    }
    else memcpy(bufferbase+outptr2, buffer, size);

    *((int *)(bufferbase+0+outptr)) = dest;
}
